
#ifndef MEM_H
#define MEM_H

#include <iostream>
#include <string>
#include <cassert>



#define VAR(x) std::cout << #x "=" << x << std::endl

template<class T> class Mem
{
private:
    char * _data;
    
    size_t _size;
    size_t _header_size;
    bool _expend;
    
    size_t compute_size(size_t vsize) const;

public:

    Mem(size_t base_size, bool expend = true);
    ~Mem();
    
    void * allocate(size_t vsize);
    void deallocate(void * obj);
    void * reallocate(void * obj, size_t vsize);
    
    void show_memory_map(const uint& width = 64) const;
    
    size_t capacity() const;
    size_t allocated_size() const;
    
};






typedef unsigned int uint;

template<class T>
Mem<T>::Mem(size_t base_size, bool expend)
    : _size(base_size), _expend(expend)
{
    _header_size = sizeof(T);
    _data = static_cast<char*>( calloc(base_size, 1) );
    
    std::cout << "Start address: " << (uint)(&_data[0]);
    std::cout << "\nSize: " << base_size << std::endl;
}

template<class T>
Mem<T>::~Mem()
{
    std::cout << "Un-deallocated size: "
                << (uint)(allocated_size())
                << "B\n";
    free(_data);
}

template<class T>
inline size_t Mem<T>::compute_size(size_t vsize) const
{
    //vsize mod _header_size must be null, so we
    //increase the block size if necessary
    return (vsize % _header_size) == 0 ? vsize :
                vsize + (_header_size - (vsize % _header_size));
}

template<class T>
void * Mem<T>::allocate(size_t vsize)
{
    size_t s = compute_size(vsize);

    void* address = 0;
    uint i = 0;
    
    while(i <= (_size - _header_size))
    {
        T * header = (T*)&_data[i];
        if( *header == 0 )
        { //ok free block
            if( i + _header_size + s  > _size )
                //not enough memory
                break;
            
            bool block_ok = true;
            
            uint j;
            
            for(j = 0; j < _header_size + s; ++j)
            { //check if this block this enough large
                if( _data[i+j] != 0 )
                {
                    block_ok = false;
                    break;
                }
            }
            
            if(block_ok)
            { //block ok
                *header = static_cast<T>(s);
                address = (void*)&_data[i + _header_size];
                break;
            }
            else
            { //block too small
                i += _header_size;
            }
        }
        else
        { //non-free block, we continue
            i += _header_size + *header;
        }    
    }

    if( address == 0 )
        std::cerr << "Allocation error: Not enough memory !\n";
    else
        std::cout << "Allocation address: " << (uint)address << " <-> "
                << ((uint)address + s - 1) << std::endl;
    show_memory_map();
    return address;
}

template<class T>
void Mem<T>::deallocate(void * obj)
{ //we get the block start and stop indexes and we "zero"
  //all the block
    uint block_start = ((uint)obj - (uint)(&_data[0])) - _header_size;
    T header = *((T*)&_data[block_start]);
    uint block_end = block_start + header + _header_size;

    for(uint i = block_start; i < block_end; ++i)
        _data[i] = 0;
    
    std::cout << "Deallocation address: " << (uint)obj << " <-> "
            << ((uint)obj + header - 1) << std::endl;
}

template<class T>
void * Mem<T>::reallocate(void * obj, size_t vsize)
{
    void * address = 0;
    
    if(vsize == 0)
    { //free mem
        deallocate(obj);
    }
    else
    {
        uint block_start = ((uint)obj - (uint)(&_data[0])) - _header_size;
        T *header = (T*)&_data[block_start];
        uint block_end = block_start + *header + _header_size;
        
        if(vsize == *header)
        { //size unchanged, nothing to do
            address = obj;
        }
        else if(vsize < *header)
        { //want smaller size
            size_t s = compute_size(vsize);
            
            if(s == *header)
            { //in reality we get the same size
              //so nohting to do
                address = obj;
            }
            else
            { //really smaller, resize block
                *header = static_cast<T>(s);
                
                block_start += _header_size + s;
                
                for(uint i = block_start; i < block_end; ++i)
                    _data[i] = 0;
            }
            
        }
        else if(vsize > *header)
        { //want bigger size
            uint block_start = ((uint)obj - (uint)(&_data[0])) - _header_size;
            T header = *((T*)&_data[block_start]);
        
            address = allocate(vsize);

            memcpy(address, obj, header);

            deallocate(obj);
        }
    }
        
    return address;
    
}

template<class T>
void Mem<T>::show_memory_map(const uint& width) const
{ //browse memory and show memory map
    std::string map;
    uint allocated = 0;
    uint i = 0;
    while( i <= (_size - _header_size) )
    {
        T * header = (T*)&_data[i];
        if( *header == 0 )
        { //free block, we display it using dashes ('-')
            map.append(_header_size, '-');
                
            i += _header_size;
        }
        else
        { //non-free block, show header ('H') and
          //allocated ('A') size
            map.append(_header_size, 'H');
            
            map.append(*header, 'A');    
            
            allocated += _header_size + *header;
            i += _header_size + *header;
        }
    }
    if(_size > i)
        map.append(_size - i, '-');
        
    std::cout << "Memory map: 0x" << std::hex << (uint)(&_data[0]) << " <-> 0x"
                                << ((uint)(&_data[0]) + _size) << std::endl;
    std::cout << "Total size: " << std::dec << (uint)_size << "B, Allocated: "
            <<  allocated << "B, Free: " << (100 * (_size - allocated) / _size) << "%\n";
    
    std::cout << " (A = allocated, H = header, - = free)\n";
    
    i = 0;
    if( map.size() > width)
        for(i = 0; i < map.size() - width; i+=width)
            std::cout << "|" << map.substr(i, width) << "|\n";
    
    std::string end = map.substr(i, width);
    end.append(width - end.size(), ' ');
    std::cout << "|" << end << "|\n";
    
}

template<class T>
inline size_t Mem<T>::capacity() const
{ //return memory capacity
    return _size;
}

template<class T>
size_t Mem<T>::allocated_size() const
{ //return allocated memory (including headers)
    uint allocated = 0;
    uint i = 0;
    while(i <= (_size - _header_size))
    {
        T * header = (T*)&_data[i];
        if( *header == 0 )
        {  
            i += _header_size;
        }
        else
        {
            allocated += _header_size + *header;
            i += _header_size + *header;
        }    
    }
    return allocated;    
}

#endif
