# File is a single publication of this mode. On the file system layer it is
# represented by an ordinary file.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com> 

package CMSsh::StorageMode::File;

our @ISA = qw(CMSsh::Mode::SinglePublication);

use strict;
use CMSsh::Mode::SinglePublication;
use CMSsh::FileSystem::File;
use CMSsh::StorageMode::Messages;
use CMSsh::StorageMode::Config;

my $Msg = CMSsh::StorageMode::Messages->new;
my $Cfg = CMSsh::StorageMode::Config->new;

################################################################################
# Common methods

sub new ($$) {
    my ($class_name, $uri) = @_;

    my $self = CMSsh::Mode::SinglePublication->new($uri);
    unless ($self) {
        warn $Msg->get('CANT_INST_PUB', $uri);
        return undef;
    }

    my $parent_dir = CMSsh::StorageMode::Directory->new($self->get_parent_uri);
    unless ($parent_dir) {
        warn $Msg->get('CANT_INST_PUB', $self->get_parent_uri);
        return undef;
    }

    my $file_path = $parent_dir->get_fs_dir->get_path . '/' . $self->{_name};
    $self->{_file} = CMSsh::FileSystem::File->new($file_path);
    unless ($self->{_file}) {
        warn $Msg->get('CANT_INST_FSO', $file_path);
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

# No CGI methods, because this mode does not have a view

################################################################################
# CMS methods

sub move ($$) {
    my ($self, $dest_dir) = @_;

    my $new_uri = $dest_dir->get_uri . '/' . $self->{_name};
    if (CMSsh::StorageMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    my $dest_fs_dir = $dest_dir->get_fs_dir;
    my $src_fs_file = $self->{_file};
    $src_fs_file->move($dest_fs_dir)
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $src_fs_file->get_path, $dest_fs_dir->get_path);
        return 0;
    };

    $self->_set_uri($new_uri);

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;
    
    my $new_uri = $self->get_parent_uri . '/' . $new_name;
    if (CMSsh::StorageMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    $self->{_file}->rename($new_name)
    or do {
        warn $Msg->get('CANT_RENAME_FSO',
                       $self->{_file}->get_path, $new_name);
        return 0;
    };

    $self->_set_uri($self->get_parent_uri . '/' . $new_name);
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    $self->{_file}->remove
    or do {
        warn $Msg->get('CANT_REMOVE_FSO', $self->{_file}->get_path);
        return 0;
    };

    return 1;
}

1
