# Directory is a wrapper for low-level Perl's I/O operations with directories.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::FileSystem::Directory;

our @ISA = qw(CMSsh::FileSystem::Object);

use strict;
use File::Path;
use CMSsh::FileSystem::Object;
use CMSsh::Messages;

my $Msg = CMSsh::Messages->new;

sub new ($$) {
    my ($class_name, $abs_path) = @_;

    my $self = CMSsh::FileSystem::Object->new($abs_path);

    unless (-d $self->{_path}) {
        warn $Msg->get('FSO_DOESNT_EXIST', $abs_path);
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub create_subdir ($$) {
    my ($self, $name) = @_;

    if ($name =~ /\//) {
        warn $Msg->get('WRONG_NAME', $name);
        return undef;
    }

    my $abs_path = $self->get_path . '/' . $name;

    if (-d $abs_path) {
        warn $Msg->get('FSO_ALRD_EXISTS', $abs_path);
        return undef;
    }

    eval { mkpath($abs_path); 1 }
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $@);
        return undef;
    };

    return CMSsh::FileSystem::Directory->new($abs_path);
}

sub move ($$) {
    my ($self, $dest_dir) = @_;

    my $new_path = $dest_dir->get_path . '/' . $self->{_name};
    if (-d $new_path) {
        warn $Msg->get('FSO_ALRD_EXISTS', $new_path);
        return 0;
    }

    File::Copy::move($self->{_path}, $new_path)
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return 0;
    };

    $self->{_path} = $new_path;

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    return 1 if ($new_name eq $self->{_name});

    if ($new_name =~ /\//) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }

    my $new_path = $self->get_parent_path . '/' . $new_name;
    if (-d $new_path) {
        warn $Msg->get('FSO_ALRD_EXISTS', $new_path);
        return 0;
    }

    File::Copy::move($self->{_path}, $new_path)
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return 0;
    };

    $self->{_path} = $new_path;
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    eval { rmtree($self->{_path}); 1 }
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $@);
        return 0;
    };

    return 1;
}

sub get_listing ($) {
    my $self = shift;

    my (@dirs, @files);
    opendir(DIR, $self->{_path})
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return undef;
    };

    foreach my $entry (readdir(DIR)) {
        next if ($entry =~ /^\.{1,2}$/);
        my $path = $self->{_path} . '/' . $entry;
        if (-d $path) {
            my $dir = CMSsh::FileSystem::Directory->new($path);
            unless ($dir) {
                warn $Msg->get('CANT_INST_FSO', $path);
                next;
            }
            push @dirs, $dir;
        } else {
            my $file = CMSsh::FileSystem::File->new($path);
            unless ($file) {
                warn $Msg->get('CANT_INST_FSO', $path);
                next;
            }
            push @files, $file;
        }
    }

    closedir DIR;

    return { dirs => \@dirs, files => \@files };
}

1
