# Document is a single publication of this mode. On the file system layer it is
# represented by a MultiMarkdown file and an optional image file (icon).
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com> 

package CMSsh::DocumentMode::Document;

our @ISA = qw(CMSsh::Mode::SinglePublication);

use strict;
use Date::Parse;
use CMSsh::Mode::SinglePublication;
use CMSsh::FileSystem::ImgFile;
use CMSsh::DocumentMode::MkdFile;

my $Msg = CMSsh::DocumentMode::Messages->new;
my $Cfg = CMSsh::DocumentMode::Config->new;

################################################################################
# Common methods

sub new ($$) {
    my ($class_name, $uri) = @_;

    my $self = CMSsh::Mode::SinglePublication->new($uri);
    unless ($self) {
        warn $Msg->get('CANT_INST_PUB', $uri);
        return undef;
    }

    my $parent_cat = CMSsh::DocumentMode::Category->new($self->get_parent_uri);
    unless ($parent_cat) {
        warn $Msg->get('CANT_INST_PUB', $self->get_parent_uri);
        return undef;
    }

# Skip category descriptions
    if ($parent_cat->get_desc_name eq $self->{_name}) {
        warn $Msg->get('PUB_DOESNT_EXIST', $uri);
        return undef;
    }

    my $mkd_path = $parent_cat->get_doc_dir->get_path . '/' .
                       $self->{_name} . '.' .
                       CMSsh::DocumentMode::MkdFile->get_ext;
    $self->{_mkd} = CMSsh::DocumentMode::MkdFile->new($mkd_path);
    unless ($self->{_mkd}) {
        warn $Msg->get('CANT_INST_FSO', $mkd_path);
        return undef;
    }

    local $SIG{__WARN__} = sub {};
    $self->{_icon} = CMSsh::FileSystem::ImgFile->new(
                         $parent_cat->get_icon_dir->get_path . '/' .
                         $self->{_name} . '.' .
                         CMSsh::FileSystem::ImgFile->get_ext);
    delete $SIG{__WARN__};

    bless $self, $class_name;
    return $self;
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

sub get_source_path ($) {
    my $self = shift;
    return $self->{_mkd}->get_htdocs_path;
}

################################################################################
# Presentation methods

sub get_preview ($) {
    my $self = shift;

    my $metadata = $self->{_mkd}->get_metadata;
    unless ($metadata) {
        warn $Msg->get('CANT_GET_MKD_MTDATA', $self->{_mkd}->get_path,
                                   $self->{_mkd}->get_err);
        return undef;
    }

    my $content = $metadata->{Summary} ?
                      '<p>' . $metadata->{Summary} . '</p>' :
                      $self->{_mkd}->get_markup;
# Empty content is not allowed for documents
    unless ($content) {
        warn $Msg->get('CANT_GET_MKD_MARKUP', $self->{_mkd}->get_path);
        return undef;
    }

    return { title => $metadata->{Title},
             content => $content,
             date => str2time($metadata->{Date}) || 0,
             href => $self->{_uri},
             icon => $self->{_icon} ?
                         $self->{_icon}->get_htdocs_path : undef, };
}

sub get_metadata ($) {
    my $self = shift;

    my $metadata = $self->{_mkd}->get_metadata;
    unless ($metadata) {
        warn $Msg->get('CANT_GET_MKD_MTDATA', $self->{_mkd}->get_path);
        return undef;
    }

    return $metadata;
}

sub get_markup ($) {
    my $self = shift;

    my $markup = $self->{_mkd}->get_markup;
    unless (defined $markup) {
        warn $Msg->get('CANT_GET_MKD_MARKUP', $self->{_mkd}->get_path);
        return undef;
    }

    return $markup;
}

################################################################################
# Administration methods

sub move ($$) {
    my ($self, $dest_cat) = @_;

    my $new_uri = $dest_cat->get_uri . '/' . $self->{_name};
    if (CMSsh::DocumentMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    my $dest_dir = $dest_cat->get_doc_dir;
    my $src_file = $self->{_mkd};
    $src_file->move($dest_dir)
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $src_file->get_path, $dest_dir->get_path);
        return 0;
    };

    if ($self->{_icon}) {
        $dest_dir = $dest_cat->get_icon_dir;
        $src_file = $self->{_icon};
        $src_file->move($dest_dir)
        or do {
            warn $Msg->get('CANT_MOVE_FSO',
                           $src_file->get_path, $dest_dir->get_path);
            return 0;
        };
    }

    $self->_set_uri($new_uri);

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    # Avoid conflicts with page numbers
    if ($new_name =~ /^\d+$/) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }
    
    my $new_uri = $self->get_parent_uri . '/' . $new_name;
    if (CMSsh::DocumentMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    $self->{_mkd}->rename($new_name)
    or do {
        warn $Msg->get('CantRnFso', $self->{_mkd}->get_path, $new_name);
        return 0;
    };

    if ($self->{_icon}) {
        $self->{_icon}->rename($new_name)
        or do {
            warn $Msg->get('CANT_RENAME_FSO',
                           $self->{_icon}->get_path, $new_name);
            return 0;
        };
    }

    $self->_set_uri($new_uri);
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    if ($self->{_icon}) {
        $self->{_icon}->remove
        or do {
            warn $Msg->get('CANT_REMOVE_FSO', $self->{_icon}->get_path);
            return 0;
        };
    }

    $self->{_mkd}->remove
    or do {
        warn $Msg->get('CANT_REMOVE_FSO', $self->{_mkd}->get_path);
        return 0;
    };

    return 1;
}

1
