# Category is a collection of documents. On the file system layer it is
# represented by a directory containing documents, a directory containing
# document icons and a description file.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::DocumentMode::Category;

our @ISA = qw(CMSsh::Mode::PublicationCollection);

use strict;
use CMSsh::Mode::PublicationCollection;
use CMSsh::FileSystem::Directory;
use CMSsh::DocumentMode::MkdFile;

my $Msg = CMSsh::DocumentMode::Messages->new;
my $Cfg = CMSsh::DocumentMode::Config->new;

################################################################################
# Common methods

sub new ($$) {
    my ($class_name, $uri) = @_;

    my $self = CMSsh::Mode::PublicationCollection->new($uri);
    unless ($self) {
        warn $Msg->get('CANT_INST_PUB', $uri);
        return undef;
    }

    foreach (['_doc_dir', 'ROOT_DOC'], ['_icon_dir', 'ROOT_ICON']) {
        my $dir_path = $Cfg->get($_->[1]) . (!$self->is_root && $self->{_uri});
        my $dir = $self->{$_->[0]}
                = CMSsh::FileSystem::Directory->new($dir_path);
        unless ($dir) {
            warn $Msg->get('CANT_INST_FSO', $dir_path);
            return undef;
        }
    }

# Root category description name is not supposed to be changed
    my $desc_name = $self->is_root ?
                        $Cfg->get('ROOT_DESC_NAME') : $self->{_name};
    my $mkd_path = $Cfg->get('ROOT_DOC') . (!$self->is_root && $self->{_uri}) .
                       "/$desc_name" . '.' .
                       CMSsh::DocumentMode::MkdFile->get_ext;
    $self->{_desc} = CMSsh::DocumentMode::MkdFile->new($mkd_path);
    unless ($self->{_desc}) {
        warn $Msg->get('CANT_INST_FSO', $mkd_path);
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

sub get_desc_name ($) {
    my $self = shift;
    return $self->{_desc}->get_basename;
}

sub get_doc_dir ($) {
    my $self = shift;
    return $self->{_doc_dir};
}

sub get_icon_dir ($) {
    my $self = shift;
    return $self->{_icon_dir};
}

################################################################################
# Presentation methods

sub get_preview ($) {
    my $self = shift;

    my $metadata = $self->{_desc}->get_metadata;
    unless ($metadata) {
        warn $Msg->get('CANT_GET_MKD_MTDATA', $self->{_desc}->get_path);
        return undef;
    }

    my $content = $metadata->{Summary} ?
                      '<p>' . $metadata->{Summary} . '</p>' :
                      $self->{_desc}->get_markup;
# Fail on undefined result only, allow empty content
    unless (defined $content) {
        warn $Msg->get('CANT_GET_MKD_MARKUP', $self->{_desc}->get_path);
        return undef;
    }

    return { title => $metadata->{Title},
             content => $content,
             href => $self->{_uri} };
}

sub get_listing ($) {
    my $self = shift;
    my @result;

    my $listing = $self->{_doc_dir}->get_listing;
    unless ($listing) {
        warn $Msg->get('CANT_INST_FSO', $self->{_doc_dir}->get_path);
        return undef;
    }

    my (@cats, @docs);

# Add single docs
    foreach my $file (@{$listing->{files}}) {
# Skip description
        next if ($file->get_basename eq $self->get_desc_name);

        my $mkd_file = CMSsh::DocumentMode::MkdFile->new($file->get_path);
        unless ($mkd_file) {
            warn $Msg->get('CANT_INST_FSO', $file->get_path);
            next;
        }

        my $doc_name = $mkd_file->get_basename;
        unless ($doc_name) {
            warn $Msg->get('CANT_MAP_FSO', $mkd_file->get_path);
            next;
        }

        my $doc_uri = $self->{_uri} . "/$doc_name";
        my $doc = CMSsh::DocumentMode::Document->new($doc_uri);
        unless ($doc) {
            warn $Msg->get('CANT_INST_PUB', $doc_uri);
            next;
        }

        push @docs, $doc;
    }

# Add subcaterogies
    foreach my $dir (@{$listing->{dirs}}) {
# Category name is always equal to directory name
        my $cat_uri = $self->{_uri} . '/' . $dir->get_name;
        my $cat = CMSsh::DocumentMode::Category->new($cat_uri);
        unless ($cat) {
            warn $Msg->get('CANT_INST_PUB', $cat_uri);
            next;
        }

        push @cats, $cat;
    }

    return { CMSsh::DocumentMode::Document->get_type => \@docs,
             CMSsh::DocumentMode::Category->get_type => \@cats };
}

sub get_inner_preview ($) {
    my $self = shift;
    my @result;

    my $listing = $self->get_listing;
    unless ($listing) {
        warn $Msg->get('COLL_LISTING_ERR', $self->{_uri});
        return undef;
    }

# Add previews of single docs
    foreach my $doc (@{$listing->{CMSsh::DocumentMode::Document->get_type}}) {
        my $item = $doc->get_preview;
        unless ($item) {
            warn $Msg->get('CANT_PREV_PUB', $doc->get_uri);
            next;
        }
        $item->{is_category} = 0;
        push @result, $item;
    }

# Add preiews of subcaterogies
    foreach my $cat (@{$listing->{CMSsh::DocumentMode::Category->get_type}}) {
        my $item = $cat->get_preview;
        unless ($item) {
            warn $Msg->get('CANT_PREV_PUB', $cat->get_uri);
            next;
        }
        $item->{is_category} = 1;
        push @result, $item;
    }

    return \@result;
}

sub get_metadata ($) {
    my $self = shift;

    my $metadata = $self->{_desc}->get_metadata;
    unless ($metadata) {
        warn $Msg->get('CANT_GET_MKD_MTDATA', $self->{_desc}->get_path);
        return undef;
    }

    return $metadata;
}

sub get_markup ($) {
    my $self = shift;
    
    my $markup = $self->{_desc}->get_markup;
    unless (defined $markup) {
        warn $Msg->get('CANT_GET_MKD_MARKUP', $self->{_desc}->get_path);
        return undef;
    }

    return $markup;
}

################################################################################
# Administration methods

sub create_subdoc ($$$) {
    my ($self, $mkd_file, $img_file) = @_;

    # Avoid conflicts with page numbers and CSS files
    if ($mkd_file->get_name =~ /^\d+$/ || $mkd_file->get_basename =~ /\./) {
        warn $Msg->get('WRONG_SRC_NAME', $mkd_file->get_name);
        return undef;
    }

    my $name = $mkd_file->get_basename;
    my $desired_uri = $self->{_uri} . '/' . $name;

    if (CMSsh::DocumentMode->find_publication($desired_uri)) {
        warn $Msg->get('OCCUPIED_URI', $desired_uri);
        return undef;
    }

    my $dest_dir = $self->get_doc_dir;
    $mkd_file->move($dest_dir)
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $mkd_file->get_path, $dest_dir->get_path);
        return undef;
    };

    if ($img_file) {
        $img_file->rename($name)
        or do {
            warn $Msg->get('CANT_RENAME_FSO', $img_file->get_path, $name);
            return undef;
        };

        $dest_dir = $self->get_icon_dir;
        $img_file->move($dest_dir)
        or do {
            warn $Msg->get('CANT_MOVE_FSO',
                           $img_file->get_path, $dest_dir->get_path);
            return undef;
        };
    }

    return CMSsh::DocumentMode::Document->new($desired_uri);
}

sub create_subcat ($$) {
    my ($self, $mkd_file) = @_;

    # Avoid conflicts with page numbers and CSS files
    if ($mkd_file->get_name =~ /^\d+$/ || $mkd_file->get_basename =~ /\./) {
        warn $Msg->get('WRONG_SRC_NAME', $mkd_file->get_name);
        return undef;
    }

    my $name = $mkd_file->get_basename;

    if (CMSsh::DocumentMode->find_publication($self->{_uri} . '/' . $name)) {
        warn $Msg->get('OCCUPIED_URI', $self->{_uri} . '/' . $name);
        return undef;
    }

    my $doc_dir = $self->{_doc_dir}->create_subdir($name);
    unless ($doc_dir) {
        warn $Msg->get('CANT_MAKE_DIR', $name, $self->{_doc_dir}->get_path);
        return undef;
    };

    my $dest_dir = $self->get_icon_dir;
    $dest_dir->create_subdir($name)
    or do {
        warn $Msg->get('CANT_MAKE_DIR', $name, $dest_dir->get_path);
        return undef;
    };

    $mkd_file->move($doc_dir)
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $mkd_file->get_path, $doc_dir->get_path);
        return undef;
    };

    return CMSsh::DocumentMode::Category->new($self->{_uri} . '/' . $name);
}

sub move ($$) {
    my ($self, $dest_cat) = @_;

    my $new_uri = $dest_cat->get_uri . '/' . $self->{_name};
    if (CMSsh::DocumentMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

# Move the directories first
    foreach (['_doc_dir',  $dest_cat->get_doc_dir],
             ['_icon_dir', $dest_cat->get_icon_dir]) {
        my $src_dir = $self->{$_->[0]};
        $src_dir->move($_->[1])
        or do {
            warn $Msg->get('CANT_MOVE_FSO',
                           $src_dir->get_path, $_->[1]->get_path);
            return 0;
        };
    }

# Description gets broken here, need to reinitialize the file
    my $mkd_path = $Cfg->get('ROOT_DOC') . $new_uri . '/' .
                       $self->get_desc_name . '.' .
                       CMSsh::DocumentMode::MkdFile->get_ext;
    $self->{_desc} = CMSsh::DocumentMode::MkdFile->new($mkd_path);
    unless ($self->{_desc}) {
        warn $Msg->get('CANT_INST_FSO', $mkd_path);
        return 0;
    }
   
    $self->_set_uri($new_uri);

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    if ($new_name =~ /^\d+$/) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }

    my $new_uri = $self->get_parent_uri . '/' . $new_name;
    if (CMSsh::DocumentMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

# Rename the directories first
    foreach (qw(_doc_dir _icon_dir)) {
        $self->{$_}->rename($new_name)
        or do {
            warn $Msg->get('CANT_RENAME_FSO',
                           $self->{$_}->get_path, $new_name);
            return 0;
        };
    }

# Description gets broken here, need to reinitialize the file
    my $mkd_path = $Cfg->get('ROOT_DOC') . $new_uri . '/' .
                       $self->get_desc_name . '.' .
                       CMSsh::DocumentMode::MkdFile->get_ext;
    $self->{_desc} = CMSsh::DocumentMode::MkdFile->new($mkd_path);
    unless ($self->{_desc}) {
        warn $Msg->get('CANT_INST_FSO', $mkd_path);
        return 0;
    }
   
# Rename the description file
    $self->{_desc}->rename($new_name)
    or do {
        warn $Msg->get('CANT_RENAME_FSO', $self->{_desc}->get_path, $new_name);
        return 0;
    };

    $self->_set_uri($new_uri);
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    if (@{$self->get_listing->{CMSsh::DocumentMode::Document->get_type}} ||
            @{$self->get_listing->{CMSsh::DocumentMode::Category->get_type}}) {
        warn $Msg->get('COLL_NOT_EMPTY', $self->{_uri});
        return 0;
    }

    foreach (qw(_desc _icon_dir _doc_dir)) {
        $self->{$_}->remove
        or do {
            warn $Msg->get('CANT_REMOVE_FSO', $self->{$_}->get_path);
            return 0;
        };
    }

    return 1;
}

1
