//
// zmixer - audio mixer
// Copyright (C) 2005 Bryan Beicker <tokiko@tokiko.net>
//
// This file is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2)
// as published by the Free Software Foundation.
//
// This file is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//

#include <stdlib.h>		// malloc(), free(), exit()
#include <unistd.h>		// EXIT_FAILURE
#include <string.h>		// strcmp()
#include "main.h"


//
// window manager is trying to close the window
//
gboolean zmixer_unload()
{
	// let the window close
	return FALSE;
}


//
// exit the program when the window dies
//
void zmixer_destroy()
{
	gtk_main_quit();
}


//
// show an error dialog and exit
//
void zmixer_error(char * message, int num, char * hint)
{
	GtkWidget * dialog;
	
	dialog = gtk_message_dialog_new(GTK_WINDOW(zmixer), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "%s returns %d\n\n%s", message, num, hint);
	gtk_window_set_title(GTK_WINDOW(dialog), "error");
	gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
	exit(EXIT_FAILURE);
}


//
// open the alsa mixer
//
void open_mixer()
{
	int err;
	
	if ((err = snd_mixer_open(&mixer_handle, 0)) < 0) {
		zmixer_error("snd_mixer_open()", err, "alsa configuration problem");
	}
	
	if ((err = snd_mixer_attach(mixer_handle, "default")) < 0) {
		zmixer_error("snd_mixer_attach()", err, "no soundcard found");
	}
	
	if ((err = snd_mixer_selem_register(mixer_handle, NULL, NULL)) < 0) {
		zmixer_error("snd_mixer_selem_register()", err, "alsa voodoo error");
	}
	
	if ((err = snd_mixer_load(mixer_handle)) < 0) {
		zmixer_error("snd_mixer_load()", err, "can not load mixer");
	}
}


//
// refresh all volume values
//
void refresh_volume()
{
	snd_mixer_elem_t * elem;
	snd_mixer_selem_id_t * sid;
	long min;
	long max;
	long value;
	void * dummy_sid;
	
	dummy_sid = malloc(snd_mixer_selem_id_sizeof());
	if (dummy_sid == NULL) {
		g_print("error: out of memory\n");
		exit(EXIT_FAILURE);
	}
	
	for (elem = snd_mixer_first_elem(mixer_handle); elem; elem = snd_mixer_elem_next(elem)) {
		if (!snd_mixer_selem_is_active(elem)) {
			continue;
		}
		
		if (!snd_mixer_selem_has_playback_volume(elem)) {
			continue;
		}
		
		//
		// FIXME: this is brain damage
		//
		// libasound has a stray assert() in src/mixer/simple.c
		// in the snd_mixer_selem_get_id() function
		//
		// assert(elem && id) should probably be assert(elem)
		// because id is assigned a new value moments later
		//
		// to get around this, we point sid to a valid address
		// before calling snd_mixer_selem_get_id()
		//
		sid = (snd_mixer_selem_id_t *)((char *) dummy_sid);
		snd_mixer_selem_get_id(elem, sid);
		
		snd_mixer_selem_set_playback_switch_all(elem, 1);
		
		snd_mixer_selem_get_playback_volume_range(elem, &min, &max);
		snd_mixer_selem_get_playback_volume(elem, SND_MIXER_SCHN_FRONT_LEFT, &value);
		
		if (strcmp(snd_mixer_selem_id_get_name(sid), "Master") == 0) {
			gtk_range_set_range(GTK_RANGE(master), (gdouble) min, (gdouble) max);
			gtk_range_set_value(GTK_RANGE(master), (gdouble) value);
			continue;
		} else if (strcmp(snd_mixer_selem_id_get_name(sid), "PCM") == 0) {
			gtk_range_set_range(GTK_RANGE(pcm), (gdouble) min, (gdouble) max);
			gtk_range_set_value(GTK_RANGE(pcm), (gdouble) value);
			continue;
		} else if (strcmp(snd_mixer_selem_id_get_name(sid), "CD") == 0) {
			gtk_range_set_range(GTK_RANGE(cd), (gdouble) min, (gdouble) max);
			gtk_range_set_value(GTK_RANGE(cd), (gdouble) value);
			continue;
		} else if (strcmp(snd_mixer_selem_id_get_name(sid), "Line") == 0) {
			gtk_range_set_range(GTK_RANGE(line), (gdouble) min, (gdouble) max);
			gtk_range_set_value(GTK_RANGE(line), (gdouble) value);
			continue;
		} else if (strcmp(snd_mixer_selem_id_get_name(sid), "Mic") == 0) {
			gtk_range_set_range(GTK_RANGE(mic), (gdouble) min, (gdouble) max);
			gtk_range_set_value(GTK_RANGE(mic), (gdouble) value);
			continue;
		}
	}
	
	free(dummy_sid);
}


//
// set a volume value
//
void set_volume(char * name, int value)
{
	snd_mixer_elem_t * elem;
	snd_mixer_selem_id_t * sid;
	void * dummy_sid;
	
	dummy_sid = malloc(snd_mixer_selem_id_sizeof());
	if (dummy_sid == NULL) {
		g_print("error: out of memory\n");
		exit(EXIT_FAILURE);
	}
	
	for (elem = snd_mixer_first_elem(mixer_handle); elem; elem = snd_mixer_elem_next(elem)) {
		if (!snd_mixer_selem_is_active(elem)) {
			continue;
		}
		
		if (!snd_mixer_selem_has_playback_volume(elem)) {
			continue;
		}
		
		//
		// FIXME: this is brain damage (see above)
		//
		sid = (snd_mixer_selem_id_t *)((char *) dummy_sid);
		snd_mixer_selem_get_id(elem, sid);
		
		if (strcmp(snd_mixer_selem_id_get_name(sid), name) == 0) {
			snd_mixer_selem_set_playback_volume_all(elem, (long) value);
			break;
		}
	}
	
	free(dummy_sid);
}


void master_value_changed(int value)
{
	set_volume("Master", value);
}


void pcm_value_changed(int value)
{
	set_volume("PCM", value);
}


void cd_value_changed(int value)
{
	set_volume("CD", value);
}


void line_value_changed(int value)
{
	set_volume("Line", value);
}


void mic_value_changed(int value)
{
	set_volume("Mic", value);
}


void zmixer_load()
{
	gtk_range_set_increments(GTK_RANGE(master), 1, 1);
	gtk_range_set_increments(GTK_RANGE(pcm), 1, 1);
	gtk_range_set_increments(GTK_RANGE(cd), 1, 1);
	gtk_range_set_increments(GTK_RANGE(line), 1, 1);
	gtk_range_set_increments(GTK_RANGE(mic), 1, 1);
	
	open_mixer();
	refresh_volume();
}
