/* Ben Lynn
 * with cut and pasted code by Jeremy Schaeffer
 */
/*
Copyright (C) 2002 Ben Lynn <blynn@stanford.edu>
    and Jeremy Schaeffer <jeremy@datye.zzn.com>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <glib.h>
#include <gtk/gtk.h>

#include <xmms/configfile.h>

#include "version.h"
#include "control.h"
#include "config.h"

#define DEFAULT_INPIPEPREFIX "/tmp/xmmspipe-in_"
#define DEFAULT_OUTPIPEPREFIX "/tmp/xmmspipe-out_"
#define DEFAULT_ACKPIPEPREFIX "/tmp/xmmspipe-ack_"
#define DEFAULT_ACKMESSAGE "OK"
#define DEFAULT_ACKAUTO 0
#define DEFAULT_ACKECHO 1
#define DEFAULT_IDFILE "/tmp/xmmspipe-id_"
#define DEFAULT_IDAUTO 0
#define DEFAULT_OUTAUTO 0

config_t config;

gchar *configfile;

static GtkWidget *config_win;

static GtkWidget *inprefix_entry, *outprefix_entry, *ackprefix_entry,
	*idprefix_entry;
static GtkWidget *acktoggle, *idtoggle, *outtoggle;

void config_init()
{
    static int done = 0;

    if (!done) {
	done = 1;
	configfile = g_strconcat(g_get_home_dir(), "/.xmms/", CONFIG_FILENAME, NULL);
	config->inprefix = g_string_new("");
	config->outprefix = g_string_new("");
	config->ackprefix = g_string_new("");
	config->ackmessage = g_string_new("");
	config->idprefix = g_string_new("");
    }
}

static void maybe_cfg_read_gstring (GString *gs, ConfigFile *cf,
	gchar *s1, gchar *s2)
{
    gchar *s;
    if (xmms_cfg_read_string (cf, s1, s2, &s)) {
	g_string_assign(gs, s);
	g_free(s);
    }
}

static void maybe_cfg_read_int (int *i, ConfigFile *cf,
	gchar *s1, gchar *s2)
{
    int j;
    if (xmms_cfg_read_int (cf, s1, s2, &j)) {
	*i = j;
    }
}

void config_load()
{
    ConfigFile *cf;

    //assign defaults first
    config_init();
    g_string_assign(config->inprefix, DEFAULT_INPIPEPREFIX);
    g_string_assign(config->outprefix, DEFAULT_OUTPIPEPREFIX);
    g_string_assign(config->ackprefix, DEFAULT_ACKPIPEPREFIX);
    g_string_assign(config->idprefix, DEFAULT_IDFILE);
    g_string_assign(config->ackmessage, DEFAULT_ACKMESSAGE);
    config->ackauto = DEFAULT_ACKAUTO;
    config->idauto = DEFAULT_IDAUTO;
    config->ackecho = DEFAULT_ACKECHO;
    config->outauto = DEFAULT_OUTAUTO;

    if ((cf = xmms_cfg_open_file (configfile)) != NULL) {
	maybe_cfg_read_gstring(config->inprefix, cf, "in", "prefix");
	maybe_cfg_read_gstring(config->outprefix, cf, "out", "prefix");
	maybe_cfg_read_gstring(config->ackprefix, cf, "ack", "prefix");
	maybe_cfg_read_gstring(config->idprefix, cf, "id", "prefix");
	maybe_cfg_read_gstring(config->ackmessage, cf, "ack", "message");
	maybe_cfg_read_int(&config->ackauto, cf, "ack", "auto");
	maybe_cfg_read_int(&config->idauto, cf, "id", "auto");
	maybe_cfg_read_int(&config->ackecho, cf, "ack", "echo");
	maybe_cfg_read_int(&config->outauto, cf, "out", "auto");
	xmms_cfg_free(cf);
    }
}

void config_save()
{
    ConfigFile *cf;
    if ((cf = xmms_cfg_open_file(configfile)) == NULL) cf = xmms_cfg_new();
    xmms_cfg_write_string(cf, "in", "prefix", config->inprefix->str);
    xmms_cfg_write_string(cf, "out", "prefix", config->outprefix->str);
    xmms_cfg_write_string(cf, "ack", "prefix", config->ackprefix->str);
    xmms_cfg_write_string(cf, "id", "prefix", config->idprefix->str);
    xmms_cfg_write_string(cf, "ack", "message", config->ackmessage->str);
    xmms_cfg_write_int(cf, "id", "auto", config->idauto);
    xmms_cfg_write_int(cf, "ack", "auto", config->ackauto);
    xmms_cfg_write_int(cf, "ack", "echo", config->ackecho);
    xmms_cfg_write_int(cf, "out", "auto", config->outauto);
    xmms_cfg_write_file(cf, configfile);
    xmms_cfg_free(cf);
}

static void set_config()
{
    g_string_assign(config->inprefix, gtk_entry_get_text(GTK_ENTRY(inprefix_entry)));
    g_string_assign(config->outprefix, gtk_entry_get_text(GTK_ENTRY(outprefix_entry)));
    g_string_assign(config->ackprefix, gtk_entry_get_text(GTK_ENTRY(ackprefix_entry)));
    g_string_assign(config->idprefix, gtk_entry_get_text(GTK_ENTRY(idprefix_entry)));
    config->outauto = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(outtoggle));
    config->idauto = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(idtoggle));
    config->ackauto = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(acktoggle));
    control_request(C_RELOAD);
}

static void xmmspipe_config_apply(GtkWidget *wid, gpointer data)
{
    set_config();
}

static void xmmspipe_config_ok(GtkWidget *wid, gpointer data)
{
    set_config();
    gtk_widget_destroy(config_win);
    config_win = NULL;
    config_save();
}

static void new_textbox_row(GtkWidget *table, gint r,
	gchar *l, GtkWidget **entry, GString *gs)
{
    GtkWidget *label;

    label = gtk_label_new(l);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, r, r + 1);

    *entry = gtk_entry_new();
    gtk_widget_set_usize(*entry, 200, -1);
    gtk_entry_set_text(GTK_ENTRY(*entry), gs->str);
    gtk_table_attach_defaults(GTK_TABLE(table), *entry, 1, 2, r, r + 1);

}

static GtkWidget *new_toggle(GtkWidget *vbox, gchar *l, gint i)
{
    GtkWidget *r;
    r = gtk_check_button_new_with_label(l);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(r), i);
    gtk_box_pack_start(GTK_BOX(vbox), r, FALSE, FALSE, 0);

    return r;
}

void xmmspipe_config(void)
{
    GtkWidget *ok_button, *apply_button, *cancel_button,
	*buttonbox,
	*label,
	*table,
   	*vbox;

    config_load();
    if (config_win) return;
    config_win = gtk_window_new(GTK_WINDOW_DIALOG);

    gtk_window_set_title(GTK_WINDOW(config_win), ("XMMSPipe Setup"));
    gtk_window_set_policy(GTK_WINDOW(config_win), FALSE, FALSE, FALSE);
    gtk_window_set_position(GTK_WINDOW(config_win), GTK_WIN_POS_MOUSE);
    gtk_container_set_border_width(GTK_CONTAINER(config_win), 5);
    gtk_signal_connect(GTK_OBJECT(config_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &config_win);

    vbox = gtk_vbox_new(FALSE, 5);

    table = gtk_table_new(4, 2, FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);
    gtk_container_add(GTK_CONTAINER(vbox), table);
    gtk_table_set_row_spacings(GTK_TABLE(table), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table), 5);

    new_textbox_row(table, 0, "Inpipe Prefix:", &inprefix_entry, config->inprefix);
    new_textbox_row(table, 1, "Outpipe Prefix:", &outprefix_entry, config->outprefix);
    new_textbox_row(table, 2, "Ackpipe Prefix:", &ackprefix_entry, config->ackprefix);
    new_textbox_row(table, 3, "ID File Prefix:", &idprefix_entry, config->idprefix);

    label = gtk_label_new("Active at launch:");
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

    outtoggle = new_toggle(vbox, "Out Pipe", config->outauto);
    acktoggle = new_toggle(vbox, "Ack Pipe", config->ackauto);
    idtoggle = new_toggle(vbox, "ID File", config->idauto);

    gtk_container_add(GTK_CONTAINER (GTK_WINDOW(config_win)), vbox);
    buttonbox = gtk_hbutton_box_new();

    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 5);
    gtk_box_pack_start(GTK_BOX(vbox), buttonbox, FALSE, FALSE, 0);

    ok_button = gtk_button_new_with_label("Ok");
    apply_button = gtk_button_new_with_label("Apply");
    cancel_button = gtk_button_new_with_label("Cancel");

    gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked",
	    GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(config_win));
    gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
	    GTK_SIGNAL_FUNC (xmmspipe_config_ok), NULL);
    gtk_signal_connect_object(GTK_OBJECT(apply_button), "clicked",
	    GTK_SIGNAL_FUNC (xmmspipe_config_apply), NULL);

    //make them accessible by keyboard
    GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(apply_button, GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);

    gtk_box_pack_start(GTK_BOX(buttonbox), ok_button, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(buttonbox), cancel_button, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(buttonbox), apply_button, TRUE, TRUE, 0);

    gtk_widget_show_all(config_win);

    //focus on ok button; Enter will trigger it
    gtk_widget_grab_default(ok_button);
}
