/*****************************************************************************
 * grab-bktr.c: bktr interface (OpenBSD/NetBSD/FreeBSD drivers)
 *****************************************************************************
 * $Id: grab-bktr.c,v 1.19 2004/09/13 15:53:04 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2004 Alainjj
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * for the moment only a slow grabdisplay is provided
 * inspired from:
 * xawtv/libng/plugins/drv0-bsd.c
 * fxtv/tvcapture.c
 * mplayer/libmpdemux/tvi_bsdbt848.c 
 * http://www.gsp.com/cgi-bin/man.cgi?section=4&topic=meteor
 *
 * if you have some problems try with "-noxv" option first
 *
*****************************************************************************/


#include "config.h"
#ifdef HAVE_BKTR

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/filio.h>
#include <sys/time.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <pthread.h>

#if HAVE_DEV_IC_BT8XX_H
#include <dev/ic/bt8xx.h>
#else
#include <machine/ioctl_meteor.h>
#include <machine/ioctl_bt848.h>
#endif

#include <X11/Xlib.h>
#ifdef HAVE_LIBXXF86DGA
# ifdef  HAVE_XFREE86_4            // In fact have XFree86 4.X.X
#  include <X11/Xproto.h>
#  define XF86DGA_MAJOR_VERSION XDGA_MAJOR_VERSION
#  define XF86DGA_MINOR_VERSION XDGA_MINOR_VERSION
# endif
# include <X11/extensions/xf86dga.h>
# include <X11/extensions/xf86dgastr.h>
#endif
#include <X11/Intrinsic.h>
extern Display *dpy;


#include "strtab.h"
#include "grab.h"
#include "memcpy.h"
#include "colorspace.h"
#include "channel.h"

static int bttvfd = -1;
static int tunerfd = -1;
extern int vbifd;
extern struct GRABBER grab_dummy, grab_bktr;
static char *buf=NULL;
int bktr_mode=-1, bktr_read=0;
static int sizeframe;

static void* get_buf(int i);
static int grab_close ();

static struct STRTAB norms[] = {
    {  0, "NTSC"      },
    {  1, "NTSC-JP"   },
    {  2, "PAL"       },
    {  3, "PAL-M"     },
    {  4, "PAL-N"     },
    {  5, "SECAM"     },
    {  6, "RSVD"      },
    { -1, NULL }
};
static int norms_map[] = {
    BT848_IFORM_F_NTSCM,
    BT848_IFORM_F_NTSCJ,
    BT848_IFORM_F_PALBDGHI,
    BT848_IFORM_F_PALM,
    BT848_IFORM_F_PALN,
    BT848_IFORM_F_SECAM,
    BT848_IFORM_F_RSVD,
};
static int norms_map2[] = {
  METEOR_FMT_NTSC,
  METEOR_FMT_NTSC,
  METEOR_FMT_PAL,
  METEOR_FMT_PAL,
  METEOR_FMT_PAL,
  METEOR_FMT_SECAM,
  METEOR_FMT_AUTOMODE /* ? */
};

static struct STRTAB inputs[] = {
    {  0, "Television"   },
    {  1, "Composite1"   },
    {  2, "S-Video"      },
    {  3, "CSVIDEO"      },
    { -1, NULL }
};
static int inputs_map[] = {
    METEOR_INPUT_DEV1,
    METEOR_INPUT_DEV0,
    METEOR_INPUT_DEV_SVIDEO,
    METEOR_INPUT_DEV2,
};

static struct meteor_pixfmt fmt[64], *cur_fmt, *ov_fmt;
static struct meteor_video ov_pos;
static int grab_audio (int mute, int volume, int *mode);

static struct meteor_pixfmt * xaw_to_fmt(video_fmt f) {
  int i;
  if(bktr_mode!=-1) return &fmt[bktr_mode];
  for(i=0;i<64;i++) {
    int b=0;
    if(fmt[i].index==-1) continue;
    switch(fmt[i].type) {
    case METEOR_PIXTYPE_RGB:
      switch(f) {
      case VIDEO_RGB15_BE:
	b = fmt[i].masks[0]==(31<<10) && 
	  fmt[i].swap_bytes==0 && fmt[i].swap_shorts==0;
	break;
      case VIDEO_RGB15_LE:
	b = fmt[i].masks[0]==(31<<10) && 
	  fmt[i].swap_bytes==1 && fmt[i].swap_shorts==0;
	break;
      case VIDEO_RGB16_BE:
	b = fmt[i].masks[0]==(31<<11) && 
	  fmt[i].swap_bytes==0 && fmt[i].swap_shorts==0;
	break;
      case VIDEO_RGB16_LE:
	b = fmt[i].masks[0]==(31<<11) && 
	  fmt[i].swap_bytes==1 && fmt[i].swap_shorts==0;
	break;
      case VIDEO_RGB24:
	b = fmt[i].masks[0]==(255<<16) && fmt[i].Bpp==3;
	break;
      case VIDEO_RGB32_BE:
	b = fmt[i].masks[0]==(255<<16) && fmt[i].Bpp==4 
	  && fmt[i].swap_shorts==0 && 
	  fmt[i].swap_bytes==0;
	break;
      case VIDEO_RGB32_LE:
	b = fmt[i].masks[0]==(255<<16) && fmt[i].Bpp==4 
	  && fmt[i].swap_shorts==1 && 
	  fmt[i].swap_bytes==1;
	break;
      default:
      }
      break;
    case METEOR_PIXTYPE_YUV_PACKED: 
      switch(f) {
      case VIDEO_UYVY:
	b=(fmt[i].swap_bytes==0);
	break;
      case VIDEO_YUYV:
	b=(fmt[i].swap_bytes==1);
	break;
      default:
      }
      break;
    case METEOR_PIXTYPE_YUV_12:
      b=f==VIDEO_YUV420;
      break;
    default:
    }
    if(b) {
      printf("@@@@@@@@@@ f=%d i=%d\n",f,i); return &fmt[i];
    }
  }
  return NULL;
}

static int xaw_to_geo(int f) {
  switch(f) {
  case VIDEO_RGB15:
  case VIDEO_RGB16:
    return METEOR_GEO_RGB16;
  case VIDEO_RGB24:
  case VIDEO_RGB32:
    return METEOR_GEO_RGB24;
  case VIDEO_YUYV:
    return METEOR_GEO_YUV_PACKED;
  default:
    return 0;
  }
}

static int fmt_available(video_fmt f) {
  struct meteor_pixfmt *pf =xaw_to_fmt(f);
  //int g=xaw_to_geo(f);
  if(pf==NULL) return 0;
  /* ANOTHER THING TO DO ? */
  return 1;
}


void dummy(int signal) {
  if(debug>3)
    fprintf(stderr,"SIGUSR1\n");
}

static int
grab_open(struct device_t *device)
{
  int i;
  
  if (-1 == (bttvfd = open (device->video, O_RDONLY))) {
    fprintf (stderr, "open %s: %s\n", device->video, strerror (errno));
    goto err;
  }
  if (-1 == (tunerfd = open (device->tuner, O_RDONLY))) {
    fprintf (stderr, "open %s: %s\n", device->tuner, strerror (errno));
    goto err;
  }
  if(device->vbi)
    if (-1 == (vbifd = open (device->vbi, O_RDONLY))) {
      fprintf (stderr, "bktr: cant't open %s for reading, %s\n",
	       device->vbi, strerror (errno));
    }
  fprintf(stderr,"supported formats\n");
  for(i=0;i<64;i++) {
    fmt[i].index=i;
    if(-1 == ioctl(bttvfd, METEORGSUPPIXFMT, &fmt[i]))
      fmt[i].index=-1;
    else {
      struct meteor_pixfmt *a=&fmt[i];
      fprintf(stderr,"(%d,type=%d,bpp=%d,"
	      "masks=0x%lx/0x%lx/0x%lx,sb=%d,ss=%d)\n",
	      a->index,a->type,a->Bpp,a->masks[0],a->masks[1],a->masks[2],
	      a->swap_bytes,a->swap_shorts);
    }
  }
  signal(SIGUSR1, dummy);
  img=0;
  nbufs=nbufs_default;
  //grab_bktr.grab_overlay=NULL;
  ov_fmt=xaw_to_fmt(overlay_format);
  if(ov_fmt==NULL) grab_bktr.grab_overlay=NULL;

#ifdef HAVE_LIBXXF86DGA
  if (have_dga) {
    int width, banksize,ramsize;
    XF86DGAGetVideoLL (dpy, XDefaultScreen (dpy), (int *) &ov_pos.addr,
		       &width, &banksize, &ramsize);
    ov_pos.width=size_img(overlay_format,width,1);
    ov_pos.banksize=banksize;
    ov_pos.ramsize=ramsize;
  }
  else
#endif
    {
      fprintf(stderr, "overlay disabled because dga is absent\n");
      grab_bktr.grab_overlay=NULL;
    }

  return 0;
 err:
  grab_close();
  return -1;
}

static int
grab_close ()
{
  if(tunerfd != -1) {
    grab_audio (1, -1, NULL);
    close(tunerfd);
    tunerfd = -1;
  }
  if(bttvfd != -1) {
    close(bttvfd);
    bttvfd = -1;
  }
  if(vbifd != -1) {
    close(vbifd);
    vbifd = -1;
  }
  return 0;
}



static int capture=0;
static void *map=NULL;
static void stop_capture(void) {
  int arg;
  if(!capture) return;
  arg=METEOR_CAP_STOP_CONT;
  if(ioctl(bttvfd, METEORCAPTUR, &arg)<0) {
      perror("ioctl METEORCAPTUR STOP_CONT");
      return;
  }
  arg=METEOR_SIG_MODE_MASK;
  if(-1 == ioctl(bttvfd, METEORSSIGNAL, &arg))
     perror("ioctl METEORSSIGNAL off");
  if(-1 == munmap(map, sizeframe)) {
    perror("munmap");
  }
  capture=0;
}


static int mustreinitgrab=1;
void *bktr_getframe(video_fmt cur_f,int width, int height) {
  static struct meteor_geomet geom;
  struct timeval tv;
  sigset_t sa_mask;
  static int last_f=-1;
  void *dest;
  
  if(mustreinitgrab || width!=geom.columns || height!=geom.rows
     || last_f!=cur_f) {
    int arg;
    last_f=cur_f;
    if(!bktr_read)  stop_capture();
    mustreinitgrab=0;
    fprintf(stderr,"init grab_scr %d %d\n",width,height);
    cur_fmt=xaw_to_fmt(cur_f);
    if(cur_fmt==NULL) {
      fprintf(stderr,"format not available by card\n");
      return NULL;
    }
    sizeframe=size_img(cur_f,width,height);
    if(-1 == ioctl(bttvfd, METEORSACTPIXFMT, cur_fmt))
      perror("ioctl METEORSACTPIXFMT");
    memset(&geom,0,sizeof(geom));
    geom.frames = 1;
    geom.oformat =  xaw_to_geo(cur_f);
    geom.columns=width;
    geom.rows=height;
    if(-1 == ioctl(bttvfd, METEORSETGEO, &geom)) {
      perror("ioctl METEORSETGEO");
      return NULL;
    }
    if(!bktr_read) {
      map = mmap(NULL, sizeframe, PROT_READ, MAP_SHARED, bttvfd, 0);
      if(map==MAP_FAILED) {
	perror("mmap");
	return NULL;
      }
      arg=SIGUSR1;
      if(-1 == ioctl(bttvfd, METEORSSIGNAL, &arg))
	perror("ioctl METEORSSIGNAL grab_scr");
      arg = METEOR_CAP_CONTINOUS ;
      printf("BEGIN CAPTURE\n");
      if(ioctl(bttvfd, METEORCAPTUR, &arg)<0) {
	perror("ioctl METEORCAPTUR CONTINUOUS");
	return NULL;
      }
	capture=1;
    }
    buf=realloc(buf,sizeframe*nbufs);
  }

  img++; if(img==nbufs) img=0;
  dest=&buf[img*sizeframe];
#ifdef _POSIX_THREAD_IS_GNU_PTH
  pthread_yield_np();
#endif
  gettimeofday (&tv, NULL);
  videostampmin = tv.tv_usec/1000000.0+tv.tv_sec;
  if(!bktr_read) {
    sigprocmask(0,NULL,&sa_mask);
    sigsuspend(&sa_mask);
    gettimeofday (&tv, NULL);
    fast_memcpy(dest,map,sizeframe);
  } else {
    read(bttvfd, dest, sizeframe);
    gettimeofday (&tv, NULL);
  }
  videostampmax = tv.tv_usec/1000000.0+tv.tv_sec;
  return dest;
}


static int
grab_tune (unsigned long freq)
{
  stop_capture();
  if (-1 == ioctl (tunerfd, TVTUNER_SETFREQ, &freq))
    perror ("ioctl TVTUNER_SETFREQ" );
  return 0;
}

static int
grab_tuned ()
{
  return 1;
}

static int
grab_input (int input, int norm)
{
  stop_capture();
  if(input!=-1)
    if(-1 == ioctl(bttvfd, METEORSINPUT,&inputs_map[input]))
      perror("ioctl METEORSINPUT");
  if(norm!=-1) {
    if(-1 == ioctl(bttvfd, BT848SFMT,&norms_map[norm]))
      perror("ioctl BT848SFMT");
    if(-1 == ioctl(bttvfd, METEORSFMT, &norms_map2[norm]))
      perror("ioctl METEORFMT");
  }
  return 0;
}

static void control(int v,int ioc,int min,int max) {
  int arg=min+v*(max-min)/65535;
  fprintf(stderr,"v=%d arg=%d min=%d max=%d\n",v,arg,min,max);
  if(-1==ioctl(tunerfd,ioc,&arg)) {
    perror("ioctl control");
  }
}

static int
grab_picture (int color, int bright, int hue, int contrast)
{
  if(color!=-1)
    control(color,BT848_SCSAT,BT848_SATUREGMIN,BT848_SATUREGMAX);
  if(bright!=-1)
    control(bright,BT848_SBRIG,BT848_BRIGHTREGMIN,BT848_BRIGHTREGMAX);
  if(hue!=-1)
    control(hue,BT848_SHUE,BT848_HUEREGMIN,BT848_HUEREGMAX);
  if(contrast!=-1)
    control(contrast,BT848_SCONT,BT848_CONTRASTREGMIN,BT848_CONTRASTREGMAX);

  return 0;
}

static int
grab_audio (int mute, int volume, int *mode)
{
  int arg;
  if (mute != -1) {
    arg = mute ? AUDIO_MUTE : AUDIO_UNMUTE;
    if (-1 == ioctl (tunerfd, BT848_SAUDIO, &arg))
      perror ("ioctl BT848_SAUDIO");
  }
  return 0;
}

static int is525(int norm) {
  printf("@@@@@ %d %d %d\n",norm,norms_map2[norm],METEOR_FMT_NTSC);
  return norms_map2[norm]==METEOR_FMT_NTSC;
}

static int issecam(int norm) {
  return norms_map2[norm]==METEOR_FMT_SECAM;
}

static void* get_buf(int i){
  return buf + sizeframe*i;
}

static int
grab_overlay (int x, int y, int width, int height, int format,
              struct OVERLAY_CLIP *oc, int count)
{
  struct meteor_geomet geom;
  struct meteor_video pos;
  struct bktr_clip        clip[BT848_MAX_CLIP_NODE];
  int arg,i;
  printf("grab_ov\n");
  if(bttvfd==-1) return -1;
  if (width == 0 || height == 0) {
    if (debug) fprintf (stderr, "bktr: overlay off\n");
    arg=METEOR_CAP_STOP_CONT;
    if( -1 == ioctl(bttvfd, METEORCAPTUR, &arg )) {
      perror("ioctl METEORCAPTUR off");
      return -1;
    }
    memset(&pos,0,sizeof(pos));
    if(-1 == ioctl(bttvfd, METEORSVIDEO, &pos)) {
      perror("ioctl METEORSVIDEO");
      return -1;
    }
    return 0;
  }
  stop_capture();
  mustreinitgrab=1;
  memcpy(&pos,&ov_pos,sizeof(pos));
  pos.addr += pos.width*y;
  pos.addr += size_img(overlay_format,x, 1);
  if(-1 == ioctl(bttvfd, METEORSVIDEO, &pos)) {
    perror("ioctl METEORSVIDEO");
    return -1;
  }
  memset(&geom,0,sizeof(geom));
  geom.frames = 1;
  geom.oformat = xaw_to_geo(overlay_format);
  geom.columns = width;
  geom.rows=height;
  arg=METEOR_CAP_STOP_CONT;
  if(-1 == ioctl(bttvfd, METEORCAPTUR, &arg)) {
    perror("ioctl METEORCAPTUR");
    return -1;
  }
  if(-1 == ioctl(bttvfd, METEORSETGEO, &geom)) {
    perror("ioctl METEORSETGEO (overlay)");
    return -1;
  }
  if(-1 == ioctl(bttvfd, METEORSACTPIXFMT, ov_fmt)) {
    perror("ioctl METEORSACTPIXFMT");
    return -1;
  }
  memset(&clip,0,sizeof(clip));
  for (i = 0; i < count && i<BT848_MAX_CLIP_NODE; i++) {
    clip[i].x_min=oc[i].y1/2;
    clip[i].x_max=oc[i].y2/2;
    clip[i].y_min=oc[i].x1;
    clip[i].y_max=oc[i].x2-1;
  }
  if(-1 == ioctl(bttvfd, BT848SCLIP, &clip)) {
    perror("ioctl BT848SCLIP");
    //return -1;
  }
  arg=METEOR_CAP_CONTINOUS;
  if(-1 == ioctl(bttvfd, METEORCAPTUR, &arg)) {
    perror("ioctl METEORCAPTUR");
    return -1;
  }
  return 0;
}


struct GRABBER grab_bktr = {
  "bktr",
  norms, inputs,
  grab_open,
  grab_close,
  grab_overlay,
  fmt_available,
  bktr_getframe,
  grab_tune,
  grab_tuned,
  grab_input,
  grab_picture,
  grab_audio,
  is525,
  issecam,
  get_buf
};

#endif
