/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2002 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/


#include "NXConnection.h"
#include "NXSettings.h"
#include "Options.h"
#include "SettingsToParameters.h"

#include <iostream>



#undef NX_CONNECTION_DEBUG



NXConnection::NXConnection()
{
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: NXConnection created." << endl << flush;
#endif
  pSession = CreateSession();
  pConnection = 0;
  pProxy = 0;
  pXServer = 0;
}

NXConnection::~NXConnection()
{
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: NXConnection's destructor called." << endl << flush;
#endif
  HandleCleanup();
}

void NXConnection::HandleCleanup()
{
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: cleaning up NXConnection." << endl << flush;
#endif
  if( pSession )
    DeleteSession( pSession );
  if( pConnection )
    DeleteServerConnection( pConnection );
  if( pProxy )
    DeleteProxyConnection( pProxy );
  if( pXServer )
    DeleteXServerConnection( pXServer );
}

int NXConnection::GetStateId() const
{
  return GetCurrentState( pSession );
}

string NXConnection::GetState() const
{
  if( ErrorOccurred() )
    return "Connection error";

  string sRet = "";

  switch( GetCurrentState( pSession ) )
  {
  case NX_ReadyForConnection:
    sRet = "Initializing connection";
    break;

  case NX_Connecting:
    sRet = "Connecting to '";
    sRet += string( GetStringParameter( pSession, NX_HostName, "unknown" ) );
    sRet += "'";
    break;

  case NX_Connected:
    sRet = "Connected to '";
    sRet += string( GetStringParameter( pSession, NX_HostName, "unknown" ) );
    sRet += "'";
    break;

  case NX_SendingVersion:
  case NX_AcceptingVersion:
    sRet = "Negotiating protocol version";
    break;

  case NX_VersionAccepted:
    sRet = "NX protocol version accepted";
    break;

  case NX_WaitingForAuth:
  case NX_SendingAuth:
  case NX_Authenticating:
    sRet = "Waiting for authentication";
    break;

  case NX_Authenticated:
  case NX_StartingSession:
    sRet = "Authentication completed";
    break;

  case NX_SendingParameters:
  case NX_Negotiating:
    sRet = "Negotiating session parameters";
    break;

  case NX_SessionAccepted:
    sRet = "Starting session";
    break;

  case NX_ConnectionEnd:
    sRet = "";
    break;

  case NX_ProxyReady:
    sRet = "Starting session";
    break;

  case NX_ProxyConnecting:
    sRet = "Initializing X protocol compression";
    break;

  case NX_ProxyNegotiating:
    sRet = "Negotiating link parameters";
    break;

  case NX_ProxyConnected:
    sRet = "Established X server connection";
    break;

  case NX_ProxyConnectionEnd:
    sRet = "";
    break;

  default:
    sRet = "";
  }

  return sRet;
}

bool NXConnection::ErrorOccurred() const
{
  return ( GetCurrentState( pSession ) == NX_Exception );
}

string NXConnection::GetError() const
{
  char *pError = GetLastError( pSession );
  string sError( pError );
  delete pError;
  return sError;
}

void NXConnection::Start()
{
  pConnection = CreateServerConnection( pSession, "nxssh" );
  if( pConnection )
    InitServerConnection( pConnection );
}

bool NXConnection::Advance( unsigned int timeout )
{
  if( ErrorOccurred() )
    return false;

  int iState = GetStateId();

  if( iState >= NX_ReadyForConnection && iState < NX_InitProxy )
  {
    if( EndServerConnection( pConnection ) != NX_Ok )
    {
      return ( AdvanceServerConnection( pConnection, timeout ) != NX_Exception );
    }
  }

  if( !pProxy && SessionNeedToStartNewProxy( pSession) == NX_Ok )
  {
#if defined( NX_CONNECTION_DEBUG )
    cout << "Info: create proxy connection." << endl << flush;
#endif
    pProxy = CreateProxyConnection( pSession );
    if( InitProxyConnection( pProxy ) == NX_Exception )
      return false;
  }

  if( EndProxyConnection( pProxy ) != NX_Ok )
  {
    return ( AdvanceProxyConnection( pProxy, timeout ) != NX_Exception );
  }

  return true;
}

bool NXConnection::Accepted() const
{
  return ( GetStateId() == NX_ProxyConnected );
}

void NXConnection::Stop()
{
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: stop connection called." << endl << flush;
#endif
  int iState = GetStateId();

  if( iState > NX_ReadyForConnection && iState < NX_InitProxy )
  {
#if defined( NX_CONNECTION_DEBUG )
    cout << "Info: terminating server connection." << endl << flush;
#endif
    TerminateServerConnection( pConnection );
  }

  if( iState > NX_InitProxy )
  {
#if defined( NX_CONNECTION_DEBUG )
    cout << "Info: terminating proxy connection." << endl << flush;
#endif
    TerminateProxyConnection( pProxy );
  }
}

bool NXConnection::SetParameters( NXSettings* sets )
{
  string sTmp = "";

  ClearParameters( pSession );

  sTmp = options->GetNXPersonalDirectory();
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: setting 'NX_PersonalDirectory' to '" << sTmp << "'." << endl << flush;
#endif
  SetStringParameter( pSession, NX_PersonalDirectory, sTmp.c_str() );

#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: checking cookie..." << endl << flush;
#endif

  sTmp = options->GetNXCookieFilePath();
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: setting 'NX_CookieTempFilePath' to '" << sTmp << "'." << endl << flush;
#endif
  SetStringParameter( pSession, NX_CookieTempFilePath, sTmp.c_str() );

  if( GetExistingSessionCookie( pSession ) != NX_Ok )
  {
    if( GenerateSessionCookie( pSession ) != NX_Ok )
    {
      cerr << "Error: session cookie NOT found... and cannot create a new one." << endl << flush;
      return false;
    }
  }
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: cookie found..." << endl << flush;
#endif

  sTmp = options->GetNXSshPath();
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: setting 'NX_SshPath' to '" << sTmp << "'." << endl << flush;
#endif
  SetStringParameter( pSession, NX_SshPath, options->GetNXSshPath().c_str() );

  sTmp = options->GetNXSshKeyPath();
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: setting 'NX_SshKeyPath' to '" << sTmp << "'." << endl << flush;
#endif
  SetStringParameter( pSession, NX_SshKeyPath, sTmp.c_str() );

  sTmp = options->GetNXSshLogPath();
#if defined( NX_CONNECTION_DEBUG )
  cout << "Info: setting 'NX_SshLogPath' to '" << sTmp << "'." << endl << flush;
#endif
  SetStringParameter( pSession, NX_SshLogPath, sTmp.c_str() );

  /* Convert other Paramaters */
  SettingsToParameters stp( sets, pSession );
  stp.Convert();

  SetStringParameter( pSession, NX_ProxyPath, options->GetNXProxyPath().c_str() );
  SetStringParameter( pSession, NX_ProxyLibraryPath, options->GetNXLibDirectory().c_str() );
  SetStringParameter( pSession, NX_ProxyMode, "S" );


  return true;
}

string NXConnection::GetLogPath() const
{
  string file_path = "";
  if( GetStateId() < NX_ProxyReady )
    file_path = options->GetNXSshLogPath();
  else
  {
    char *pLogPath = GetStringParameter( pSession, NX_ProxyLogPath, "" );
    file_path = pLogPath;
    delete pLogPath;
  }

  return file_path;
}

