/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <X11/X.h>
#include <X11/Xatom.h>

#include "NXproto.h"
#include "NXpack.h"
#include "NXmitshm.h"

#include "ServerChannel.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "Compressor.h"
#include "Decompressor.h"

#include "Statistics.h"
#include "Alerts.h"
#include "Socket.h"

#include "Split.h"

//
// Set the verbosity level. You also
// need to define OPCODES in Misc.cpp
// if you want literals instead of
// opcodes' numbers.
//

#define PANIC
#define WARNING
#undef  OPCODES
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Define this to log when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// Define these to hide server extensions.
//

#define HIDE_MIT_SHM_EXTENSION
#define HIDE_BIG_REQUESTS_EXTENSION
#define HIDE_XFree86_Bigfont_EXTENSION
#undef  HIDE_SHAPE_EXTENSION
#undef  HIDE_XKEYBOARD_EXTENSION

//
// Here are the static members.
//

#ifdef REFERENCES

int ServerChannel::references_ = 0;

#endif

ServerChannel::ServerChannel(Transport *transport, Compressor *compressor,
                                 Decompressor *decompressor)

  : Channel(transport, compressor, decompressor),
        readBuffer_(transport_, this)
{
  //
  // Create X channels' specific caches
  // only if protocol is older than 3.
  //

  if (control -> isProtoStep3() == 0)
  {
    clientCache_ = new ClientCache();
    serverCache_ = new ServerCache();

    if (clientCache_ == NULL || serverCache_ == NULL)
    {
      #ifdef PANIC
      *logofs << "ServerChannel: PANIC! Failed to create channel's caches.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to create channel's caches.\n";

      HandleCleanup();
    }
  }

  //
  // Current sequence and opcodes known
  // by client and X server.
  //

  clientOpcode_ = 0;
  serverOpcode_ = 0;

  clientSequence_ = 0;
  serverSequence_ = 0;

  //
  // Save the last motion event here and
  // flush it only when timeout expires.
  //

  lastMotion_[0] = '\0';
  lastMotionTs_  = nullTimestamp();

  //
  // Save sequence numbers of split commits
  // in order to mask errors.
  //

  for (int i = 0; i < MAX_COMMIT_SEQUENCE_QUEUE; i++)
  {
    commitSequenceQueue_[i] = 0;
  }

  syncCounter_ = 0;
  firstSyncTs_ = nullTimestamp();

  //
  // Do we enable or not sending of expose
  // events to the X client.
  //

  enableExpose_         = 1;
  enableGraphicsExpose_ = 1;
  enableNoExpose_       = 1;

  //
  // Track data of image currently being
  // decompressed.
  //

  imageState_ = NULL;

  //
  // Track MIT-SHM resources.
  //

  shmemState_ = NULL;

  //
  // Use per-client data to unpack images.
  //

  for (int i = 0; i < CONNECTIONS_LIMIT; i++)
  {
    unpackState_[i] = NULL;
  }

  //
  // Unpack tight images.
  //

  tightState_ = NULL;

  #ifdef REFERENCES
  *logofs << "ServerChannel: Created new object at " 
          << this << " for FD#" << fd_ << " out of "
          << ++references_ << " allocated channels.\n"
          << logofs_flush;
  #endif
}

ServerChannel::~ServerChannel()
{
  #ifdef TEST
  *logofs << "ServerChannel: Freeing image state information.\n"
          << logofs_flush;
  #endif

  handleImageStateRemove();

  #ifdef TEST
  *logofs << "ServerChannel: Freeing shared memory information.\n"
          << logofs_flush;
  #endif

  handleShmemStateRemove();

  #ifdef TEST
  *logofs << "ServerChannel: Freeing unpack state information.\n"
          << logofs_flush;
  #endif

  for (int i = 0; i < CONNECTIONS_LIMIT; i++)
  {
    handleUnpackStateRemove(i);
  }

  #ifdef TEST
  *logofs << "ServerChannel: Freeing tight state information.\n"
          << logofs_flush;
  #endif

  handleTightStateRemove();

  #ifdef TEST
  *logofs << "ServerChannel: Freeing channel caches.\n"
          << logofs_flush;
  #endif

  if (control -> isProtoStep3() == 0)
  {
    delete clientCache_;
    delete serverCache_;
  }

  #ifdef REFERENCES
  *logofs << "ServerChannel: Deleted object at " 
          << this << " for FD#" << fd_ << " out of "
          << --references_ << " allocated channels.\n"
          << logofs_flush;
  #endif
}

//
// Beginning of handleRead().
//

int ServerChannel::handleRead(EncodeBuffer& encodeBuffer)
{
  #ifdef TEST
  *logofs << "handleRead: Called for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Pointer to located message and
  // its size in bytes.
  //

  const unsigned char *inputMessage;
  unsigned int inputLength;

  //
  // Keep handling X messages until
  // it's time to yield.
  //

  int yield = 0;
  int loops = 0;
  int reads = 0;

  //
  // Set when message is found in
  // cache.
  //

  int hit;

  while (yield == 0)
  {

  #ifdef TEST

  *logofs << "handleRead: Initial time spent handling messages is "
          << control -> getTimeInARow() << " Ms.\n"
          << logofs_flush;

  *logofs << "handleRead: Initial bytes to write on proxy link are "
          << control -> getBytesInARow() << " bytes.\n"
          << logofs_flush;

  #endif

  //
  // Before reading from socket handle any
  // data we left in the read buffer.
  //

  if (pending_ == 0)
  {
    if (readBuffer_.readMessage() < 0)
    {
      return -1;
    }
  }

  reads++;

  #if defined(INFO) || defined(TEST)
  *logofs << "handleRead: Getting messages from FD#" << fd_ << " with "
          << readBuffer_.getLength() << " bytes in the read buffer.\n"
          << logofs_flush;
  #endif

  //
  // Extract any complete message which 
  // is available in the buffer.
  //

  while (yield == 0 && (inputMessage = readBuffer_.getMessage(inputLength)) != NULL)
  {
    loops++;

    hit = 0;

    if (firstReply_)
    {
      imageByteOrder_ = inputMessage[30];
      bitmapBitOrder_ = inputMessage[31];
      scanlineUnit_   = inputMessage[32];
      scanlinePad_    = inputMessage[33];

      firstReply_ = 0;

      encodeBuffer.encodeValue((unsigned int) inputMessage[0], 8);
      encodeBuffer.encodeValue((unsigned int) inputMessage[1], 8);
      encodeBuffer.encodeValue(GetUINT(inputMessage + 2, bigEndian_), 16);
      encodeBuffer.encodeValue(GetUINT(inputMessage + 4, bigEndian_), 16);
      encodeBuffer.encodeValue(GetUINT(inputMessage + 6, bigEndian_), 16);

      if (ServerCache::lastInitReply.compare(inputLength - 8, inputMessage + 8))
      {
        encodeBuffer.encodeBool(1);
      }
      else
      {
        encodeBuffer.encodeBool(0);

        for (unsigned int i = 8; i < inputLength; i++)
        {
          encodeBuffer.encodeValue((unsigned int) inputMessage[i], 8);
        }
      }

      #if defined(TEST) || defined(OPCODES)

      int bits = encodeBuffer.getBits();

      *logofs << "handleRead: Handled first reply. " << inputLength
              << " bytes in, " << bits << " bits (" << ((float) bits) / 8
              << " bytes) out.\n" << logofs_flush;

      #endif

      if (firstAgent_)
      {
        cerr << "Info" << ": Established X server connection.\n" ;

        firstAgent_ = 0;
      }
    }
    else
    {
      //
      // Check if this is a reply.
      //

      if (*inputMessage == X_Reply)
      {
        int bits = 0;

        unsigned char inputOpcode = *inputMessage;

        unsigned short int requestSequenceNum;
        unsigned char      requestOpcode;
        unsigned int       requestData[3];

        unsigned int sequenceNum = GetUINT(inputMessage + 2, bigEndian_);

        //
        // To increase responsiveness, we could have forwarded
        // the sync requests to the X server. At the same time,
        // we don't want the dummy replies to disturb the proxy
        // link, so we must intercept them before entering the
        // encode loop.
        //

        if (control -> AgentSyncPropagate == 1)
        {
          if (sequenceQueue_.peek(requestSequenceNum, requestOpcode,
                                      requestData[0], requestData[1], requestData[2]) &&
                                          requestSequenceNum == sequenceNum &&
                                              requestOpcode == X_GetInputFocus &&
                                                  requestData[0] == opcodeStore_ -> sync)
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleRead: Received tainted X_GetInputFocus reply "
                    << "for request OPCODE#" << requestData[0] << " with "
                    << "sequence " << sequenceNum << ".\n"
                    << logofs_flush;
            #endif

            sequenceQueue_.pop(requestSequenceNum, requestOpcode,
                                   requestData[0], requestData[1], requestData[2]);

            syncCounter_--;

            if (syncCounter_ == 0)
            {
              #ifdef DEBUG
              *logofs << "handleRead: Cleared first sync timestamp "
                      << "for FD#" << fd_ << ".\n" << logofs_flush;
              #endif

              firstSyncTs_ = nullTimestamp();
            }
            else if (syncCounter_ < 0)
            {
              #ifdef WARNING
              *logofs << "handleRead: WARNING! Reached negative value "
                      << "for sync counter.\n" << logofs_flush;
              #endif

              syncCounter_ = 0;
            }

            continue;
          }
        }

        //
        // Encode opcode and difference between
        // current sequence and the last one.
        //

        encodeBuffer.encodeOpcodeValue(inputOpcode, serverCache_ -> opcodeCache);

        serverOpcode_ = inputOpcode;

        unsigned int sequenceDiff = sequenceNum - serverSequence_;

        serverSequence_ = sequenceNum;

        #ifdef DEBUG
        *logofs << "handleRead: Last server sequence number for FD#" 
                << fd_ << " is " << serverSequence_ << " with "
                << "difference " << sequenceDiff << ".\n"
                << logofs_flush;
        #endif

        if (serverSequenceReset_ == 1)
        {
          encodeBuffer.encodeValue(serverSequence_, 32, 16);

          #if defined(INFO) || defined(TEST)
          *logofs << "handleRead: Reset last server sequence number to "
                  << serverSequence_ << " for FD#" 
                  << fd_ << ".\n" << logofs_flush;
          #endif

          serverSequenceReset_ = 0;
        }
        else
        {
          encodeBuffer.encodeCachedValue(sequenceDiff, 16,
                             serverCache_ -> replySequenceCache, 7);
        }

        //
        // Now handle the data part.
        //

        if (sequenceQueue_.peek(requestSequenceNum, requestOpcode) &&
                                    requestSequenceNum == sequenceNum)
        {
          //
          // We've found the request that generated this reply.
          // It is possible to compress the reply based on the
          // specific request type.
          //

          sequenceQueue_.pop(requestSequenceNum, requestOpcode,
                                 requestData[0], requestData[1], requestData[2]);

          //
          // If differential compression is disabled
          // then use the most simple encoding.
          //

          if (control -> LocalDeltaCompression == 0)
          {
            int result = handleFastReadReply(encodeBuffer, requestOpcode,
                                                 inputMessage, inputLength);
            if (result < 0)
            {
              return -1;
            }
            else if (result > 0)
            {
              if (pending_ == 0 && control -> isTimeToYield(yield_in_channel) == 1)
              {
                #if defined(INFO) || defined(TEST)
                *logofs << "handleRead: Giving up because of "
                        << control -> getBytesInARow() << " bytes to write after "
                        << control -> getTimeInARow() << " Ms for FD#" << fd_
                        << ".\n" << logofs_flush;
                #endif

                yield = 1;
              }

              continue;
            }
          }

          switch (requestOpcode)
          {
          case X_AllocColor:
            {
              const unsigned char *nextSrc = inputMessage + 8;
              for (unsigned int i = 0; i < 3; i++)
              {
                unsigned int colorValue = GetUINT(nextSrc, bigEndian_);
                nextSrc += 2;
                if (colorValue == requestData[i])
                  encodeBuffer.encodeBool(1);
                else
                {
                  encodeBuffer.encodeBool(0);
                  encodeBuffer.encodeValue(colorValue - colorValue, 16, 6);
                }
              }
              unsigned int pixel = GetULONG(inputMessage + 16, bigEndian_);
              encodeBuffer.encodeValue(pixel, 32, 9);

              //
              // Don't set priority in case of RDP
              // sessions. NX desktop uses a specific
              // request to pipeline multiple replies.
              //

              if (control -> SessionMode != SESSION_RDP)
              {
                priority_++;
              }
            }
            break;
          case X_GetAtomName:
            {
              unsigned int nameLength = GetUINT(inputMessage + 8, bigEndian_);
              encodeBuffer.encodeValue(nameLength, 16, 6);
              const unsigned char *nextSrc = inputMessage + 32;

              if (control -> isProtoStep3() == 1)
              {
                serverCache_ -> getAtomNameTextCompressor.reset();
                for (unsigned int i = 0; i < nameLength; i++)
                {
                  serverCache_ -> getAtomNameTextCompressor.
                        encodeChar(*nextSrc++, encodeBuffer);
                }
              }
              else
              {
                //
                // This is an error. Should not encode data
                // using the remote side cache. It didn't
                // cause problems so far only because cache
                // was used per-channel.
                //

                clientCache_ -> internAtomTextCompressor.reset();
                for (unsigned int i = 0; i < nameLength; i++)
                {
                  clientCache_ -> internAtomTextCompressor.
                        encodeChar(*nextSrc++, encodeBuffer);
                }
              }

              priority_++;
            }
            break;
          case X_GetGeometry:
            {
              //
              // TODO: This obtains a satisfactory 10:1, but
              // could be cached to leverage the big amount
              // of such requests issued by QT clients.
              //

              encodeBuffer.encodeCachedValue(inputMessage[1], 8,
                                             serverCache_ -> depthCache);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                  29, serverCache_ -> getGeometryRootCache, 9);
              const unsigned char *nextSrc = inputMessage + 12;
              for (unsigned int i = 0; i < 5; i++)
              {
                encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                   *serverCache_ -> getGeometryGeomCache[i], 8);
                nextSrc += 2;
              }

              priority_++;
            }
            break;
          case X_GetInputFocus:
            {
              //
              // Is it a real X_GetInputFocus or a
              // masqueraded reply?
              //

              if (requestData[0] == X_GetInputFocus)
              {
                encodeBuffer.encodeValue((unsigned int) inputMessage[1], 2);
                encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                29, serverCache_ -> getInputFocusWindowCache);

                priority_++;
              }
              else
              {
                //
                // TODO: We are not setting priority in case
                // of replies other than real X_GetInputFocus
                // or X_NXGetUnpackParameters. We should check
                // once again that this is OK.
                //

                #if defined(INFO) || defined(TEST)
                *logofs << "handleRead: Received tainted X_GetInputFocus reply "
                        << "for request OPCODE#" << requestData[0] << " with "
                        << "sequence " << sequenceNum << ".\n"
                        << logofs_flush;
                #endif

                //
                // Don't encode any data in case of sync
                // messages or any other reply for which
                // opcode is enough.
                //

                if (requestData[0] == opcodeStore_ -> getUnpackParameters)
                {
                  for (int i = 0; i < PACK_METHOD_LIMIT; i++)
                  {
                    encodeBuffer.encodeBool(control -> LocalUnpackMethods[i]);
                  }

                  priority_++;
                }
                else if (requestData[0] == opcodeStore_ -> getShmemParameters)
                {
                  if (handleShmemInit(encodeBuffer, requestOpcode, requestData[1],
                                          inputMessage, inputLength) < 0)
                  {
                    return -1;
                  }

                  priority_++;
                }

                //
                // Account this data to the original opcode.
                //

                requestOpcode = requestData[0];
              }
            }
            break;
          case X_GetKeyboardMapping:
            {
              unsigned int keysymsPerKeycode = (unsigned int) inputMessage[1];
              if (ServerCache::getKeyboardMappingLastMap.compare(inputLength - 32,
                        inputMessage + 32) && (keysymsPerKeycode ==
                            ServerCache::getKeyboardMappingLastKeysymsPerKeycode))
              {
                encodeBuffer.encodeBool(1);

                priority_++;

                break;
              }
              ServerCache::getKeyboardMappingLastKeysymsPerKeycode = keysymsPerKeycode;
              encodeBuffer.encodeBool(0);
              unsigned int numKeycodes =
              (((inputLength - 32) / keysymsPerKeycode) >> 2);
              encodeBuffer.encodeValue(numKeycodes, 8);
              encodeBuffer.encodeValue(keysymsPerKeycode, 8, 4);
              const unsigned char *nextSrc = inputMessage + 32;
              unsigned char previous = 0;
              for (unsigned int count = numKeycodes * keysymsPerKeycode;
                   count; --count)
              {
                unsigned int keysym = GetULONG(nextSrc, bigEndian_);
                nextSrc += 4;
                if (keysym == NoSymbol)
                  encodeBuffer.encodeBool(1);
                else
                {
                  encodeBuffer.encodeBool(0);
                  unsigned int first3Bytes = (keysym >> 8);
                  encodeBuffer.encodeCachedValue(first3Bytes, 24,
                             serverCache_ -> getKeyboardMappingKeysymCache, 9);
                  unsigned char lastByte = (unsigned char) (keysym & 0xff);
                  encodeBuffer.encodeCachedValue(lastByte - previous, 8,
                           serverCache_ -> getKeyboardMappingLastByteCache, 5);
                  previous = lastByte;
                }
              }

              priority_++;
            }
            break;
          case X_GetModifierMapping:
            {
              encodeBuffer.encodeValue((unsigned int) inputMessage[1], 8);
              const unsigned char *nextDest = inputMessage + 32;
              if (ServerCache::getModifierMappingLastMap.compare(inputLength - 32,
                                                nextDest))
              {
                encodeBuffer.encodeBool(1);

                priority_++;

                break;
              }
              encodeBuffer.encodeBool(0);
              for (unsigned int count = inputLength - 32; count; count--)
              {
                unsigned char next = *nextDest++;
                if (next == 0)
                  encodeBuffer.encodeBool(1);
                else
                {
                  encodeBuffer.encodeBool(0);
                  encodeBuffer.encodeValue(next, 8);
                }
              }

              priority_++;
            }
            break;
          case X_GetProperty:
            {
              if (control -> isProtoStep3() == 1)
              {
                MessageStore *messageStore = serverStore_ ->
                                     getReplyStore(X_GetProperty);

                hit = handleEncode(encodeBuffer, serverCache_, messageStore,
                                       requestOpcode, inputMessage, inputLength);

                priority_++;

                break;
              }

              unsigned char format = (unsigned int) inputMessage[1];
              encodeBuffer.encodeCachedValue(format, 8,
                                       serverCache_ -> getPropertyFormatCache);
              unsigned int numBytes = GetULONG(inputMessage + 16, bigEndian_);
              encodeBuffer.encodeValue(numBytes, 32, 9);
              if (format == 16)
                numBytes <<= 1;
              else if (format == 32)
                numBytes <<= 2;
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                  29, serverCache_ -> getPropertyTypeCache, 9);
              encodeBuffer.encodeValue(GetULONG(inputMessage + 12, bigEndian_),
                                       32, 9);
              const unsigned char *nextSrc = inputMessage + 32;
              if (format == 8 && numBytes < SERVER_TEXT_CACHE_SIZE)
              {
                if (requestData[0] == XA_RESOURCE_MANAGER)
                {
                  if (ServerCache::xResources.compare(numBytes, inputMessage + 32))
                  {
                    encodeBuffer.encodeBool(1);

                    priority_++;

                    break;
                  }
                  encodeBuffer.encodeBool(0);
                }
                serverCache_ -> getPropertyTextCompressor.reset();
                for (unsigned int i = 0; i < numBytes; i++)
                {
                  unsigned char nextChar;
                  serverCache_ -> getPropertyTextCompressor.encodeChar(
                                    nextChar = *nextSrc++, encodeBuffer);
                  if (nextChar == 10)
                  {
                    serverCache_ -> getPropertyTextCompressor.reset(nextChar);
                  }
                }
              }
              else
              {
                unsigned int compressedDataSize = 0;
                unsigned char *compressedData   = NULL;

                int compressed = handleCompress(encodeBuffer, requestOpcode, inputMessage,
                                                    inputLength, 32, compressedData,
                                                        compressedDataSize);
                if (compressed < 0)
                {
                  return -1;
                }
              }

              priority_++;
            }
            break;
          case X_GetSelectionOwner:
            {
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                29, serverCache_ -> getSelectionOwnerCache, 9);
              priority_++;
            }
            break;
          case X_GetWindowAttributes:
            {
              encodeBuffer.encodeValue((unsigned int) inputMessage[1], 2);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                           29, serverCache_ -> visualCache);
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 12, bigEndian_),
                         16, serverCache_ -> getWindowAttributesClassCache, 3);
              encodeBuffer.encodeCachedValue(inputMessage[14], 8,
                           serverCache_ -> getWindowAttributesBitGravityCache);
              encodeBuffer.encodeCachedValue(inputMessage[15], 8,
                           serverCache_ -> getWindowAttributesWinGravityCache);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 16, bigEndian_),
                        32, serverCache_ -> getWindowAttributesPlanesCache, 9);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 20, bigEndian_),
                         32, serverCache_ -> getWindowAttributesPixelCache, 9);
              encodeBuffer.encodeBool((unsigned int) inputMessage[24]);
              encodeBuffer.encodeBool((unsigned int) inputMessage[25]);
              encodeBuffer.encodeValue((unsigned int) inputMessage[26], 2);
              encodeBuffer.encodeBool((unsigned int) inputMessage[27]);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 28, bigEndian_),
                                         29, serverCache_ -> colormapCache, 9);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 32, bigEndian_),
                        32, serverCache_ -> getWindowAttributesAllEventsCache);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 36, bigEndian_),
                       32, serverCache_ -> getWindowAttributesYourEventsCache);
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 40, bigEndian_),
                    16, serverCache_ -> getWindowAttributesDontPropagateCache);

              priority_++;
            }
            break;
          case X_GrabKeyboard:
          case X_GrabPointer:
            {
              encodeBuffer.encodeValue((unsigned int) inputMessage[1], 3);

              priority_++;
            }
            break;
          case X_InternAtom:
            {
              encodeBuffer.encodeValue(GetULONG(inputMessage + 8, bigEndian_), 29, 9);

              //
              // We set priority. This could penalize those
              // well written clients using XInternAtoms()
              // to pipeline multiple replies.
              //

              priority_++;
            }
            break;
          case X_ListExtensions:
            {
              encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 8);
              unsigned int numExtensions = (unsigned int) inputMessage[1];
              encodeBuffer.encodeValue(numExtensions, 8);
              const unsigned char *nextSrc = inputMessage + 32;

              for (; numExtensions; numExtensions--)
              {
                unsigned int length = (unsigned int) (*nextSrc++);

                encodeBuffer.encodeValue(length, 8);

                #ifdef HIDE_MIT_SHM_EXTENSION

                if (!strncmp((char *) nextSrc, "MIT-SHM", 7))
                {
                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Hiding MIT-SHM extension in reply.\n"
                          << logofs_flush;
                  #endif

                  memcpy((unsigned char *) nextSrc, "NO-MIT-", 7);
                }

                #endif

                #ifdef HIDE_BIG_REQUESTS_EXTENSION

                if (!strncmp((char *) nextSrc, "BIG-REQUESTS", 12))
                {
                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Hiding BIG-REQUESTS extension in reply.\n"
                          << logofs_flush;
                  #endif

                  memcpy((unsigned char *) nextSrc, "NO-BIG-REQUE", 12);
                }

                #endif

                #ifdef HIDE_XKEYBOARD_EXTENSION

                if (!strncmp((char *) nextSrc, "XKEYBOARD", 9))
                {
                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Hiding XKEYBOARD extension in reply.\n"
                          << logofs_flush;
                  #endif

                  memcpy((unsigned char *) nextSrc, "NO-XKEYBO", 9);
                }

                #endif

                #ifdef HIDE_XFree86_Bigfont_EXTENSION

                if (!strncmp((char *) nextSrc, "XFree86-Bigfont", 15))
                {
                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Hiding XFree86-Bigfont extension in reply.\n"
                          << logofs_flush;
                  #endif

                  memcpy((unsigned char *) nextSrc, "NO-XFree86-Bigf", 15);
                }

                #endif

                #ifdef HIDE_SHAPE_EXTENSION

                if (!strncmp((char *) nextSrc, "SHAPE", 5))
                {
                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Hiding SHAPE extension in reply.\n"
                          << logofs_flush;
                  #endif

                  memcpy((unsigned char *) nextSrc, "NO-SH", 5);
                }

                #endif

                //
                // Check if user disabled RENDER extension.
                //

                if (control -> AgentHideRender == 1 &&
                        !strncmp((char *) nextSrc, "RENDER", 6))
                {
                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Hiding RENDER extension in reply.\n"
                          << logofs_flush;
                  #endif

                  memcpy((unsigned char *) nextSrc, "NO-REN", 6);
                }

                for (; length; length--)
                {
                  encodeBuffer.encodeValue((unsigned int) (*nextSrc++), 8);
                }
              }

              priority_++;
            }
            break;
          case X_ListFonts:
            {
              MessageStore *messageStore = serverStore_ ->
                                   getReplyStore(X_ListFonts);

              if (handleEncode(encodeBuffer, serverCache_, messageStore,
                                   inputMessage, inputLength))
              {
                priority_++;

                hit = 1;

                break;
              }

              encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 8);
              unsigned int numFonts = GetUINT(inputMessage + 8, bigEndian_);
              encodeBuffer.encodeValue(numFonts, 16, 6);

              //
              // Use differential encoding at startup
              // (better compression) and ZLIB encoding
              // later (reduced size of cached message).
              //

              if (control -> isStartup() == 1)
              {
                // Differential encoding.
                encodeBuffer.encodeBool(1);

                const unsigned char* nextSrc = inputMessage + 32;
                for (; numFonts; numFonts--)
                {
                  unsigned int length = (unsigned int) (*nextSrc++);
                  encodeBuffer.encodeValue(length, 8);
                  serverCache_ -> getPropertyTextCompressor.reset();
                  for (; length; length--)
                  {
                    serverCache_ -> getPropertyTextCompressor.encodeChar(
                                       *nextSrc++, encodeBuffer);
                  }
                }
              }
              else
              {
                // Plain data compression.
                encodeBuffer.encodeBool(0);

                unsigned int compressedDataSize = 0;
                unsigned char *compressedData   = NULL;

                int compressed = handleCompress(encodeBuffer, requestOpcode, inputMessage,
                                                    inputLength, messageStore -> dataOffset,
                                                        compressedData, compressedDataSize);
                if (compressed < 0)
                {
                  return -1;
                }
                else if (compressed > 0)
                {
                  //
                  // Update size according to result of image compression.
                  //

                  handleUpdate(messageStore, inputLength - messageStore ->
                                   dataOffset, compressedDataSize);
                }
              }

              priority_++;
            }
            break;
          case X_LookupColor:
          case X_AllocNamedColor:
            {
              const unsigned char *nextSrc = inputMessage + 8;
              if (requestOpcode == X_AllocNamedColor)
              {
                encodeBuffer.encodeValue(GetULONG(nextSrc, bigEndian_), 32, 9);
                nextSrc += 4;
              }
              unsigned int count = 3;
              do
              {
                unsigned int exactColor = GetUINT(nextSrc, bigEndian_);
                encodeBuffer.encodeValue(exactColor, 16, 9);
                unsigned int visualColor = GetUINT(nextSrc + 6, bigEndian_) -
                exactColor;
                encodeBuffer.encodeValue(visualColor, 16, 5);
                nextSrc += 2;
              }
              while (--count);

              priority_++;
            }
            break;
          case X_QueryBestSize:
            {
              encodeBuffer.encodeValue(GetUINT(inputMessage + 8, bigEndian_), 16, 8);
              encodeBuffer.encodeValue(GetUINT(inputMessage + 10, bigEndian_), 16, 8);

              priority_++;
            }
            break;
          case X_QueryColors:
            {
              //
              // Use ZLIB encoding at startup (when dif-
              // ferential encoding is quite ineffective)
              // and differential encoding later.
              //

              if (control -> isStartup() == 0)
              {
                // Differential encoding.
                encodeBuffer.encodeBool(1);

                unsigned int numColors = ((inputLength - 32) >> 3);
                const unsigned char *nextSrc = inputMessage + 40;
                unsigned char *nextDest = (unsigned char *) inputMessage + 38;
                for (unsigned int c = 1; c < numColors; c++)
                {
                  for (unsigned int i = 0; i < 6; i++)
                    *nextDest++ = *nextSrc++;
                  nextSrc += 2;
                }
                unsigned int colorsLength = numColors * 6;
                if (serverCache_ -> queryColorsLastReply.compare(colorsLength,
                                                                     inputMessage + 32))
                  encodeBuffer.encodeBool(1);
                else
                {
                  const unsigned char *nextSrc = inputMessage + 32;
                  encodeBuffer.encodeBool(0);
                  encodeBuffer.encodeValue(numColors, 16, 5);
                  for (numColors *= 3; numColors; numColors--)
                  {
                    encodeBuffer.encodeValue(GetUINT(nextSrc, bigEndian_), 16);
                    nextSrc += 2;
                  }
                }
              }
              else
              {
                // Plain data compression.
                encodeBuffer.encodeBool(0);

                // Reply length.
                unsigned int numColors = ((inputLength - 32) >> 3);
                encodeBuffer.encodeValue(numColors, 16, 5);

                unsigned int compressedDataSize = 0;
                unsigned char *compressedData   = NULL;

                int compressed = handleCompress(encodeBuffer, requestOpcode, inputMessage,
                                                    inputLength, 32, compressedData,
                                                        compressedDataSize);
                if (compressed < 0)
                {
                  return -1;
                }
              }

              priority_++;
            }
            break;
          case X_QueryExtension:
            {
              if (requestData[0] == X_QueryExtension)
              {
                //
                // Value in requestData[0] will be nonzero
                // if the request is for an extension that
                // we should hide to the X client.
                //

                if (requestData[1])
                {
                  encodeBuffer.encodeBool(0);
                  encodeBuffer.encodeValue(0, 8);
                }
                else
                {
                  encodeBuffer.encodeBool((unsigned int) inputMessage[8]);
                  encodeBuffer.encodeValue((unsigned int) inputMessage[9], 8);
                }

                encodeBuffer.encodeValue((unsigned int) inputMessage[10], 8);
                encodeBuffer.encodeValue((unsigned int) inputMessage[11], 8);

                if (requestData[2] == X_NXInternalShapeExtension)
                {
                  opcodeStore_ -> shapeExtension = inputMessage[9];

                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Shape extension opcode for FD#" << fd_
                          << " is " << (unsigned int) opcodeStore_ -> shapeExtension
                          << ".\n" << logofs_flush;
                  #endif
                }
                else if (requestData[2] == X_NXInternalRenderExtension)
                {
                  opcodeStore_ -> renderExtension = inputMessage[9];

                  #if defined(INFO) || defined(TEST)
                  *logofs << "handleRead: Render extension opcode for FD#" << fd_
                          << " is " << (unsigned int) opcodeStore_ -> renderExtension
                          << ".\n" << logofs_flush;
                  #endif
                }

                priority_++;
              }
              else
              {
                #if defined(INFO) || defined(TEST)
                *logofs << "handleRead: Received tainted X_QueryExtension reply "
                        << "for request OPCODE#" << requestData[0] << " with "
                        << "sequence " << sequenceNum << ".\n"
                        << logofs_flush;
                #endif

                if (requestData[0] == opcodeStore_ -> getShmemParameters)
                {
                  if (handleShmemInit(encodeBuffer, requestOpcode, requestData[1],
                                          inputMessage, inputLength) < 0)
                  {
                    return -1;
                  }

                  priority_++;
                }

                //
                // Account this data to the original opcode.
                //

                requestOpcode = requestData[0];
              }
            }
            break;
          case X_QueryFont:
            {
              MessageStore *messageStore = serverStore_ ->
                                   getReplyStore(X_QueryFont);

              if (handleEncode(encodeBuffer, serverCache_, messageStore,
                                   inputMessage, inputLength))
              {
                priority_++;

                hit = 1;

                break;
              }

              //
              // Use differential encoding at startup
              // (better compression) and ZLIB encoding
              // later (reduced size of cached message).
              //

              if (control -> isStartup() == 1)
              {
                // Differential encoding.
                encodeBuffer.encodeBool(1);

                unsigned int numProperties = GetUINT(inputMessage + 46, bigEndian_);
                unsigned int numCharInfos = GetULONG(inputMessage + 56, bigEndian_);
                encodeBuffer.encodeValue(numProperties, 16, 8);
                encodeBuffer.encodeValue(numCharInfos, 32, 10);
                handleEncodeCharInfo(inputMessage + 8, encodeBuffer);
                handleEncodeCharInfo(inputMessage + 24, encodeBuffer);
                encodeBuffer.encodeValue(GetUINT(inputMessage + 40, bigEndian_), 16, 9);
                encodeBuffer.encodeValue(GetUINT(inputMessage + 42, bigEndian_), 16, 9);
                encodeBuffer.encodeValue(GetUINT(inputMessage + 44, bigEndian_), 16, 9);
                encodeBuffer.encodeBool((unsigned int) inputMessage[48]);
                encodeBuffer.encodeValue((unsigned int) inputMessage[49], 8);
                encodeBuffer.encodeValue((unsigned int) inputMessage[50], 8);
                encodeBuffer.encodeBool((unsigned int) inputMessage[51]);
                encodeBuffer.encodeValue(GetUINT(inputMessage + 52, bigEndian_), 16, 9);
                encodeBuffer.encodeValue(GetUINT(inputMessage + 54, bigEndian_), 16, 9);
                const unsigned char *nextSrc = inputMessage + 60;
                unsigned int index;

                int end = 0;

                if (ServerCache::queryFontFontCache.lookup(numProperties * 8 +
                                          numCharInfos * 12, nextSrc, index))
                {
                  encodeBuffer.encodeBool(1);
                  encodeBuffer.encodeValue(index, 4);

                  end = 1;
                }

                if (end == 0)
                {
                  encodeBuffer.encodeBool(0);
                  for (; numProperties; numProperties--)
                  {
                    encodeBuffer.encodeValue(GetULONG(nextSrc, bigEndian_), 32, 9);
                    encodeBuffer.encodeValue(GetULONG(nextSrc + 4, bigEndian_), 32, 9);
                    nextSrc += 8;
                  }
                  for (; numCharInfos; numCharInfos--)
                  {
                    handleEncodeCharInfo(nextSrc, encodeBuffer);

                    nextSrc += 12;
                  }
                }
              }
              else
              {
                // Plain data compression.
                encodeBuffer.encodeBool(0);

                // Reply length.
                encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 16);

                unsigned int compressedDataSize = 0;
                unsigned char *compressedData   = NULL;

                int compressed = handleCompress(encodeBuffer, requestOpcode, inputMessage,
                                                    inputLength, messageStore -> dataOffset,
                                                        compressedData, compressedDataSize);
                if (compressed < 0)
                {
                  return -1;
                }
                else if (compressed > 0)
                {
                  //
                  // Update size according to result of image compression.
                  //

                  handleUpdate(messageStore, inputLength - messageStore ->
                                   dataOffset, compressedDataSize);
                }
              }

              priority_++;
            }
            break;
          case X_QueryPointer:
            {
              encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                 29, serverCache_ -> queryPointerRootCache, 9);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_),
                                29, serverCache_ -> queryPointerChildCache, 9);
              unsigned int rootX = GetUINT(inputMessage + 16, bigEndian_);
              unsigned int rootY = GetUINT(inputMessage + 18, bigEndian_);
              unsigned int eventX = GetUINT(inputMessage + 20, bigEndian_);
              unsigned int eventY = GetUINT(inputMessage + 22, bigEndian_);
              eventX -= rootX;
              eventY -= rootY;
              encodeBuffer.encodeCachedValue(
                             rootX - serverCache_ -> motionNotifyLastRootX, 16,
                                    serverCache_ -> motionNotifyRootXCache, 8);
              serverCache_ -> motionNotifyLastRootX = rootX;
              encodeBuffer.encodeCachedValue(
                             rootY - serverCache_ -> motionNotifyLastRootY, 16,
                                    serverCache_ -> motionNotifyRootYCache, 8);
              serverCache_ -> motionNotifyLastRootY = rootY;
              encodeBuffer.encodeCachedValue(eventX, 16,
                                   serverCache_ -> motionNotifyEventXCache, 8);
              encodeBuffer.encodeCachedValue(eventY, 16,
                                   serverCache_ -> motionNotifyEventYCache, 8);
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 24, bigEndian_),
                                   16, serverCache_ -> motionNotifyStateCache);
              priority_++;
            }
            break;
          case X_QueryTree:
            {
              //
              // TODO: This was very inefficient. In practice
              // it just copied data on the output. Now obtains
              // an average 7:1 compression. Could be cached.
              //

              if (control -> isProtoStep3() == 1)
              {
                unsigned int children = GetUINT(inputMessage + 16, bigEndian_);

                encodeBuffer.encodeValue(children, 16, 8);

                encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_), 29,
                                   serverCache_ -> queryTreeWindowCache);

                encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_), 29,
                                   serverCache_ -> queryTreeWindowCache);

                const unsigned char *next = inputMessage + 32;

                for (unsigned int i = 0; i < children; i++)
                {
                  encodeBuffer.encodeCachedValue(GetULONG(next + (i * 4), bigEndian_), 29,
                                     serverCache_ -> queryTreeWindowCache);
                }
              }
              else
              {
                encodeBuffer.encodeValue(inputMessage[1], 8);
                encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32);
                for (unsigned int i = 8; i < inputLength; i++)
                {
                  encodeBuffer.encodeValue((unsigned int) inputMessage[i], 8);
                }
              }

              priority_++;
            }
            break;
          case X_TranslateCoords:
            {
              encodeBuffer.encodeBool((unsigned int) inputMessage[1]);
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                             29, serverCache_ -> translateCoordsChildCache, 9);
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 12, bigEndian_),
                                 16, serverCache_ -> translateCoordsXCache, 8);
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 14, bigEndian_),
                                 16, serverCache_ -> translateCoordsYCache, 8);
              priority_++;
            }
            break;
          case X_GetImage:
            {
              MessageStore *messageStore = serverStore_ ->
                                   getReplyStore(X_GetImage);

              if (handleEncode(encodeBuffer, serverCache_, messageStore,
                                   inputMessage, inputLength))
              {
                priority_++;

                hit = 1;

                break;
              }

              // Depth.
              encodeBuffer.encodeCachedValue(inputMessage[1], 8,
                                 serverCache_ -> depthCache);
              // Reply length.
              encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 9);

              // Visual.
              encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_), 29,
                                 serverCache_ -> visualCache);

              unsigned int compressedDataSize = 0;
              unsigned char *compressedData   = NULL;

              int compressed = handleCompress(encodeBuffer, requestOpcode, inputMessage,
                                                  inputLength, messageStore -> dataOffset,
                                                      compressedData, compressedDataSize);
              if (compressed < 0)
              {
                return -1;
              }
              else if (compressed > 0)
              {
                //
                // Update size according to result of image compression.
                //

                handleUpdate(messageStore, inputLength - messageStore ->
                                 dataOffset, compressedDataSize);
              }

              priority_++;
            }
            break;
            case X_GetPointerMapping:
            {
              encodeBuffer.encodeValue(inputMessage[1], 8, 4);
              encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 4);
              for (unsigned int i = 32; i < inputLength; i++)
                encodeBuffer.encodeValue((unsigned int) inputMessage[i], 8, 4);

              priority_++;
            }
            break;
            case X_GetKeyboardControl:
            {
              encodeBuffer.encodeValue(inputMessage[1], 8, 2);
              encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 8);
              for (unsigned int i = 8; i < inputLength; i++)
                encodeBuffer.encodeValue((unsigned int) inputMessage[i], 8, 4);

              priority_++;
            }
            break;
          default:
            {
              #ifdef PANIC
              *logofs << "ServerChannel: PANIC! No matching request with OPCODE#"
                      << requestOpcode << " for reply with sequence number "
                      << requestSequenceNum << ".\n" << logofs_flush;
              #endif

              cerr << "Error" << ": No matching request with OPCODE#"
                   << requestOpcode << " for reply with sequence number "
                   << requestSequenceNum << ".\n";

              return -1;
            }
          }

          bits = encodeBuffer.getBits();

          #if defined(TEST) || defined(OPCODES)

          const char *cacheString = (hit ? "cached " : "");

          *logofs << "handleRead: Handled " << cacheString << "reply to OPCODE#"
                  << (unsigned int) requestOpcode << " (" << DumpOpcode(requestOpcode)
                  << ") for FD#" << fd_ << " sequence " << serverSequence_
                  << ". " << inputLength << " bytes in, " << bits << " bits ("
                  << ((float) bits) / 8 << " bytes) out.\n" << logofs_flush;

          #endif

        } // End of if (sequenceQueue_.peek(requestSequenceNum, requestOpcode) && ...
        else
        {
          //
          // We didn't push the request opcode.
          // Check if fast encoding is required.
          //

          requestOpcode = X_Reply;

          if (control -> LocalDeltaCompression == 0)
          {
            int result = handleFastReadReply(encodeBuffer, requestOpcode,
                                                 inputMessage, inputLength);
            if (result < 0)
            {
              return -1;
            }
            else if (result > 0)
            {
              if (pending_ == 0 && control -> isTimeToYield(yield_in_channel) == 1)
              {
                #if defined(INFO) || defined(TEST)
                *logofs << "handleRead: Giving up because of "
                        << control -> getBytesInARow() << " bytes to write after "
                        << control -> getTimeInARow() << " Ms for FD#" << fd_
                        << ".\n" << logofs_flush;
                #endif

                yield = 1;
              }

              continue;
            }
          }

          //
          // Group all replies whose opcode was not
          // pushed in sequence number queue under
          // the category 'generic reply'.
          //

          #ifdef DEBUG
          *logofs << "handleRead: Identified generic reply.\n"
                  << logofs_flush;
          #endif

          MessageStore *messageStore = serverStore_ ->
                               getReplyStore(X_NXInternalGenericReply);

          hit = handleEncode(encodeBuffer, serverCache_, messageStore,
                                 requestOpcode, inputMessage, inputLength);

          priority_++;

          bits = encodeBuffer.getBits();

          #if defined(TEST) || defined(OPCODES)

          const char *cacheString = (hit ? "cached " : "");

          *logofs << "handleRead: Handled " << cacheString << "generic reply "
                  << "OPCODE#" << X_NXInternalGenericReply << " for FD#" << fd_
                  << " sequence " << serverSequence_ << ". " << inputLength
                  << " bytes in, " << bits << " bits (" << ((float) bits) / 8
                  << " bytes) out.\n" << logofs_flush;

          #endif
        }

        if (control -> CollectStatistics)
        {
          if (hit)
          {
            statistics -> addCachedReply(requestOpcode);
          }

          statistics -> addReplyBits(requestOpcode, inputLength << 3, bits);
        }

        control -> addOutputInARow(inputLength);

        control -> addBitsInARow(bits);

      } // End of if (inputMessage[0] == 1) ...
      else
      {
        //
        // Event or error.
        //

        unsigned char inputOpcode = *inputMessage;

        unsigned int inputSequence = GetUINT(inputMessage + 2, bigEndian_);

        //
        // Check if this is an event which we can discard.
        //

        if ((inputOpcode == Expose && enableExpose_ == 0) ||
                (inputOpcode == GraphicsExpose && enableGraphicsExpose_ == 0) ||
                    (inputOpcode == NoExpose && enableNoExpose_ == 0))
        {
          continue;
        }
        else if (shmemState_ != NULL && shmemState_ -> enabled == 1 &&
                     inputOpcode == shmemState_ -> event)
        {
          checkShmemEvent(inputOpcode, inputSequence, inputMessage);

          continue;
        }
        else if (inputOpcode == MotionNotify)
        {
          //
          // Save the motion event and send when:
          //
          // - Another event or error is received.
          // - Motion timeout is elapsed.
          //
          // If a previous motion event is found, replace it
          // with the new one. Don't reset the timeout, though,
          // so we still have a motion every MotionTimeout ms.
          //

          if (lastMotion_[0] == '\0')
          {
            lastMotionTs_ = getTimestamp();
          }

          memcpy(lastMotion_, inputMessage, 32);

          #if defined(INFO) || defined(TEST)
          *logofs << "handleRead: Saved suppressed motion event for FD#"
                  << fd_ << ".\n" << logofs_flush;
          #endif

          continue;
        }
        else if (inputOpcode == X_Error)
        {
          //
          // Check if this is an error that matches a sequence
          // number for which we were expecting a reply.
          //

          unsigned short int dummySequenceNum;
          unsigned char dummyOpcode;

          if (sequenceQueue_.peek(dummySequenceNum, dummyOpcode) &&
                  ((unsigned int) dummySequenceNum == inputSequence))
          {
            sequenceQueue_.pop(dummySequenceNum, dummyOpcode);
          }

          //
          // Check if error is due to an image commit
          // generated at the end of a split.
          //

          int suppress = 0;

          for (int i = 0; i < MAX_COMMIT_SEQUENCE_QUEUE; i++)
          {
            #ifdef DEBUG
            *logofs << "handleRead: Checking committed image's sequence number "
                    << commitSequenceQueue_[i] << ".\n" << logofs_flush;
            #endif

            if (commitSequenceQueue_[i] == serverSequence_)
            {
              #ifdef WARNING
              *logofs << "handleRead: WARNING! Suppressing error on OPCODE#"
                      << (unsigned int) opcodeStore_ -> commitSplit << " for FD#"
                      << fd_ << " with sequence " << inputSequence << " at position "
                      << i << ".\n" << logofs_flush;
              #endif

              cerr << "Warning" << ": Suppressing error on commit split "
                   << "with sequence " << inputSequence << " at position "
                   << i << ".\n";

              suppress = 1;

              break;
            }
          }

          if (suppress == 1)
          {
            continue;
          }

          //
          // Check if it's an error generated by a request
          // concerning shared memory support.
          //

          if (shmemState_ != NULL && (shmemState_ -> sequence ==
                  inputSequence || (shmemState_ -> enabled == 1 &&
                      (shmemState_ -> opcode == *(inputMessage + 10) ||
                           shmemState_ -> error == *(inputMessage + 1)))))
          {
            if (checkShmemError(*(inputMessage + 1), inputSequence, inputMessage) > 0)
            {
              continue;
            }
          }
        }

        //
        // Check if user pressed the CTRL+ALT+SHIFT+ESC key
        // sequence because was unable to kill the session
        // through the normal procedure.
        //

        if (inputOpcode == KeyPress && *(inputMessage + 1) == 0x09 &&
                (GetUINT(inputMessage + 28, bigEndian_) & 0x0d == 0x0d))
        {
          #ifdef PANIC
          *logofs << "handleRead: PANIC! Received sequence CTRL+ALT+SHIFT+ESC "
                  << "for FD#"<< fd_ << ". Showing abort dialog.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Received sequence CTRL+ALT+SHIFT+ESC. "
               << "Showing abort dialog.\n";

          HandleAlert(CLOSE_UNRESPONSIVE_X_SERVER_ALERT);
        }

        //
        // We are going to handle an event or error
        // that's not a mouse motion. Prepend any
        // saved motion to it.
        //

        if (lastMotion_[0] != '\0')
        {
          if (handleMotion(encodeBuffer, 1) < 0)
          {
            #ifdef PANIC
            *logofs << "handleRead: PANIC! Can't encode motion event for FD#"
                    << fd_ << ".\n" << logofs_flush;
            #endif

            cerr << "Error" << ": Can't encode motion event for FD#"
                 << fd_ << ".\n";

            return -1;
          }
        }

        //
        // Encode opcode and difference between
        // current sequence and the last one.
        //

        encodeBuffer.encodeOpcodeValue(inputOpcode, serverCache_ -> opcodeCache);

        serverOpcode_ = inputOpcode;

        unsigned int sequenceDiff = inputSequence - serverSequence_;

        serverSequence_ = inputSequence;

        #ifdef DEBUG
        *logofs << "handleRead: Last server sequence number for FD#" 
                << fd_ << " is " << serverSequence_ << " with "
                << "difference " << sequenceDiff << ".\n"
                << logofs_flush;
        #endif

        if (serverSequenceReset_ == 1)
        {
          encodeBuffer.encodeValue(serverSequence_, 32, 16);

          #if defined(INFO) || defined(TEST)
          *logofs << "handleRead: Reset last server sequence number to "
                  << serverSequence_ << " for FD#" 
                  << fd_ << ".\n" << logofs_flush;
          #endif

          serverSequenceReset_ = 0;
        }
        else
        {
          encodeBuffer.encodeCachedValue(sequenceDiff, 16,
                             serverCache_ -> eventSequenceCache, 7);
        }

        //
        // If differential compression is disabled
        // then use the most simple encoding.
        //

        if (control -> LocalDeltaCompression == 0)
        {
          int result = handleFastReadEvent(encodeBuffer, inputOpcode,
                                               inputMessage, inputLength);
          if (result < 0)
          {
            return -1;
          }
          else if (result > 0)
          {
            if (pending_ == 0 && control -> isTimeToYield(yield_in_channel) == 1)
            {
              #if defined(INFO) || defined(TEST)
              *logofs << "handleRead: Giving up because of "
                      << control -> getBytesInARow() << " bytes to write after "
                      << control -> getTimeInARow() << " Ms for FD#" << fd_
                      << ".\n" << logofs_flush;
              #endif

              yield = 1;
            }

            continue;
          }
        }

        switch (inputOpcode)
        {
        case X_Error:
          {
            unsigned char errorCode = *(inputMessage + 1);

            encodeBuffer.encodeCachedValue(errorCode, 8,
                               serverCache_ -> errorCodeCache);

            if (errorCode != 11 && errorCode != 8 &&
                    errorCode != 15 && errorCode != 1)
            {
              encodeBuffer.encodeValue(GetULONG(inputMessage + 4, bigEndian_), 32, 16);
            }

            if (errorCode >= 18)
            {
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 8, bigEndian_), 16,
                                 serverCache_ -> errorMinorCache);
            }

            encodeBuffer.encodeCachedValue(inputMessage[10], 8,
                               serverCache_ -> errorMajorCache);

            if (errorCode >= 18)
            {
              const unsigned char *nextSrc = inputMessage + 11;
              for (unsigned int i = 11; i < 32; i++)
                encodeBuffer.encodeValue(*nextSrc++, 8);
            }
          }
          break;
        case ButtonPress:
        case ButtonRelease:
        case KeyPress:
        case KeyRelease:
        case MotionNotify:
        case EnterNotify:
        case LeaveNotify:
          {
            unsigned char detail = inputMessage[1];
            if (*inputMessage == MotionNotify)
              encodeBuffer.encodeBool((unsigned int) detail);
            else if ((*inputMessage == EnterNotify) || (*inputMessage == LeaveNotify))
              encodeBuffer.encodeValue((unsigned int) detail, 3);
            else if (*inputMessage == KeyRelease)
            {
              if (detail == serverCache_ -> keyPressLastKey)
                encodeBuffer.encodeBool(1);
              else
              {
                encodeBuffer.encodeBool(0);
                encodeBuffer.encodeValue((unsigned int) detail, 8);
              }
            }
            else if ((*inputMessage == ButtonPress) || (*inputMessage == ButtonRelease))
              encodeBuffer.encodeCachedValue(detail, 8,
                                             serverCache_ -> buttonCache);
            else
              encodeBuffer.encodeValue((unsigned int) detail, 8);
            unsigned int timestamp = GetULONG(inputMessage + 4, bigEndian_);
            unsigned int timestampDiff =
            timestamp - serverCache_ -> lastTimestamp;
            serverCache_ -> lastTimestamp = timestamp;
            encodeBuffer.encodeCachedValue(timestampDiff, 32,
                                serverCache_ -> motionNotifyTimestampCache, 9);
            int skipRest = 0;
            if (*inputMessage == KeyRelease)
            {
              skipRest = 1;
              for (unsigned int i = 8; i < 31; i++)
              {
                if (inputMessage[i] != serverCache_ -> keyPressCache[i - 8])
                {
                  skipRest = 0;
                  break;
                }
              }
              encodeBuffer.encodeBool(skipRest);
            }

            if (!skipRest)
            {
              const unsigned char *nextSrc = inputMessage + 8;
              for (unsigned int i = 0; i < 3; i++)
              {
                encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 29,
                                   *serverCache_ -> motionNotifyWindowCache[i], 6);
                nextSrc += 4;
              }
              unsigned int rootX  = GetUINT(inputMessage + 20, bigEndian_);
              unsigned int rootY  = GetUINT(inputMessage + 22, bigEndian_);
              unsigned int eventX = GetUINT(inputMessage + 24, bigEndian_);
              unsigned int eventY = GetUINT(inputMessage + 26, bigEndian_);
              eventX -= rootX;
              eventY -= rootY;
              encodeBuffer.encodeCachedValue(rootX -
                    serverCache_ -> motionNotifyLastRootX, 16,
                    serverCache_ -> motionNotifyRootXCache, 6);
              serverCache_ -> motionNotifyLastRootX = rootX;
              encodeBuffer.encodeCachedValue(rootY -
                    serverCache_ -> motionNotifyLastRootY, 16,
                    serverCache_ -> motionNotifyRootYCache, 6);
              serverCache_ -> motionNotifyLastRootY = rootY;
              encodeBuffer.encodeCachedValue(eventX, 16,
                                   serverCache_ -> motionNotifyEventXCache, 6);
              encodeBuffer.encodeCachedValue(eventY, 16,
                                   serverCache_ -> motionNotifyEventYCache, 6);
              encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 28, bigEndian_),
                                   16, serverCache_ -> motionNotifyStateCache);
              if ((*inputMessage == EnterNotify) || (*inputMessage == LeaveNotify))
                encodeBuffer.encodeValue((unsigned int) inputMessage[30], 2);
              else
                encodeBuffer.encodeBool((unsigned int) inputMessage[30]);
              if ((*inputMessage == EnterNotify) || (*inputMessage == LeaveNotify))
                encodeBuffer.encodeValue((unsigned int) inputMessage[31], 2);
              else if (*inputMessage == KeyPress)
              {
                serverCache_ -> keyPressLastKey = detail;
                for (unsigned int i = 8; i < 31; i++)
                {
                  serverCache_ -> keyPressCache[i - 8] = inputMessage[i];
                }
              }
            }
          }
          break;
        case ColormapNotify:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                             29, serverCache_ -> colormapNotifyWindowCache, 8);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                           29, serverCache_ -> colormapNotifyColormapCache, 8);
            encodeBuffer.encodeBool((unsigned int) inputMessage[12]);
            encodeBuffer.encodeBool((unsigned int) inputMessage[13]);
          }
          break;
        case ConfigureNotify:
          {
            const unsigned char *nextSrc = inputMessage + 4;
            for (unsigned int i = 0; i < 3; i++)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 29,
                                 *serverCache_ -> configureNotifyWindowCache[i], 9);
              nextSrc += 4;
            }
            for (unsigned int j = 0; j < 5; j++)
            {
              encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                 *serverCache_ -> configureNotifyGeomCache[j], 8);
              nextSrc += 2;
            }
            encodeBuffer.encodeBool(*nextSrc);
          }
          break;
        case CreateNotify:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                        29, serverCache_ -> createNotifyWindowCache, 9);
            unsigned int window = GetULONG(inputMessage + 8, bigEndian_);
            encodeBuffer.encodeValue(window -
                             serverCache_ -> createNotifyLastWindow, 29, 5);
            serverCache_ -> createNotifyLastWindow = window;
            const unsigned char* nextSrc = inputMessage + 12;
            for (unsigned int i = 0; i < 5; i++)
            {
              encodeBuffer.encodeValue(GetUINT(nextSrc, bigEndian_), 16, 9);
              nextSrc += 2;
            }
            encodeBuffer.encodeBool(*nextSrc);
          }
          break;
        case Expose:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_), 29,
                               serverCache_ -> exposeWindowCache, 9);
            const unsigned char *nextSrc = inputMessage + 8;
            for (unsigned int i = 0; i < 5; i++)
            {
              encodeBuffer.encodeCachedValue(GetUINT(nextSrc, bigEndian_), 16,
                                 *serverCache_ -> exposeGeomCache[i], 6);
              nextSrc += 2;
            }
          }
          break;
        case FocusIn:
        case FocusOut:
          {
            encodeBuffer.encodeValue((unsigned int) inputMessage[1], 3);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                                    29, serverCache_ -> focusInWindowCache, 9);
            encodeBuffer.encodeValue((unsigned int) inputMessage[8], 2);
          }
          break;
        case KeymapNotify:
          {
          if (ServerCache::lastKeymap.compare(31, inputMessage + 1))
              encodeBuffer.encodeBool(1);
            else
            {
              encodeBuffer.encodeBool(0);
              const unsigned char *nextSrc = inputMessage + 1;
              for (unsigned int i = 1; i < 32; i++)
                encodeBuffer.encodeValue((unsigned int) *nextSrc++, 8);
            }
          }
          break;
        case MapNotify:
        case UnmapNotify:
        case DestroyNotify:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                                   29, serverCache_ -> mapNotifyEventCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                                  29, serverCache_ -> mapNotifyWindowCache, 9);
            if ((*inputMessage == MapNotify) || (*inputMessage == UnmapNotify))
              encodeBuffer.encodeBool((unsigned int) inputMessage[12]);
          }
          break;
        case NoExpose:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                                 29, serverCache_ -> noExposeDrawableCache, 9);
            encodeBuffer.encodeCachedValue(GetUINT(inputMessage + 8, bigEndian_), 16,
                                           serverCache_ -> noExposeMinorCache);
            encodeBuffer.encodeCachedValue(inputMessage[10], 8,
                                           serverCache_ -> noExposeMajorCache);
          }
          break;
        case PropertyNotify:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                             29, serverCache_ -> propertyNotifyWindowCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                               29, serverCache_ -> propertyNotifyAtomCache, 9);
            unsigned int timestamp = GetULONG(inputMessage + 12, bigEndian_);
            unsigned int timestampDiff =
            timestamp - serverCache_ -> lastTimestamp;
            serverCache_ -> lastTimestamp = timestamp;
            encodeBuffer.encodeValue(timestampDiff, 32, 9);
            encodeBuffer.encodeBool((unsigned int) inputMessage[16]);
          }
          break;
        case ReparentNotify:
          {
            const unsigned char* nextSrc = inputMessage + 4;
            for (unsigned int i = 0; i < 3; i++)
            {
              encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_),
                     29, serverCache_ -> reparentNotifyWindowCache, 9);
              nextSrc += 4;
            }
            encodeBuffer.encodeValue(GetUINT(nextSrc, bigEndian_), 16, 6);
            encodeBuffer.encodeValue(GetUINT(nextSrc + 2, bigEndian_), 16, 6);
            encodeBuffer.encodeBool((unsigned int)inputMessage[20]);
          }
          break;
        case SelectionClear:
          {
            unsigned int timestamp = GetULONG(inputMessage + 4, bigEndian_);
            unsigned int timestampDiff = timestamp - serverCache_ -> lastTimestamp;
            serverCache_ -> lastTimestamp = timestamp;
            encodeBuffer.encodeValue(timestampDiff, 32, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                           29, serverCache_ -> selectionClearWindowCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_),
                           29, serverCache_ -> selectionClearAtomCache, 9);
          }
          break;
        case SelectionRequest:
          {
            unsigned int timestamp = GetULONG(inputMessage + 4, bigEndian_);
            unsigned int timestampDiff = timestamp - serverCache_ -> lastTimestamp;
            serverCache_ -> lastTimestamp = timestamp;
            encodeBuffer.encodeValue(timestampDiff, 32, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 8, bigEndian_),
                           29, serverCache_ -> selectionClearWindowCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 12, bigEndian_),
                           29, serverCache_ -> selectionClearWindowCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 16, bigEndian_),
                           29, serverCache_ -> selectionClearAtomCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 20, bigEndian_),
                           29, serverCache_ -> selectionClearAtomCache, 9);
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 24, bigEndian_),
                           29, serverCache_ -> selectionClearAtomCache, 9);
          }
          break;
        case VisibilityNotify:
          {
            encodeBuffer.encodeCachedValue(GetULONG(inputMessage + 4, bigEndian_),
                           29, serverCache_ -> visibilityNotifyWindowCache, 9);
            encodeBuffer.encodeValue((unsigned int) inputMessage[8], 2);
          }
          break;
        default:
          {
            #ifdef TEST
            *logofs << "handleRead: Using generic event compression "
                    << "for OPCODE#" << (unsigned int) inputOpcode
                    << ".\n" << logofs_flush;
            #endif

            if (control -> isProtoStep3() == 1)
            {
              encodeBuffer.encodeCachedValue(*(inputMessage + 1), 8,
                           serverCache_ -> genericEventCharCache);

              for (unsigned int i = 0; i < 14; i++)
              {
                encodeBuffer.encodeCachedValue(GetUINT(inputMessage + i * 2 + 4, bigEndian_),
                                   16, *serverCache_ -> genericEventIntCache[i]);
              }
            }
            else
            {
              encodeBuffer.encodeValue(inputMessage[1], 8);
              for (unsigned int i = 4; i < inputLength; i++)
              {
                encodeBuffer.encodeValue((unsigned int) inputMessage[i], 8);
              }
            }
          }
        }

        //
        // Set priority in the following cases.
        // Including KeyPress and KeyRelease we
        // greatly increase traffic because of
        // auto-repeat, managed by X server.
        //

        switch (inputOpcode)
        {
          case X_Error:
          case ButtonPress:
          case ButtonRelease:
          {
            priority_++;
          }
        }

        int bits = encodeBuffer.getBits();

        #if defined(TEST) || defined(OPCODES)

        if (*inputMessage == X_Error)
        {
          unsigned char code = *(inputMessage + 1);

          *logofs << "handleRead: Handled error ERR_CODE#"
                  << (unsigned int) code << " for FD#" << fd_;

          *logofs << " RES_ID#" << GetULONG(inputMessage + 4, bigEndian_);

          *logofs << " MIN_OP#" << GetUINT(inputMessage + 8, bigEndian_);

          *logofs << " MAJ_OP#" << (unsigned int) *(inputMessage + 10);

          *logofs << " sequence " << inputSequence << ". " << inputLength
                  << " bytes in, " << bits << " bits (" << ((float) bits) / 8
                  << " bytes) out.\n" << logofs_flush;
        }
        else
        {
          *logofs << "handleRead: Handled event OPCODE#"
                  << (unsigned int) *inputMessage << " for FD#" << fd_
                  << " sequence " << inputSequence << ". " << inputLength
                  << " bytes in, " << bits << " bits (" << ((float) bits) / 8
                  << " bytes) out.\n" << logofs_flush;
        }

        #endif

        if (control -> CollectStatistics)
        {
          statistics -> addEventBits(*inputMessage, inputLength << 3, bits);
        }

        control -> addOutputInARow(inputLength);

        control -> addBitsInARow(bits);

      } // End of if (inputMessage[0] == X_Reply) ... else ...

    } // End of if (firstReply_) ... else ...

    //
    // If we consumed too many ticks or produced
    // enough data, then leave remaining messages
    // in the read buffer.
    //

    if (pending_ == 0 && control -> isTimeToYield(yield_in_channel) == 1)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up because of "
              << control -> getBytesInARow() << " bytes to write after "
              << control -> getTimeInARow() << " Ms for FD#" << fd_
              << ".\n" << logofs_flush;
      #endif

      yield = 1;
    }

  } // End of while (yield == 0 && (inputMessage = readBuffer_.getMessage(inputLength)) != 0) ...

  //
  // Set pending flag if we quit the loop before
  // consuming all messages and bytes left in the
  // read buffer are considered enough to make a
  // complete message.
  //

  if (yield == 1)
  {
    pending_ = readBuffer_.checkMessage();
  }
  else
  {
    //
    // Give up if we just needed to handle
    // messages left in the read buffer
    //

    if (pending_ > 0)
    {
      pending_ = 0;

      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up because of no more "
              << "pending messages for FD#" << fd_
              << ".\n" << logofs_flush;
      #endif

      yield = 1;
    }

    //
    // Don't perform more reads if there
    // is prioritized data.
    //

    else if (priority_ > 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up with " << priority_
              << " prioritized messages " << "for FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      yield = 1;
    }

    //
    // Check if we read enough data.
    //

    else if (reads >= control -> ServerReadsInARowLimit ||
                loops >= control -> ServerMessagesInARowLimit)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up because of limit of "
              << control -> ServerReadsInARowLimit << " reads and "
              << control -> ServerMessagesInARowLimit
              << " messages for FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      yield = 1;
    }

    //
    // If it's not time to yield, check if
    // we can read more bytes from socket.
    //

    else if (transport_ -> readable() <= 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleRead: Giving up with no bytes "
              << "available on FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      yield = 1;
    }

    #if defined(INFO) || defined(TEST)

    if (yield == 1)
    {
      *logofs << "handleRead: Performed " << reads << " reads "
              << "and handled " << loops << " messages producing "
              << control -> getBytesInARow() << " bytes in "
              << control -> getTimeInARow() << " Ms for FD#"
              << fd_ << ".\n" << logofs_flush;
    }
    else
    {
      *logofs << "handleRead: Still reading messages with "
              << transport_ -> readable() << " bytes available "
              << "for FD#" << fd_ << ".\n" << logofs_flush;
    }

    #endif

  } // End of if (yield == 1) ... else ...

  } // End of while (yield == 0) ...

  //
  // Flush motion event if a null
  // timeout was provided.
  //

  if (lastMotion_[0] != '\0')
  {
    if (handleMotion(encodeBuffer, 0) < 0)
    {
      #ifdef PANIC
      *logofs << "handleRead: PANIC! Can't encode motion event for FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't encode motion event for FD#"
           << fd_ << ".\n";

      return -1;
    }
  }

  //
  // Reset the read buffer.
  //

  readBuffer_.partialReset();

  return 1;
}

//
// End of handleRead().
//

//
// Beginning of handleWrite().
//

int ServerChannel::handleWrite(const unsigned char *message, unsigned int length)
{
  #ifdef TEST
  *logofs << "handleWrite: Called for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Create the buffer from which to 
  // decode messages.
  //

  DecodeBuffer decodeBuffer(message, length);

  #if defined(INFO) || defined(TEST)
  *logofs << "handleWrite: Cumulative length of decode "
          << "buffer is " << length << " bytes.\n"
          << logofs_flush;
  #endif

  if (firstRequest_)
  {
    unsigned char *outputMessage = writeBuffer_.addMessage(length);
    unsigned char *nextDest = outputMessage;
    unsigned int  nextByte;

    for (unsigned int i = 0; i < length; i++)
    {
      decodeBuffer.decodeValue(nextByte, 8);

      *nextDest++ = (unsigned char) nextByte;
    }

    if (*outputMessage == 0x42)
    {
      setBigEndian(1);
    }
    else
    {
      setBigEndian(0);
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "handleWrite: First request detected.\n" << logofs_flush;
    #endif

    firstRequest_ = 0;
  }
  else
  {
    unsigned char outputOpcode;

    unsigned char *outputMessage;
    unsigned int outputLength;

    //
    // Set if we should flush immediately
    // any data accumulated in the write
    // buffer.
    //

    unsigned int outputFlush = 0;

    //
    // Set when message is found in cache.
    //

    int hit;

    while (decodeBuffer.decodeOpcodeValue(outputOpcode, clientCache_ -> opcodeCache, 1))
    {
      hit = 0;

      clientOpcode_ = outputOpcode;

      //
      // Splits are sent by client proxy outside the
      // normal read loop. As we 'insert' splits in
      // the real client-server X protocol, we must
      // avoid to increment the sequence number or
      // our clients would get confused.
      //

      if (outputOpcode != opcodeStore_ -> split)
      {
        if (clientSequenceReset_ == 1)
        {
          unsigned int sequenceNum;

          decodeBuffer.decodeValue(sequenceNum, 32, 16);

          #if defined(INFO) || defined(TEST)
          *logofs << "handleWrite: Synchronizing sequence "
                  << clientSequence_ << " with remote sequence "
                  << sequenceNum << " for FD#" << fd_ << ".\n"
                  << logofs_flush;
          #endif

          if (handleResetSequence(sequenceNum) < 0)
          {
            return -1;
          }

          clientSequenceReset_ = 0;
        }
        else
        {
          clientSequence_++;
          clientSequence_ &= 0xffff;
        }

        #ifdef DEBUG
        *logofs << "handleWrite: Last client sequence number for FD#" 
                << fd_ << " is " << clientSequence_ << ".\n"
                << logofs_flush;
        #endif
      }
      else
      {
        //
        // It's a split, not a normal
        // burst of proxy data.
        //

        handleUnsplit(decodeBuffer);

        continue;
      }

      //
      // Is differential encoding disabled?
      //

      if (control -> RemoteDeltaCompression == 0)
      {
        int result = handleFastWriteRequest(decodeBuffer, outputOpcode,
                                                outputMessage, outputLength);
        if (result < 0)
        {
          return -1;
        }
        else if (result > 0)
        {
          continue;
        }
      }

      //
      // General-purpose temp variables for
      // decoding ints and chars.
      //

      unsigned int value;
      unsigned char cValue;

      #ifdef DEBUG
      *logofs << "handleWrite: Going to handle request OPCODE#"
              << (unsigned int) outputOpcode << " (" << DumpOpcode(outputOpcode)
              << ") for FD#" << fd_ << " sequence " << clientSequence_
              << ".\n" << logofs_flush;
      #endif

      switch (outputOpcode)
      {
      case X_AllocColor:
        {
          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> colormapCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned char *nextDest = outputMessage + 8;
          unsigned int colorData[3];

          for (unsigned int i = 0; i < 3; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *(clientCache_ -> allocColorRGBCache[i]), 4);
            PutUINT(value, nextDest, bigEndian_);
            colorData[i] = value;
            nextDest += 2;
          }

          sequenceQueue_.push(clientSequence_, outputOpcode,
                                  colorData[0], colorData[1], colorData[2]);

          //
          // Don't flush in case of RDP sessions.
          //

          if (control -> SessionMode != SESSION_RDP)
          {
            outputFlush++;
          }
        }
        break;
      case X_ReparentWindow:
        {
          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeValue(value, 16, 11);
          PutUINT(value, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeValue(value, 16, 11);
          PutUINT(value, outputMessage + 14, bigEndian_);
        }
        break;
      case X_ChangeProperty:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ChangeProperty);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          unsigned char format;
          decodeBuffer.decodeCachedValue(format, 8,
                             clientCache_ -> changePropertyFormatCache);
          unsigned int dataLength;
          decodeBuffer.decodeValue(dataLength, 32, 6);
          outputLength = 24 + RoundUp4(dataLength * (format >> 3));
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 2);
          outputMessage[1] = (unsigned char) value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> changePropertyPropertyCache, 9);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> changePropertyTypeCache, 9);
          PutULONG(value, outputMessage + 12, bigEndian_);
          outputMessage[16] = format;
          PutULONG(dataLength, outputMessage + 20, bigEndian_);
          unsigned char *nextDest = outputMessage + 24;

          if (format == 8)
          {
            clientCache_ -> changePropertyTextCompressor.reset();

            for (unsigned int i = 0; i < dataLength; i++)
            {
              *nextDest++ = clientCache_ -> changePropertyTextCompressor.
                                                  decodeChar(decodeBuffer);
            }
          }
          else if (format == 32)
          {
            for (unsigned int i = 0; i < dataLength; i++)
            {
              decodeBuffer.decodeCachedValue(value, 32,
                                 clientCache_ -> changePropertyData32Cache);

              PutULONG(value, nextDest, bigEndian_);

              nextDest += 4;
            }
          }
          else
          {
            for (unsigned int i = 0; i < dataLength; i++)
            {
              decodeBuffer.decodeValue(value, 16);

              PutUINT(value, nextDest, bigEndian_);

              nextDest += 2;
            }
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_SendEvent:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_SendEvent);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 44;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeBool(value);
          *(outputMessage + 1) = value;
          decodeBuffer.decodeBool(value);
          if (value)
          {
            decodeBuffer.decodeBool(value);
          }
          else
          {
            decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          }
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 32,
                       clientCache_ -> sendEventMaskCache, 9);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(*(outputMessage + 12), 8,
                       clientCache_ -> sendEventCodeCache);
          decodeBuffer.decodeCachedValue(*(outputMessage + 13), 8,
                       clientCache_ -> sendEventByteDataCache);
          decodeBuffer.decodeValue(value, 16, 4);
          clientCache_ -> sendEventLastSequence += value;
          clientCache_ -> sendEventLastSequence &= 0xffff;
          PutUINT(clientCache_ -> sendEventLastSequence, outputMessage + 14, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 32,
                       clientCache_ -> sendEventIntDataCache);
          PutULONG(value, outputMessage + 16, bigEndian_);

          for (unsigned int i = 20; i < 44; i++)
          {
            decodeBuffer.decodeCachedValue(cValue, 8,
                               clientCache_ -> sendEventEventCache);
            *(outputMessage + i) = cValue;
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_ChangeWindowAttributes:
        {
          unsigned int numAttrs;
          decodeBuffer.decodeValue(numAttrs, 4);
          outputLength = 12 + (numAttrs << 2);
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned int bitmask;
          decodeBuffer.decodeCachedValue(bitmask, 15,
                                     clientCache_ -> createWindowBitmaskCache);
          PutULONG(bitmask, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          unsigned int mask = 0x1;
          for (unsigned int i = 0; i < 15; i++)
          {
            if (bitmask & mask)
            {
              decodeBuffer.decodeCachedValue(value, 32,
                                 *clientCache_ -> createWindowAttrCache[i]);
              PutULONG(value, nextDest, bigEndian_);
              nextDest += 4;
            }
            mask <<= 1;
          }
        }
        break;
      case X_ClearArea:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ClearArea);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeBool(value);
          outputMessage[1] = (unsigned char) value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned char *nextDest = outputMessage + 8;
          for (unsigned int i = 0; i < 4; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> clearAreaGeomCache[i], 8);
            PutUINT(value, nextDest, bigEndian_);
            nextDest += 2;
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_CloseFont:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 29, 5);
          clientCache_ -> lastFont += value;
          clientCache_ -> lastFont &= 0x1fffffff;
          PutULONG(clientCache_ -> lastFont, outputMessage + 4, bigEndian_);
        }
        break;
      case X_ConfigureWindow:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ConfigureWindow);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned int bitmask;
          decodeBuffer.decodeCachedValue(bitmask, 7,
                                  clientCache_ -> configureWindowBitmaskCache);
          PutUINT(bitmask, outputMessage + 8, bigEndian_);
          unsigned int mask = 0x1;
          for (unsigned int i = 0; i < 7; i++)
          {
            if (bitmask & mask)
            {
              unsigned char* nextDest = writeBuffer_.addMessage(4);
              outputLength += 4;
              decodeBuffer.decodeCachedValue(value, CONFIGUREWINDOW_FIELD_WIDTH[i],
                                 *clientCache_ -> configureWindowAttrCache[i], 8);
              PutULONG(value, nextDest, bigEndian_);
              nextDest += 4;
            }
            mask <<= 1;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_ConvertSelection:
        {
          outputLength = 24;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> convertSelectionRequestorCache, 9);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned char* nextDest = outputMessage + 8;
          for (unsigned int i = 0; i < 3; i++)
          {
            decodeBuffer.decodeCachedValue(value, 29,
                               *(clientCache_ -> convertSelectionAtomCache[i]), 9);
            PutULONG(value, nextDest, bigEndian_);
            nextDest += 4;
          }
          decodeBuffer.decodeValue(value, 32, 4);
          clientCache_ -> convertSelectionLastTimestamp += value;
          PutULONG(clientCache_ -> convertSelectionLastTimestamp,
                   nextDest, bigEndian_);
        }
        break;
      case X_CopyArea:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_CopyArea);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 28;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 12, bigEndian_);
          unsigned char *nextDest = outputMessage + 16;
          for (unsigned int i = 0; i < 6; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> copyAreaGeomCache[i], 8);
            PutUINT(value, nextDest, bigEndian_);
            nextDest += 2;
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_CopyGC:
        {
          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 23,
                             clientCache_ -> createGCBitmaskCache);
          PutULONG(value, outputMessage + 12, bigEndian_);
        }
        break;
      case X_CopyPlane:
        {
          outputLength = 32;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 12, bigEndian_);
          unsigned char *nextDest = outputMessage + 16;
          for (unsigned int i = 0; i < 6; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> copyPlaneGeomCache[i], 8);
            PutUINT(value, nextDest, bigEndian_);
            nextDest += 2;
          }
          decodeBuffer.decodeCachedValue(value, 32,
                             clientCache_ -> copyPlaneBitPlaneCache, 10);
          PutULONG(value, outputMessage + 28, bigEndian_);
        }
        break;
      case X_CreateGC:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_CreateGC);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned int offset = 8;
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + offset, bigEndian_);
          offset += 4;
          unsigned int bitmask;
          decodeBuffer.decodeCachedValue(bitmask, 23,
                                         clientCache_ -> createGCBitmaskCache);
          PutULONG(bitmask, outputMessage + offset, bigEndian_);
          unsigned int mask = 0x1;
          for (unsigned int i = 0; i < 23; i++)
          {
            if (bitmask & mask)
            {
              unsigned char* nextDest = writeBuffer_.addMessage(4);
              outputLength += 4;
              unsigned int fieldWidth = CREATEGC_FIELD_WIDTH[i];
              if (fieldWidth <= 4)
                decodeBuffer.decodeValue(value, fieldWidth);
              else
                decodeBuffer.decodeCachedValue(value, fieldWidth,
                                   *clientCache_ -> createGCAttrCache[i]);
              PutULONG(value, nextDest, bigEndian_);
            }
            mask <<= 1;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_ChangeGC:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ChangeGC);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned int offset = 8;
          unsigned int bitmask;
          decodeBuffer.decodeCachedValue(bitmask, 23,
                                         clientCache_ -> createGCBitmaskCache);
          PutULONG(bitmask, outputMessage + offset, bigEndian_);
          unsigned int mask = 0x1;
          for (unsigned int i = 0; i < 23; i++)
          {
            if (bitmask & mask)
            {
              unsigned char* nextDest = writeBuffer_.addMessage(4);
              outputLength += 4;
              unsigned int fieldWidth = CREATEGC_FIELD_WIDTH[i];
              if (fieldWidth <= 4)
                decodeBuffer.decodeValue(value, fieldWidth);
              else
                decodeBuffer.decodeCachedValue(value, fieldWidth,
                                   *clientCache_ -> createGCAttrCache[i]);
              PutULONG(value, nextDest, bigEndian_);
            }
            mask <<= 1;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_CreatePixmap:
        {
          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(cValue, 8, clientCache_ -> depthCache);
          outputMessage[1] = cValue;
          decodeBuffer.decodeBool(value);
          if (!value)
          {
            decodeBuffer.decodeValue(value, 29, 4);
            clientCache_ -> createPixmapLastPixmap += value;
            clientCache_ -> createPixmapLastPixmap &= 0x1fffffff;
          }
          PutULONG(clientCache_ -> createPixmapLastPixmap, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16, clientCache_ -> createPixmapXCache, 8);
          PutUINT(value, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16, clientCache_ -> createPixmapYCache, 8);
          PutUINT(value, outputMessage + 14, bigEndian_);
        }
        break;
      case X_CreateWindow:
        {
          outputLength = 32;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeCachedValue(cValue, 8, clientCache_ -> depthCache);
          outputMessage[1] = cValue;
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          unsigned int i;
          for (i = 0; i < 6; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> createWindowGeomCache[i], 8);
            PutUINT(value, nextDest, bigEndian_);
            nextDest += 2;
          }
          decodeBuffer.decodeCachedValue(value, 29, clientCache_ -> visualCache);
          PutULONG(value, outputMessage + 24, bigEndian_);
          unsigned int bitmask;
          decodeBuffer.decodeCachedValue(bitmask, 15,
                                     clientCache_ -> createWindowBitmaskCache);
          PutULONG(bitmask, outputMessage + 28, bigEndian_);
          unsigned int mask = 0x1;
          for (i = 0; i < 15; i++)
          {
            if (bitmask & mask)
            {
              nextDest = writeBuffer_.addMessage(4);
              outputLength += 4;
              decodeBuffer.decodeCachedValue(value, 32,
                                 *clientCache_ -> createWindowAttrCache[i]);
              PutULONG(value, nextDest, bigEndian_);
            }
            mask <<= 1;
          }
          writeBuffer_.unregisterPointer();
        }
        break;
      case X_DeleteProperty:
        {
          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeValue(value, 29, 9);
          PutULONG(value, outputMessage + 8, bigEndian_);
        }
        break;
      case X_FillPoly:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_FillPoly);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          unsigned int numPoints;
          decodeBuffer.decodeCachedValue(numPoints, 14,
                                 clientCache_ -> fillPolyNumPointsCache, 4);
          outputLength = 16 + (numPoints << 2);
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeValue(value, 2);
          outputMessage[12] = (unsigned char) value;
          unsigned int relativeCoordMode;
          decodeBuffer.decodeBool(relativeCoordMode);
          outputMessage[13] = (unsigned char) relativeCoordMode;
          unsigned char *nextDest = outputMessage + 16;
          unsigned int pointIndex = 0;
          for (unsigned int i = 0; i < numPoints; i++)
          {
            if (relativeCoordMode)
            {
              decodeBuffer.decodeCachedValue(value, 16,
                                 *clientCache_ -> fillPolyXRelCache[pointIndex], 8);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
              decodeBuffer.decodeCachedValue(value, 16,
                                 *clientCache_ -> fillPolyYRelCache[pointIndex], 8);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
            }
            else
            {
              unsigned int x, y;
              decodeBuffer.decodeBool(value);
              if (value)
              {
                decodeBuffer.decodeValue(value, 3);
                x = clientCache_ -> fillPolyRecentX[value];
                y = clientCache_ -> fillPolyRecentY[value];
              }
              else
              {
                decodeBuffer.decodeCachedValue(x, 16,
                                   *clientCache_ -> fillPolyXAbsCache[pointIndex], 8);
                decodeBuffer.decodeCachedValue(y, 16,
                                   *clientCache_ -> fillPolyYAbsCache[pointIndex], 8);
                clientCache_ -> fillPolyRecentX[clientCache_ -> fillPolyIndex] = x;
                clientCache_ -> fillPolyRecentY[clientCache_ -> fillPolyIndex] = y;
                clientCache_ -> fillPolyIndex++;
                if (clientCache_ -> fillPolyIndex == 8)
                  clientCache_ -> fillPolyIndex = 0;
              }
              PutUINT(x, nextDest, bigEndian_);
              nextDest += 2;
              PutUINT(y, nextDest, bigEndian_);
              nextDest += 2;
            }

            if (control -> isProtoStep3() == 1)
            {
              if (++pointIndex == 10) pointIndex = 0;
            }
            else
            {
              if (pointIndex + 1 < 10) pointIndex++;
            }
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_FreeColors:
        {
          unsigned int numPixels;
          decodeBuffer.decodeValue(numPixels, 16, 4);
          outputLength = 12 + (numPixels << 2);
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> colormapCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeValue(value, 32, 4);
          PutULONG(value, outputMessage + 8, bigEndian_);
          unsigned char* nextDest = outputMessage + 12;
          while (numPixels)
          {
            decodeBuffer.decodeValue(value, 32, 8);
            PutULONG(value, nextDest, bigEndian_);
            nextDest += 4;
            numPixels--;
          }
        }
        break;
      case X_FreeCursor:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29, clientCache_ -> cursorCache, 9);
          PutULONG(value, outputMessage + 4, bigEndian_);
        }
        break;
      case X_FreeGC:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
        }
        break;
      case X_FreePixmap:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeBool(value);
          if (!value)
          {
            decodeBuffer.decodeValue(value, 29, 4);
            clientCache_ -> createPixmapLastPixmap += value;
            clientCache_ -> createPixmapLastPixmap &= 0x1fffffff;
          }
          PutULONG(clientCache_ -> createPixmapLastPixmap, outputMessage + 4, bigEndian_);
        }
        break;
      case X_GetAtomName:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 29, 9);
          PutULONG(value, outputMessage + 4, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GetGeometry:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GetInputFocus:
        {
          outputLength = 4;
          outputMessage = writeBuffer_.addMessage(outputLength);

          sequenceQueue_.push(clientSequence_, outputOpcode, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GetModifierMapping:
        {
          outputLength = 4;
          outputMessage = writeBuffer_.addMessage(outputLength);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GetKeyboardMapping:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 8);
          outputMessage[4] = value;
          decodeBuffer.decodeValue(value, 8);
          outputMessage[5] = value;

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GetProperty:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_GetProperty);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            //
            // Save a reference to identify the reply.
            //

            unsigned int property = GetULONG(outputMessage + 8, bigEndian_);

            sequenceQueue_.push(clientSequence_, outputOpcode, property);

            outputFlush++;

            break;
          }

          outputLength = 24;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeBool(value);
          outputMessage[1] = (unsigned char) value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          unsigned int property;
          decodeBuffer.decodeValue(property, 29, 9);
          PutULONG(property, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeValue(value, 29, 9);
          PutULONG(value, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeValue(value, 32, 2);
          PutULONG(value, outputMessage + 16, bigEndian_);
          decodeBuffer.decodeValue(value, 32, 8);
          PutULONG(value, outputMessage + 20, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode, property);

          handleSave(messageStore, outputMessage, outputLength);

          outputFlush++;
        }
        break;
      case X_GetSelectionOwner:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                           clientCache_ -> getSelectionOwnerSelectionCache, 9);
          PutULONG(value, outputMessage + 4, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GrabButton:
      case X_GrabPointer:
        {
          outputLength = 24;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeBool(value);
          outputMessage[1] = (unsigned char) value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> grabButtonEventMaskCache);
          PutUINT(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeBool(value);
          outputMessage[10] = (unsigned char) value;
          decodeBuffer.decodeBool(value);
          outputMessage[11] = (unsigned char) value;
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> grabButtonConfineCache, 9);
          PutULONG(value, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> cursorCache, 9);
          PutULONG(value, outputMessage + 16, bigEndian_);
          if (outputOpcode == X_GrabButton)
          {
            decodeBuffer.decodeCachedValue(cValue, 8,
                               clientCache_ -> grabButtonButtonCache);
            outputMessage[20] = cValue;
            decodeBuffer.decodeCachedValue(value, 16,
                               clientCache_ -> grabButtonModifierCache);
            PutUINT(value, outputMessage + 22, bigEndian_);
          }
          else
          {
            decodeBuffer.decodeValue(value, 32, 4);
            clientCache_ -> grabKeyboardLastTimestamp += value;
            PutULONG(clientCache_ -> grabKeyboardLastTimestamp,
                     outputMessage + 20, bigEndian_);

            sequenceQueue_.push(clientSequence_, outputOpcode);

            outputFlush++;
          }
        }
        break;
      case X_GrabKeyboard:
        {
          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeBool(value);
          outputMessage[1] = (unsigned char) value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeValue(value, 32, 4);
          clientCache_ -> grabKeyboardLastTimestamp += value;
          PutULONG(clientCache_ -> grabKeyboardLastTimestamp, outputMessage + 8,
                   bigEndian_);
          decodeBuffer.decodeBool(value);
          outputMessage[12] = (unsigned char) value;
          decodeBuffer.decodeBool(value);
          outputMessage[13] = (unsigned char) value;

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GrabServer:
      case X_UngrabServer:
      case X_NoOperation:
        {
          #ifdef DEBUG
          *logofs << "handleWrite: Managing (probably tainted) X_NoOperation request for FD#" 
                  << fd_ << ".\n" << logofs_flush;
          #endif

          outputLength = 4;
          outputMessage = writeBuffer_.addMessage(outputLength);
        }
        break;
      case X_PolyText8:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyText8);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> polyTextCacheX, block);
          clientCache_ -> polyTextLastX += value;
          clientCache_ -> polyTextLastX &= 0xffff;
          PutUINT(clientCache_ -> polyTextLastX, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> polyTextCacheY, block);
          clientCache_ -> polyTextLastY += value;
          clientCache_ -> polyTextLastY &= 0xffff;
          PutUINT(clientCache_ -> polyTextLastY, outputMessage + 14, bigEndian_);
          unsigned int addedLength = 0;
          writeBuffer_.registerPointer(&outputMessage);
          for (;;)
          {
            decodeBuffer.decodeBool(value);
            if (!value)
              break;
            unsigned int textLength;
            decodeBuffer.decodeValue(textLength, 8);
            if (textLength == 255)
            {
              addedLength += 5;
              unsigned char *nextSegment = writeBuffer_.addMessage(5);
              *nextSegment = (unsigned char) textLength;
              decodeBuffer.decodeCachedValue(value, 29,
                                 clientCache_ -> polyTextFontCache);
              PutULONG(value, nextSegment + 1, 1);
            }
            else
            {
              addedLength += (textLength + 2);
              unsigned char *nextSegment =
              writeBuffer_.addMessage(textLength + 2);
              *nextSegment = (unsigned char) textLength;
              unsigned char *nextDest = nextSegment + 1;
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> polyTextDeltaCache);
              *nextDest++ = cValue;
              clientCache_ -> polyTextTextCompressor.reset();
              while (textLength)
              {
                *nextDest++ = clientCache_ -> polyTextTextCompressor.decodeChar(decodeBuffer);
                textLength--;
              }
            }
          }
          outputLength += addedLength;
          unsigned int mod4 = (addedLength & 0x3);
          if (mod4)
          {
            unsigned int extra = 4 - mod4;
            unsigned char *nextDest = writeBuffer_.addMessage(extra);
            for (unsigned int i = 0; i < extra; i++)
              *nextDest++ = 0;
            outputLength += extra;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PolyText16:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyText16);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> polyTextCacheX, block);
          clientCache_ -> polyTextLastX += value;
          clientCache_ -> polyTextLastX &= 0xffff;
          PutUINT(clientCache_ -> polyTextLastX, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> polyTextCacheY, block);
          clientCache_ -> polyTextLastY += value;
          clientCache_ -> polyTextLastY &= 0xffff;
          PutUINT(clientCache_ -> polyTextLastY, outputMessage + 14, bigEndian_);
          unsigned int addedLength = 0;
          writeBuffer_.registerPointer(&outputMessage);
          for (;;)
          {
            decodeBuffer.decodeBool(value);
            if (!value)
              break;
            unsigned int textLength;
            decodeBuffer.decodeValue(textLength, 8);
            if (textLength == 255)
            {
              addedLength += 5;
              unsigned char *nextSegment = writeBuffer_.addMessage(5);
              *nextSegment = (unsigned char) textLength;
              decodeBuffer.decodeCachedValue(value, 29, clientCache_ -> polyTextFontCache);
              PutULONG(value, nextSegment + 1, 1);
            }
            else
            {
              addedLength += (textLength * 2 + 2);
              unsigned char *nextSegment =
              writeBuffer_.addMessage(textLength * 2 + 2);
              *nextSegment = (unsigned char) textLength;
              unsigned char *nextDest = nextSegment + 1;
              decodeBuffer.decodeCachedValue(cValue, 8, clientCache_ -> polyTextDeltaCache);
              *nextDest++ = cValue;
              clientCache_ -> polyTextTextCompressor.reset();
              textLength <<= 1;
              while (textLength)
              {
                *nextDest++ =
                  clientCache_ -> polyTextTextCompressor.decodeChar(decodeBuffer);
                textLength--;
              }
            }
          }
          outputLength += addedLength;

          unsigned int mod4 = (addedLength & 0x3);
          if (mod4)
          {
            unsigned int extra = 4 - mod4;
            unsigned char *nextDest = writeBuffer_.addMessage(extra);
            for (unsigned int i = 0; i < extra; i++)
              *nextDest++ = 0;
            outputLength += extra;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_ImageText8:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ImageText8);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          unsigned int textLength;
          decodeBuffer.decodeCachedValue(textLength, 8,
                       clientCache_ -> imageTextLengthCache, 4);
          outputLength = 16 + RoundUp4(textLength);
          outputMessage = writeBuffer_.addMessage(outputLength);
          outputMessage[1] = (unsigned char) textLength;
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                       clientCache_ -> imageTextCacheX, block);
          clientCache_ -> imageTextLastX += value;
          clientCache_ -> imageTextLastX &= 0xffff;
          PutUINT(clientCache_ -> imageTextLastX, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                       clientCache_ -> imageTextCacheY, block);
          clientCache_ -> imageTextLastY += value;
          clientCache_ -> imageTextLastY &= 0xffff;
          PutUINT(clientCache_ -> imageTextLastY, outputMessage + 14, bigEndian_);
          unsigned char *nextDest = outputMessage + 16;
          clientCache_ -> imageTextTextCompressor.reset();
          for (unsigned int j = 0; j < textLength; j++)
            *nextDest++ = clientCache_ -> imageTextTextCompressor.decodeChar(decodeBuffer);

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_ImageText16:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_ImageText16);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          int block = 8;

          if (control -> isProtoStep2() == 1)
          {
            block = 0;
          }

          unsigned int textLength;
          decodeBuffer.decodeCachedValue(textLength, 8,
                       clientCache_ -> imageTextLengthCache, 4);
          outputLength = 16 + RoundUp4(textLength * 2);
          outputMessage = writeBuffer_.addMessage(outputLength);
          outputMessage[1] = (unsigned char) textLength;
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                       clientCache_ -> imageTextCacheX, block);
          clientCache_ -> imageTextLastX += value;
          clientCache_ -> imageTextLastX &= 0xffff;
          PutUINT(clientCache_ -> imageTextLastX, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                       clientCache_ -> imageTextCacheY, block);
          clientCache_ -> imageTextLastY += value;
          clientCache_ -> imageTextLastY &= 0xffff;
          PutUINT(clientCache_ -> imageTextLastY, outputMessage + 14, bigEndian_);
          unsigned char *nextDest = outputMessage + 16;
          clientCache_ -> imageTextTextCompressor.reset();
          for (unsigned int j = 0; j < textLength * 2; j++)
            *nextDest++ = clientCache_ -> imageTextTextCompressor.decodeChar(decodeBuffer);

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_InternAtom:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_InternAtom);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            sequenceQueue_.push(clientSequence_, outputOpcode);

            outputFlush++;

            break;
          }

          unsigned int nameLength;
          decodeBuffer.decodeValue(nameLength, 16, 6);
          outputLength = RoundUp4(nameLength) + 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          PutUINT(nameLength, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeBool(value);
          outputMessage[1] = (unsigned char) value;
          unsigned char *nextDest = outputMessage + 8;
          clientCache_ -> internAtomTextCompressor.reset();
          for (unsigned int i = 0; i < nameLength; i++)
          {
            *nextDest++ = clientCache_ -> internAtomTextCompressor.decodeChar(decodeBuffer);
          }

          sequenceQueue_.push(clientSequence_, outputOpcode);

          handleSave(messageStore, outputMessage, outputLength);

          outputFlush++;
        }
        break;
      case X_ListExtensions:
        {
          outputLength = 4;
          outputMessage = writeBuffer_.addMessage(outputLength);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_ListFonts:
        {
          unsigned int textLength;
          decodeBuffer.decodeValue(textLength, 16, 6);
          outputLength = 8 + RoundUp4(textLength);
          outputMessage = writeBuffer_.addMessage(outputLength);
          PutUINT(textLength, outputMessage + 6, bigEndian_);
          decodeBuffer.decodeValue(value, 16, 6);
          PutUINT(value, outputMessage + 4, bigEndian_);
          unsigned char* nextDest = outputMessage + 8;
          clientCache_ -> polyTextTextCompressor.reset();
          for (unsigned int i = 0; i < textLength; i++)
          {
            *nextDest++ = clientCache_ -> polyTextTextCompressor.decodeChar(decodeBuffer);
          }

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_LookupColor:
      case X_AllocNamedColor:
        {
          unsigned int textLength;
          decodeBuffer.decodeValue(textLength, 16, 6);
          outputLength = 12 + RoundUp4(textLength);
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                             clientCache_ -> colormapCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          PutUINT(textLength, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          clientCache_ -> polyTextTextCompressor.reset();
          for (unsigned int i = 0; i < textLength; i++)
          {
            *nextDest++ = clientCache_ -> polyTextTextCompressor.decodeChar(decodeBuffer);
          }

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_MapWindow:
      case X_UnmapWindow:
      case X_MapSubwindows:
      case X_GetWindowAttributes:
      case X_DestroyWindow:
      case X_DestroySubwindows:
      case X_QueryPointer:
      case X_QueryTree:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> windowCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          if (outputOpcode == X_QueryPointer ||
                  outputOpcode == X_GetWindowAttributes ||
                      outputOpcode == X_QueryTree)
          {
            sequenceQueue_.push(clientSequence_, outputOpcode);

            outputFlush++;
          }
        }
        break;
      case X_OpenFont:
        {
          unsigned int nameLength;
          decodeBuffer.decodeValue(nameLength, 16, 7);
          outputLength = RoundUp4(12 + nameLength);
          outputMessage = writeBuffer_.addMessage(outputLength);
          PutUINT(nameLength, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeValue(value, 29, 5);
          clientCache_ -> lastFont += value;
          clientCache_ -> lastFont &= 0x1fffffff;
          PutULONG(clientCache_ -> lastFont, outputMessage + 4, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          clientCache_ -> openFontTextCompressor.reset();
          for (; nameLength; nameLength--)
          {
            *nextDest++ = clientCache_ -> openFontTextCompressor.
                                decodeChar(decodeBuffer);
          }
        }
        break;
      case X_PolyFillRectangle:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyFillRectangle);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0, lastWidth = 0, lastHeight = 0;
          unsigned int numRectangles = 0;

          for (;;)
          {
            outputLength += 8;
            writeBuffer_.addMessage(8);
            unsigned char *nextDest = outputMessage + 12 +
            (numRectangles << 3);
            numRectangles++;
            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillRectangleCacheX[index], 8);
            value += lastX;
            PutUINT(value, nextDest, bigEndian_);
            lastX = value;
            nextDest += 2;
            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillRectangleCacheY[index], 8);
            value += lastY;
            PutUINT(value, nextDest, bigEndian_);
            lastY = value;
            nextDest += 2;
            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillRectangleCacheWidth[index], 8);
            value += lastWidth;
            PutUINT(value, nextDest, bigEndian_);
            lastWidth = value;
            nextDest += 2;
            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillRectangleCacheHeight[index], 8);
            value += lastHeight;
            PutUINT(value, nextDest, bigEndian_);
            lastHeight = value;
            nextDest += 2;

            if (++index == 4) index = 0;

            decodeBuffer.decodeBool(value);

            if (!value) break;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PolyFillArc:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyFillArc);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0,
                       lastWidth = 0, lastHeight = 0,
                       lastAngle1 = 0, lastAngle2 = 0;

          unsigned int numArcs = 0;

          for (;;)
          {
            outputLength += 12;
            writeBuffer_.addMessage(12);

            unsigned char *nextDest = outputMessage + 12 +
            (numArcs * 12);
            numArcs++;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillArcCacheX[index], 8);
            value += lastX;
            PutUINT(value, nextDest, bigEndian_);
            lastX = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillArcCacheY[index], 8);
            value += lastY;
            PutUINT(value, nextDest, bigEndian_);
            lastY = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillArcCacheWidth[index], 8);
            value += lastWidth;
            PutUINT(value, nextDest, bigEndian_);
            lastWidth = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillArcCacheHeight[index], 8);
            value += lastHeight;
            PutUINT(value, nextDest, bigEndian_);
            lastHeight = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillArcCacheAngle1[index], 8);
            value += lastAngle1;
            PutUINT(value, nextDest, bigEndian_);
            lastAngle1 = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyFillArcCacheAngle2[index], 8);
            value += lastAngle2;
            PutUINT(value, nextDest, bigEndian_);
            lastAngle2 = value;
            nextDest += 2;

            if (++index == 2) index = 0;

            decodeBuffer.decodeBool(value);

            if (!value) break;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PolyArc:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyArc);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          writeBuffer_.registerPointer(&outputMessage);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0,
                       lastWidth = 0, lastHeight = 0,
                       lastAngle1 = 0, lastAngle2 = 0;

          unsigned int numArcs = 0;

          for (;;)
          {
            outputLength += 12;
            writeBuffer_.addMessage(12);

            unsigned char *nextDest = outputMessage + 12 +
            (numArcs * 12);
            numArcs++;

            decodeBuffer.decodeCachedValue(value, 16,
                           *clientCache_ -> polyArcCacheX[index], 8);
            value += lastX;
            PutUINT(value, nextDest, bigEndian_);
            lastX = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyArcCacheY[index], 8);
            value += lastY;
            PutUINT(value, nextDest, bigEndian_);
            lastY = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyArcCacheWidth[index], 8);
            value += lastWidth;
            PutUINT(value, nextDest, bigEndian_);
            lastWidth = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyArcCacheHeight[index], 8);
            value += lastHeight;
            PutUINT(value, nextDest, bigEndian_);
            lastHeight = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyArcCacheAngle1[index], 8);
            value += lastAngle1;
            PutUINT(value, nextDest, bigEndian_);
            lastAngle1 = value;
            nextDest += 2;

            decodeBuffer.decodeCachedValue(value, 16,
                         *clientCache_ -> polyArcCacheAngle2[index], 8);
            value += lastAngle2;
            PutUINT(value, nextDest, bigEndian_);
            lastAngle2 = value;
            nextDest += 2;

            if (++index == 2) index = 0;

            decodeBuffer.decodeBool(value);

            if (!value) break;
          }
          writeBuffer_.unregisterPointer();

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PolyPoint:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyPoint);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          unsigned int numPoints;
          decodeBuffer.decodeValue(numPoints, 16, 4);
          outputLength = (numPoints << 2) + 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          unsigned int relativeCoordMode;
          decodeBuffer.decodeBool(relativeCoordMode);
          outputMessage[1] = (unsigned char) relativeCoordMode;
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;

          for (unsigned int i = 0; i < numPoints; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> polyPointCacheX[index], 8);
            lastX += value;
            PutUINT(lastX, nextDest, bigEndian_);
            nextDest += 2;
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> polyPointCacheY[index], 8);
            lastY += value;
            PutUINT(lastY, nextDest, bigEndian_);
            nextDest += 2;

            if (++index == 2) index = 0;
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PolyLine:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolyLine);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          unsigned int numPoints;
          decodeBuffer.decodeValue(numPoints, 16, 4);
          outputLength = (numPoints << 2) + 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          unsigned int relativeCoordMode;
          decodeBuffer.decodeBool(relativeCoordMode);
          outputMessage[1] = (unsigned char) relativeCoordMode;
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;

          unsigned int index = 0;
          unsigned int lastX = 0, lastY = 0;

          for (unsigned int i = 0; i < numPoints; i++)
          {
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> polyLineCacheX[index], 8);
            lastX += value;
            PutUINT(lastX, nextDest, bigEndian_);
            nextDest += 2;
            decodeBuffer.decodeCachedValue(value, 16,
                               *clientCache_ -> polyLineCacheY[index], 8);
            lastY += value;
            PutUINT(lastY, nextDest, bigEndian_);
            nextDest += 2;

            if (++index == 2) index = 0;
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PolyRectangle:
        {
          unsigned int numRectangles;
          decodeBuffer.decodeValue(numRectangles, 16, 3);
          outputLength = (numRectangles << 3) + 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          for (unsigned int i = 0; i < numRectangles; i++)
            for (unsigned int k = 0; k < 4; k++)
            {
              decodeBuffer.decodeCachedValue(value, 16,
                                *clientCache_ -> polyRectangleGeomCache[k], 8);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
            }
        }
        break;
      case X_PolySegment:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PolySegment);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          unsigned int numSegments;
          decodeBuffer.decodeValue(numSegments, 16, 4);
          outputLength = (numSegments << 3) + 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;

          for (numSegments *= 2; numSegments; numSegments--)
          {
            unsigned int index;
            decodeBuffer.decodeBool(index);
            unsigned int x;
            decodeBuffer.decodeCachedValue(x, 16,
                               clientCache_ -> polySegmentCacheX, 6);
            x += clientCache_ -> polySegmentLastX[index];
            PutUINT(x, nextDest, bigEndian_);
            nextDest += 2;

            unsigned int y;
            decodeBuffer.decodeCachedValue(y, 16,
                               clientCache_ -> polySegmentCacheY, 6);
            y += clientCache_ -> polySegmentLastY[index];
            PutUINT(y, nextDest, bigEndian_);
            nextDest += 2;

            clientCache_ -> polySegmentLastX[clientCache_ -> polySegmentCacheIndex] = x;
            clientCache_ -> polySegmentLastY[clientCache_ -> polySegmentCacheIndex] = y;

            if (clientCache_ -> polySegmentCacheIndex == 1)
              clientCache_ -> polySegmentCacheIndex = 0;
            else
              clientCache_ -> polySegmentCacheIndex = 1;
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_PutImage:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_PutImage);

          hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                 outputOpcode, outputMessage, outputLength);

          if (outputOpcode == X_PutImage)
          {
            handleImage(outputOpcode, outputMessage, outputLength, outputFlush);
          }
        }
        break;
      case X_QueryBestSize:
        {
          outputLength = 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 2);
          outputMessage[1] = (unsigned char)value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeValue(value, 16, 8);
          PutUINT(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeValue(value, 16, 8);
          PutUINT(value, outputMessage + 10, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_QueryColors:
        {
          // Differential or plain data compression?
          decodeBuffer.decodeBool(value);

          if (value)
          {
            unsigned int numColors;
            decodeBuffer.decodeValue(numColors, 16, 5);
            outputLength = (numColors << 2) + 8;
            outputMessage = writeBuffer_.addMessage(outputLength);
            decodeBuffer.decodeCachedValue(value, 29,
                               clientCache_ -> colormapCache);
            PutULONG(value, outputMessage + 4, bigEndian_);
            unsigned char *nextDest = outputMessage + 8;
            unsigned int predictedPixel = clientCache_ -> queryColorsLastPixel;
            for (unsigned int i = 0; i < numColors; i++)
            {
              unsigned int pixel;
              decodeBuffer.decodeBool(value);
              if (value)
                pixel = predictedPixel;
              else
                decodeBuffer.decodeValue(pixel, 32, 9);
              PutULONG(pixel, nextDest, bigEndian_);
              if (i == 0)
                clientCache_ -> queryColorsLastPixel = pixel;
              predictedPixel = pixel + 1;
              nextDest += 4;
            }
          }
          else
          {
            // Request length.
            unsigned int requestLength;
            decodeBuffer.decodeValue(requestLength, 16, 10);
            outputLength = (requestLength << 2);
            outputMessage = writeBuffer_.addMessage(outputLength);

            const unsigned char *compressedData = NULL;
            unsigned int compressedDataSize = 0;

            int decompressed = handleDecompress(decodeBuffer, outputOpcode, outputMessage,
                                                    outputLength, 4, compressedData,
                                                        compressedDataSize);
            if (decompressed < 0)
            {
              return -1;
            }
          }

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_QueryExtension:
        {
          unsigned int nameLength;
          decodeBuffer.decodeValue(nameLength, 16, 6);
          outputLength = 8 + RoundUp4(nameLength);
          outputMessage = writeBuffer_.addMessage(outputLength);
          PutUINT(nameLength, outputMessage + 4, bigEndian_);
          unsigned char *nextDest = outputMessage + 8;
          for (unsigned int i = 0; i < nameLength; i++)
          {
            decodeBuffer.decodeValue(value, 8);
            *nextDest++ = (unsigned char) value;
          }

          unsigned int hide = 0;

          #ifdef HIDE_MIT_SHM_EXTENSION

          if (!strncmp((char *) outputMessage + 8, "MIT-SHM", 7))
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Going to hide MIT-SHM extension in reply.\n"
                    << logofs_flush;
            #endif

            hide = 1;
          }

          #endif

          #ifdef HIDE_BIG_REQUESTS_EXTENSION

          if (!strncmp((char *) outputMessage + 8, "BIG-REQUESTS", 12))
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Going to hide BIG-REQUESTS extension in reply.\n"
                    << logofs_flush;
            #endif

            hide = 1;
          }

          #endif

          #ifdef HIDE_XKEYBOARD_EXTENSION

          else if (!strncmp((char *) outputMessage + 8, "XKEYBOARD", 9))
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Going to hide XKEYBOARD extension in reply.\n"
                    << logofs_flush;
            #endif

            hide = 1;
          }

          #endif

          #ifdef HIDE_XFree86_Bigfont_EXTENSION

          else if (!strncmp((char *) outputMessage + 8, "XFree86-Bigfont", 15))
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Going to hide XFree86-Bigfont extension in reply.\n"
                    << logofs_flush;
            #endif

            hide = 1;
          }

          #endif

          //
          // This is if you want to experiment disabling SHAPE extensions.
          //

          #ifdef HIDE_SHAPE_EXTENSION

          if (!strncmp((char *) outputMessage + 8, "SHAPE", 5))
          {
            #ifdef DEBUG
            *logofs << "handleWrite: Going to hide SHAPE extension in reply.\n"
                    << logofs_flush;
            #endif

            hide = 1;
          }

          #endif

          //
          // Check if user disabled RENDER extension.
          //

          if (control -> AgentHideRender == 1 &&
                  strncmp((char *) outputMessage + 8, "RENDER", 6) == 0)
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Going to hide RENDER extension in reply.\n"
                    << logofs_flush;
            #endif

            hide = 1;
          }

          unsigned int extension = 0;

          if (strncmp((char *) outputMessage + 8, "SHAPE", 5) == 0)
          {
            extension = X_NXInternalShapeExtension;
          }
          else if (strncmp((char *) outputMessage + 8, "RENDER", 6) == 0)
          {
            extension = X_NXInternalRenderExtension;
          }

          sequenceQueue_.push(clientSequence_, outputOpcode,
                                  outputOpcode, hide, extension);
          outputFlush++;
        }
        break;
      case X_QueryFont:
        {
          outputLength = 8;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 29, 5);
          clientCache_ -> lastFont += value;
          clientCache_ -> lastFont &= 0x1fffffff;
          PutULONG(clientCache_ -> lastFont, outputMessage + 4, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_SetClipRectangles:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_SetClipRectangles);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            break;
          }

          unsigned int numRectangles;
          decodeBuffer.decodeValue(numRectangles, 13, 4);
          outputLength = (numRectangles << 3) + 12;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeValue(value, 2);
          outputMessage[1] = (unsigned char) value;
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                                   clientCache_ -> setClipRectanglesXCache, 8);
          PutUINT(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                                   clientCache_ -> setClipRectanglesYCache, 8);
          PutUINT(value, outputMessage + 10, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          for (unsigned int i = 0; i < numRectangles; i++)
          {
            for (unsigned int k = 0; k < 4; k++)
            {
              decodeBuffer.decodeCachedValue(value, 16,
                                 *clientCache_ -> setClipRectanglesGeomCache[k], 8);
              PutUINT(value, nextDest, bigEndian_);
              nextDest += 2;
            }
          }

          handleSave(messageStore, outputMessage, outputLength);
        }
        break;
      case X_SetDashes:
        {
          unsigned int numDashes;
          decodeBuffer.decodeCachedValue(numDashes, 16,
                                      clientCache_ -> setDashesLengthCache, 5);
          outputLength = 12 + RoundUp4(numDashes);
          outputMessage = writeBuffer_.addMessage(outputLength);
          PutUINT(numDashes, outputMessage + 10, bigEndian_);
          decodeBuffer.decodeXidValue(value, clientCache_ -> gcCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                                      clientCache_ -> setDashesOffsetCache, 5);
          PutUINT(value, outputMessage + 8, bigEndian_);
          unsigned char *nextDest = outputMessage + 12;
          for (unsigned int i = 0; i < numDashes; i++)
          {
            decodeBuffer.decodeCachedValue(cValue, 8,
                                clientCache_ -> setDashesDashCache_[i & 1], 5);
            *nextDest++ = cValue;
          }
        }
        break;
      case X_SetSelectionOwner:
        {
          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                                    clientCache_ -> setSelectionOwnerCache, 9);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 29,
                           clientCache_ -> getSelectionOwnerSelectionCache, 9);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 32,
                           clientCache_ -> setSelectionOwnerTimestampCache, 9);
          PutULONG(value, outputMessage + 12, bigEndian_);
        }
        break;
      case X_TranslateCoords:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_TranslateCoords);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            sequenceQueue_.push(clientSequence_, outputOpcode);

            outputFlush++;

            break;
          }

          outputLength = 16;
          outputMessage = writeBuffer_.addMessage(outputLength);
          decodeBuffer.decodeCachedValue(value, 29,
                                   clientCache_ -> translateCoordsSrcCache, 9);
          PutULONG(value, outputMessage + 4, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 29,
                                  clientCache_ -> translateCoordsDestCache, 9);
          PutULONG(value, outputMessage + 8, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                                     clientCache_ -> translateCoordsXCache, 8);
          PutUINT(value, outputMessage + 12, bigEndian_);
          decodeBuffer.decodeCachedValue(value, 16,
                                     clientCache_ -> translateCoordsYCache, 8);
          PutUINT(value, outputMessage + 14, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          handleSave(messageStore, outputMessage, outputLength);

          outputFlush++;
        }
        break;
      case X_GetImage:
        {
          MessageStore *messageStore = clientStore_ ->
                               getRequestStore(X_GetImage);

          if (handleDecode(decodeBuffer, clientCache_, messageStore,
                               outputMessage, outputLength))
          {
            sequenceQueue_.push(clientSequence_, outputOpcode);

            outputFlush++;

            break;
          }

          outputLength = 20;
          outputMessage = writeBuffer_.addMessage(outputLength);
          // Format.
          unsigned int format;
          decodeBuffer.decodeValue(format, 2);
          outputMessage[1] = (unsigned char) format;
          // Drawable.
          decodeBuffer.decodeXidValue(value, clientCache_ -> drawableCache);
          PutULONG(value, outputMessage + 4, bigEndian_);
          // X.
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> putImageXCache, 8);
          clientCache_ -> putImageLastX += value;
          clientCache_ -> putImageLastX &= 0xffff;
          PutUINT(clientCache_ -> putImageLastX, outputMessage + 8, bigEndian_);
          // Y.
          decodeBuffer.decodeCachedValue(value, 16,
                             clientCache_ -> putImageYCache, 8);
          clientCache_ -> putImageLastY += value;
          clientCache_ -> putImageLastY &= 0xffff;
          PutUINT(clientCache_ -> putImageLastY, outputMessage + 10, bigEndian_);
          // Width.
          unsigned int width;
          decodeBuffer.decodeCachedValue(width, 16,
                             clientCache_ -> putImageWidthCache, 8);
          PutUINT(width, outputMessage + 12, bigEndian_);
          // Height.
          unsigned int height;
          decodeBuffer.decodeCachedValue(height, 16,
                             clientCache_ -> putImageHeightCache, 8);
          PutUINT(height, outputMessage + 14, bigEndian_);
          // Plane mask.
          decodeBuffer.decodeCachedValue(value, 32,
                             clientCache_ -> getImagePlaneMaskCache, 5);
          PutULONG(value, outputMessage + 16, bigEndian_);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          handleSave(messageStore, outputMessage, outputLength);

          outputFlush++;
        }
        break;
      case X_GetPointerMapping:
        {
          outputLength = 4;
          outputMessage = writeBuffer_.addMessage(outputLength);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      case X_GetKeyboardControl:
        {
          outputLength = 4;
          outputMessage = writeBuffer_.addMessage(outputLength);

          sequenceQueue_.push(clientSequence_, outputOpcode);

          outputFlush++;
        }
        break;
      default:
        {
          if (outputOpcode == opcodeStore_ -> sync)
          {
            if (control -> AgentSyncPropagate == 1)
            {
              if (syncCounter_ == 0)
              {
                #ifdef DEBUG
                *logofs << "handleWrite: Set first sync timestamp for FD#"
                        << fd_ << ".\n" << logofs_flush;
                #endif

                firstSyncTs_ = getTimestamp();
              }

              syncCounter_++;

              //
              // Force a reply from X server
              // to get any outstanding event.
              //

              #if defined(INFO) || defined(TEST)
              *logofs << "handleWrite: Sending X_GetInputFocus request for FD#" 
                      << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                      << ".\n" << logofs_flush;
              #endif

              outputOpcode = X_GetInputFocus;

              outputLength  = 4;
              outputMessage = writeBuffer_.addMessage(outputLength);

              sequenceQueue_.push(clientSequence_, outputOpcode, opcodeStore_ -> sync);
            }
            else
            {
              #if defined(INFO) || defined(TEST)
              *logofs << "handleWrite: Sending X_NoOperation request for FD#" 
                      << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                      << ".\n" << logofs_flush;
              #endif

              outputOpcode = X_NoOperation;

              outputLength  = 4;
              outputMessage = writeBuffer_.addMessage(outputLength);
            }

            outputFlush++;
          }
          else if (outputOpcode == opcodeStore_ -> karma)
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Sending X_NoOperation request for FD#" 
                    << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            outputOpcode = X_NoOperation;

            outputLength  = 4;
            outputMessage = writeBuffer_.addMessage(outputLength);
          }
          else if (outputOpcode == opcodeStore_ -> shapeExtension)
          {
            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXInternalShapeExtension);

            hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                   outputOpcode, outputMessage, outputLength);
          }
          else if (outputOpcode == opcodeStore_ -> renderExtension &&
                       control -> isProtoStep3() == 1)
          {
            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXInternalRenderExtension);

            hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                   outputOpcode, outputMessage, outputLength);
          }
          else if (outputOpcode == opcodeStore_ -> putPackedImage)
          {
            #ifdef DEBUG
            *logofs << "handleWrite: Decoding packed image request for FD#"
                    << fd_ << ".\n" << logofs_flush;
            #endif

            //
            // TODO: Due to the way tight decoding is
            // implemented, we need to flush the main
            // write buffer before decoding the image.
            //

            if (tightState_ != NULL)
            {
              handleFlush(flush_if_any);
            }

            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXPutPackedImage);

            hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                   outputOpcode, outputMessage, outputLength);

            if (outputOpcode == opcodeStore_ -> putPackedImage)
            {
              handleImage(outputOpcode, outputMessage, outputLength, outputFlush);
            }
          }
          else if (outputOpcode == opcodeStore_ -> setUnpackColormap)
          {
            #ifdef DEBUG
            *logofs << "handleWrite: Decoding set unpack colormap request "
                    << "for FD#" << fd_ << ".\n" << logofs_flush;
            #endif

            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXSetUnpackColormap);

            hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                   outputOpcode, outputMessage, outputLength);
            //
            // Message could have been split.
            //

            if (outputOpcode == opcodeStore_ -> setUnpackColormap)
            {
              handleColormap(outputOpcode, outputMessage, outputLength);
            }
          }
          else if (outputOpcode == opcodeStore_ -> setUnpackAlpha &&
                       control -> isProtoStep4() == 1)
          {
            #ifdef DEBUG
            *logofs << "handleWrite: Decoding unpack alpha request for FD#"
                    << fd_ << ".\n" << logofs_flush;
            #endif

            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXSetUnpackAlpha);

            hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                   outputOpcode, outputMessage, outputLength);
            //
            // Message could have been split.
            //

            if (outputOpcode == opcodeStore_ -> setUnpackAlpha)
            {
              handleAlpha(outputOpcode, outputMessage, outputLength);
            }
          }
          else if (outputOpcode == opcodeStore_ -> setUnpackGeometry)
          {
            #ifdef DEBUG
            *logofs << "handleWrite: Decoding set unpack geometry request "
                    << "for FD#" << fd_ << ".\n" << logofs_flush;
            #endif

            if (control -> isProtoStep4() == 1)
            {
              MessageStore *messageStore = clientStore_ ->
                                   getRequestStore(X_NXSetUnpackGeometry);

              hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                     outputOpcode, outputMessage, outputLength);
            }
            else
            {
              outputLength  = 24;
              outputMessage = writeBuffer_.addMessage(outputLength);

              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> depthCache);
              *(outputMessage + 4) = cValue;
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> depthCache);
              *(outputMessage + 5) = cValue;
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> depthCache);
              *(outputMessage + 6) = cValue;
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> depthCache);
              *(outputMessage + 7) = cValue;
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> depthCache);
              *(outputMessage + 8) = cValue;
              decodeBuffer.decodeCachedValue(cValue, 8,
                                 clientCache_ -> depthCache);
              *(outputMessage + 9) = cValue;

              decodeBuffer.decodeValue(value, 32);
              PutULONG(value, outputMessage + 12, bigEndian_);
              decodeBuffer.decodeValue(value, 32);
              PutULONG(value, outputMessage + 16, bigEndian_);
              decodeBuffer.decodeValue(value, 32);
              PutULONG(value, outputMessage + 20, bigEndian_);
            }

            handleGeometry(outputOpcode, outputMessage, outputLength);
          }
          else if (outputOpcode == opcodeStore_ -> freeUnpack &&
                       control -> isProtoStep4() == 1)
          {
            #ifdef DEBUG
            *logofs << "handleWrite: Decoding free unpack request "
                    << "for FD#" << fd_ << ".\n" << logofs_flush;
            #endif

            decodeBuffer.decodeCachedValue(cValue, 8,
                         clientCache_ -> resourceCache);

            #ifdef DEBUG
            *logofs << "handleWrite: Freeing unpack state for client "
                    << (unsigned int) cValue << ".\n" << logofs_flush;
            #endif

            handleUnpackStateRemove(cValue);

            #ifdef TEST
            *logofs << "handleWrite: Sending X_NoOperation request for FD#" 
                    << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            outputLength  = 4;
            outputMessage = writeBuffer_.addMessage(outputLength);

            outputOpcode = X_NoOperation;
          }
          else if (outputOpcode == opcodeStore_ -> startSplit ||
                       outputOpcode == opcodeStore_ -> endSplit)
          {
            #ifdef TEST
            *logofs << "handleWrite: Sending X_NoOperation request for FD#" 
                    << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            outputLength  = 4;
            outputMessage = writeBuffer_.addMessage(outputLength);

            outputOpcode = X_NoOperation;
          }
          else if (outputOpcode == opcodeStore_ -> commitSplit)
          {
            int result = handleCommit(decodeBuffer, outputOpcode,
                                          outputMessage, outputLength);
            //
            // Check if message has been successfully
            // extracted form the split store. In this
            // case post-process it in the usual way.
            //

            if (result > 0)
            {
              if (outputOpcode == opcodeStore_ -> putPackedImage ||
                      outputOpcode == X_PutImage)
              {
                handleImage(outputOpcode, outputMessage, outputLength, outputFlush);
              }
              else if (outputOpcode == opcodeStore_ -> setUnpackColormap)
              {
                handleColormap(outputOpcode, outputMessage, outputLength);
              }
              else if (outputOpcode == opcodeStore_ -> setUnpackAlpha)
              {
                handleAlpha(outputOpcode, outputMessage, outputLength);
              }
            }
            else if (result < 0)
            {
              return -1;
            }
          }
          else if (outputOpcode == opcodeStore_ -> setExposeEvents)
          {
            //
            // Send expose events according to agent's wish.
            //

            decodeBuffer.decodeBool(enableExpose_);
            decodeBuffer.decodeBool(enableGraphicsExpose_);
            decodeBuffer.decodeBool(enableNoExpose_);

            #ifdef TEST
            *logofs << "handleWrite: Sending X_NoOperation request for FD#" 
                    << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            outputLength  = 4;
            outputMessage = writeBuffer_.addMessage(outputLength);

            outputOpcode = X_NoOperation;
          }
          else if (outputOpcode == opcodeStore_ -> getUnpackParameters)
          {
            //
            // Client proxy needs the list of supported
            // unpack methods. We would need an encode
            // buffer, but this is in proxy, not here in
            // channel.
            //

            #if defined(INFO) || defined(TEST)
            *logofs << "handleWrite: Sending X_GetInputFocus request for FD#" 
                    << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            outputOpcode = X_GetInputFocus;

            outputLength  = 4;
            outputMessage = writeBuffer_.addMessage(outputLength);

            sequenceQueue_.push(clientSequence_, outputOpcode,
                                    opcodeStore_ -> getUnpackParameters);

            outputFlush++;
          }
          else if (outputOpcode == opcodeStore_ -> getControlParameters ||
                       outputOpcode == opcodeStore_ -> getCleanupParameters ||
                           outputOpcode == opcodeStore_ -> getImageParameters)
          {
            #ifdef TEST
            *logofs << "handleWrite: Sending X_NoOperation request for FD#" 
                    << fd_ << " due to OPCODE#" << (unsigned int) outputOpcode
                    << ".\n" << logofs_flush;
            #endif

            outputLength  = 4;
            outputMessage = writeBuffer_.addMessage(outputLength);

            outputOpcode = X_NoOperation;
          }
          else if (outputOpcode == opcodeStore_ -> getShmemParameters)
          {
            if (control -> isProtoStep4() == 1)
            {
              if (handleShmemInit(decodeBuffer, outputOpcode,
                                      outputMessage, outputLength) < 0)
              {
                return -1;
              }

              outputFlush++;
            }
            else
            {
              #ifdef WARNING
              *logofs << "handleWrite: WARNING! Decoding unsupported shmem "
                      << "request OPCODE#" << (unsigned int) outputOpcode
                      << " for FD#" << fd_ << " as generic request.\n"
                      << logofs_flush;
              #endif

              MessageStore *messageStore = clientStore_ ->
                                   getRequestStore(X_NXInternalGenericRequest);

              hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                     outputOpcode, outputMessage, outputLength);
            }
          }
          else
          {
            MessageStore *messageStore = clientStore_ ->
                                 getRequestStore(X_NXInternalGenericRequest);

            hit = handleDecode(decodeBuffer, clientCache_, messageStore,
                                   outputOpcode, outputMessage, outputLength);

            if (control -> isProtoStep3() == 1 &&
                    outputOpcode > 127)
            {
              outputFlush++;
            }
          }
        }
      } // End of switch on opcode.

      //
      // A packed image request can generate more than just
      // a single X_PutImage. Write buffer is handled inside
      // handleUnpack(). Cannot simply assume that the final
      // opcode and size must be put at the buffer offset as
      // as buffer could have been grown or could have been
      // replaced by a scratch buffer. The same is true in
      // the case of a shared memory image.
      //

      if (outputOpcode != 0)
      {
        //
        // Commit opcode and size to the buffer.
        //

        *outputMessage = (unsigned char) outputOpcode;

        PutUINT(outputLength >> 2, outputMessage + 2, bigEndian_);

        #if defined(TEST) || defined(OPCODES)
        *logofs << "handleWrite: Handled request OPCODE#"
                << (unsigned int) outputOpcode << " ("
                << DumpOpcode(outputOpcode) << ") for FD#"
                << fd_ << " sequence " << clientSequence_
                << ". "  << outputLength << " bytes out.\n"
                << logofs_flush;
        #endif
      }
      #if defined(TEST) || defined(OPCODES)
      else
      {
        //
        // In case of shared memory images the log doesn't
        // reflect the actual opcode of the request that is
        // going to be written. It would be possible to find
        // the opcode of the original request received from
        // the remote proxy in member imageState_ -> opcode,
        // but we have probably already deleted the struct.
        //

        *logofs << "handleWrite: Handled image request for FD#"
                  << fd_ << " new sequence " << clientSequence_
                  << ". " << outputLength << " bytes out.\n"
                  << logofs_flush;
      }
      #endif

      //
      // Check if we produced enough data. We need to
      // decode all provided messages. Just update the
      // finish flag in case of failure.
      //

      if (outputFlush > 0)
      {
        handleFlush(flush_if_any);

        outputFlush = 0;
      }
      else
      {
        handleFlush(flush_if_needed);
      }

    } // End of while (decodeBuffer.decodeOpcodeValue(outputOpcode, 8, ...

  } // End of if (firstRequest_) ... else ...

  //
  // Write any remaining data to the X connection.
  //

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }

  //
  // Reset offset at which we read the
  // last event looking for the shared
  // memory completion.
  //

  if (shmemState_ != NULL)
  {
    shmemState_ -> checked = 0;
  }

  return 1;
}

//
// End of handleWrite().
//

//
// Other members.
//

int ServerChannel::handleSplit(EncodeBuffer& encodeBuffer, MessageStore *store, 
                                   const unsigned char *buffer, const unsigned int size)
{
  #ifdef PANIC
  *logofs << "ServerChannel: PANIC! Function handleSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int ServerChannel::handleSplit(EncodeBuffer &encodeBuffer, int packetLimit)
{
  #ifdef PANIC
  *logofs << "ServerChannel: PANIC! Function handleSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int ServerChannel::handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                     unsigned char *buffer, const unsigned int size)
{
  //
  // Find out if message has been split.
  //

  unsigned int value;

  decodeBuffer.decodeBool(value);

  if (value == 0)
  {
    return 0;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleUnsplit: Message of size " << size
          << " has been split.\n" << logofs_flush;
  #endif

  //
  // Get MD5 of the message being split.
  //

  md5_byte_t *checksum = NULL;

  if (control -> isProtoStep3() == 1 &&
          (control -> ImageCacheEnableLoad > 0 ||
               control -> ImageCacheEnableSave > 0))
  {
    checksum = new md5_byte_t[MD5_LENGTH];

    for (unsigned int i = 0; i < MD5_LENGTH; i++)
    {
      decodeBuffer.decodeValue(value, 8);

      if (checksum != NULL)
      {
        checksum[i] = (unsigned char) value;
      }
    }
  }

  clientStore_ -> getSplitStore() -> add(store, store -> lastAdded, checksum, buffer, size);

  delete [] checksum;

  #if defined(INFO) || defined(TEST)
  *logofs << "handleUnsplit: Added message to the split store.\n"
          << logofs_flush;
  #endif

  return 1;
}

int ServerChannel::handleUnsplit(DecodeBuffer &decodeBuffer)
{
  #if defined(INFO) || defined(TEST)
  *logofs << "handleUnsplit: Going to handle unsplit of messages "
          << " for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  if (clientStore_ -> getSplitStore() -> receive(decodeBuffer) < 0)
  {
    #ifdef PANIC
    *logofs << "handleUnsplit: PANIC! Receive of split for FD#" << fd_
            << " failed.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Receive of split for FD#" << fd_
         << " failed.\n";

    return -1;
  }

  //
  // Note that we don't need the client id at X server
  // side and, thus, we don't provide it at the time
  // we add split to the split store.
  //

  #if defined(INFO) || defined(TEST)
  if (clientStore_ -> getSplitStore() -> getLastPosition() != nothing)
  {
    *logofs << "handleUnsplit: Remote agent should commit object at "
            << "position " << clientStore_ -> getSplitStore() ->
               getLastPosition() << " to the X server now.\n"
            << logofs_flush;
  }
  #endif

  #if defined(INFO) || defined(TEST)
  *logofs << "handleUnsplit: There are still "
          << clientStore_ -> getSplitStore() -> getSize()
          << " messages to be transferred for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int ServerChannel::handleAbortSplit(EncodeBuffer &encodeBuffer)
{
  if (firstRequest_ == 1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleSplit: Delaying sending of abort splits "
            << "for FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleSplit: Going to send abort split messages "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  int totalAborts = 0;

  T_splits *splits = clientStore_ -> getSplitStore() -> getSplits();

  for (T_splits::iterator i = splits -> begin(); i != splits -> end(); i++)
  {
    Split *split = *i;

    //
    // If message has the abort flag set this means
    // that we have still to check if a matching
    // image exists on disk.
    //

    if (split -> getAbort() == 1)
    {
      if (clientStore_ -> getSplitStore() -> find(split) == 1)
      {
        //
        // A matching image was found. We send an
        // abort message followed by the checksum
        // of message we want to restore.
        //

        md5_byte_t *checksum = split -> getChecksum();

        #ifdef TEST
        *logofs << "handleSplit: Sending abort split for checksum ["
                << DumpChecksum(checksum) << "].\n" << logofs_flush;
        #endif

        encodeBuffer.encodeOpcodeValue(opcodeStore_ -> abortSplit, serverCache_ -> opcodeCache);

        serverOpcode_ = opcodeStore_ -> abortSplit;

        for (unsigned int i = 0; i < MD5_LENGTH; i++)
        {
          encodeBuffer.encodeValue((unsigned int) checksum[i], 8);
        }

        //
        // Update statistics for this special opcode.
        //

        int bits = encodeBuffer.getBits();

        #if defined(TEST) || defined(OPCODES)
        *logofs << "handleSplit: Handled event OPCODE#"
                << (unsigned int) opcodeStore_ -> abortSplit << " ("
                << DumpOpcode(opcodeStore_ -> split) << ")" << " for FD#"
                << fd_ << " sequence none. 0 bytes in, " << bits << " bits ("
                << ((float) bits) / 8 << " bytes) out.\n" << logofs_flush;
        #endif

        if (control -> CollectStatistics)
        {
          statistics -> addEventBits(opcodeStore_ -> abortSplit, 0, bits);
        }

        control -> addBitsInARow(bits);

        totalAborts++;
      }

      //
      // Be sure we skip this message the
      // next time this function is called.
      //

      split -> resetAbort();
    }
  }

  clientStore_ -> getSplitStore() -> resetAbort();

  if (totalAborts > 0)
  {
    //
    // Send event to remote proxy
    // as soon as possible.
    //

    priority_++;

    return 1;
  }

  return 0;
}

int ServerChannel::handleAbortSplit(DecodeBuffer &decodeBuffer)
{
  #ifdef PANIC
  *logofs << "ServerChannel: PANIC! Function handleAbortSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleAbortSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int ServerChannel::handleCommit(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                    unsigned char *&buffer, unsigned int &size)
{
  //
  // Get request type and position of image to commit.
  //

  unsigned char request;

  decodeBuffer.decodeOpcodeValue(request, clientCache_ -> opcodeCache);

  clientOpcode_ = request;

  unsigned int diffCommit;

  decodeBuffer.decodeValue(diffCommit, 32, 5);

  int lastCommit = clientStore_ -> getSplitStore() -> getLastCommit();

  lastCommit += diffCommit;

  clientStore_ -> getSplitStore() -> setLastCommit(lastCommit);

  unsigned int position = lastCommit;

  #if defined(INFO) || defined(TEST)
  *logofs << "handleCommit: Committing request OPCODE#"
          << (unsigned int) request << " with object at position "
          << position << " for FD#" << fd_ << " due to OPCODE#"
          << (unsigned int) opcodeStore_ -> commitSplit
          << ".\n" << logofs_flush;
  #endif

  MessageStore *store = clientStore_ -> getRequestStore(request);

  if (store == NULL)
  {
    #ifdef PANIC
    *logofs << "handleCommit: PANIC! Can't commit request OPCODE#"
            << request << " No message store found.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't commit request with opcode "
         << request << " No message store found.\n";

    return -1;
  }

  Message *message = store -> get(position);

  //
  // Commit can come after store has been
  // reset. In this case simply ignore it.
  //

  if (message == NULL)
  {
    #ifdef WARNING
    *logofs << "handleCommit: WARNING! Can't retrieve object of type "
            << (unsigned int) request << " from message store "
            << "at position " << position << ".\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Can't retrieve object of type "
         << (unsigned int) request << " from message store "
         << "at position " << position << ".\n";

    opcode = X_NoOperation;

    size   = 4;
    buffer = writeBuffer_.addMessage(size);

    return 0;
  }

  //
  // Allocate a buffer of the final length in
  // uncompressed form and expand message from
  // cache to the outgoing buffer.
  //

  size = store -> plainSize(position);

  buffer = writeBuffer_.addMessage(size);

  #ifdef TEST
  *logofs << "handleCommit: Prepared an outgoing buffer of "
          << size << " bytes.\n" << logofs_flush;
  #endif

  store -> unparse(message, buffer, size, bigEndian_);

  #ifdef DUMP

  store -> dump(message);

  #endif

  //
  // Unlock message so that we can remove
  // it or save on disk at shutdown.
  //

  store -> unlock(position);

  #if defined(INFO) || defined(TEST)

  checkLocks(store);

  #endif

  //
  // Now in the write buffer there is
  // an instance of this operation.
  //

  opcode = request;

  //
  // Save the sequence number to be able
  // to mask any error generated by the
  // request.

  for (int i = 0; i < MAX_COMMIT_SEQUENCE_QUEUE - 1; i++)
  {
    commitSequenceQueue_[i + 1] = commitSequenceQueue_[i];
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleCommit: Saved " << clientSequence_
          << " as last sequence number of image to commit.\n"
          << logofs_flush;
  #endif

  commitSequenceQueue_[0] = clientSequence_;

  return 1;
}

int ServerChannel::handleGeometry(unsigned char &opcode, unsigned char *&buffer,
                                      unsigned int &size)
{
  //
  // Replace the old geometry and taint
  // the message into a X_NoOperation.
  //

  int client = 0;

  if (control -> isProtoStep4() == 1)
  {
    client = *(buffer + 1);
  }

  #ifdef TEST
  *logofs << "handleGeometry: Setting new unpack geometry "
          << "for client " << client << ".\n"
          << logofs_flush;
  #endif

  handleUpdateAgentClients(client);

  handleUnpackStateInit(client);

  handleUnpackAllocGeometry(client);

  unpackState_[client] -> geometry -> depth1_bpp  = *(buffer + 4);
  unpackState_[client] -> geometry -> depth4_bpp  = *(buffer + 5);
  unpackState_[client] -> geometry -> depth8_bpp  = *(buffer + 6);
  unpackState_[client] -> geometry -> depth16_bpp = *(buffer + 7);
  unpackState_[client] -> geometry -> depth24_bpp = *(buffer + 8);
  unpackState_[client] -> geometry -> depth32_bpp = *(buffer + 9);

  unpackState_[client] -> geometry -> red_mask   = GetULONG(buffer + 12, bigEndian_);
  unpackState_[client] -> geometry -> green_mask = GetULONG(buffer + 16, bigEndian_);
  unpackState_[client] -> geometry -> blue_mask  = GetULONG(buffer + 20, bigEndian_);

  #ifdef TEST
  *logofs << "handleGeometry: Sending X_NoOperation request for FD#" 
          << fd_ << " due to OPCODE#" << (unsigned int) opcode
          << ".\n" << logofs_flush;
  #endif

  writeBuffer_.removeMessage(size - 4);

  size   = 4;
  opcode = X_NoOperation;

  return 1;
}

int ServerChannel::handleColormap(unsigned char &opcode, unsigned char *&buffer,
                                      unsigned int &size)
{
  //
  // Replace the old colormap and taint
  // the message into a X_NoOperation.
  //

  int client = 0;

  if (control -> isProtoStep4() == 1)
  {
    client = *(buffer + 1);
  }

  #ifdef TEST
  *logofs << "handleColormap: Setting new unpack colormap "
          << "for client " << client << ".\n"
          << logofs_flush;
  #endif

  handleUpdateAgentClients(client);

  handleUnpackStateInit(client);

  handleUnpackAllocColormap(client);

  unsigned int entries = GetULONG(buffer + 4, bigEndian_);

  if (size == entries * 4 + 8)
  {
    if (unpackState_[client] -> colormap -> entries != entries &&
            unpackState_[client] -> colormap -> data != NULL)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleColormap: Freeing previously "
              << "allocated unpack colormap.\n"
              << logofs_flush;
      #endif

      delete [] unpackState_[client] -> colormap -> data;

      unpackState_[client] -> colormap -> data    = NULL;
      unpackState_[client] -> colormap -> entries = 0;
    }

    if (entries > 0)
    {
      if (unpackState_[client] -> colormap -> data == NULL)
      {
        unpackState_[client] -> colormap -> data = new unsigned int[entries];
      }

      if (unpackState_[client] -> colormap -> data != NULL)
      {
        unpackState_[client] -> colormap -> entries = entries;

        #ifdef DEBUG
        *logofs << "handleColormap: Size of new colormap "
                << "data is " << entries * sizeof(unsigned int)
                << ".\n" << logofs_flush;
        #endif

        memcpy((unsigned char *) unpackState_[client] -> colormap -> data,
                   buffer + 8, entries * sizeof(unsigned int));

        #if defined(DEBUG) && defined(DUMP)

        *logofs << "handleColormap: Dumping colormap entries:\n"
                << logofs_flush;

        const unsigned int *p = (unsigned int *) buffer + 8;

        for (unsigned int i = 0; i < entries; i++)
        {
          *logofs << "handleColormap: [" << i << "] ["
                  << (void *) p[i] << "].\n"
                  << logofs_flush;
        }

        #endif
      }
      else
      {
        #ifdef PANIC
        *logofs << "handleColormap: PANIC! Can't allocate "
                << entries << " entries for unpack colormap "
                << "for FD#" << fd_ << ".\n" << logofs_flush;
        #endif
      }
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "handleColormap: PANIC! Bad size " << size
            << " for set unpack colormap message for FD#"
            << fd_ << " with " << entries << " entries.\n"
            << logofs_flush;
    #endif
  }

  #ifdef TEST
  *logofs << "handleColormap: Sending X_NoOperation request for FD#" 
          << fd_ << " due to OPCODE#" << (unsigned int) opcode
          << ".\n" << logofs_flush;
  #endif

  writeBuffer_.removeMessage(size - 4);

  size   = 4;
  opcode = X_NoOperation;

  return 1;
}

int ServerChannel::handleAlpha(unsigned char &opcode, unsigned char *&buffer,
                                      unsigned int &size)
{
  //
  // Check the version, even if we don't
  // have this message in versions prior
  // than 1.3.
  //
 
  int client = 0;

  if (control -> isProtoStep4() == 1)
  {
    client = *(buffer + 1);
  }

  #ifdef TEST
  *logofs << "handleAlpha: Setting new unpack alpha "
          << "for client " << client << ".\n"
          << logofs_flush;
  #endif

  handleUpdateAgentClients(client);

  handleUnpackStateInit(client);

  handleUnpackAllocAlpha(client);

  unsigned int entries = GetULONG(buffer + 4, bigEndian_);

  if (size == RoundUp4(entries) + 8)
  {
    if (unpackState_[client] -> alpha -> entries != entries &&
            unpackState_[client] -> alpha -> data != NULL)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleAlpha: Freeing previously allocated "
              << "unpack alpha data.\n" << logofs_flush;
      #endif

      delete [] unpackState_[client] -> alpha -> data;

      unpackState_[client] -> alpha -> data    = NULL;
      unpackState_[client] -> alpha -> entries = 0;
    }

    if (entries > 0)
    {
      if (unpackState_[client] -> alpha -> data == NULL)
      {
        unpackState_[client] -> alpha -> data = new unsigned char[entries];
      }

      if (unpackState_[client] -> alpha -> data != NULL)
      {
        unpackState_[client] -> alpha -> entries = entries;

        #ifdef DEBUG
        *logofs << "handleAlpha: Size of new alpha data is "
                << entries * sizeof(unsigned char) << ".\n"
                << logofs_flush;
        #endif

        memcpy((unsigned char *) unpackState_[client] -> alpha -> data,
                   buffer + 8, entries * sizeof(unsigned char));

        #if defined(DEBUG) && defined(DUMP)

        *logofs << "handleAlpha: Dumping alpha entries:\n"
                << logofs_flush;

        const unsigned char *p = buffer + 8;

        for (unsigned int i = 0; i < entries; i++)
        {
          *logofs << "handleAlpha: [" << i << "] ["
                  << (void *) ((int) p[i]) << "].\n"
                  << logofs_flush;
        }

        #endif
      }
      else
      {
        #ifdef PANIC
        *logofs << "handleAlpha: PANIC! Can't allocate "
                << entries << " entries for unpack alpha data "
                << "for FD#" << fd_ << ".\n" << logofs_flush;
        #endif
      }
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "handleAlpha: PANIC! Bad size " << size
            << " for set unpack alpha message for FD#"
            << fd_ << " with " << entries << " entries.\n"
            << logofs_flush;
    #endif
  }

  #ifdef TEST
  *logofs << "handleAlpha: Sending X_NoOperation request for FD#" 
          << fd_ << " due to OPCODE#" << (unsigned int) opcode
          << ".\n" << logofs_flush;
  #endif

  writeBuffer_.removeMessage(size - 4);

  size   = 4;
  opcode = X_NoOperation;

  return 1;
}

int ServerChannel::handleImage(unsigned char &opcode, unsigned char *&buffer,
                                   unsigned int &size, unsigned int &flush)
{
  int result = 1;

  //
  // Save the original opcode together with
  // the image state so we can later find if
  // this is a plain or a packed image when
  // moving data to the shared memory area.
  //

  handleImageStateAlloc(opcode);

  if (opcode == opcodeStore_ -> putPackedImage)
  {
    //
    // Some packed images are unpacked to a series
    // of X_ChangeGC and X_PolyFillRectangle so we
    // need to update our sequence counter.
    //

    unsigned char method = *(buffer + 12);

    unsigned int requests = 1;

    if (method == PACK_RDP_TEXT)
    {
      unsigned int width = GetUINT(buffer + 28, bigEndian_);

      requests = (width * 2);

      clientSequence_ += requests;
      clientSequence_ &= 0xffff;

      #ifdef DEBUG
      *logofs << "handleImage: Updated last client sequence number "
              << "for FD#" << fd_ << " to " << clientSequence_
              << ".\n" << logofs_flush;
      #endif
    }

    //
    // Unpack the image and put X_PutImage or any
    // other graphical request in a new buffer.
    // Save the expected output size, so, in case
    // of error, we can still update the statistics.
    //

    int length = GetULONG(buffer + 20, bigEndian_);

    #ifdef TEST
    *logofs << "handleImage: Sending requests for FD#" << fd_
            << " due to OPCODE#" << (unsigned int) opcode << " with "
            << GetULONG(buffer + 16, bigEndian_) << " bytes packed "
            << "and " << GetULONG(buffer + 20, bigEndian_)
            << " bytes unpacked.\n" << logofs_flush;
    #endif

    if (control -> CollectStatistics)
    {
      statistics -> addPackedBytesIn(size);
    }

    result = handleUnpack(opcode, buffer, size);

    if (result < 0)
    {
      //
      // Recover from error. Send as many X_NoOperation
      // as needed to keep the sequence counter in sync
      // with the remote peer.
      //

      for (unsigned int i = 0; i < requests; i++)
      {
        size   = 4;
        buffer = writeBuffer_.addMessage(size);

        *buffer = X_NoOperation;

        PutUINT(size >> 2, buffer + 2, bigEndian_);

        #ifdef PANIC
        *logofs << "handleImage: PANIC! Sending X_NoOperation for FD#"
                << fd_ << " to recover from failed unpack.\n"
                << logofs_flush;
        #endif
      }

      //
      // Set output length to reflect the amount of data
      // that would have been produced by unpacking the 
      // image. This is advisable to keep the counters
      // in sync with those at remote proxy and doesn't
      // have any effect on the size of data sent to the
      // X server as the latter is taken from the actual
      // content of the write buffer.
      //

      size = length;

      if (method != PACK_RDP_TEXT)
      {
        size += 24;
      }
    }

    if (control -> CollectStatistics)
    {
      statistics -> addPackedBytesOut(size);
    }

    //
    // Refrain the write loop from putting
    // opcode and size in the output buffer.
    //

    opcode = 0;
  }

  //
  // Now image is unpacked as a X_PutImage
  // in write buffer. Check if we can send
  // the image using the MIT-SHM extension.
  //

  if (result > 0)
  {
    result = handleShmem(opcode, buffer, size, flush);

    //
    // We already put opcode and size in
    // the resulting buffer.
    //

    if (result > 0)
    {
      opcode = 0;
    }
  }

  return 1;
}

int ServerChannel::handleMotion(EncodeBuffer& encodeBuffer, int forceMotion)
{
  if (lastMotion_[0] == '\0')
  {
    #ifdef DEBUG
    *logofs << "handleMotion: No motion events to send for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (forceMotion == 0)
  {
    int diffTs = diffTimestamp(lastMotionTs_, getTimestamp());

    if (diffTs < (control -> MotionTimeout -
                      control -> LatencyTimeout))
    {
      #ifdef DEBUG
      *logofs << "handleMotion: It's not time to flush "
              << "motion event yet for FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      return 0;
    }
  }

  //
  // Replicate code from read loop. When have
  // time and wish, try to split everything
  // in functions.
  //

  const unsigned char *buffer = lastMotion_;
  unsigned char opcode = *lastMotion_;
  unsigned int size = 32;

  if (GetUINT(buffer + 2, bigEndian_) < serverSequence_)
  {
    PutUINT(serverSequence_, (unsigned char *) buffer + 2, bigEndian_);
  }

  encodeBuffer.encodeOpcodeValue(opcode, serverCache_ -> opcodeCache);

  serverOpcode_ = opcode;

  unsigned int sequenceNum = GetUINT(buffer + 2, bigEndian_);

  unsigned int sequenceDiff = sequenceNum - serverSequence_;

  serverSequence_ = sequenceNum;

  #ifdef DEBUG
  *logofs << "handleRead: Last server sequence number for FD#" 
          << fd_ << " is " << serverSequence_ << " with "
          << "difference " << sequenceDiff << ".\n"
          << logofs_flush;
  #endif

  if (serverSequenceReset_ == 1)
  {
    encodeBuffer.encodeValue(serverSequence_, 32, 16);

    #if defined(INFO) || defined(TEST)
    *logofs << "handleMotion: Reset last server sequence number to "
            << serverSequence_ << " for FD#" 
            << fd_ << ".\n" << logofs_flush;
    #endif

    serverSequenceReset_ = 0;
  }
  else
  {
    encodeBuffer.encodeCachedValue(sequenceDiff, 16,
                       serverCache_ -> eventSequenceCache, 7);
  }

  //
  // Timeout is elapsed so it's a good
  // idea to send immediately the event
  // to proxy.
  //

  if (forceMotion == 0)
  {
    priority_++;
  }

  //
  // If we fast encoded the message
  // then skip the rest.
  //

  if (control -> LocalDeltaCompression == 0)
  {
    int result = handleFastReadEvent(encodeBuffer, opcode,
                                         buffer, size);

    #if defined(INFO) || defined(TEST)
    *logofs << "handleMotion: Sent saved motion event for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    lastMotion_[0] = '\0';

    #if defined(INFO) || defined(TEST)
    *logofs << "handleMotion: Reset last motion event for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    if (result < 0)
    {
      return -1;
    }
    else if (result > 0)
    {
      return 1;
    }
  }

  //
  // This should be just the part specific
  // for motion events but is currently a
  // copy-paste of code from read loop.
  //

  unsigned char detail = buffer[1];
  if (*buffer == MotionNotify)
    encodeBuffer.encodeBool((unsigned int) detail);
  else if ((*buffer == EnterNotify) || (*buffer == LeaveNotify))
    encodeBuffer.encodeValue((unsigned int) detail, 3);
  else if (*buffer == KeyRelease)
  {
    if (detail == serverCache_ -> keyPressLastKey)
      encodeBuffer.encodeBool(1);
    else
    {
      encodeBuffer.encodeBool(0);
      encodeBuffer.encodeValue((unsigned int) detail, 8);
    }
  }
  else if ((*buffer == ButtonPress) || (*buffer == ButtonRelease))
    encodeBuffer.encodeCachedValue(detail, 8,
                                   serverCache_ -> buttonCache);
  else
    encodeBuffer.encodeValue((unsigned int) detail, 8);
  unsigned int timestamp = GetULONG(buffer + 4, bigEndian_);
  unsigned int timestampDiff = timestamp - serverCache_ -> lastTimestamp;
  serverCache_ -> lastTimestamp = timestamp;
  encodeBuffer.encodeCachedValue(timestampDiff, 32,
                      serverCache_ -> motionNotifyTimestampCache, 9);
  int skipRest = 0;
  if (*buffer == KeyRelease)
  {
    skipRest = 1;
    for (unsigned int i = 8; i < 31; i++)
    {
      if (buffer[i] != serverCache_ -> keyPressCache[i - 8])
      {
        skipRest = 0;
        break;
      }
    }
    encodeBuffer.encodeBool(skipRest);
  }
  if (!skipRest)
  {
    const unsigned char *nextSrc = buffer + 8;
    for (unsigned int i = 0; i < 3; i++)
    {
      encodeBuffer.encodeCachedValue(GetULONG(nextSrc, bigEndian_), 29,
                         *serverCache_ -> motionNotifyWindowCache[i], 6);
      nextSrc += 4;
    }
    unsigned int rootX = GetUINT(buffer + 20, bigEndian_);
    unsigned int rootY = GetUINT(buffer + 22, bigEndian_);
    unsigned int eventX = GetUINT(buffer + 24, bigEndian_);
    unsigned int eventY = GetUINT(buffer + 26, bigEndian_);
    eventX -= rootX;
    eventY -= rootY;
    encodeBuffer.encodeCachedValue(rootX -
          serverCache_ -> motionNotifyLastRootX, 16,
          serverCache_ -> motionNotifyRootXCache, 6);
    serverCache_ -> motionNotifyLastRootX = rootX;
    encodeBuffer.encodeCachedValue(rootY -
          serverCache_ -> motionNotifyLastRootY, 16,
          serverCache_ -> motionNotifyRootYCache, 6);
    serverCache_ -> motionNotifyLastRootY = rootY;
    encodeBuffer.encodeCachedValue(eventX, 16,
                         serverCache_ -> motionNotifyEventXCache, 6);
    encodeBuffer.encodeCachedValue(eventY, 16,
                         serverCache_ -> motionNotifyEventYCache, 6);
    encodeBuffer.encodeCachedValue(GetUINT(buffer + 28, bigEndian_),
                         16, serverCache_ -> motionNotifyStateCache);
    if ((*buffer == EnterNotify) || (*buffer == LeaveNotify))
      encodeBuffer.encodeValue((unsigned int) buffer[30], 2);
    else
      encodeBuffer.encodeBool((unsigned int) buffer[30]);
    if ((*buffer == EnterNotify) || (*buffer == LeaveNotify))
      encodeBuffer.encodeValue((unsigned int) buffer[31], 2);
    else if (*buffer == KeyPress)
    {
      serverCache_ -> keyPressLastKey = detail;
      for (unsigned int i = 8; i < 31; i++)
      {
        serverCache_ -> keyPressCache[i - 8] = buffer[i];
      }
    }
  }

  //
  // Print info about achieved compression
  // and update the statistics.
  //

  int bits = encodeBuffer.getBits();

  #if defined(TEST) || defined(OPCODES)
  *logofs << "handleMotion: Handled event OPCODE#" << (unsigned int) buffer[0]
          << " for FD#" << fd_ << " sequence " << sequenceNum << ". "
          << size << " bytes in, " << bits << " bits (" << ((float) bits) / 8
          << " bytes) out.\n" << logofs_flush;
  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addEventBits(*buffer, size << 3, bits);
  }

  control -> addOutputInARow(size);

  control -> addBitsInARow(bits);

  #if defined(INFO) || defined(TEST)
  *logofs << "handleMotion: Sent saved motion event for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  lastMotion_[0] = '\0';

  #if defined(INFO) || defined(TEST)
  *logofs << "handleMotion: Reset last motion event for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int ServerChannel::handleConfiguration()
{
  #ifdef TEST
  *logofs << "ServerChannel: Setting new buffer parameters "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  readBuffer_.setSize(control -> ServerInitialReadSize,
                          control -> ServerMaximumReadSize);

  writeBuffer_.setSize(control -> TransportXBufferSize,
                           control -> TransportXBufferThreshold,
                               control -> TransportMaximumBufferSize);

  transport_ -> setSize(control -> TransportXBufferSize,
                            control -> TransportXBufferThreshold,
                                control -> TransportMaximumBufferSize);
  return 1;
}

int ServerChannel::handleFinish()
{
  #ifdef TEST
  *logofs << "ServerChannel: Finishing connection for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  pending_ = 0;
  finish_  = 1;

  //
  // Reset motion events.
  //

  lastMotion_[0] = '\0';

  transport_ -> fullReset();

  return 1;
}

int ServerChannel::handleReset()
{
  #ifdef TEST
  *logofs << "handleReset: Resetting server channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Reset state in base channel class.
  //

  Channel::handleReset();

  //
  // Reallocate new caches. In protocol version 3
  // this is done in proxy that later propagates
  // pointers to channels
  //

  if (control -> isProtoStep3() == 0)
  {
    if (clientCache_ != NULL)
    {
      delete clientCache_;

      clientCache_ = new ClientCache();
    }

    if (serverCache_ != NULL)
    {
      delete serverCache_;

      serverCache_ = new ServerCache();
    }

    #ifdef TEST
    *logofs << "handleReset: Replaced encode caches in channel "
            << "for FD#" << fd_ << ".\n" << logofs_flush;
    #endif
  }

  //
  // Force channel to synchronize sequence
  // counters with the remote end.
  //

  clientSequenceReset_ = 1;
  serverSequenceReset_ = 1;

  //
  // Remove from queue any request for which
  // we were waiting for a reply.
  //

  unsigned short int requestSequenceNum;
  unsigned char      requestOpcode;
  unsigned int       requestData[3];

  while (sequenceQueue_.pop(requestSequenceNum, requestOpcode,
                                requestData[0], requestData[1], requestData[2]))
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleReset: Discarding reference to request "
            << (unsigned int) requestOpcode << " with sequence "
            << requestSequenceNum << ". Request data is "
            << requestData[0] << " " << requestData[1] << " "
            << requestData[2] << ".\n" << logofs_flush;
    #endif
  }

  clientOpcode_ = 0;
  serverOpcode_ = 0;

  syncCounter_ = 0;
  firstSyncTs_ = nullTimestamp();

  //
  // We don't need to reset:
  //
  // 1. Any saved motion event.
  //
  // 2. Sequence numbers of split commits
  //    saved to suppress errors.
  //
  // 3. Last unpack colormap and alpha channel.
  //
  // 4. Current unpack geometry.
  //
  // 5. Agent's settings for expose events.
  //

  #ifdef TEST
  *logofs << "handleReset: Completed reset of server channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int ServerChannel::handleUnpack(unsigned char &opcode, unsigned char *&buffer,
                                    unsigned int &size)
{
  int client = 0;

  if (control -> isProtoStep4() == 1)
  {
    client = *(buffer + 1);
  }

  #ifdef DEBUG
  *logofs << "handleUnpack: Unpacking image for client " << client
          << " with method " << (unsigned int) *(buffer + 12)
          << ".\n" << logofs_flush;
  #endif

  handleUpdateAgentClients(client);

  handleUnpackStateInit(client);

  T_geometry *geometryState = unpackState_[client] -> geometry;
  T_colormap *colormapState = unpackState_[client] -> colormap;
  T_alpha    *alphaState    = unpackState_[client] -> alpha;

  if (geometryState == NULL)
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Missing geometry unpacking "
            << "image for client " << client << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Missing geometry unpacking "
         << "image for client " << client << ".\n";

    return -1;
  }

  //
  // Get image data from buffer.
  //

  imageState_ -> drawable  = GetULONG(buffer + 4,  bigEndian_);
  imageState_ -> gcontext  = GetULONG(buffer + 8,  bigEndian_);

  imageState_ -> method    = *(buffer + 12);

  imageState_ -> format    = *(buffer + 13);
  imageState_ -> srcDepth  = *(buffer + 14);
  imageState_ -> dstDepth  = *(buffer + 15);

  imageState_ -> srcLength = GetULONG(buffer + 16,  bigEndian_);
  imageState_ -> dstLength = GetULONG(buffer + 20,  bigEndian_);

  imageState_ -> srcX      = GetUINT(buffer + 24, bigEndian_);
  imageState_ -> srcY      = GetUINT(buffer + 26, bigEndian_);
  imageState_ -> srcWidth  = GetUINT(buffer + 28, bigEndian_);
  imageState_ -> srcHeight = GetUINT(buffer + 30, bigEndian_);

  imageState_ -> dstX      = GetUINT(buffer + 32, bigEndian_);
  imageState_ -> dstY      = GetUINT(buffer + 34, bigEndian_);
  imageState_ -> dstWidth  = GetUINT(buffer + 36, bigEndian_);
  imageState_ -> dstHeight = GetUINT(buffer + 38, bigEndian_);

  #ifdef TEST
  *logofs << "handleUnpack: Source X is " << imageState_ -> srcX
          << " Y is " << imageState_ -> srcY << " width is "
          << imageState_ -> srcWidth << " height is "
          << imageState_ -> srcHeight << ".\n"
          << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "handleUnpack: Destination X is " << imageState_ -> dstX
          << " Y is " << imageState_ -> dstY << " width is "
          << imageState_ -> dstWidth << " height is "
          << imageState_ -> dstHeight << ".\n"
          << logofs_flush;
  #endif

  if (imageState_ -> srcX != 0 || imageState_ -> srcY != 0)
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Unsupported source coordinates "
            << "in unpack request.\n" << logofs_flush;
    #endif

    return -1;
  }
  else if ((imageState_ -> method == PACK_RDP_PLAIN_256_COLORS ||
               imageState_ -> method == PACK_RDP_COMPRESSED_256_COLORS ||
                   imageState_ -> method == PACK_COLORMAP_256_COLORS) &&
                       (colormapState == NULL || colormapState -> data == NULL))
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Cannot find any unpack colormap.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  //
  // Field srcLength carries size of image data in
  // packed format. Field dstLength is size of the
  // image in the original X bitmap format.
  //

  unsigned int srcDataOffset = 40;

  unsigned int srcSize = imageState_ -> srcLength;

  unsigned int removeSize = size;

  unsigned char *srcData = buffer + srcDataOffset;

  //
  // Handle the special case of RDP text.
  //

  if (imageState_ -> method == PACK_RDP_TEXT)
  {
    unsigned int dstElements = (srcSize - 16) / 20;

    //
    // It should match dstWidth.
    //

    if (dstElements != imageState_ -> dstWidth)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! RDP text elements look "
              << dstElements << " while they should be "
              << imageState_ -> dstWidth << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": RDP text elements look "
           << dstElements << " while they should be "
           << imageState_ -> dstWidth << ".\n";

      return -1;
    }

    //
    // Message represents a RDP glyph that is displayed
    // by translating it in a series of X_CreatePixmap,
    // X_ChangeGC, X_PutImage and X_PolyFillRectangle
    // messages. Refer to libXcompext implementation to
    // find out the exact mapping.
    //

    unsigned int dstSize = ((12 + (4 * 3)) +
                                (12 + (4 * 3) +
                                     12 + (1 * 8)) * dstElements);

    if (dstSize != imageState_ -> dstLength)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! RDP text has destination size "
              << dstSize << " with " << dstElements << " elements and "
              << "reported size " << imageState_ -> dstLength << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": RDP text has destination size "
           << dstSize << " with " << dstElements << " elements and "
           << "reported size " << imageState_ -> dstLength << ".\n";
 
      return -1;
    }

    unsigned char *dstData = writeBuffer_.addScratchMessage(dstSize);

    if (dstData == NULL)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! Can't allocate " << dstSize
              << " bytes for RDP text.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate " << dstSize
           << " bytes for RDP text.\n";

      writeBuffer_.removeMessage(removeSize);

      return -1;
    }

    int result = UnpackRDPText(imageState_ -> drawable, imageState_ -> gcontext,
                                   bigEndian_, srcData, srcSize, dstData, dstSize);

    writeBuffer_.removeMessage(removeSize);

    if (result <= 0)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! Failed to unpack RDP text.\n"
              << logofs_flush;
      #endif

      writeBuffer_.removeScratchMessage();

      return -1;
    }

    buffer = dstData;
    size   = dstSize;

    return 1;
  }

  //
  // Get source and destination bits per pixel.
  //

  int srcBitsPerPixel = GetBitsPerPixel(imageState_ -> method);

  if (srcBitsPerPixel <= 0)
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Can't identify "
            << "source bits per pixel for method "
            << (unsigned int) imageState_ -> method
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't identify "
         << "source bits per pixel for method "
         << (unsigned int) imageState_ -> method
         << ".\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "handleUnpack: Source bits per pixel are "
          << srcBitsPerPixel << " source data size is "
          << srcSize << ".\n" << logofs_flush;
  #endif

  int dstBitsPerPixel = GetBitsPerPixel(geometryState, imageState_ -> dstDepth);

  if (dstBitsPerPixel <= 0)
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Can't identify "
            << "destination bits per pixel for depth "
            << (unsigned int) imageState_ -> dstDepth
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't identify "
         << "destination bits per pixel for depth "
         << (unsigned int) imageState_ -> dstDepth
         << ".\n";

    return -1;
  }

  //
  // Destination is a PutImage request.
  //

  unsigned int dstDataOffset = 24;

  //
  // Output buffer size must match the number of input
  // pixels multiplied by the number of bytes per pixel
  // of current geometry.
  //

  size = (RoundUp4(imageState_ -> dstWidth * dstBitsPerPixel / 8) *
                       imageState_ -> dstHeight) + dstDataOffset;

  #ifdef TEST
  *logofs << "handleUnpack: Destination bits per pixel are "
          << dstBitsPerPixel << " destination data size is "
          << size - dstDataOffset << ".\n" << logofs_flush;
  #endif

  //
  // General case initialization. Anyway, if the image is
  // tight encoded and the uncompressed image is bigger
  // than 256K, we will need to generate multiple image
  // requests.
  //

  unsigned int dstSize = size - dstDataOffset;

  imageState_ -> dstLines = imageState_ -> dstHeight;

  unsigned char *dstData;

  //
  // Size of the final output buffer had to be stored
  // in the offset field of XImage/NXPackedImage.
  //

  #ifdef WARNING

  if (dstSize != imageState_ -> dstLength)
  {
    *logofs << "handleUnpack: WARNING! Destination size mismatch "
            << "with reported " << imageState_ -> dstLength
            << " and actual " << dstSize << ".\n"
            << logofs_flush;
  }

  #endif

  //
  // TODO: Check if the encoding side is broken. It
  // is not simple to figure out if the problem is
  // in the RFB agent or in the Xvnc server. In any
  // case we need to skip the decompression in case
  // of manifest size error.
  //

  if (size > MESSAGE_DATA_LIMIT &&
         imageState_ -> method == PACK_RFB_HEXTILE)
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Failed to unpack image "
            << "with method '" << (unsigned int) imageState_ -> method
            << "' and requested size " << size << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Failed to unpack image "
         << "with method '" << (unsigned int) imageState_ -> method
         << "' and requested size " << size << ".\n";

    writeBuffer_.removeMessage(removeSize);

    return -1;
  }
  else if (imageState_ -> method == PACK_RFB_TIGHT_PLAIN ||
               imageState_ -> method == PACK_RFB_TIGHT_COMPRESSED)
  {
    //
    // The case of tight encoding is also special.
    // The unpack procedure can generate messages
    // that exceed the maximum allowed size of a
    // single X request. To simplify things we let
    // the decompressor handle the write buffer by
    // itself.
    //

    //
    // TODO: Size of unpacked data to be written is
    // reported to be 262144 when multiple messages
    // are generated and this is wrong as it should
    // take into account the first 24 bytes of the
    // X_PutImage.
    //

    handleTightStateAlloc();

    unsigned int finalSize = 0;

    if (size > MESSAGE_DATA_LIMIT)
    {
      imageState_ -> dstLines = (MESSAGE_DATA_LIMIT - dstDataOffset) /
                                    RoundUp4(imageState_ -> dstWidth * dstBitsPerPixel / 8);

      size = imageState_ -> dstLines *
                 RoundUp4(imageState_ -> dstWidth * dstBitsPerPixel / 8) +
                     dstDataOffset;

      buffer = writeBuffer_.addScratchMessage(size);

      dstData = new unsigned char [dstSize];

      //
      // Calculate the total size of the messages
      // to be generated inside UnpackTight.
      //

      finalSize = ((unsigned int) (imageState_ -> dstHeight /
                       imageState_ -> dstLines)) * size;
 
      if (imageState_ -> dstHeight % imageState_ -> dstLines > 0)
      {
        finalSize += dstDataOffset +
                         RoundUp4(imageState_ -> dstWidth * dstBitsPerPixel / 8) *
                             (imageState_ -> dstHeight % imageState_ -> dstLines);
      }
    }
    else
    {
      buffer = writeBuffer_.addScratchMessage(size);

      dstData = buffer + dstDataOffset;
    }

    int result;

    result = tightState_ -> UnpackTight(imageState_ -> method, colormapState,
                                            srcData, srcSize, imageState_ -> dstHeight,
                                                imageState_ -> dstWidth, dstData, dstSize,
                                                    imageState_ -> dstHeight, imageState_ -> dstWidth,
                                                        dstBitsPerPixel, imageState_ -> drawable,
                                                            imageState_ -> gcontext, imageState_ -> dstX,
                                                                imageState_ -> dstY, imageState_ -> dstDepth,
                                                                    &writeBuffer_, transport_,
                                                                        bigEndian_, buffer);
    if (result < 0)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! Failed to unpack image "
              << "with method '" << (unsigned int) imageState_ -> method
              << "' and requested size " << size << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to unpack image "
           << "with method '" << (unsigned int) imageState_ -> method
           << "' and requested size " << size << ".\n";

      return -1;
    }

    //
    // Final size is not set yet if we
    // didn't split the unpacked image.
    //

    if (finalSize == 0)
    {
      *buffer = (unsigned char) X_PutImage;

      *(buffer + 1) = imageState_ -> format;

      PutUINT(size >> 2, buffer + 2, bigEndian_);

      PutULONG(imageState_ -> drawable, buffer + 4,  bigEndian_);
      PutULONG(imageState_ -> gcontext, buffer + 8,  bigEndian_);

      PutUINT(imageState_ -> dstWidth, buffer + 12, bigEndian_);
      PutUINT(imageState_ -> dstLines, buffer + 14, bigEndian_);

      PutUINT(imageState_ -> dstX, buffer + 16, bigEndian_);
      PutUINT(imageState_ -> dstY, buffer + 18, bigEndian_);

      *(buffer + 20) = 0;
      *(buffer + 21) = imageState_ -> dstDepth;

      finalSize = size;
    }

    if (finalSize > size)
    {
      size = finalSize;
    }

    writeBuffer_.removeMessage(removeSize);

    return 1;
  }

  //
  // The decoding algorithm has put the packed image
  // in the plain write buffer. The basic idea is to
  // use the scratch buffer to uncompress the image.
  //

  if (imageState_ -> method != PACK_RDP_COMPRESSED_256_COLORS)
  {
    buffer = writeBuffer_.addScratchMessage(size);

    dstData = buffer + dstDataOffset;
  }
  else
  {
    //
    // If data contains a compressed RDP bitmap which
    // must be first decompressed to be unpacked, we
    // invert the logic. Allocate data to be unpacked
    // in scratch buffer and use the write buffer for
    // the final.
    //

    unsigned int rdpSize = imageState_ -> srcWidth * imageState_ -> srcHeight *
                               imageState_ -> srcDepth / 8;

    unsigned char *rdpData = writeBuffer_.addScratchMessage(rdpSize);

    if (rdpData == NULL)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! Can't allocate " << rdpSize
              << " bytes for compressed RDP image.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate " << rdpSize
           << " bytes for compressed RDP image.\n";

      writeBuffer_.removeMessage(removeSize);

      return -1;
    }

    //
    // Please note that in the case of RDP we
    // actually use dstHeight, not srcHeight.
    //

    if (UnpackRDPTo8(rdpData, imageState_ -> srcWidth,
                         imageState_ -> dstHeight, srcData, srcSize) < 0)
    {
      #ifdef PANIC
      *logofs << "handleUnpack: PANIC! Can't decompress RDP image.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't decompress RDP image.\n";

      writeBuffer_.removeMessage(removeSize);

      writeBuffer_.removeScratchMessage();

      return -1;
    }

    //
    // Now we have the uncompressed RDP bitmap
    // into the write buffer's scratch message
    // Prepare the write buffer to contain the
    // unpacked image.
    //

    writeBuffer_.removeMessage(removeSize);

    buffer = writeBuffer_.addMessage(size);

    dstData = buffer + dstDataOffset;

    srcData = rdpData;
    srcSize = rdpSize;
  }

  //
  // Unpack image into the buffer.
  //

  *buffer = (unsigned char) X_PutImage;

  *(buffer + 1) = imageState_ -> format;

  PutUINT(size >> 2, buffer + 2, bigEndian_);

  PutULONG(imageState_ -> drawable, buffer + 4,  bigEndian_);
  PutULONG(imageState_ -> gcontext, buffer + 8,  bigEndian_);

  PutUINT(imageState_ -> dstWidth, buffer + 12, bigEndian_);
  PutUINT(imageState_ -> dstLines, buffer + 14, bigEndian_);

  PutUINT(imageState_ -> dstX, buffer + 16, bigEndian_);
  PutUINT(imageState_ -> dstY, buffer + 18, bigEndian_);

  *(buffer + 20) = 0;
  *(buffer + 21) = imageState_ -> dstDepth;

  #ifdef TEST
  *logofs << "handleUnpack: Write buffer size is "
          << writeBuffer_.getLength() << " scratch size is "
          << writeBuffer_.getScratchLength() << ".\n"
          << logofs_flush;
  #endif

  int result = 0;

  switch (imageState_ -> method)
  {
    case PACK_RDP_PLAIN_256_COLORS:
    case PACK_RDP_COMPRESSED_256_COLORS:
    {
      result = Unpack8(geometryState, colormapState, srcBitsPerPixel,
                           imageState_ -> srcWidth, imageState_ -> srcHeight, srcData,
                               srcSize, dstBitsPerPixel, imageState_ -> dstWidth,
                                   imageState_ -> dstHeight, dstData, dstSize);
      break;
    }
    case PACK_RFB_HEXTILE:
    {
      result = UnpackHextile(imageState_ -> srcWidth, imageState_ -> srcHeight, srcData,
                                 srcSize, dstBitsPerPixel, imageState_ -> dstWidth,
                                     imageState_ -> dstHeight, dstData, dstSize);
      break;
    }
    case PACK_JPEG_8_COLORS:
    case PACK_JPEG_64_COLORS:
    case PACK_JPEG_256_COLORS:
    case PACK_JPEG_512_COLORS:
    case PACK_JPEG_4K_COLORS:
    case PACK_JPEG_32K_COLORS:
    case PACK_JPEG_64K_COLORS:
    case PACK_JPEG_256K_COLORS:
    case PACK_JPEG_2M_COLORS:
    case PACK_JPEG_16M_COLORS:
    {
      result = UnpackJpeg(geometryState, imageState_ -> method, srcData,
                              srcSize, dstBitsPerPixel, imageState_ -> dstWidth,
                                  imageState_ -> dstHeight, dstData, dstSize);
      break;
    }
    case PACK_PNG_8_COLORS:
    case PACK_PNG_64_COLORS:
    case PACK_PNG_256_COLORS:
    case PACK_PNG_512_COLORS:
    case PACK_PNG_4K_COLORS:
    case PACK_PNG_32K_COLORS:
    case PACK_PNG_64K_COLORS:
    case PACK_PNG_256K_COLORS:
    case PACK_PNG_2M_COLORS:
    case PACK_PNG_16M_COLORS:
    {
      result = UnpackPng(geometryState, imageState_ -> method, srcData,
                             srcSize, dstBitsPerPixel, imageState_ -> dstWidth,
                                 imageState_ -> dstHeight, dstData, dstSize);
      break;
    }
    case PACK_TIGHT_8_COLORS:
    case PACK_TIGHT_64_COLORS:
    case PACK_TIGHT_256_COLORS:
    case PACK_TIGHT_512_COLORS:
    case PACK_TIGHT_4K_COLORS:
    case PACK_TIGHT_32K_COLORS:
    case PACK_TIGHT_64K_COLORS:
    case PACK_TIGHT_256K_COLORS:
    case PACK_TIGHT_2M_COLORS:
    case PACK_TIGHT_16M_COLORS:
    {
      handleTightStateAlloc();

      result = tightState_ -> UnpackTight(geometryState, colormapState,
                                              imageState_ -> method, srcData, srcSize,
                                                  imageState_ -> dstDepth, dstBitsPerPixel,
                                                      imageState_ -> dstWidth, imageState_ -> dstHeight,
                                                          dstData, dstSize);
      break;
    }
    default:
    {
      const T_colormask *colorMask = GetColorMask(imageState_ -> method);

      switch (imageState_ -> method)
      {
        case PACK_MASKED_8_COLORS:
        case PACK_MASKED_64_COLORS:
        case PACK_MASKED_256_COLORS:
        {
          result = Unpack8(geometryState, colorMask, imageState_ -> srcDepth,
                               imageState_ -> srcWidth, imageState_ -> srcHeight,
                                   srcData, srcSize, imageState_ -> dstDepth,
                                       imageState_ -> dstWidth, imageState_ -> dstHeight,
                                           dstData, dstSize);
          break;
        }
        case PACK_MASKED_512_COLORS:
        case PACK_MASKED_4K_COLORS:
        case PACK_MASKED_32K_COLORS:
        case PACK_MASKED_64K_COLORS:
        {
          result = Unpack16(geometryState, colorMask, imageState_ -> srcDepth,
                                imageState_ -> srcWidth, imageState_ -> srcHeight,
                                    srcData, srcSize, imageState_ -> dstDepth,
                                        imageState_ -> dstWidth, imageState_ -> dstHeight,
                                            dstData, dstSize);
          break;
        }
        case PACK_MASKED_256K_COLORS:
        case PACK_MASKED_2M_COLORS:
        case PACK_MASKED_16M_COLORS:
        {
          result = Unpack24(geometryState, colorMask, imageState_ -> srcDepth,
                                imageState_ -> srcWidth, imageState_ -> srcHeight,
                                    srcData, srcSize, imageState_ -> dstDepth,
                                        imageState_ -> dstWidth, imageState_ -> dstHeight,
                                            dstData, dstSize);
          break;
        }
        default:
        {
          break;
        }
      }
    }
  }

  if (imageState_ -> method != PACK_RDP_COMPRESSED_256_COLORS)
  {
    writeBuffer_.removeMessage(removeSize);
  }
  else
  {
    writeBuffer_.removeScratchMessage();
  }

  if (result <= 0)
  {
    #ifdef PANIC
    *logofs << "handleUnpack: PANIC! Failed to unpack image "
            << "with method '" << (unsigned int) imageState_ -> method
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Failed to unpack image "
         << "with method '" << (unsigned int) imageState_ -> method
         << "'.\n";

    writeBuffer_.removeScratchMessage();

    return -1;
  }

  //
  // Alpha channel is used only on some 32 bits pixmaps
  // and only if render extension is in use. Presently
  // we don't have an efficient way to know in advance
  // if mask must be applied or not to the image. If an
  // alpha channel is set, the function will check if
  // the size of the alpha data matches the size of the
  // image. In the worst case we'll create an useless
  // alpha plane for a pixmap that doesn't need it.
  //

  if (alphaState != NULL && alphaState -> data != NULL && dstBitsPerPixel == 32)
  {
    UnpackAlpha(alphaState, dstData, dstSize, bigEndian_);
  }

  return 1;
}

//
// Handling of MIT-SHM extension has been plugged late in
// the design, so we have to make some assumptions. Image
// is a X_PutImage request contained either in the scratch
// buffer or in the normal write buffer. We need to move
// the image data to the shared memory segment and replace
// the X_PutImage request with a X_ShmPutImage.
//

int ServerChannel::handleShmem(unsigned char &opcode, unsigned char *&buffer,
                                   unsigned int &size, unsigned int &flush)
{
  if (shmemState_ == NULL || shmemState_ -> enabled != 1)
  {
    #ifdef TEST

    if (shmemState_ != NULL)
    {
      *logofs << "handleShmem: PANIC! Shared memory "
              << "state found but support is not enabled "
              << "for FD#" << fd_ << " in stage "
              << shmemState_ -> stage << ".\n"
              << logofs_flush;
    }

    #endif

    return 0;
  }

  //
  // Ignore null requests and requests that will not result
  // in a single X_PutImage. To conform with the other func-
  // tions, we get the opcode passed as a parameter. It can
  // be zero if we don't want the write loop to put opcode
  // and length in the resulting buffer. Anyway we are only
  // interested in the original opcode of the request, that
  // is stored in the image state.
  //

  unsigned char *dstData    = buffer + 24;
  unsigned int  dstDataSize = size   - 24;

  if (dstDataSize == 0 || dstDataSize > MESSAGE_DATA_LIMIT ||
          (imageState_ -> opcode == opcodeStore_ -> putPackedImage &&
               imageState_ -> method == PACK_RDP_TEXT))
  {
    #ifdef TEST
    *logofs << "handleShmem: Ignoring image with opcode "
            << (unsigned int) imageState_ -> opcode
            << " and size " << size << " for FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "handleShmem: Handling image with opcode "
          << (unsigned int) imageState_ -> opcode
          << " and size " << size << " for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Get image data from buffer.
  //

  if (imageState_ -> opcode == X_PutImage)
  {
    //
    // We still need to get the image's data.
    //

    imageState_ -> format    = *(buffer + 1);

    imageState_ -> drawable  = GetULONG(buffer + 4,  bigEndian_);
    imageState_ -> gcontext  = GetULONG(buffer + 8,  bigEndian_);

    imageState_ -> dstWidth  = GetUINT(buffer + 12, bigEndian_);
    imageState_ -> dstHeight = GetUINT(buffer + 14, bigEndian_);

    imageState_ -> srcX      = 0;
    imageState_ -> srcY      = 0;

    imageState_ -> srcWidth  = imageState_ -> dstWidth;
    imageState_ -> srcHeight = imageState_ -> dstHeight;

    imageState_ -> dstX      = GetUINT(buffer + 16, bigEndian_);
    imageState_ -> dstY      = GetUINT(buffer + 18, bigEndian_);

    imageState_ -> leftPad   = *(buffer + 20);
    imageState_ -> dstDepth  = *(buffer + 21);

    imageState_ -> dstLines  = imageState_ -> dstHeight;

    imageState_ -> dstLength = size - 24;
  }

  //
  // If image can't fit in the available
  // space check if the completion event
  // is arrived.
  //

  #if defined(INFO) || defined(TEST)

  if (isTimestamp(shmemState_ -> last) == 0 &&
          shmemState_ -> offset != 0)
  {
    *logofs << "handleShmem: PANIC! No timestamp for sequence "
            << shmemState_ -> sequence << " with offset "
            << shmemState_ -> offset << ".\n"
            << logofs_flush;
  }

  #endif

  if (shmemState_ -> offset + imageState_ -> dstLength >
          shmemState_ -> size)
  {
    if (isTimestamp(shmemState_ -> last) == 1 &&
            handleShmemEvent() <= 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleShmem: WARNING! Missing completion "
              << "after " << diffTimestamp(shmemState_ -> last,
                 getTimestamp()) << " Ms for shared memory "
              << "for FD#" << fd_ << ".\n" << logofs_flush;
      #endif

      return 0;
    }
  }

  //
  // Let image start at current offset
  // in the shared segment.
  //

  #ifdef TEST
  *logofs << "handleShmem: Copying " << dstDataSize
          << " bytes to shared memory at offset "
          << shmemState_ -> offset << " for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  memcpy((unsigned char *) shmemState_ -> address +
             shmemState_ -> offset, dstData, dstDataSize);

  //
  // Get rid of the original X_PutImage
  // request.
  //

  if (writeBuffer_.getScratchData() != NULL)
  {
    writeBuffer_.removeScratchMessage();
  }
  else
  {
    writeBuffer_.removeMessage(size);
  }

  //
  // Add a X_ShmPutImage request to the
  // write buffer.
  //

  buffer = writeBuffer_.addMessage(40);

  *buffer = shmemState_ -> opcode;

  *(buffer + 1) = X_ShmPutImage;

  PutUINT(40 >> 2, buffer + 2, bigEndian_);

  PutULONG(imageState_ -> drawable, buffer + 4,  bigEndian_);
  PutULONG(imageState_ -> gcontext, buffer + 8,  bigEndian_);

  PutUINT(imageState_ -> dstWidth, buffer + 12, bigEndian_);
  PutUINT(imageState_ -> dstLines, buffer + 14, bigEndian_);

  PutUINT(imageState_ -> srcX, buffer + 16, bigEndian_);
  PutUINT(imageState_ -> srcY, buffer + 18, bigEndian_);

  PutUINT(imageState_ -> dstWidth,  buffer + 20, bigEndian_);
  PutUINT(imageState_ -> dstLines,  buffer + 22, bigEndian_);

  PutUINT(imageState_ -> dstX, buffer + 24, bigEndian_);
  PutUINT(imageState_ -> dstY, buffer + 26, bigEndian_);

  *(buffer + 28) = imageState_ -> dstDepth;
  *(buffer + 29) = imageState_ -> format;
  *(buffer + 30) = 1;

  PutULONG(shmemState_ -> segment, buffer + 32, bigEndian_);
  PutULONG(shmemState_ -> offset,  buffer + 36, bigEndian_);

  shmemState_ -> offset += dstDataSize;

  shmemState_ -> sequence = clientSequence_;
  shmemState_ -> last     = getTimestamp();

  #ifdef TEST
  *logofs << "handleShmem: Saved shared image "
          << "sequence " << shmemState_ -> sequence
          << " for FD#" << fd_ << " with offset "
          << shmemState_ -> offset << ".\n"
          << logofs_flush;
  #endif

  if (control -> TransportShmemThreshold > 0 &&
          (int) dstDataSize > control ->
              TransportShmemThreshold)
  {
    flush++;
  }

  return 1;
}

int ServerChannel::handleResetSequence(unsigned short int sequenceNum)
{
  //
  // We can't be sure X server has gotten the last
  // request, but we can't neither take the risk
  // to add a spare sequence.
  //

  if (sequenceNum == clientSequence_)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleResetSequence: Keeping last client sequence "
            << clientSequence_ << " for FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // Connection reset could be raised at any time,
  // for example because of a failure decoding a
  // request. We must handle those cases in which
  // request was not sent to X server but sequence
  // counter was indeed incremented. The only way
  // to do so is to find out the real value known
  // by X server.
  //

  unsigned int size = 16;
  unsigned char *buffer = writeBuffer_.addMessage(size);

  buffer[0] = X_GetMotionEvents;
  buffer[1] = 0;

  PutUINT(size >> 2, buffer + 2, bigEndian_);

  PutULONG(0, buffer + 4, bigEndian_);

  PutULONG(0, buffer + 8,  bigEndian_);
  PutULONG(0, buffer + 12, bigEndian_);

  #if defined(INFO) || defined(TEST)
  *logofs << "handleResetSequence: Sending bad X_GetMotionEvents for FD#" 
          << fd_ << " to get last X server's sequence.\n"
          << logofs_flush;
  #endif

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }
  else if (transport_ -> force() < 0)
  {
    return -1;
  }

  T_timestamp startTs = getTimestamp();

  int found = 0;

  while (found == 0)
  {
    if (pending_ == 0)
    {
      if (readBuffer_.readMessage() < 0)
      {
        return -1;
      }
    }
    else
    {
      pending_ = 0;
    }

    const unsigned char *error;

    while (found == 0 && (error = readBuffer_.getMessage(size)) != NULL)
    {
      //
      // Note that we can't be sure to get the error
      // as an incomplete request could have been
      // sent just before the reset.
      //

      if (error[0] == X_Error && error[1] == BadWindow &&
              error[10] == X_GetMotionEvents &&
                  (int) GetULONG(error + 4, bigEndian_) == 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleResetSequence: Successfully identified error "
                << "with sequence " << clientSequence_
                << " for FD#" << fd_ << ".\n" << logofs_flush;
        #endif

        found = 1;
      }
      #if defined(INFO) || defined(TEST)
      else
      {
        *logofs << "handleResetSequence: Got message OPCODE#" << (unsigned int) error[0]
                << " with sequence " << GetUINT(error + 2, bigEndian_)
                << " for FD#" << fd_ << ".\n" << logofs_flush;

        if (error[0] == X_Error)
        {
          *logofs << "handleResetSequence: Error is " << (unsigned int) error[1]
                  << " major is " << (unsigned int) error[10] << " value is "
                  << GetULONG(error + 4, bigEndian_) << ".\n"
                  << logofs_flush;
        }
      }
      #endif

      clientSequence_ = GetUINT(error + 2, bigEndian_);
    }

    readBuffer_.partialReset();

    if (diffTimestamp(startTs, getTimestamp()) > control -> ResetTimeout)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleResetSequence: Doubtfully assuming last X server's "
              << "sequence " << clientSequence_ << " for FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      found = 1;
    }
    else if (transport_ -> wait(control -> ResetTimeout) < 0)
    {
      return -1;
    }
  }

  //
  // We must synchronize sequence number of X client
  // with sequence number known by X server. This is
  // achieved by sending a X_NoOperation for any re-
  // quest we have lost because of reset.
  //

  clientSequence_++;
  clientSequence_ &= 0xffff;

  while (clientSequence_ != sequenceNum)
  {
    size   = 4;
    buffer = writeBuffer_.addMessage(size);

    *buffer = X_NoOperation;

    PutUINT(size >> 2, buffer + 2, bigEndian_);

    #if defined(INFO) || defined(TEST)
    *logofs << "handleResetSequence: Sending X_NoOperation for FD#"
            << fd_ << " to synchronize client sequence "
            << clientSequence_ << ".\n" << logofs_flush;
    #endif

    clientSequence_++;
    clientSequence_ &= 0xffff;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "handleResetSequence: Reset last client sequence number to "
          << clientSequence_ << " for FD#" 
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

//
// Use a simple encoding. Need to handle X_ListExtensions
// and X_QueryExtension to hide MIT-SHM and RENDER, in
// reply.
//

int ServerChannel::handleFastWriteRequest(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                              unsigned char *&buffer, unsigned int &size)
{
  //
  // We have 3 new NX requests in
  // the latest protocol version.
  //

  if ((opcode >= X_NXFirstOpcode &&
           opcode <= (X_NXLastOpcode - 3)) ||
               opcode == X_ListExtensions ||
                   opcode == X_QueryExtension)
  {
    return 0;
  }
  else if (control -> isProtoStep4() == 1 &&
               opcode > (X_NXLastOpcode - 3) &&
                   opcode <= X_NXLastOpcode)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "handleFastWriteRequest: Decoding raw request OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  buffer = writeBuffer_.addMessage(4);

  memcpy(buffer, decodeBuffer.decodeMemory(4), 4);

  size = GetUINT(buffer + 2, bigEndian_) << 2;

  writeBuffer_.registerPointer(&buffer);

  if (writeBuffer_.getAvailable() < size - 4 ||
          (int) size >= control -> TransportWriteThreshold)
  {
    #ifdef DEBUG
    *logofs << "handleFastWriteRequest: Using scratch buffer for OPCODE#"
            << (unsigned int) opcode << " with size " << size << " and "
            << writeBuffer_.getLength() << " bytes in buffer.\n"
            << logofs_flush;
    #endif

    //
    // The procedure moving data to shared memory
    // assumes that the full message is stored in
    // the scratch buffer. We can safely let the
    // scratch buffer inherith the decode buffer
    // at the first byte read for this message.
    //

    writeBuffer_.removeMessage(4);

    buffer = writeBuffer_.addScratchMessage(((unsigned char *)
                             decodeBuffer.decodeMemory(size - 4)) - 4, size);
  }
  else
  {
    writeBuffer_.addMessage(size - 4);

    memcpy(buffer + 4, decodeBuffer.decodeMemory(size - 4), size - 4);
  }

  //
  // Opcode could have been tainted by the client
  // proxy. Replace the original opcode with the
  // one sent in the decode buffer.
  //

  *buffer = opcode;

  writeBuffer_.unregisterPointer();

  unsigned int flush = 0;

  if (opcode == X_PutImage)
  {
    handleImage(opcode, buffer, size, flush);
  }

  #if defined(TEST) || defined(OPCODES)

  if (opcode != 0)
  {
    *logofs << "handleFastWriteRequest: Handled request "
            << "OPCODE#" << (unsigned int) opcode << " ("
            << DumpOpcode(opcode) << ") for FD#" << fd_
            << " sequence " << clientSequence_ << ". "
            << size << " bytes out.\n" << logofs_flush;
  }
  else
  {
    *logofs << "handleFastWriteRequest: Handled image "
            << "request for FD#" << fd_ << " sequence "
            << clientSequence_ << ". " << size
            << " bytes out.\n" << logofs_flush;
  }

  #endif

  switch (opcode)
  {
    //
    // Don't flush in case of RDP sessions.
    //
    // case X_AllocColor:
    //

    case X_GetAtomName:
    case X_GetGeometry:
    case X_GetInputFocus:
    case X_GetModifierMapping:
    case X_GetKeyboardMapping:
    case X_GetProperty:
    case X_GetSelectionOwner:
    case X_GrabPointer:
    case X_GrabKeyboard:
    case X_InternAtom:
    case X_ListExtensions:
    case X_ListFonts:
    case X_LookupColor:
    case X_AllocNamedColor:
    case X_QueryPointer:
    case X_GetWindowAttributes:
    case X_QueryTree:
    case X_QueryBestSize:
    case X_QueryColors:
    case X_QueryFont:
    case X_TranslateCoords:
    case X_GetImage:
    case X_GetPointerMapping:
    case X_GetKeyboardControl:
    {
      flush++;

      break;
    }
    default:
    {
      if (opcode == X_AllocColor &&
              control -> SessionMode != SESSION_RDP)
      {
        flush++;

        break;
      }
      else if (control -> isProtoStep3() == 1 &&
                   opcode != opcodeStore_ -> shapeExtension &&
                       opcode != opcodeStore_ -> renderExtension &&
                           opcode > 127)
      {
        flush++;

        break;
      }
    }
  }

  if (flush > 0)
  {
    handleFlush(flush_if_any);
  }
  else
  {
    handleFlush(flush_if_needed);
  }

  return 1;
}

//
// Use the simplest encoding except for replies that
// need to be managed some way.
//

int ServerChannel::handleFastReadReply(EncodeBuffer &encodeBuffer, const unsigned char &opcode,
                                           const unsigned char *&buffer, const unsigned int &size)
{
  //
  // If we pushed a X_GetInputFocus in the sequence
  // queue this means that the original message was
  // a NX request for which we have to provide a NX
  // reply.
  //

  if ((opcode >= X_NXFirstOpcode &&
           opcode <= X_NXLastOpcode) ||
               opcode == X_QueryExtension ||
                   opcode == X_ListExtensions ||
                       opcode == X_GetInputFocus)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "handleFastReadReply: Encoding raw reply OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_
          << " with size " << size << ".\n"
          << logofs_flush;
  #endif

  encodeBuffer.encodeMemory(buffer, size);

  //
  // Send back replies as soon as possible.
  //

  switch (opcode)
  {
    case X_AllocColor:
    {
      if (control -> SessionMode != SESSION_RDP)
      {
        priority_++;
      }

      break;
    }
    default:
    {
      priority_++;

      break;
    }
  }

  int bits = encodeBuffer.getBits();

  #if defined(TEST) || defined(OPCODES)
  *logofs << "handleFastReadReply: Handled raw reply OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_ << " sequence "
          << serverSequence_ << ". " << size << " bytes in, "
          << bits << " bits (" << ((float) bits) / 8
          << " bytes) out.\n" << logofs_flush;
  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addReplyBits(opcode, size << 3, bits);
  }

  control -> addOutputInARow(size);

  control -> addBitsInARow(bits);

  return 1;
}

int ServerChannel::handleFastReadEvent(EncodeBuffer &encodeBuffer, const unsigned char &opcode,
                                           const unsigned char *&buffer, const unsigned int &size)
{
  #ifdef DEBUG
  *logofs << "handleFastReadEvent: Encoding raw "
          << (opcode == X_Error ? "error" : "event") << " OPCODE#"
          << (unsigned int) opcode << " for FD#" << fd_
          << " with size " << size << ".\n"
          << logofs_flush;
  #endif

  encodeBuffer.encodeMemory(buffer, size);

  switch (opcode)
  {
    case X_Error:
    case ButtonPress:
    case ButtonRelease:
    {
      priority_++;
    }
  }

  int bits = encodeBuffer.getBits();

  #if defined(TEST) || defined(OPCODES)

  if (opcode == X_Error)
  {
    unsigned char code = *(buffer + 1);

    *logofs << "handleFastReadEvent: Handled error ERR_CODE#"
            << (unsigned int) code << " for FD#" << fd_;

    *logofs << " RES_ID#" << GetULONG(buffer + 4, bigEndian_);

    *logofs << " MIN_OP#" << GetUINT(buffer + 8, bigEndian_);

    *logofs << " MAJ_OP#" << (unsigned int) *(buffer + 10);

    *logofs << " sequence " << serverSequence_ << ". " << size
            << " bytes in, " << bits << " bits (" << ((float) bits) / 8
            << " bytes) out.\n" << logofs_flush;
  }
  else
  {
    *logofs << "handleFastReadEvent: Handled event OPCODE#"
            << (unsigned int) *buffer << " for FD#" << fd_
            << " sequence " << serverSequence_ << ". " << size
            << " bytes in, " << bits << " bits (" << ((float) bits) / 8
            << " bytes) out.\n" << logofs_flush;
  }

  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addEventBits(opcode, size << 3, bits);
  }

  control -> addOutputInARow(size);

  control -> addBitsInARow(bits);

  return 1;
}

//
// Handle the MIT-SHM initialization
// messages exchanged with the remote
// proxy. 
//

int ServerChannel::handleShmemInit(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                                       const unsigned int stage, const unsigned char *buffer,
                                           const unsigned int size)
{
  #ifdef TEST
  *logofs << "handleShmemInit: Returning shmem reply for "
          << "stage " << stage << ".\n" << logofs_flush;
  #endif

  if (opcode == X_QueryExtension)
  {
    encodeBuffer.encodeValue(stage, 2);

    shmemState_ -> present = *(buffer + 8);
    shmemState_ -> opcode  = *(buffer + 9);
    shmemState_ -> event   = *(buffer + 10);
    shmemState_ -> error   = *(buffer + 11);

    #ifdef TEST
    *logofs << "handleShmemInit: Extension present is "
            << shmemState_ -> present << " base opcode "
            << (unsigned int) shmemState_ -> opcode << " base event "
            << (unsigned int) shmemState_ -> event << " base error "
            << (unsigned int) shmemState_ -> error << ".\n"
            << logofs_flush;
    #endif
  }
  else if (opcode == X_GetInputFocus)
  {
    encodeBuffer.encodeValue(stage, 2);

    encodeBuffer.encodeBool(0);

    if (shmemState_ -> present == 1 &&
            shmemState_ -> address != NULL &&
                shmemState_ -> segment > 0 &&
                    shmemState_ -> id > 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleShmemInit: Using shared memory support in X server.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Using shared memory support in X server.\n";

      shmemState_ -> enabled = 1;

      encodeBuffer.encodeBool(1);
    }
    else
    {
      #ifdef WARNING
      *logofs << "handleShmemInit: WARNING! Not using shared memory "
              << "support in X server for FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Not using shared memory support "
           << "in X server.\n";

      handleShmemStateRemove();

      encodeBuffer.encodeBool(0);
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "handleShmemInit: PANIC! Conversation error "
            << "handling shared memory support for FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Conversation error handling "
         << "shared memory support.\n";

    return -1;
  }

  return 1;
}

int ServerChannel::handleShmemInit(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                       unsigned char *&buffer, unsigned int &size)
{
  //
  // We need to query and initialize MIT-SHM on
  // the real X server. To do this we'll need 3
  // requests. At the end we'll have to encode
  // the final reply for the X client side.
  //

  handleShmemStateAlloc();

  unsigned int stage;

  decodeBuffer.decodeValue(stage, 2);

  unsigned int expected = shmemState_ -> stage + 1;

  if (stage != expected || stage > 2)
  {
    #ifdef PANIC
    *logofs << "handleShmemInit: PANIC! Unexpected stage "
            << stage << " in handling shared memory "
            << "support for FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Unexpected stage "
         << stage << " in handling shared memory "
         << "support for FD#" << fd_ << ".\n";

    return -1;
  }

  switch (stage)
  {
    case 0:
    {
      unsigned int enableClient;
      unsigned int enableServer;

      decodeBuffer.decodeBool(enableClient);
      decodeBuffer.decodeBool(enableServer);

      unsigned int clientSegment;
      unsigned int serverSegment;

      decodeBuffer.decodeValue(clientSegment, 29, 9);
      decodeBuffer.decodeValue(serverSegment, 29, 9);

      shmemState_ -> segment = serverSegment;

      #if defined(INFO) || defined(TEST)
      *logofs << "handleShmemInit: Sending X_QueryExtension request "
              << "for FD#" << fd_ << " due to OPCODE#" << (unsigned int)
                 opcodeStore_ -> getShmemParameters << " in stage "
              << stage << ".\n" << logofs_flush;
      #endif

      opcode = X_QueryExtension;

      size   = 16;
      buffer = writeBuffer_.addMessage(size);

      PutUINT(7, buffer + 4, bigEndian_);

      //
      // Simply make the query fail if shared
      // memory support is disabled by user.
      //

      if (control -> AgentShmemServer == 1 &&
              control -> AgentShmemServerSize > 0 &&
                  enableServer == 1)
      {
        memcpy(buffer + 8, "MIT-SHM", 7);
      }
      else
      {
        memcpy(buffer + 8, "NO-MIT-", 7);
      }

      sequenceQueue_.push(clientSequence_, opcode,
                              opcodeStore_ -> getShmemParameters, stage);

      //
      // Save the sequence number so we can
      // later identify any matching X error
      // received from server.
      //

      shmemState_ -> sequence = clientSequence_;

      break;
    }
    case 1:
    {
      if (shmemState_ -> present == 1)
      {
        //
        // Make the segment read-write for everybody on
        // Cygwin (to avoid any lack of support or any
        // performance issue) and on MacOS/X (where the
        // 0600 mask doesn't seem to work).
        //

        #if defined(__CYGWIN32__) || defined(__APPLE__)

        int permissions = 0777;

        #else

        int permissions = 0600;

        #endif

        shmemState_ -> size = control -> AgentShmemServerSize;

        shmemState_ -> id = shmget(IPC_PRIVATE, shmemState_ -> size,
                                       IPC_CREAT | permissions);

        if (shmemState_ -> id >= 0)
        {
          #ifdef TEST
          *logofs << "handleShmemInit: Allocated shared memory "
                  << "segment of " << shmemState_ -> size
                  << " bytes with id " << shmemState_ -> id
                  << ".\n" << logofs_flush;
          #endif


          shmemState_ -> address = shmat(shmemState_ -> id, 0, 0);

          if (shmemState_ -> address != NULL)
          {
            #if defined(INFO) || defined(TEST)
            *logofs << "handleShmemInit: Sending X_ShmAttach request "
                    << "for FD#" << fd_ << " due to OPCODE#" << (unsigned int)
                       opcodeStore_ -> getShmemParameters << " in stage "
                    << stage << ".\n" << logofs_flush;
            #endif

            opcode = shmemState_ -> opcode;

            size   = 16;
            buffer = writeBuffer_.addMessage(size);

            *(buffer + 1) = X_ShmAttach;

            PutULONG(shmemState_ -> segment, buffer + 4, bigEndian_);
            PutULONG(shmemState_ -> id, buffer + 8, bigEndian_);

            *(buffer + 12) = 1;

            shmemState_ -> sequence = clientSequence_;

            break;
          }
          else
          {
            #ifdef WARNING
            *logofs << "handleShmemInit: WARNING! Can't attach the shared "
                    << "memory segment. Error is " << EGET() << " '"
                    << ESTR() << "'.\n" << logofs_flush;
            #endif

            cerr << "Warning" << ": Can't attach the shared memory "
                 << "segment. Error is " << EGET() << " '"
                 << ESTR() << "'.\n";
          }
        }
        else
        {
          #ifdef WARNING
          *logofs << "handleShmemInit: WARNING! Can't create the shared "
                  << "memory segment. Error is " << EGET() << " '"
                  << ESTR() << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Can't create the shared memory "
               << "segment. Error is " << EGET() << " '"
               << ESTR() << "'.\n";
        }
      }

      if (shmemState_ -> present != 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleShmemInit: Resetting shared memory "
                << "presence flag for FD#" << fd_ << ".\n"
                << logofs_flush;
        #endif

        shmemState_ -> present = 0;
      }

      #if defined(INFO) || defined(TEST)
      *logofs << "handleShmemInit: Sending X_NoOperation request "
              << "for FD#" << fd_ << " due to OPCODE#" << (unsigned int)
                 opcodeStore_ -> getShmemParameters << " in stage "
              << stage << ".\n" << logofs_flush;
      #endif

      opcode = X_NoOperation;

      size   = 4;
      buffer = writeBuffer_.addMessage(size);

      break;
    }
    default:
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "handleShmemInit: Sending X_GetInputFocus request "
              << "for FD#" << fd_ << " due to OPCODE#" << (unsigned int)
                 opcodeStore_ -> getShmemParameters << " in stage "
              << stage << ".\n" << logofs_flush;
      #endif

      opcode = X_GetInputFocus;

      size   = 4;
      buffer = writeBuffer_.addMessage(size);

      sequenceQueue_.push(clientSequence_, opcode,
                              opcodeStore_ -> getShmemParameters, stage);
      break;
    }
  }

  shmemState_ -> stage += 1;

  return 1;
}

//
// Wait for a MIT-SHM image completion
// event from the real X server.
//

int ServerChannel::handleShmemEvent()
{
  #ifdef TEST
  *logofs << "handleShmemEvent: Waiting for shared memory "
          << "sequence " << shmemState_ -> sequence
          << " for X server FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  //
  // TODO: When we have to force data to the X socket
  // we should also, at the same time, read events and
  // replies that have become available. To do that we
  // need to modify the wait() and force() methods in
  // transport to select the socket for both reading
  // and writing.
  //

  if (transport_ -> force() < 0)
  {
    return -1;
  }

  unsigned int timeout = control -> ShmemTimeout;

  T_timestamp startTs = getTimestamp();

  const unsigned char *event;

  for (;;)
  {
    if (pending_ == 0)
    {
      if (transport_ -> wait(timeout) <= 0)
      {
        break;
      }
      else if (readBuffer_.readMessage() < 0)
      {
        return -1;
      }
    }
    else
    {
      pending_ = 0;
    }

    event = readBuffer_.checkMessage(shmemState_ -> checked, shmemState_ -> event,
                                         shmemState_ -> sequence);

    if (event != NULL)
    {
      #ifdef TEST
      *logofs << "handleShmemEvent: Reset shared memory sequence "
              << shmemState_ -> sequence << " for FD#" << fd_
              << " after " << diffTimestamp(startTs, getTimestamp())
              << " Ms.\n" << logofs_flush;
      #endif

      shmemState_ -> sequence = 0;
      shmemState_ -> offset   = 0;
      shmemState_ -> last     = nullTimestamp();

      break;
    }

    timeout = control -> ShmemTimeout -
                  diffTimestamp(startTs, getTimestamp());

    if (timeout <= 0)
    {
      break;
    }
  }

  pending_ = readBuffer_.checkMessage();

  if (isTimestamp(shmemState_ -> last) == 1)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "handleShmemEvent: WARNING! Can't reset shared "
            << "memory sequence for FD#" << fd_ << " after "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "handleShmemEvent: Spent "
          << diffTimestamp(startTs, getTimestamp()) << " Ms "
          << "waiting for shared memory sequence for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 1;
}

int ServerChannel::checkShmemEvent(unsigned char event, unsigned short sequence,
                                       const unsigned char *buffer)
{
  if (isTimestamp(shmemState_ -> last) == 1 &&
          sequence == shmemState_ -> sequence)
  {
    #ifdef TEST
    *logofs << "checkShmemEvent: Reset shared memory sequence "
            << shmemState_ -> sequence << " for FD#" << fd_
            << " after " << diffTimestamp(shmemState_ -> last,
               getTimestamp()) << " Ms.\n" << logofs_flush;
    #endif

    shmemState_ -> sequence = 0;
    shmemState_ -> offset   = 0;
    shmemState_ -> last     = nullTimestamp();
  }
  #ifdef TEST
  else
  {
    *logofs << "checkShmemEvent: Skipping unhandled shared memory "
            << "image sequence " << sequence << " for FD#"
            << fd_ << ".\n" << logofs_flush;
  }
  #endif

  return 1;
}

int ServerChannel::checkShmemError(unsigned char error, unsigned short sequence,
                                       const unsigned char *buffer)
{
  #ifdef WARNING

  *logofs << "checkShmemError: WARNING! Failed operation for "
          << "FD#" << fd_ << " in stage " << shmemState_ -> stage
          << " with ERR_CODE#" << (unsigned int) *(buffer + 1);

  *logofs << " RES_ID#" << GetULONG(buffer + 4, bigEndian_);

  *logofs << " MIN_OP#" << GetUINT(buffer + 8, bigEndian_);

  *logofs << " MAJ_OP#" << (unsigned int) *(buffer + 10);

  *logofs << " sequence " << sequence << ".\n";

  *logofs << logofs_flush;

  #endif

  cerr << "Warning" << ": Failed shared memory operation "
       << "with ERR_CODE#" << (unsigned int) error;

  cerr << " RES_ID#" << GetULONG(buffer + 4, bigEndian_);

  cerr << " MIN_OP#" << GetUINT(buffer + 8, bigEndian_);

  cerr << " MAJ_OP#" << (unsigned int) *(buffer + 10);

  cerr << ".\n";

  //
  // If enabled flag is <= 0 we are still
  // in the inizialization phase. In this
  // case force presence to false.
  //

  if (shmemState_ -> enabled != 1)
  {
    if (shmemState_ -> present != 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "checkShmemError: Resetting shared memory "
              << "presence flag for FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      shmemState_ -> present = 0;
    }

    return 0;
  }

  if (shmemState_ -> sequence == sequence)
  {
    //
    // Reset the sequence and timestamp only
    // if shared memory is already enabled
    // and failure is not likely to cause
    // any further X error.
    //

    shmemState_ -> sequence = 0;
    shmemState_ -> offset   = 0;
    shmemState_ -> last     = nullTimestamp();
  }

  return 1;
}

int ServerChannel::handleShmemStateAlloc()
{
  if (shmemState_ == NULL)
  {
    shmemState_ = new T_shmem_state();

    shmemState_ -> stage   = -1;
    shmemState_ -> present = -1;
    shmemState_ -> enabled = -1;

    shmemState_ -> segment = -1;
    shmemState_ -> id      = -1;
    shmemState_ -> address = NULL;
    shmemState_ -> size    = 0;

    shmemState_ -> opcode = 0xff;
    shmemState_ -> event  = 0xff;
    shmemState_ -> error  = 0xff;

    shmemState_ -> sequence = 0;
    shmemState_ -> offset   = 0;
    shmemState_ -> last     = nullTimestamp();

    shmemState_ -> checked  = 0;
  }

  return 1;
}

int ServerChannel::handleShmemStateRemove()
{
  if (shmemState_ != NULL)
  {
    if (shmemState_ -> address != NULL)
    {
      shmdt((char *) shmemState_ -> address);
    }

    if (shmemState_ -> id > 0)
    {
      shmctl(shmemState_ -> id, IPC_RMID, 0);
    }

    delete shmemState_;

    shmemState_ = NULL;
  }

  return 1;
}

int ServerChannel::handleUnpackStateInit(int client)
{
  if (unpackState_[client] == NULL)
  {
    unpackState_[client] = new T_unpack_state();

    if (unpackState_[client] == NULL)
    {
      #ifdef PANIC
      *logofs << "handleUnpackStateInit: PANIC! Can't allocate "
              << "memory for unpack state in context [A].\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate memory for "
           << "unpack state in context [A].\n";

      HandleAbort();
    }

    unpackState_[client] -> geometry = NULL;
    unpackState_[client] -> colormap = NULL;
    unpackState_[client] -> alpha    = NULL;
  }

  return 1;
}

int ServerChannel::handleUnpackAllocGeometry(int client)
{
  if (unpackState_[client] -> geometry == NULL)
  {
    unpackState_[client] -> geometry = new T_geometry();

    if (unpackState_[client] -> geometry == NULL)
    {
      #ifdef PANIC
      *logofs << "handleUnpackAllocGeometry: PANIC! Can't allocate "
              << "memory for unpack state in context [B].\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate memory for "
           << "unpack state in context [B].\n";

      HandleAbort();
    }

    unpackState_[client] -> geometry -> depth1_bpp  = 4;
    unpackState_[client] -> geometry -> depth4_bpp  = 4;
    unpackState_[client] -> geometry -> depth8_bpp  = 8;
    unpackState_[client] -> geometry -> depth16_bpp = 16;
    unpackState_[client] -> geometry -> depth24_bpp = 32;
    unpackState_[client] -> geometry -> depth32_bpp = 32;
  }

  return 1;
}

int ServerChannel::handleUnpackAllocColormap(int client)
{
  if (unpackState_[client] -> colormap == NULL)
  {
    unpackState_[client] -> colormap = new T_colormap();

    if (unpackState_[client] -> colormap == NULL)
    {
      #ifdef PANIC
      *logofs << "handleUnpackAllocColormap: PANIC! Can't allocate "
              << "memory for unpack state in context [C].\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate memory for "
           << "unpack state in context [C].\n";

      HandleAbort();
    }

    unpackState_[client] -> colormap -> entries = 0;
    unpackState_[client] -> colormap -> data    = NULL;
  }

  return 1;
}

int ServerChannel::handleUnpackAllocAlpha(int client)
{
  if (unpackState_[client] -> alpha == NULL)
  {
    unpackState_[client] -> alpha = new T_alpha();

    if (unpackState_[client] -> alpha == NULL)
    {
      #ifdef PANIC
      *logofs << "handleUnpackAllocAlpha: PANIC! Can't allocate "
              << "memory for unpack state in context [D].\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate memory for "
           << "unpack state in context [D].\n";

      HandleAbort();
    }

    unpackState_[client] -> alpha -> entries = 0;
    unpackState_[client] -> alpha -> data    = NULL;
  }

  return 1;
}

int ServerChannel::handleUnpackStateRemove(int client)
{
  if (unpackState_[client] != NULL)
  {
    delete unpackState_[client] -> geometry;

    if (unpackState_[client] -> colormap != NULL)
    {
      delete unpackState_[client] -> colormap -> data;
    }

    delete unpackState_[client] -> colormap;

    if (unpackState_[client] -> alpha != NULL)
    {
      delete unpackState_[client] -> alpha -> data;
    }

    delete unpackState_[client] -> alpha;

    delete unpackState_[client];

    unpackState_[client] = NULL;
  }

  return 1;
}

void ServerChannel::handleEncodeCharInfo(const unsigned char *nextSrc, EncodeBuffer &encodeBuffer)
{
  unsigned int value = GetUINT(nextSrc, bigEndian_) |
                           (GetUINT(nextSrc + 10, bigEndian_) << 16);

  encodeBuffer.encodeCachedValue(value, 32,
                     *serverCache_ -> queryFontCharInfoCache[0], 6);

  nextSrc += 2;

  for (unsigned int i = 1; i < 5; i++)
  {
    unsigned int value = GetUINT(nextSrc, bigEndian_);

    nextSrc += 2;

    encodeBuffer.encodeCachedValue(value, 16,
                       *serverCache_ -> queryFontCharInfoCache[i], 6);
  }
}

void ServerChannel::setBigEndian(int flag)
{
  bigEndian_ = flag;

  readBuffer_.setBigEndian(flag);
}
