/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * The Mobile Mesh Routing Protocol
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id: MmRouter.h,v 1.8 2004/04/23 16:14:41 br1 Exp $
 * 
 * NOTES:

   ToDo/Bugs

   - Sequence Number Wrap around

   - Startup database request

   Is this a kludge?
   We previously assumed that each interface had a unique IP address.
   Now that we support tunneling between border routers, we have a 
   point-to-point tunnel interface to each peering border router. 
   Each tunnel interface could have a unique local address, but that means
   if we had 'n' border routers, we would have n(n-1) tunnels and 
   therefore we would need 2n(n-1) addresses...that's alot of addresses:
   For n=100, we would need 19800 addresses! Instead we allow a border router 
   to have only one local tunnel address shared by all tunnel
   interfaces...hence for n=100, we need only 100 addresses.
   The problem now is that the 99 tunnel interfaces on one border router 
   all have the same addresses (of course they have unique point-to-point
   destination addresses). 
   
   We now need a mechanism to 1) aggregate links we
   learn of on tunnel interfaces so that our Lsp lists all tunnel neighbors
   for our single local tunnel address, 2) determine an interface name based 
   upon a local address and possibly a point-to-point destination address (
   this is needed when creating routes ).
   
*/
#ifndef __MmRouter_h
#define __MmRouter_h 

#include <fstream>
#include <strstream>
#include <cmath>
#include <cstring>
#include <csignal>
#include <cstdio>
#include <list>
#include <vector>
#include <set>
#include <map>
#include <algo.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <net/ethernet.h>
#include <netdb.h>
#include <net/route.h>

#include <MmMsg.h>
#include <UtInterface.h>
#include <UtDebug.h>
#include <UtString.h>
#include <UtTime.h>
#include <UtUdpSocket.h>
#include <UtUnixSocket.h>
#include <UtUtil.h>


/* Abstraction of a Route...
 * Note: Assumes addresses are in HOST order!
 */
class Route {
public:
  enum Type_t { Host, Network };
private:
  String   cIfname;
  Type_t cType;
  unsigned int cDstAddr;
  unsigned int cNetmask;
  unsigned int cMetric;
  bool cHasGw;
  unsigned int cGwAddr;

public:
  unsigned int finalDest;

  Route(const String& ifname, Type_t t, unsigned int dst, unsigned int netmask,
	unsigned int metric) :
    cIfname(ifname), cType(t), cDstAddr(dst), cNetmask(netmask), 
    cMetric(metric), cHasGw(false), cGwAddr(0)  {}

  Route(const String& ifname, Type_t t, unsigned int dst, unsigned int netmask, 
	unsigned int metric, unsigned int gw) :
    cIfname(ifname), cType(t), cDstAddr(dst), cNetmask(netmask), 
    cMetric(metric), cHasGw(true), cGwAddr(gw) {}

  Route() : 
    cIfname(""), cType(Host), cDstAddr(0), cNetmask(0), cMetric(0), cHasGw(false), cGwAddr(0)  {}

  String Ifname() const { return cIfname; }
  Type_t Type() const { return cType; }
  unsigned int DstAddr() const { return cDstAddr; }
  unsigned int Netmask() const { return cNetmask; }
  unsigned int Metric() const { return cMetric; }
  bool HasGw() const { return cHasGw; }
  unsigned int GwAddr() const { return cGwAddr; }

  operator rtentry() const {
    struct rtentry rt;
    memset (&rt, 0, sizeof rt);
    rt.rt_dst     = InetAddr(cDstAddr);
    rt.rt_gateway = InetAddr(cGwAddr);
    rt.rt_genmask = InetAddr(cNetmask);
    rt.rt_flags   = RTF_UP;
    rt.rt_metric  = (short)cMetric;
    if(cType == Host) rt.rt_flags |= RTF_HOST;
    if(cHasGw) rt.rt_flags |= RTF_GATEWAY;

    //### This is why C strings are sooo bad...we have to expose a pointer to memory owned
    //### by this Route...BEWARE...make sure the Route object is still valid (in scope) when
    //### using the rtentry!
    rt.rt_dev = (char*)cIfname.c_str();
    return(rt);
  }

  String Dump() const {
    String s;
    s += String("<Route> ");
    if(cType == Host) 
      s += String("Host ");
    else
      s += String("Network ");
    s += DumpAddr(cDstAddr) + " / ";
    s += DumpAddr(cNetmask) + " ";
    s += String("metric=") + String((int)cMetric) + " ";
    if(cHasGw) 
      s += String("gw=") + DumpAddr(cGwAddr) + " ";

    s += cIfname + " ";
    // s += "\n";
    return(s);
  }
  
  int operator ==(const Route& rhs) const {
    if((cIfname  == rhs.cIfname)  &&
       (cType    == rhs.cType)    &&
       (cDstAddr == rhs.cDstAddr) &&
       (cNetmask == rhs.cNetmask) &&
       (cMetric  == rhs.cMetric)  &&
       (cHasGw   == rhs.cHasGw)   &&
       ((cHasGw ? (cGwAddr == rhs.cGwAddr) : true))) {
      return(1);
    }
    return(0);
  }

  int operator <(const Route& rhs) const {
    if(cIfname < rhs.cIfname)     return(1);
    if(!(cIfname == rhs.cIfname)) return(0);

    // Here if the same interface name
    if(cType    <  rhs.cType)     return(1);
    if(cType    != rhs.cType)     return(0);

    // Here if same interface name and route type
    if(cDstAddr < rhs.cDstAddr)   return(1);
    if(cDstAddr != rhs.cDstAddr)  return(0);

    // Here if same interface name, route type, and dst
    if(cNetmask < rhs.cNetmask)   return(1);
    if(cNetmask != rhs.cNetmask)  return(0);

    // Here if same interface name, route type, dst, and netmask
    if(cMetric < rhs.cMetric)   return(1);
    if(cMetric != rhs.cMetric)  return(0);

    // Here if same interface name, route type, dst, netmask, and metric
    if(((int)cHasGw)   < ((int)rhs.cHasGw)) return(1);
    if(cHasGw != rhs.cHasGw)                return(0);
    
    // Here if same interface name, route type, dst, netmask, metric, and hasGw
    if(!cHasGw)                             return(0);
    if(cGwAddr < rhs.cGwAddr)               return(1);

    return(0);
  }

  
};

 
class Router {
  enum { MAX_INTERFACES = 1024 }; // If you have more than 1024 interfaces how do you keep track of the cables??
  UdpSocket* cSockets[MAX_INTERFACES];     // Maps file descriptors to their UdpSocket's
  Interface* cInterfaces[MAX_INTERFACES];  // Maps file descriptors to their Interface's 
  UnixSocket* cReportSockets[MAX_INTERFACES]; // Maps unix socket file desc's to their UnixSocket's
  Interface* cReportInterfaces[MAX_INTERFACES]; // Maps unix socket file desc's to Interface's
  bool       cReportReplies[MAX_INTERFACES];    // Do we expect a ping reply from corresponding report socket
  list<Interface> cIfList;
  fd_set cFds;
  int cFdLimit;
  unsigned int cId;
  unsigned int cSeq;
  InetAddr   cBroadcastAddr;
  UdpSocket  cRouteSocket;   // Used to add/delete routes in the kernel
  UnixSocket* cConfigSocket;     // Allows others to send us add/delete/list interface commands
  UnixSocket* cImportSocket;     // Allows others to send us external route add / delete / list commands
  UnixSocket* cExportSocket;     // Tell others outside our cloud about who can be reached within our cloud
  UnixSocket* cDumpSocket;       // Dump our Lsp database to registered clients

  typedef std::set<String, less<String> > ExportClientSet;
  ExportClientSet cExportClientSet;  // Client paths who want to hear exports

  typedef std::set<String, less<String> > DatabaseDumpClientSet;
  DatabaseDumpClientSet cDatabaseDumpClientSet;  // Client paths who want to hear LSP database dumps

  typedef std::set<Route, less<Route> > RouteSet;
  RouteSet cRoutes;         // Routes that we have added to the kernel

  list<ExtRoute> cExtRouteList;  // External routes that we are configured to advertise


  // Keys are neighbor interfaces
  typedef std::set<Neighbor, less<Neighbor> > NeighborSet;

  // Keys are local Interface pointers, values are sets of neighbors we've heard
  typedef std::map<Interface*, NeighborSet, less<Interface*> >     IfNeighborMap;
  IfNeighborMap  cIfNeighborMap;

  // Keys are local addrs, values are sets of neighbors we've heard
  // *NOTE: multiple tunnel interfaces share the same local ip address 
  // We merge tunnel interfaces with the same address when creating our Lsp
  typedef std::map<unsigned int, NeighborSet, less<unsigned int> > IfAddrNeighborMap;
  IfAddrNeighborMap  cIfAddrNeighborMap;

  // Keys are interface addresses of neighbor interfaces; values are their router id's
  typedef std::map<unsigned int, unsigned int , less<unsigned int> > RouterIdMap;
  RouterIdMap    cRouterIdMap;


  // Keys are router Id's, values are Lsp's generated by corresponding router
  typedef std::map< unsigned int, Lsp, less< unsigned int> > LspMap;
  LspMap cLspMap;
  LspMap cInvertedLspMap;


  // Keys are router id's
  typedef std::map<unsigned int, Time, less<unsigned int> > TimeMap;

  TimeMap cTxTimeMap;

  Time cPrunePeriod;

  String cDebugFile;

public:
  class Config {
    unsigned short cPort;         // Udp port used by this routing protocol
    double       cAlpha;        // Fish-eye routing base: tx delay = pow(alpha,hops);
    unsigned int cMinTxDelay;   // Limits the minimum fish-eye tx delay
    unsigned int cMaxTxDelay;   // Limits the maximum fish-eye tx delay
    unsigned int cMaxAge;       // How old an LSP can get before it dies
    unsigned int cUpdatePeriod;  // Seconds between building our own lsp, recomputing routes, and sending our own lsp
    bool cAddDefaultGw;		// should default gw be added to the routing table
    
    
  public:
  String cDefaultGwScript;	// script to execute after default gw is added
    Config() : cPort(20471), cAlpha(1), cMinTxDelay(1), cMaxTxDelay(10), cMaxAge(120), cUpdatePeriod(10) { };
    unsigned short Port() const { return cPort; }
    double       Alpha() const { return cAlpha; }
    unsigned int MinTxDelay() const { return cMinTxDelay; }
    unsigned int MaxTxDelay() const { return cMaxTxDelay; }
    unsigned int MaxAge() const { return cMaxAge; }
    unsigned int UpdatePeriod() const { return cUpdatePeriod; }
    unsigned int addDefaultGw() const { return cAddDefaultGw; }
    String defaultGwScript() { return cDefaultGwScript; }
    
    unsigned short& Port()  { return cPort; }
    double&       Alpha()  { return cAlpha; }
    unsigned int& MinTxDelay()  { return cMinTxDelay; }
    unsigned int& MaxTxDelay()  { return cMaxTxDelay; }
    unsigned int& MaxAge()  { return cMaxAge; }
    unsigned int& UpdatePeriod()  { return cUpdatePeriod; }
    bool& addDefaultGw() { return cAddDefaultGw; }

    String Dump() const {
      String s;
      s += String("<Config> ");
      s += String("port=") + String((int)Port()) + " ";
      s += String("alpha=") + String(Alpha()) + " ";
      s += String("minTxDelay=") + String((int)MinTxDelay()) + " ";
      s += String("maxTxDelay=") + String((int)MaxTxDelay()) + " ";
      s += String("maxAge=") + String((int)MaxAge()) + " ";
      s += String("updatePeriod=") + String((int)UpdatePeriod()) + " ";
     return(s);
    }
  };

private:
  Config cConfig;

  /* Adds a route to the kernel's forwarding table
   * Returns true on success
   * Otherwise a warning is issued and returns false
   */
  bool KernelAdd(const Route& r) {
    if(Debug("KernelAdd")) {
      Report::Debug(String("KernelAdd() - ") + r.Dump());
    }
    rtentry rt = r;
    if(ioctl(cRouteSocket.Fd(), SIOCADDRT, &rt) < 0) {
      char buf[1024];
      sprintf(buf,"Router::KernelAdd() - ioctl(SIOCADDRT) failed: %s",strerror(errno));
      Report::Warn(buf);
      return(false);
    }
    return(true);
  }

  /* Deletes a route from the kernel's forwarding table
   * Returns true on success
   * Otherwise a warning is issued and returns false
   */
  bool KernelDelete(const Route& r) {
    if(Debug("KernelDelete")) {
      Report::Debug(String("KernelDelete() - ") + r.Dump());
    }
    rtentry rt = r;
    if(ioctl(cRouteSocket.Fd(), SIOCDELRT, &rt) < 0) {
      char buf[1024];
      sprintf(buf,"Router::KernelDelete() - ioctl(SIOCDELRT) failed: %s",strerror(errno));
      Report::Warn(buf);
      return(false);
    }
    return(true);
  }

  /* Turns IP forwarding on or off in the kernel
   */
  void KernelForwarding(bool on) {
    String path = String("/proc/sys/net/ipv4/ip_forward");
    String s = (on ? "1" : "0");
    s += "\n";
    int fd;
    if((fd = open(path.str(),O_WRONLY)) == -1) {
      Report::Error(String("Router::KernelForwarding() - could not open '") + path + "' :" + strerror(errno));
    }
    if(write(fd,s.str(),s.length()) != ((int)s.length())) {
      Report::Error(String("Router::KernelForwarding() - could not write '") + path + "' ");
    }
    close(fd);
  }

  /* Scan all stored Lsp's and populates a map which allows us to 
   * find the router id of any interface address. Note: interface addresses MUST be unique.
   */
  void PopulateRouterIdMap() {
    cRouterIdMap.clear();  // Empty previous contents

    LspMap::iterator i;
    for(i = cLspMap.begin(); i != cLspMap.end(); i++) {
      Lsp&  lsp = (*i).second; 

      // Iterate across the router's interfaces clearing the neighbor lists
      list<Lsp::Face>::iterator j;
      for(j = lsp.FaceList().begin(); j != lsp.FaceList().end(); j++) {
	cRouterIdMap[(*j).Addr()] = lsp.RouterId();
      }
    }
  }

  /* Attempts to find the router id associated with the given interface...
   * Returns true upon success.
   */
  bool LookupRouterId(unsigned int addr, unsigned int& routerId) {
    RouterIdMap::iterator it = cRouterIdMap.find(addr);
    if(it != cRouterIdMap.end()) {
      routerId = (*it).second;
      return(true);
    }
    if(Debug("LookupRouterId")) {
      Report::Debug(String("Router::LookupRouterId() - failed to find router id for ")
		    + DumpAddr(addr));
    }
    return(false);
  }

  /* An Lsp includes a list of neighbors that a node can hear.  We
   * need the inverse: a list of neighbors that can hear this node.
   * Also, we need to make sure the neighbors have bi-directional
   * links with this node unless the interface allows uni-directional
   * links. 
   */
  void InvertLspMap() {
    // Make sure we start with an empty inverted lsp map and link set
    cInvertedLspMap.clear();

    // For each Lsp in our map, create a corresponding Lsp in our inverted map with 
    // the same local interfaces and external route list but no neighbor lists
    LspMap::iterator i;
    for(i = cLspMap.begin(); i != cLspMap.end(); i++) {
      Lsp  lsp = (*i).second;  // Make a copy

      // Iterate across the router's interfaces clearing the neighbor lists
      list<Lsp::Face>::iterator j;
      for(j = lsp.FaceList().begin(); j != lsp.FaceList().end(); j++) {
	(*j).NeighborList().clear();
      }
      cInvertedLspMap[lsp.RouterId()] = lsp;
    }


    // Now iterate across all saved Lsp packets and append the local interface
    // to its neighbor's inverted Lsp
    for(i = cLspMap.begin(); i != cLspMap.end(); i++) {
      const Lsp& lsp = (*i).second;

      // Iterate across the router's interfaces
      list<Lsp::Face>::const_iterator j;
      for(j = lsp.FaceList().begin(); j != lsp.FaceList().end(); j++) {
	unsigned int addr = (*j).Addr();
	const list<Neighbor>& neighborList = (*j).NeighborList();

	// Interate across this interface's neighbor list
	list<Neighbor>::const_iterator k;
	for(k = neighborList.begin(); k != neighborList.end(); k++) {
	  unsigned int neighborAddr = (*k).Addr();
	  unsigned int neighborMetric = (*k).Metric();
	  unsigned int neighborRouterId;

	  // Try to lookup the router id of this neighbor's interface...
	  // If we don't know it, that's ok, skip it
	  if(!LookupRouterId(neighborAddr,neighborRouterId)) continue;

	  // Attempt to find an inverted Lsp for this neighbor...
	  // If we don't have one, that's fine just skip it
	  LspMap::iterator jj = cInvertedLspMap.find(neighborRouterId);
	  if(jj == cInvertedLspMap.end()) continue;
	  Lsp& invLsp = (*jj).second;
	  
	  // Find the interface in the neighbors inverted Lsp map
	  list<Lsp::Face>::iterator ii;
	  bool found = false;
	  for(ii = invLsp.FaceList().begin(); ii != invLsp.FaceList().end(); ii++) {
	    if((*ii).Addr() == neighborAddr) {
	      found = true;
	      break;
	    }
	  }
	  if(!found) {
	    Report::Warn("Router::InvertLspMap() - Neighbor's inverted Lsp is missing an interface!");
	  }
	  else {
	    // ii now points to the appropriate interface of the inverted Lsp
	    // Append the interface that heard the Lsp to the inverted Lsp
	    (*ii).NeighborList().push_back(Neighbor(addr,neighborMetric));
	  }
	}
      }
    }
  }

  /* A simple way of debugging ComputeRoutes with arbitrary lsp database.
   */
  void MockPopulateLspMap() {
    cId = 0xC0A80501;

    unsigned int a1 = 0xC0A80101;
    unsigned int a2 = 0xC0A80102;
    unsigned int a3 = 0xC0A80103;
    unsigned int a4 = 0xC0A80104;
    unsigned int a5 = 0xC0A80105;

    unsigned int b1 = 0xC0A80201;
    unsigned int b2 = 0xC0A80202;
    unsigned int b3 = 0xC0A80203;

    unsigned int c1 = 0xC0A80301;
    unsigned int c2 = 0xC0A80302;
    unsigned int c3 = 0xC0A80303;

    unsigned int d1 = 0xC0A80401;
    unsigned int d2 = 0xC0A80402;

    unsigned int e1 = 0xC0A80501;
    unsigned int e2 = 0xC0A80502;
    unsigned int e3 = 0xC0A80503;

    unsigned int f1 = 0xC0A80601;
    unsigned int f2 = 0xC0A80602;

    

    // Node A
    Lsp lspA(a1,1,2,3);
    lspA.FaceList().push_back(Lsp::Face());
    lspA.FaceList().back().Addr() = a1;

    lspA.FaceList().push_back(Lsp::Face());
    lspA.FaceList().back().Addr() = a2;
    lspA.FaceList().back().NeighborList().push_back(Neighbor(f2,1));  // Who I hear
    
    lspA.FaceList().push_back(Lsp::Face());
    lspA.FaceList().back().Addr() = a3;
    
    lspA.FaceList().push_back(Lsp::Face());
    lspA.FaceList().back().Addr() = a4;

    lspA.FaceList().push_back(Lsp::Face());
    lspA.FaceList().back().Addr() = a5;
    lspA.FaceList().back().NeighborList().push_back(Neighbor(e3,1));  // Border Interface
    
    cLspMap[lspA.RouterId()] = lspA;

    // Node B
    Lsp lspB(b1,1,2,3);
    lspB.FaceList().push_back(Lsp::Face());
    lspB.FaceList().back().Addr() = b1;

    lspB.FaceList().push_back(Lsp::Face());
    lspB.FaceList().back().Addr() = b2;
    lspB.FaceList().back().NeighborList().push_back(Neighbor(a2,1));
    lspB.FaceList().back().NeighborList().push_back(Neighbor(f2,1));
    
    lspB.FaceList().push_back(Lsp::Face());
    lspB.FaceList().back().Addr() = b3;
    lspB.FaceList().back().NeighborList().push_back(Neighbor(a3,1));
    
    lspB.ExtRouteList().push_back(ExtRoute(0x82010000,0xFFFF0000,30));
    cLspMap[lspB.RouterId()] = lspB;

    // Node C
    Lsp lspC(c1,1,2,3);
    lspC.FaceList().push_back(Lsp::Face());
    lspC.FaceList().back().Addr() = c1;
    lspC.FaceList().back().NeighborList().push_back(Neighbor(e2,1));

    lspC.FaceList().push_back(Lsp::Face());
    lspC.FaceList().back().Addr() = c2;
    lspC.FaceList().back().NeighborList().push_back(Neighbor(d1,1));

    lspC.FaceList().push_back(Lsp::Face());
    lspC.FaceList().back().Addr() = c3;
    lspC.FaceList().back().NeighborList().push_back(Neighbor(b1,1));

    cLspMap[lspC.RouterId()] = lspC;

    // Node D
    Lsp lspD(d1,1,2,3);
    lspD.FaceList().push_back(Lsp::Face());
    lspD.FaceList().back().Addr() = d1;
    lspD.FaceList().back().NeighborList().push_back(Neighbor(c2,1));

    lspD.FaceList().push_back(Lsp::Face());
    lspD.FaceList().back().Addr() = d2;
    //    lspD.FaceList().back().NeighborList().push_back(Neighbor(a3,1));

    cLspMap[lspD.RouterId()] = lspD;

    // Node E
    Lsp lspE(e1,1,2,3);
    lspE.FaceList().push_back(Lsp::Face());
    lspE.FaceList().back().Addr() = e1;
    lspE.FaceList().back().NeighborList().push_back(Neighbor(f1,1));

    lspE.FaceList().push_back(Lsp::Face());
    lspE.FaceList().back().Addr() = e2;
    lspE.FaceList().back().NeighborList().push_back(Neighbor(c1,1));

    lspE.FaceList().push_back(Lsp::Face());
    lspE.FaceList().back().Addr() = e3;
    lspE.FaceList().back().NeighborList().push_back(Neighbor(a5,1));  // Border Interface
    
    lspE.ExtRouteList().push_back(ExtRoute(0x82010000,0xFFFF0000,20));

    cLspMap[lspE.RouterId()] = lspE;

    // Node F
    Lsp lspF(f1,1,2,3);
    lspF.FaceList().push_back(Lsp::Face());
    lspF.FaceList().back().Addr() = f1;
    lspF.FaceList().back().NeighborList().push_back(Neighbor(e1,1));

    lspF.FaceList().push_back(Lsp::Face());
    lspF.FaceList().back().Addr() = f2;
    lspF.FaceList().back().NeighborList().push_back(Neighbor(a2,1));
    lspF.FaceList().back().NeighborList().push_back(Neighbor(b2,1));

    lspF.ExtRouteList().push_back(ExtRoute(0x82000000,0xFF000000,10));
    cLspMap[lspF.RouterId()] = lspF;
  }

  
  /* Pretty prints the given LspMap to a String */
  String DumpLspMap(LspMap& lspMap) {
    String s;
    LspMap::iterator it;
    int i;
    for(it = lspMap.begin(), i = 0; it != lspMap.end(); it++, i++) {
      s += String("\nlsp[") + String(i) + String("] =") + (*it).second.Dump();
    }
    return(s);
  }

  /* Pretty prints our NeighborSet to a String */
  String DumpNeighborSet() {
    String s;
    
    IfNeighborMap::iterator i;
    for(i = cIfNeighborMap.begin(); i != cIfNeighborMap.end(); i++) {
      Interface* f = (*i).first;
      NeighborSet::iterator it;
      NeighborSet& ns = (*i).second;
      s += String("\ninterface=") + DumpAddr(f->Addr());
      int j;
      for(it = ns.begin(), j=0; it != ns.end(); it++, j++) {
	s += String("\n  neighbor[") + String(j) + String("] = ") + (*it).Dump();
      }
    }
    return(s);
  }

  /* Nested class for dealing with IP addres/netmask pairs.
   * Really this is just a convenience.
   */
  class NetAddr {
    unsigned int cAddr;
    unsigned int cNetmask;
  public:
    NetAddr(unsigned int addr, unsigned int netmask) :
      cAddr(addr), cNetmask(netmask) {};
    NetAddr() : cAddr(0), cNetmask(0) {};
    unsigned int& Addr() { return cAddr; }
    unsigned int& Netmask() { return cNetmask; }
    unsigned int Addr() const { return cAddr; }
    unsigned int Netmask() const { return cNetmask; }
    int operator ==(const NetAddr& rhs) const {
      return((cAddr == rhs.cAddr) && (cNetmask == rhs.cNetmask));
    }
    
    int operator <(const NetAddr& rhs) const {
      if(cAddr < rhs.cAddr)     return(1);
      if(!(cAddr == rhs.cAddr)) return(0);
      
      // Here if the same addr
      if(cNetmask  <  rhs.cNetmask)     return(1);
      return(0);
    }
  };
    
  /* Nested class for storing path attributes...
   * We need this when we run Djykstra on our LspMap.
   */
  class PathInfo {
    unsigned int cCost;          // Total cost to the destination
    unsigned int cForwardingIf;  // Our local interface address to forward towards the destination
    unsigned int cFirstHopAddr;  // The address of the first hop interface (ie. IP gateway)
    unsigned int cArrivingIf;    // The interface address on the arriving router
    bool cDirect;                // True if we reach this router directly (ie. its one hop away)
  public:
    PathInfo() : cCost(0), cForwardingIf(0), cFirstHopAddr(0), cArrivingIf(0), cDirect(false) {};
    PathInfo(unsigned int cost, unsigned int forwardingIf, unsigned int firstHopAddr,
	     unsigned int arrivingIf, bool direct) : 
      cCost(cost), cForwardingIf(forwardingIf), cFirstHopAddr(firstHopAddr), 
      cArrivingIf(arrivingIf), cDirect(direct) {};
    unsigned int  Cost()         const { return cCost;         }
    unsigned int  ForwardingIf() const { return cForwardingIf; }
    unsigned int  FirstHopAddr() const { return cFirstHopAddr; }
    unsigned int  ArrivingIf()   const { return cArrivingIf;   }
    bool          Direct()       const { return cDirect;       }
    unsigned int& Cost()         { return cCost;         }
    unsigned int& ForwardingIf() { return cForwardingIf; }
    unsigned int& FirstHopAddr() { return cFirstHopAddr; }
    unsigned int& ArrivingIf()   { return cArrivingIf;   }
    bool&         Direct()       { return cDirect;       }

    String Dump() const {
      String s;
      s += String("forIf=") + DumpAddr(ForwardingIf()) + " ";
      s += String("firstHop=") + DumpAddr(FirstHopAddr()) + " ";
      s += String("cost=") + String((int)Cost()) + " ";
      s += String("direct=") + String((int)Direct()) + " ";
      s += String("arrIf=") + DumpAddr(ArrivingIf()) + " ";
      //s += "\n";
      return s;
    }

  };

  // Keys are router id's, values are PathInfo's */
  typedef std::map<unsigned int, PathInfo, less<unsigned int> > PathInfoMap;

  // Keys are external route address/netmasks, values are PathInfo's */
  typedef std::map<NetAddr, PathInfo, less<NetAddr> > ExtPathInfoMap;

  /* Returns the name of an IP interface using the given addresses
   * First, tries to find a point-to-point match...if none,
   * then tries to find any interface with the given local address
   */
  String LookupIfname(unsigned int ifaddr, unsigned int dstaddr) {
    for(int fd=0; fd < cFdLimit; fd++) {
      Interface* f = cReportInterfaces[fd];
      if(f && f->HasDestAddr() && (f->Addr() == ifaddr) && (f->DestAddr() == dstaddr)) {
	return(f->Name());
      }
    }
    for(int fd=0; fd < cFdLimit; fd++) {
      Interface* f = cReportInterfaces[fd];
      if(f && (f->Addr() == ifaddr)) {
	return(f->Name());
      }
    }
    Report::Error(String("Router::LookupIfname() - failed for ") + DumpAddr(ifaddr));
    return("");
  }
  
  /* Builds a set of shortest paths from the given root to all nodes
   * using Dijkstra's algorithm. 
   *
   * The caller may elect not to compute shortest paths to external routes;
   * this will speed things up a little when the caller doesn't care about
   * shortest paths to external routes (which is the case when computing which
   * border router has the shortest *Host* route).
   */
  RouteSet ComputeRoutes(unsigned int root, bool doExternalRoutes = true) {

    if(Debug("ComputeRoutes")) {
      Report::Debug(String("ComputeRoutes() - LspMap:") +  DumpLspMap(cLspMap));
      Report::Debug(String("ComputeRoutes() - Inverted LspMap:") + DumpLspMap(cInvertedLspMap));
      Report::Debug(String("ComputeRoutes() - Neighbors:") + DumpNeighborSet());
    }
    
    PathInfoMap path;
    PathInfoMap tent;
    ExtPathInfoMap extPath;

    RouteSet routes;
 
    // Start with ourself as the root of the tree
    unsigned int n = root; 
    path[n] = PathInfo();  // Initialize root's PathInfo
    do {
      const PathInfo& npi = path[n];
      LspMap::iterator it = cInvertedLspMap.find(n);
      if(it != cInvertedLspMap.end()) {
	// We have an Lsp for this router...
	const Lsp& lsp = (*it).second;
	list<Lsp::Face>::const_iterator j;
	for(j = lsp.FaceList().begin(); j != lsp.FaceList().end(); j++) {
	  unsigned int addr = (*j).Addr();
	  const list<Neighbor>& neighborList = (*j).NeighborList();
	  if(Debug("ComputeRoutes")) {
	    Report::Debug(String("ComputeRoutes() - face: ") + (*j).Dump());
	  }

	  // Interate across this interface's neighbor list
	  list<Neighbor>::const_iterator k;
	  for(k = neighborList.begin(); k != neighborList.end(); k++) {
	    unsigned int neighborAddr = (*k).Addr();
	    unsigned int neighborMetric = (*k).Metric();
	    unsigned int neighborRouterId;

	    // Attempt to lookup the router id of the neighbor; skip it if we don't know it
	    if(!LookupRouterId(neighborAddr,neighborRouterId)) continue;

	    if(Debug("ComputeRoutes")) {
	      Report::Debug(String("ComputeRoutes() - neighbor: ") + DumpAddr(neighborAddr));
	    }

	    // If we already have a shortest path to this node, ignore it
	    if(path.find(neighborRouterId) != path.end()) {
	      continue;
	    }

	    PathInfo mpi;
	    mpi.Cost() = npi.Cost() + neighborMetric;
	    mpi.ForwardingIf() = (npi.ForwardingIf() ? npi.ForwardingIf() : addr);
	    mpi.FirstHopAddr() = (npi.FirstHopAddr() ? npi.FirstHopAddr() : neighborAddr);
	    mpi.ArrivingIf()   = neighborAddr;
	    mpi.Direct()       = (n == root);

	    PathInfoMap::iterator ii = tent.find(neighborRouterId);
	    if(ii != tent.end()) {
	      // We already have a tentative shortest path...overwrite it if this path is shorter
	      if(mpi.Cost() < (*ii).second.Cost()) {
		(*ii).second = mpi;
	      }
	    }
	    else {
	      // This is the first tentative path to this node
	      tent[neighborRouterId] = mpi;
	    }
	  }
	}
	if(doExternalRoutes) {
	  // Handle this router's external routes, if any
	  list<ExtRoute>::const_iterator x;
	  for(x = lsp.ExtRouteList().begin(); x != lsp.ExtRouteList().end(); x++) {
	    NetAddr na((*x).Addr(),(*x).Netmask());

	    PathInfo xpi;
	    xpi.Cost() = npi.Cost() + (*x).Metric();
	    xpi.ForwardingIf() = npi.ForwardingIf();
	    xpi.FirstHopAddr() = npi.FirstHopAddr();
	    xpi.ArrivingIf()   = n; //br1: default route final destination
	    ExtPathInfoMap::iterator y = extPath.find(na);
	    if(y != extPath.end()) {
	      // We already have seen this external route
	      // Overwrite it if we found a lower cost path
	      if(xpi.Cost() < (*y).second.Cost()) {
		(*y).second = xpi;
	      }
	    }
	    else {
	      // We have not seen this external route before, create it
	      extPath[na] = xpi;
	    }
	  }
	}
      }
      // We're done when we have no more tentative paths to explore
      if(tent.size() == 0) break;

      // Find the lowest cost tentative path and explore from there
      //### For now, we'll perform a linear search...a priority queue should replace this
      PathInfoMap::iterator jj = tent.begin();
      PathInfoMap::iterator min = jj;
      for(; jj != tent.end(); jj++) {
	if((*jj).second.Cost() < (*min).second.Cost()) min = jj;
      }

      n = (*min).first;
      path[n] = (*min).second;
      tent.erase(min);

    } while(1);

    
    // Dump the paths
    if(Debug("ComputeRoutes")) {
      PathInfoMap::iterator it;
      int i;
      for(it = path.begin(), i=0; it != path.end(); it++, i++) {
	unsigned int routerId = (*it).first;
	const PathInfo& pi = (*it).second;
	Report::Debug(String("ComputeRoutes() - path[") + String(i) + "] " +
		      DumpAddr(routerId) + " -> " + pi.Dump());
      }
      ExtPathInfoMap::iterator k;
      for(k = extPath.begin(), i=0; k != extPath.end(); k++, i++) {
	const PathInfo& pi = (*k).second;
	Report::Debug(String("ComputeRoutes() - extPath[") + String(i) + "] " +
		      DumpAddr((*k).first.Addr()) +  " / " +
		      DumpAddr((*k).first.Netmask()) + " -> " + pi.Dump());
      }
    }
    
    // We have computed shortest paths to all destination routers and all
    // externally advertised routes.
    // Now, translate the paths into a set of Route objects

    /* We want direct routes to only our neighbors' interfaces that we
     * forward to.  For example, our neighbor B may have interfaces 1,
     * 2, and 3. We may have a direct link to interfaces 1 and 2 but
     * not to interface 3. When we computed the paths above, we may
     * have selected interface 2 as the interface that we would
     * forward any traffic that needed to go to B. This means that we
     * want a direct route to interface 2 on neighbor B and indirect
     * (or IP gateway) routes to interfaces 1 and 3. We could
     * conceivably add a direct route to interface 1 also, but, we
     * refrain from doing this because it creates the following
     * problem.
     *
     * Linux does not do a good job when it comes to deleting multiple
     * routes to the same host address (with the same metric) - you
     * can't specify which route gets deleted; the first route that
     * the kernel finds that matches the host address get whacked,
     * regardless of whether we supply a gateway address.  So, we want
     * to avoid adding multiple routes to the same host.
     *
     */

    // If we aren't computing our own routes then ain't now way we'll know ifnames
    bool doIfname = (root == cId);

    // Translate paths to direct or gateway routes
    PathInfoMap::iterator it;
    for(it = path.begin(); it != path.end(); it++) {
      unsigned int routerId = (*it).first;
      const PathInfo& pi = (*it).second;
      // Ignore our own route...this should have been added when the system booted
      // or when the interface was configured.
      if(pi.ForwardingIf() == 0) continue;
      
      String ifname = (doIfname ? LookupIfname(pi.ForwardingIf(),pi.ArrivingIf()) : String(""));
      
      // If this router is a neighbor of ours, add a direct route
      // to the interface that we forward to
      if(pi.Direct()) {
	Route r = Route(ifname,Route::Host,pi.ArrivingIf(),0xFFFFFFFF,pi.Cost());
	if(Debug("ComputeRoutes")) {
	  Report::Debug(String("ComputeRoutes() - Inserting Direct Route: ") + r.Dump());
	}
	routes.insert(r);
      }

      // We want to add a gateway route for each of the routers interfaces,
      // except for an interface that we have a direct route to.
      LspMap::iterator j = cLspMap.find(routerId);
      if(j == cLspMap.end()) {
	String s;
	s += String("Router::ComputeRoutes() - routerId=") + DumpAddr(routerId) + " pathInfo=" + pi.Dump();
	Report::Error(s);
      }
      const Lsp& lsp = (*j).second;
      list<Lsp::Face>::const_iterator k;
      for(k = lsp.FaceList().begin(); k != lsp.FaceList().end(); k++) {
	unsigned int addr = (*k).Addr();
	if(pi.Direct() && (pi.ArrivingIf() == addr)) {
	  if(Debug("ComputeRoutes")) {
	    Report::Debug(String("ComputeRoutes() - Skipping..already have a direct route to:  ")
			  + DumpAddr(addr));
	  }
	  continue; // Already have a direct route?
	}
	Route r = Route(ifname,Route::Host,addr,0xFFFFFFFF,pi.Cost(),pi.FirstHopAddr());
	if(Debug("ComputeRoutes")) {
	  Report::Debug(String("ComputeRoutes() - Inserting Gateway Route: ") + r.Dump());
	}
	routes.insert(r);
      }
    }

    if(doExternalRoutes) {
      // Translate external routes to gateway routes
      ExtPathInfoMap::iterator ii;
      for(ii = extPath.begin(); ii != extPath.end(); ii++) {
	const NetAddr& na  = (*ii).first;
	const PathInfo& pi = (*ii).second;
	// Ignore our own external routes...this should have been added
	// when the system booted, when the interface was configured, or by some
	// other routing protocol (eg. GATED)
	if(pi.ForwardingIf() == 0) continue;

	String ifname = (doIfname ? LookupIfname(pi.ForwardingIf(),pi.ArrivingIf()) : String(""));
	Route r(ifname,Route::Network,na.Addr(),na.Netmask(),pi.Cost(),pi.FirstHopAddr());
	if(Debug("ComputeRoutes")) {
	  Report::Debug(String("ComputeRoutes() - Inserting External Route: ") + r.Dump());
	}
	//br1: memorize final destination of defautl route
	r.finalDest=pi.ArrivingIf();
	//Report::Debug(String("* ext r: ") + DumpAddr(pi.ArrivingIf()) );
	//Report::Debug(String("* ext fd: ") + r.finalDest );

	routes.insert(r);
      }
    }

    return(routes);
  }

  /* Iterates across the Lsp map and decrements the age of all packets
   * by the given amount.  Any Lsp's that have expired are removed
   * from the Lsp map. Also, we must remove corresponding entries from
   * the transmit time map.
   */
  void PruneOldLsps(unsigned int delta) {
    LspMap::iterator it;
    for(it = cLspMap.begin(); it != cLspMap.end(); ) {
      Lsp& lsp = (*it).second;
      unsigned short& age = lsp.Age();
      if(age <= delta) {
	Report::Warn(String("PruneOldLsps() - lsp from ") + DumpAddr(lsp.RouterId()) +
			" expired...deleting.");
	TimeMap::iterator j = cTxTimeMap.find((*it).first);
	if(j != cTxTimeMap.end()) {
	  cTxTimeMap.erase(j);
	}
	cLspMap.erase(it++);
      }
      else {
	age -= delta;
	it++;
	if(Debug("PruneOldLsps")) {
	  Report::Debug(String("PruneOldLsps() - lsp from ") + DumpAddr(lsp.RouterId()) +
			" age=" + String((int)age));
	}
      }
    }
  }

  /* Attempts to send the given route to all clients who previously
   *  expressed interest in hearing reachability info. We assume that
   *  any client who sent us a list command would like to hear future
   *  adds and deletes.
   */
  void ExportRoute(const Route& r, bool add) {
    String s;
    s += (add ? String("a ") : String("d "));
    s += DumpAddr(r.DstAddr()) + " ";
    s += DumpAddr(r.Netmask()) + " ";
    if(add)
      s +=  String((int)r.Metric());
    s += " \n";

    // Send the reachability to all known clients
    ExportClientSet::iterator it;
    for(it = cExportClientSet.begin(); it != cExportClientSet.end();) {
      UnixDatagram reply(s,(*it));
      if(!cExportSocket->Send(reply)) {
	// The client must have gone away...remove him
	cExportClientSet.erase(it++);
      }
      else {
	it++;
      }
    }
  }



  bool DatabaseDump(const String& addr) {
    // Send the begin dump datagram
    UnixDatagram beginDg("{\n",addr);
    if(!cDumpSocket->Send(beginDg)) return false;

    // Send all Lsp's
    LspMap::iterator it;
    for(it = cLspMap.begin(); it != cLspMap.end(); it++) {
      UnixDatagram dg((*it).second.Encode(),addr);
      if(!cDumpSocket->Send(dg)) return false;
    }

    // Send the end dump datagram
    UnixDatagram endDg("}\n",addr);
    if(!cDumpSocket->Send(endDg)) return(false);
    return(true);
  }

  /* Attempts to send the contents of the Lsp database to all clients
   *  who previously expressed interest in hearing it. We assume that
   *  any client who sent us a list command would like to hear future
   *  adds and deletes.
   */
  void DatabaseDump() {
    // Send the database to all known clients
    DatabaseDumpClientSet::iterator it;
    for(it = cDatabaseDumpClientSet.begin(); it != cDatabaseDumpClientSet.end();) {
      if(!DatabaseDump(*it)) {
	// The client must have gone away...remove him
	cDatabaseDumpClientSet.erase(it++);
      }
      else {
	it++;
      }
    }
  }


  /* Coordinates the steps required to update the kernel routing
   * table.  First, it populated the router id map. Next, it builds a set of
   * routes.  Then, it efficiently updates the kernel routing table by
   * performing set operations to determine which IP routes need to be
   * deleted and which need to be added.  */
  void UpdateRoutingTable() {

    PopulateRouterIdMap();
    InvertLspMap();

    // Get the set of routes based upon the current link state database
    RouteSet routes = ComputeRoutes(cId);

    if(Debug("UpdateRoutingTable")) {
      Report::Debug("UpdateRoutingTable() - Had:");
      RouteSet::iterator it;
      for(it = cRoutes.begin(); it != cRoutes.end(); it++) {
	Report::Debug((*it).Dump());
      }
    }


    if(Debug("UpdateRoutingTable")) {
      Report::Debug("UpdateRoutingTable() - Computed:");
      RouteSet::iterator it;
      for(it = routes.begin(); it != routes.end(); it++) {
	Report::Debug((*it).Dump());
      }
    }

    // Figure out which routes are now stale by performing a set difference:
    //   stale = cRoutes - routes
    list<Route> stale;
    back_insert_iterator< list<Route> > si(stale);
    set_difference(cRoutes.begin(),cRoutes.end(),
		   routes.begin(),routes.end(),
		   si);

    if(Debug("UpdateRoutingTable")) {
      Report::Debug("UpdateRoutingTable() - Stale:");
      list<Route>::iterator it;
      for(it = stale.begin(); it != stale.end(); it++) {
	Report::Debug((*it).Dump());
      }
    }

    // Delete the stale routes
    list<Route>::iterator it;
    for(it = stale.begin(); it != stale.end(); it++) {
      KernelDelete((*it));
      if ( (*it).Netmask() == 0 && (*it).DstAddr() == 0 && cConfig.addDefaultGw() ) {
		Report::Debug(String("*** DEL default route to ") + DumpAddr((*it).finalDest) );
		char cmdbuffer[64];
		sprintf( cmdbuffer, "%s %s del &", cConfig.defaultGwScript().c_str(), DumpAddr((*it).finalDest).c_str() );
		system(cmdbuffer);
	}
      ExportRoute((*it),false);
    }

    // Figure out which routes are new
    //   found = routes - cRoutes
    //list<Route> found(routes.size());
    list<Route> found;
    back_insert_iterator< list<Route> > fi(found);

    set_difference(routes.begin(),routes.end(),
		   cRoutes.begin(),cRoutes.end(),
		   fi); 

    if(Debug("UpdateRoutingTable")) {
      Report::Debug("UpdateRoutingTable() - Found:");
      list<Route>::iterator it;
      for(it = found.begin(); it != found.end(); it++) {
	Report::Debug((*it).Dump());
      }
    }

    // Add the newly found routes
    // Note: We must add direct routes before gateway routes, otherwise
    // we'll get an error message about the gateway not being reachable
    for(it = found.begin(); it != found.end(); it++) {
      if(!(*it).HasGw())
	if(!KernelAdd((*it))) {
	  routes.erase((*it)); // If we could not add the route then don't hold onto it
	}
	else {
	  ExportRoute((*it),true);
	}
    }
    for(it = found.begin(); it != found.end(); it++) {
      if((*it).HasGw()) {
      	// default gw
	if ( (*it).Netmask() == 0 && (*it).DstAddr() == 0 ) {
		if ( cConfig.addDefaultGw() ) {
			if(!KernelAdd((*it))) {
				routes.erase((*it)); // If we could not add the route then don't hold onto it
			} else {
				ExportRoute((*it),true);
				
				Report::Debug(String("*** ADD default route to ") + DumpAddr((*it).finalDest) );
				char cmdbuffer[64];
				sprintf( cmdbuffer, "%s add %s &", cConfig.defaultGwScript().c_str(), DumpAddr((*it).finalDest).c_str() );
				system(cmdbuffer);
			}
		}
	}
	// other routes
      	else {
		if(!KernelAdd((*it))) {
	  		routes.erase((*it)); // If we could not add the route then don't hold onto it
		}
		else {
			ExportRoute((*it),true);
			
			char cmdbuffer[64];
			sprintf( cmdbuffer, "%s del %s &", cConfig.defaultGwScript().c_str(), DumpAddr((*it).finalDest).c_str() );
			system(cmdbuffer);
		}
	}
      }
    }

    // Overwrite our old table
    cRoutes = routes;
  }

  /* Looks for a file named "mmrp-seq-<port>" which stores the sequence number
   * of the last LSP sent by this router.
   * If the file exists, and the time elapse since the file was last modified is less
   * than MAX_AGE, then the sequence number contained in the file is returned.
   * Otherwise, zero is returned.
   *
   */
  void InitializeSequenceNumber() {
    // Start seq at zero...maybe modify below
    cSeq = 0;

    struct stat fs;
    String fn = String("mmrp-seq-") + String((int)cConfig.Port());
    if(stat(fn.c_str(), &fs)) return;
    time_t now = time(0);
    
    // In case someone reset the system clock, don't bother
    if(fs.st_mtime > now) return;
    
    // If the sequence number was written a long time ago, don't bother
    if(int(fs.st_mtime + cConfig.MaxAge()) < now ) return;
    
    // Here if we want to use the last sequence number
    FILE* f;
    if((f = fopen(fn.c_str(),"r"))) {
      unsigned int seq;
      int n = fscanf(f,"%u",&seq);
      fclose(f);
      if(n == 1) {
	cSeq = seq + 1;
      }
    }
  }

  /* Saves the current sequence number to a file...this
   * allows a router that crashes to restart and pickup where it left off
   * without having to wait for its previous LSP to be age'd out.
   */
  void SaveSequenceNumber() {
    String fn = String("mmrp-seq-") + String((int)cConfig.Port());
    FILE* f;
    if((f = fopen(fn.c_str(),"w"))) {
      fprintf(f,"%u\n",cSeq);
      fclose(f);
    }
  }

  /* Reads configuration information from the stream
   * Verifies that interfaces exist and gets system interface info (ie. hw addr, ip addr, etc.)
   */
  void ReadConfiguration(istream& is) {
    list<Interface> sysList = Interface::Discover();

    String err("Invalid line in config file: ");
    String s;
    Config config;
    while(s.Getline(is)) {
      String line = s;
      String type     = s.Token(&s);
      if(!type.length()) {
	continue; // Ignore blank lines
      }
      if(type[0] == '#') {
	continue; // Ignore lines that begin with a '#' comment
      }
      else if(type == "port") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.Port() = v.Int();
      }
      else if(type == "alpha") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.Alpha() = v.Double();
	if((config.Alpha() < 1.0) || (config.Alpha() > 2.0)) {
	  Report::Error(err + line + " Alpha must be >= 1.0 and <= 2.0 ");
	}
      }
      else if(type == "minTxDelay") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.MinTxDelay() = v.Int();
      }
      else if(type == "maxTxDelay") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.MaxTxDelay() = v.Int();
      }
      else if(type == "maxAge") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.MaxAge() = v.Int();
      }
      else if(type == "updatePeriod") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.UpdatePeriod() = v.Int();
      }
      else if(type == "encryption") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	isEncrypted = v.Int();
	// isEncrypted is statically defined in UtUdpSocket.h
      }
      else if(type == "interface") {
	String ifname   = s.Token(&s);
	if(!ifname.length()) {
	  Report::Error(err + line);
	}
	AddInterface(ifname);
      }
      else if(type == "external") {
	String addr    = s.Token(&s);
	String netmask = s.Token(&s);
	String metric  = s.Token(&s);
	if(!addr.length() || !netmask.length() || !metric.length()) {
	  Report::Error(err + line);
	}
	InetAddr a(addr);
	InetAddr nm(netmask);
	cExtRouteList.push_back(ExtRoute(a.Addr(),nm.Addr(),metric.Int()));
      }
      else if(type == "addDefaultGw") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.addDefaultGw() = v.Int();
      }
      else if(type == "defaultGwScript") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	config.cDefaultGwScript = v;
      }
      else if(type != "") {
	Report::Error(err + line);
      }
      cConfig = config;  // Update after every line parsed...AddInterface looks at cConfig!
    }
  }

  /* Returns the well known name of the Unix socket that the
   * LinkDiscover program listens on.
   */
  String ReportSocketName(const String& ifname) {
    String s = String("ln-") + ifname;
    return(s);
  }
  
  /* When we start up, a LinkDiscover program may have already discovered a 
   * bunch of neighbors (links). We can get a list of the current links
   * by asking for it.
   */
  void QueryLinkNeighbors(UnixSocket& reportSocket, Interface& f) {
    String err = String("Link discovery program on interface '") + 
      f.Name() + "' is not responding. " +
      "Please restart it before running mmrp.";
    
    // Send a list request to LinkDiscover program on this interface
    String s = "l\n";
    UnixDatagram reply(s,ReportSocketName(f.Name()));
    if(!reportSocket.Send(reply)) {
      Report::Error(err);
    }
  }

  /* A crude test to determine if each LinkDiscover is still operational
   * Send a ping request to each...we harvest ping replies asynchronously
   * later. If we didn't hear a ping reply since last time we checked, we exit.
   */
  void CheckLinkDiscovers() {
    int fd;
    for(fd=0; fd < cFdLimit; fd++) {
      Interface* f = cReportInterfaces[fd];
      if(f) {
	UnixSocket* reportSocket = cReportSockets[fd];
	bool& expectReply = cReportReplies[fd];
	String err = String("Link discovery program on interface '") + 
	  f->Name() + "' is not responding. " +
	  "Please restart it before running mmrp.";
	
	if(expectReply) {
	  Report::Error(err);
	}
	expectReply = true;
	
	// Send a ping request to LinkDiscover program on this interface
	String s = "q\n";
	UnixDatagram reply(s,ReportSocketName(f->Name()));
	if(!reportSocket->Send(reply)) {
	  Report::Error(err);
	}
      }
    }
  }

  /* Merge neighbors from different interfaces sharing the same address
   * 
   */
  void MergeNeighbors() {
    // Clear out old contents...
    cIfAddrNeighborMap.clear();
    
    // Add empty neighbor sets for all active interfaces
    int fd;
    for(fd=0; fd < cFdLimit; fd++) {
      if(cReportSockets[fd] && cReportInterfaces[fd]) {
	cIfAddrNeighborMap[cReportInterfaces[fd]->Addr()];
      }
    }

    IfNeighborMap::iterator i;
    for(i = cIfNeighborMap.begin(); i != cIfNeighborMap.end(); i++) {
      Interface* f = (*i).first;
      NeighborSet::iterator it;
      NeighborSet& ns = (*i).second;
      for(it = ns.begin(); it != ns.end(); it++) {
	cIfAddrNeighborMap[f->Addr()].insert((*it));
      }
    }
  }

   
  /* Builds an Lsp based upon the links we have been told exist
   * by our local interfaces. 
   *
   */
  Lsp BuildLocalLsp() {
    list<Lsp::Face> faces;

    MergeNeighbors();

    IfAddrNeighborMap::iterator i;
    for(i = cIfAddrNeighborMap.begin(); i != cIfAddrNeighborMap.end(); i++) {
      unsigned int addr = (*i).first;
      NeighborSet::iterator it;
      NeighborSet& ns = (*i).second;
      list<Neighbor> neighbors;
      for(it = ns.begin(); it != ns.end(); it++) {
	neighbors.push_back((*it));
      }
      faces.push_back(Lsp::Face(addr,neighbors));
    }

    Lsp lsp(cId, cSeq++, 0, cConfig.MaxAge(), &faces, &cExtRouteList);

    if(Debug("BuildLocalLsp")) {
      Report::Debug(String("BuildLocalLsp() - local lsp: ") + lsp.Dump());
      Report::Debug(String("BuildLocalLsp() - Neighbors:") + DumpNeighborSet());
    }

    return(lsp);
  }

  /* Computes the amount of time in seconds that an Lsp with the given
   * number of hops should be held until it is transmitted.
   *
   * For fish-eye routing, the tx delay grows as a function of the
   * number of hops.  To be practical, we limit the maximum tx delay;
   * otherwise large diameter networks may take way too long to get
   * updates from the other end.
   *
   * We add a random component to help prevent synchronizing the
   * network.  The reader is directed to "The Synchronization of
   * Periodic Routing Messages" by Sally Floyd and Van Jacobson;
   *   http://www.aciri.org/floyd/papers/sync_94.ps.Z 
   * Sally recommends drawing the delay from a uniformly distributed
   * interval [0.5Tp,1.5Tp] where Tp is the average period.
   *
   * I'm not following this recommendation exactly, but I will add
   * some randomness. It seems reasonable to simply add a random
   * number uniformly distributed between [0,0.5Tp] which results in a
   * delay uniformly distributed [Tp,1.5Tp] with an average period of
   * 1.25Tp. This delay has less variance than what Sally recommends
   * but it's probably sufficient.
   *  
   */
  Time ComputeTxDelay(unsigned int hops) {
    double t = pow(cConfig.Alpha(), hops); 
    t = ((t < cConfig.MinTxDelay()) ? cConfig.MinTxDelay() : t);
    t = ((t > cConfig.MaxTxDelay()) ? cConfig.MaxTxDelay() : t);
    double r = t * (double(random()) / double(RAND_MAX)) * 0.5;
    t += r;
    return(Time(t));
  }
   

  /* One of our local interfaces heard an Lsp from a neighbor's
   * interface. Check to see if this is the most recent Lsp, if so,
   * store it.
   *
   */
  void HandleLsp(Lsp& lsp, const InetAddr& src, Interface *f) {
    // Disregard any packets that have our Hancock
    if(lsp.RouterId() == cId) return;
    if(Debug("HandleLsp")) {
      String s;
      s += String("HandleLsp() - From ") +
	String(src) + " on " + DumpAddr(f->Addr()) + " : " + lsp.Dump();
      Report::Debug(s);
    }
    

    // We must decrement the age by at least 1 second at each hop,
    // otherwise we can have an Lsp wander around the network forever.
    unsigned short& age = lsp.Age();
    if(age == 0) return; // We don't handle packets that have expired
    age -= 1;

    // We must increment the number of hops that this Lsp has travelled
    // Note: Avoid wraparound...though highly unlikely that a packet
    // bopped around 64k hops!
    unsigned short& hops = lsp.Hops();
    if(hops != 0xFFFFFFFF) 
      hops += 1;

    // Compute when we should rebroadcast this Lsp based on its hop count
    Time txTime = Time::Now() + ComputeTxDelay(lsp.Hops());

    LspMap::iterator it = cLspMap.find(lsp.RouterId());
    if(it == cLspMap.end()) {
      // We don't have an Lsp from this router yet...go ahead and save it
      cLspMap[lsp.RouterId()] = lsp;
      cTxTimeMap[lsp.RouterId()] = txTime;
    }
    else {
      if( ( (*it).second.Seq() < lsp.Seq() ) ||
          ( ((*it).second.Seq() == lsp.Seq()) && ((*it).second.Hops() > lsp.Hops()) ) ) {
	// Either the lsp in our map is older than the one we just got or
	// it has the same sequence number but with a higher hop count
	// In either case, store the lsp we just heard
	(*it).second = lsp;

	/* Only update the transmit time if it is earlier than the one we have
	 * This prevents a rather insidious problem caused by newer packets
	 * showing up, overwriting older ones, and never getting transmitted.
	 */
	if(txTime < cTxTimeMap[lsp.RouterId()]) {
	  cTxTimeMap[lsp.RouterId()] = txTime;
	}
      }
    }
  }

  /* Determine what type of message this is, decode it, and dispatch it */
  void HandleReceive(UdpDatagram& dg, Interface* f) {
    String d = dg.GetData();
    InetAddr src = dg.GetAddr();
    
    if(d.length() < 2) {
      char buf[1024];
      sprintf(buf,"Router::HandleReceive() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    // Make sure we're not hearing one of our own!
    for(int fd=0; fd < cFdLimit; fd++) {
      Interface* f = cInterfaces[fd];
      if(f && (f->Addr() == src.Addr())) {
	return;
      }
    }

    unsigned char version = d[0];
    if(version != MsgTypes::Version) {
      char buf[1024];
      sprintf(buf,"Router::HandleReceive() - Invalid version from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char type = d[1];
    switch(type) {
    case MsgTypes::tLsp:
      {
	Lsp lsp;
	if(lsp.Decode(d))
	  HandleLsp(lsp,src,f);
	break;
      }
    default:
      {
	char buf[1024];
	sprintf(buf,"Router::HandleReceive() - Invalid message type from %s, discarding.",
		String(src).c_str());
	Report::Warn(buf);
      }
    }
  }

  /* Handle link reports from an interface's LinkDiscover program
   * We are a client and are implicitly registered to receive link
   * status changes as a result of sending a list command 'l' to
   * the LinkDiscover server program.
   *
   * Reports are in the following format:
   *
   *  Add a link: 
   *     "a <addr> <metric>"
   *
   *  Delete a previously added link:
   *     "d <addr> <netmask>
   *
   *  Listing an existing link:
   *     "l <addr> <metric>"
   *
   *  The server may inform us that it is going away by sending an 'e' command.
   *
   */
  void HandleReport(UnixDatagram& dg, Interface* f, bool& expectReply) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();

    if(Debug("HandleReport")) {
      Report::Debug(String("HandleReport() - From: ") + String(dg.GetAddr()) + "  '" + dg.GetData() + "' ");
    }

    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Router::HandleReport() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char rep = d[0];

    String err("Router::HandleReport() - Invalid command format.");

    // Consume report
    d.Token(&d);
    switch(rep) {
    case 'l':
    case 'a':
      {
	// Add a link
	String addr    = d.Token(&d);
	String metric  = d.Token(&d);
	cout << metric;
	if(!addr.length() || !metric.length()) {
	  Report::Warn(err);
	  return;
	}
	NeighborSet& ns = cIfNeighborMap[f];
	unsigned int a = InetAddr(addr).Addr();
	unsigned int m = metric.Int();
	Neighbor n(a,m);

	NeighborSet::iterator it;
	it = ns.find(n);
	if (it != ns.end()) {
		// need to replace to update metric
		ns.erase(n);
	}
	ns.insert(n);

	if(Debug("HandleReport")) {
	  Report::Debug(String("Router::HandleReport() - adding link:") + n.Dump());
	  Report::Debug(String("Router::HandleReport() - Neighbors:") + DumpNeighborSet());
	}
	break;
      }
    case 'd':
      {
	// Delete a link
	String addr    = d.Token(&d);
	if(!addr.length()) {
	  Report::Warn(err);
	  return;
	}
	unsigned int a = InetAddr(addr).Addr();
	NeighborSet& ns = cIfNeighborMap[f];
	ns.erase(Neighbor(a,0));
	if(Debug("HandleReport")) {
	  Report::Debug(String("Router::HandleReport() - deleting link: ") + addr);
	  Report::Debug(String("Router::HandleReport() - Neighbors: ") + DumpNeighborSet());
	}
	break;
      }
    case 'r':
      {
	// Ping reply
	expectReply = false;
	break;
      }
    case 'e':
      {
	Report::Error(String("Exiting due to discovery program shutdown on interface ") 
		      + f->Name() + ".");
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }

  /* We allow clients to send us commands to tell us about
   * an interface that we should start/stop running the routing protocol on.
   *
   * Note: The client is responsible for ensuring that a discovery program
   * is running on the interface before asking us to add it...otherwise we'll exit!
   *
   * Supported commands are,
   *
   *  Add an interface: 
   *     "a <ifname>"
   *
   *  Delete a previously added interface:
   *     "d <ifname>"
   *
   *  List all interfaces that we are running the protocol on:
   *     "l"
   *
   */
  void HandleConfig(UnixDatagram& dg) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();

    if(Debug("HandleConfig")) {
      Report::Debug(String("HandleConfig() - From: ") + String(dg.GetAddr()) + "  '" + dg.GetData() + "' ");
    }

    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Router::HandleConfig() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char cmd = d[0];
    
    String err("Router::HandleConfig() - Invalid command format.");

    // Consume cmd
    d.Token(&d);
    switch(cmd) {
    case 'a':
      {
	// Add an interface...
	String ifname  = d.Token(&d);
	if(!ifname.length()) {
	  Report::Warn(err);
	  return;
	}
	AddInterface(ifname);
	break;
      }
    case 'd':
      {
	// Delete an interface...
	String ifname  = d.Token(&d);
	if(!ifname.length()) {
	  Report::Warn(err);
	  return;
	}
	DeleteInterface(ifname);
	break;
      }
    case 'l':
      {
	// List all interfaces and return them
	String s = "\n";
	list<Interface>::iterator it;
	for(it = cIfList.begin(); it != cIfList.end(); it++) {
	  s += (*it).Name() + " \n";
	}
	UnixDatagram reply(s,dg.GetAddr());
	cConfigSocket->Send(reply);
	break;
      }
    case 'q':
      {
	// Ping request...send a ping reply
	String s = "r\n";
	UnixDatagram reply(s,dg.GetAddr());
	cConfigSocket->Send(reply);
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }



  /* We allow clients to send us commands to alter the list of external routes that we
   *  advertise. This allows other routing protocols (eg. OSPF) to introduce routes they
   *  know of into the Mobile Mesh Routing Protocol cloud.
   *
   * Supported commands are,
   *
   *  Add an external route: 
   *     "a <addr> <netmask> <metric>"
   *
   *  Delete a previously added external route:
   *     "d <addr> <netmask>
   *
   *  List all external routes:
   *     "l"
   *
   */
  void HandleImport(UnixDatagram& dg) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();

    if(Debug("HandleImport")) {
      Report::Debug(String("HandleImport() - From: ") + String(dg.GetAddr()) + "  '" + dg.GetData() + "' ");
    }

    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Router::HandleImport() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char cmd = d[0];
    
    String err("Router::HandleImport() - Invalid command format.");

    // Consume cmd
    d.Token(&d);
    switch(cmd) {
    case 'a':
      {
	// Add an external route advertisement...
	String addr    = d.Token(&d);
	String netmask = d.Token(&d);
	String metric  = d.Token(&d);
	if(!addr.length() || !netmask.length() || !metric.length()) {
	  Report::Warn(err);
	  return;
	}
	InetAddr a(addr);
	InetAddr nm(netmask);

	// If we already have this route, just update the metric
	bool found = false;
	list<ExtRoute>::iterator it;
	for(it = cExtRouteList.begin(); it != cExtRouteList.end(); it++) {
	  if(((*it).Addr() == a.Addr()) && ((*it).Netmask() == nm.Addr())) {
	    (*it) = ExtRoute(a.Addr(),nm.Addr(),metric.Int());
	    found = true;
	    break;
	  }
	}
	if(!found) 
	  cExtRouteList.push_back(ExtRoute(a.Addr(),nm.Addr(),metric.Int()));
	break;
      }
    case 'd':
      {
	// Delete an external route advertisement...
	String addr    = d.Token(&d);
	String netmask = d.Token(&d);
	if(!addr.length() || !netmask.length() ) {
	  Report::Warn(err);
	  return;
	}
	InetAddr a(addr);
	InetAddr nm(netmask);

	// Look for this route and erase it if found
	list<ExtRoute>::iterator it;
	for(it = cExtRouteList.begin(); it != cExtRouteList.end(); it++) {
	  if(((*it).Addr() == a.Addr()) && ((*it).Netmask() == nm.Addr())) {
	    cExtRouteList.erase(it);
	    break;
	  }
	}
	break;
      }
    case 'l':
      {
	// List all external routes and return them
	String s = "\n";
	list<ExtRoute>::iterator it;
	for(it = cExtRouteList.begin(); it != cExtRouteList.end(); it++) {
	  s += DumpAddr((*it).Addr()) + " "; 
	  s += DumpAddr((*it).Netmask()) + " ";
	  s +=  String((int)(*it).Metric()) + " \n";
	}
	UnixDatagram reply(s,dg.GetAddr());
	cImportSocket->Send(reply);
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }

  /* We can provide output to clients to inform them of the addresses
   * that are reachable within our cloud. This allows some other thing
   * to perform some form of exterior routing.  
   *
   *  There following command is supported,
   *
   *  List reachability info for all addresses in the cloud:
   *   "l"
   *  This command will return a new line delimited list of:
   *    "l <addr> <metric>"
   *
   *
   *  The format of reachability output information is as follows,
   *
   *  The following address is now in the cloud:
   *   "a <addr> <metric>"
   *
   *  The following address in no longer in the cloud:
   *   "d <addr>"
   *
   */
  void HandleExport(UnixDatagram& dg) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();

    if(Debug("HandleExport")) {
      Report::Debug(String("HandleExport() - From: ") + String(dg.GetAddr()) + "  '" + dg.GetData() + "' ");
    }

    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Router::HandleExport() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char cmd = d[0];
    
    String err("Router::HandleExport() - Invalid command format.");

    // Consume cmd
    d.Token(&d);
    switch(cmd) {
    case 'l':
      {
	// List all members of the cloud and return them
	String s = "\n";
	RouteSet::iterator it;
	for(it = cRoutes.begin(); it != cRoutes.end(); it++) {
	  s += String("l ");
	  s += DumpAddr((*it).DstAddr()) + " "; 
	  s += DumpAddr((*it).Netmask()) + " ";
	  s +=  String((int)(*it).Metric()) + " \n";
	}
	UnixDatagram reply(s,dg.GetAddr());
	cExportSocket->Send(reply);
	cExportClientSet.insert(dg.GetAddr()); // Remember who wants to hear exports
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }
  
  /* We can dump the contents of our LSP database to clients.
   * This allows some other thing to use the database for other
   * purposes..eg. to display the current topology.
   * I don't like that we're exposing this stuff but at least it
   * keeps the router general. 
   *
   *  The following command is supported,
   *
   *  List database of Lsp's:
   *   "l"
   *  This command will return a datagram for each Lsp delimited by curly braces:
   *    "{\n"
   *    "...encoded Lsp 1"
   *    "...encoded Lsp N"
   *    "}\n"
   *
   */
  void HandleDatabaseDump(UnixDatagram& dg) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();
    
    if(Debug("HandleDatabaseDump")) {
      Report::Debug(String("HandleDatabaseDump() - From: ") + String(dg.GetAddr()) + "  '" + dg.GetData() + "' ");
    }
    
    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Router::HandleDatabaseDump() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }
    
    unsigned char cmd = d[0];
    
    String err("Router::HandleDatabaseDump() - Invalid command format.");
    
    // Consume cmd
    d.Token(&d);
    switch(cmd) {
    case 'l':
      {
	if(DatabaseDump(dg.GetAddr())) {
	  cDatabaseDumpClientSet.insert(dg.GetAddr()); // Remember who wants to hear database dumps
	}
	break;
      }
    case 'q':
      {
	cDatabaseDumpClientSet.insert(dg.GetAddr()); // Remember who wants to hear reports

	// Ping request...send a ping reply
	String s = "r\n";
	UnixDatagram reply(s,dg.GetAddr());
	cDumpSocket->Send(reply);
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }
  
  /* Iterates across the tx time map and finds earliest time to transmit
   * an Lsp.
   */
  Time EarliestTxTime() const {
    TimeMap::const_iterator it;
    Time min = Time::Eternity();
    for(it = cTxTimeMap.begin(); it != cTxTimeMap.end(); it++) {
      if(double((*it).second) < double(min)) {
	min = (*it).second;
      }
    }
    return(min);
  }
  

  // Broadcasts an Lsp out each interface...except border interface(if we have one!)
  void BroadcastLsp(const Lsp& lsp) {
    UdpDatagram dg(lsp.Encode(), cBroadcastAddr);
    int fd;
    for(fd=0; fd < cFdLimit; fd++) {
      if(cSockets[fd]) {
	if(Debug("BroadcastLsp")) {
	  Report::Debug(String("BroadcastLsp() out ") + DumpAddr(cInterfaces[fd]->Addr()) +
			" " + lsp.Dump());
	}
	cSockets[fd]->Send(dg);
      }
    }
  }
  
  /* Attempts to add the interface with the given name...
   * We error if the system does not have an interface with the given name
   * 
   * Note: If there's no discovery program running on the interface it will
   * eventually be detected and we will exit; but we don't do that here.
   */
  void AddInterface(const String& ifname) {
    list<Interface> sysList = Interface::Discover();

    // Make sure we don't already have this interface
    list<Interface>::iterator it;
    for(it = cIfList.begin(); it != cIfList.end(); it++) {
      if((*it).Name() == ifname) {
	String err = String("AddInterface() - Already have interface ") + 
	  ifname + ", ignoring request to add. ";
	Report::Warn(err);
	return;
      }
    }
    
    // Make sure this is a valid interface!
    bool found = false;
    for(it = sysList.begin(); it != sysList.end(); it++) {
      if((*it).Name() == ifname) {
	found = true;
	break;
      }
    }
    if(!found) {
      String err = String("AddInterface() - Invalid interface: ") + 
	ifname + ", make sure interface is up with 'ifconfig'. ";
      Report::Error(err);
    }

    // Give ourselves a router id if we don't have one already
    if(!cId) {
      cId = (*it).Addr();
      srandom(cId);         // Seed the random number generator
    }

    // Save interface
    cIfList.push_back((*it));
    Interface& f = cIfList.back();

    // Create a Udp socket and bind it to the interface
    // Populate map which maps file descriptors to socket pointer
    // Populate map which maps file descriptors to interface pointer
    // Add file descriptor to our set
    // Create a Unix socket to hear link reports from link layer

    UdpSocket* s = new UdpSocket(InetAddr(InetAddr::ANY,cConfig.Port()),1);
    s->BindDevice(f.Name());
    cSockets[s->Fd()] = s;
    cInterfaces[s->Fd()] = &f;
    FD_SET(s->Fd(),&cFds);
    if(s->Fd() >= cFdLimit) cFdLimit = s->Fd() + 1;
    
    UnixSocket* r = new UnixSocket();
    cReportSockets[r->Fd()] = r;
    cReportInterfaces[r->Fd()] = &f;
    FD_SET(r->Fd(),&cFds);
    if(r->Fd() >= cFdLimit) cFdLimit = r->Fd() + 1;
    
    // Get the current list of neighbors on this interface
    // by asking for it from the Link Discovery program running on the interface
    QueryLinkNeighbors(*r,f); 
  }

  /* Stop running our protocol on the given interface
   */
  void DeleteInterface(const String& ifname) {
    list<Interface> sysList = Interface::Discover();

    // Make sure we have this interface
    list<Interface>::iterator it;
    bool found = false;
    for(it = cIfList.begin(); it != cIfList.end(); it++) {
      if((*it).Name() == ifname) {
	found = true;
	break;
      }
    }
    if(!found) {
      String err = String("DeleteInterface() - Unknown interface ") + 
	ifname + ", ignoring request to delete. ";
      Report::Warn(err);
      return;
    }
    
    // Delete any Udp or Unix socket's associated with the interface
    for(int fd = 0; fd < cFdLimit; fd++) {
      if(cInterfaces[fd] && cInterfaces[fd]->Name() == ifname) {
	delete(cSockets[fd]);
	cSockets[fd] = 0;
	cInterfaces[fd] = 0;
	FD_CLR(fd,&cFds);
      }
      if(cReportInterfaces[fd] && cReportInterfaces[fd]->Name() == ifname) {
	delete(cReportSockets[fd]);
	cReportSockets[fd] = 0;
	cReportInterfaces[fd] = 0;
	FD_CLR(fd,&cFds);
      }
    }

    // Purge any neighbors for this interface
    cIfNeighborMap.erase(&(*it));

    // Remove interface from our list
    cIfList.erase(it);
  }


public:
  
  Router(const String& ifname, const String& configFile, const String& debugFile) : 
    cId(0), cSeq(0), cRouteSocket(InetAddr("")), 
    cPrunePeriod(5.0),
    cDebugFile(debugFile)
  { 
    // Initialize all socket pointers and interface pointers to null
    for(int i=0; i < MAX_INTERFACES; i++) {
      cSockets[i] = 0;
      cInterfaces[i] = 0;
      cReportSockets[i] = 0;
      cReportInterfaces[i] = 0;
      cReportReplies[i] = false;
    }
    
    
    FD_ZERO(&cFds);

    cFdLimit = 0;

    if(cDebugFile.length())
      Debug::Load(cDebugFile);

      if (ifname.length()) {
    		AddInterface(ifname);
	}
	
    std::ifstream is(configFile.c_str());
    if(!is) {
      Report::Error(String("Missing configuration file: ") + configFile);
    }
    ReadConfiguration(is);
    InitializeSequenceNumber();
    KernelForwarding(true);

    cConfigSocket = new UnixSocket(String("mmrp-c") + String((int)cConfig.Port()));
    cImportSocket = new UnixSocket(String("mmrp-i") + String((int)cConfig.Port()));
    cExportSocket = new UnixSocket(String("mmrp-x") + String((int)cConfig.Port()));
    cDumpSocket   = new UnixSocket(String("mmrp-t") + String((int)cConfig.Port()));

    int fd;
    fd = cConfigSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd >= cFdLimit) cFdLimit = fd + 1;

    fd = cImportSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd >= cFdLimit) cFdLimit = fd + 1;

    fd = cExportSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd >= cFdLimit) cFdLimit = fd + 1;

    fd = cDumpSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd >= cFdLimit) cFdLimit = fd + 1;

    cBroadcastAddr = InetAddr(InetAddr::BCAST,cConfig.Port());

    // Debug
    if(Debug("Verbose")) {
      Report::Debug("Starting protocol on these interfaces:");
      list<Interface>::iterator it;
      for(it = cIfList.begin(); it != cIfList.end(); it++) {
	Report::Debug(String("  ") + (*it).Dump());
      }
      Report::Debug(String("Config parameters:\n   ") + cConfig.Dump());
      Report::Debug(String("External routes:\n   "));
      list<ExtRoute>::iterator ii;
      for(ii = cExtRouteList.begin(); ii != cExtRouteList.end(); ii++) {
	String s = "   ";
	s += DumpAddr((*ii).Addr()) + " "; 
	s += DumpAddr((*ii).Netmask()) + " ";
	s +=  String((int)(*ii).Metric()) + " \n";
	Report::Debug(s);
      }
    }
  }

  ~Router() {
    if(Debug("~Router")) {
      Report::Debug("~Router!!!");
    }

    // Delete our Unix sockets
    delete(cConfigSocket);
    cConfigSocket = 0;
    delete(cImportSocket);
    cImportSocket = 0;
    delete(cExportSocket);
    cExportSocket = 0;
    delete(cDumpSocket);
    cDumpSocket = 0;

    // Close all Udp sockets by deleting 'em
    for(int i=0; i < MAX_INTERFACES; i++) {
      if(cSockets[i]) {
	delete(cSockets[i]);
	cSockets[i] = 0;
      }
    }
    // Remove any changes we made to the kernel routing table
    RouteSet::iterator it;
    for(it = cRoutes.begin(); it != cRoutes.end(); it++) {
      KernelDelete((*it));
    }
    cRoutes.clear();
  }

  /* Main event loop for the Router...
   *
   */
  void Run() {

#if 0
    MockPopulateLspMap();
    UpdateRoutingTable();
    cerr << "\nEnter a character<return> to quit. "; 
    cin >> c;
    return;

#endif
    Time now = Time::Now();
    Time updateTime = now + cConfig.UpdatePeriod();  
    Time pruneTime = now + cPrunePeriod + Time(0.5);  // prune when we are not busy...


    while(1) {
      Time txTime = EarliestTxTime();
      
      Time earliest = Min(updateTime,Min(pruneTime,txTime));
      Time timeleft = earliest - Time::Now();


      fd_set fds = cFds;      
      timeval tv = timeval(timeleft);
      int n;

      if(Debug("Run")) {
	Report::Debug(String("Run() - timeleft=") + timeleft.Str());
      }

      if((n=select(cFdLimit, &fds, 0, 0, &tv)) == -1) {
	char buf[1024];
	sprintf(buf,"Router::Run() - select failed: %s",strerror(errno));
	Report::Error(buf);
      }

      if(n > 0) {
	int fd;
	for(fd=0; fd < cFdLimit; fd++) {
	  if(FD_ISSET(fd,&fds)) {
	    if(cSockets[fd]) {
	      // Handle network receive
	      UdpDatagram dg = cSockets[fd]->Recv();
	      if(Debug("Run")) {
		Report::Debug(String("Run() - From ") + String(dg.GetAddr()) + 
			      " on " + cInterfaces[fd]->Name());
	      }
	      HandleReceive(dg,cInterfaces[fd]);
	    }
	    else if(cReportSockets[fd]) {
	      // Handle link report
	      UnixDatagram dg = cReportSockets[fd]->Recv();
	      if(Debug("Run")) {
		Report::Debug(String("Run() - From ") + String(dg.GetAddr()));
	      }
	      HandleReport(dg,cReportInterfaces[fd],cReportReplies[fd]);
	    }
	  }
	}
	if(FD_ISSET(cConfigSocket->Fd(),&fds)) {
	  UnixDatagram dg = cConfigSocket->Recv();
	  HandleConfig(dg);
	}
	if(FD_ISSET(cImportSocket->Fd(),&fds)) {
	  UnixDatagram dg = cImportSocket->Recv();
	  HandleImport(dg);
	}
	if(FD_ISSET(cExportSocket->Fd(),&fds)) {
	  UnixDatagram dg = cExportSocket->Recv();
	  HandleExport(dg);
	}
	if(FD_ISSET(cDumpSocket->Fd(),&fds)) {
	  UnixDatagram dg = cDumpSocket->Recv();
	  HandleDatabaseDump(dg);
	}
      }

      Time now = Time::Now();
      if(double(earliest) <= double(now)) {
	// Time to do something!

	// Should we prune oldies?
	if(double(pruneTime) <= double(now)) {
	  pruneTime = pruneTime + cPrunePeriod;
	  PruneOldLsps(cPrunePeriod);
	  CheckLinkDiscovers(); // Covenient place to do this
	  Debug::Update();  // Convenient place to check if debug flags have changed
	}


	// Should we build our lsp, send it, and update our routing table?
	if(double(updateTime) <= double(now)) {
	  updateTime = updateTime + cConfig.UpdatePeriod();
	  Lsp lsp = BuildLocalLsp();
	  cLspMap[cId] = lsp;
	  BroadcastLsp(lsp);
	  SaveSequenceNumber();
	  UpdateRoutingTable();
	  DatabaseDump();
	}

	// Should we transmit someone else's Lsp?
	if(double(txTime) <= double(now)) {
	  // Iterate over the tx time map...send any that need to go
	  TimeMap::iterator it;
	  for(it = cTxTimeMap.begin(); it != cTxTimeMap.end();) {
	    if(double((*it).second) < double(now)) {
	      // Make sure we have this Lsp!!
	      unsigned int routerId = (*it).first;
	      LspMap::const_iterator j = cLspMap.find(routerId);
	      if(j != cLspMap.end()) {
		BroadcastLsp((*j).second);
	      }
	      else {
		Report::Error("Router::Run() - tx time map and lsp map are out of sync!");
	      }
	      // Remove this entry from the tx time map
	      // NOTE: the trick to not invalidate the iterator is to post-increment
	      cTxTimeMap.erase(it++);
	    }
	    else {
	      it++;
	    }
	  }
	}
      }
    }
  }
      
  
};

#endif
