/*
 * linux/arch/arm/mach-omap/generic.c
 *
 * Code common to all OMAP machines.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>

#include <asm/hardware.h>
#include <asm/system.h>
#include <asm/pgtable.h>
#include <asm/mach/map.h>
#include <asm/arch/ck.h>
#include <asm/io.h>

#include "generic.h"

int
cpu_type(void)
{
	int processor_type;
	processor_type = inl(OMAP1510_ID_CODE_REG);
	if (processor_type == 0) {
		return OMAP1509;
	} else {
		return OMAP1510;
	}
}

int
omap1510_dsp_idle(void)
{
	unsigned long tmp;
	u32 boot_vector;
	u8 c55_start[] =
	{
		0xfb, 0x51, 0x00, 0xf5,  0x34, 0x04,		/* MOV #245, port(#3404h) */
		0xfb, 0x51, 0x00, 0xa0,  0x34, 0x04,		/* MOV #160, port(#3404h) */
                0xe6, 0x00, 0x00, 0x98,                         /* MOV #0,mmap(IER0)      */
                0xe6, 0x8a, 0x00, 0x98,                         /* MOV #0,mmap(IER1)      */
                0x46, 0xb3,                                     /* BSET ST1_INTM */
                0xfb, 0x02, 0xff, 0xff, 0x98,                   /* MOV #0xffff, mmap(IFR0) */
                0xfb, 0x8c, 0xff, 0xff, 0x98,                   /* MOV #0xffff, mmap(IFR1) */
                0xe6, 0x51, 0x00, 0x40, 0x04,                   /* MOV #0, port(#4004h)    */
		0x46, 0x86, 					/* BCLR ST3_HOMP       */
		0x46, 0x96,					/* BCLR ST3_HOMR       */
		0xfb, 0x51, 0x00, 0xff, 0x00, 0x01,		/* MOV #255, port(#0001h) */
		0x46, 0x87,					/* BSET ST3_HOMP       */
		0x46, 0x97,					/* BSET ST3_HOMR       */
		0x7a, 0x00, 0x00, 0x0c,				/* loop: IDLE          */
		0x4a, 0x7a,					/* b loop	       */
		0x20, 0x20, 0x20, 0x20				/* NOP, NOP, NOP, NOP  */
	};

	tmp = inw(0xfffece10);  /* read ARM_RSTCT1 register */
	tmp &= ~(0x06);		// put the DSP into reset
	outw(tmp, 0xfffece10);

	if (0 == (tmp & 0x4)) {   // MPUI in reset
		tmp |= 0x4;
		outw(tmp, 0xfffece10);
		ck_enable(api_ck);
	}
	tmp = inw(0xe1008008);  /* read DSP_IDLECT2 register */
	if (0 == (tmp & 0x2)) {  // DSP CLKM enable
		tmp |= 0x2;
		outw(tmp, 0xe1008008);
	}
	tmp = inw(0xe1008014);  /* read DSP_RSTCT2 register */
	if (0 == (tmp & 0x1)) {  // DSP PER_EN bit
		tmp |= 0x1;
		outw(tmp, 0xe1008014);
	}
	tmp = inw(0xfffece00);  /* read ARM_CKCTL register */
	tmp |= 0x2000;
	outw(tmp, 0xfffece00);
	// Write C55 code at reset vector.
	if (cpu_type() == OMAP1509) {
		boot_vector = 0x4c000;
	}
	else {
		boot_vector = 0x10000;
	}
	memcpy((void *)(OMAP1510_DSP_BASE + boot_vector),
	       &c55_start,
	       sizeof(c55_start));
	outw(0x5, 0xfffec918);   // Set DSP boot mode
	tmp = inw(0xfffece10);   // take DSP out of reset
	tmp |= 0x6;
	outw(tmp, 0xfffece10);
}

/*
 * Common OMAP I/O mapping
 *
 * The machine specific code may provide the extra mapping besides the
 * default mapping provided here.
 */

#define MAP_DESC(base,start,size,domain,r,w,c,b) { base,start,size,domain,r,w,c,b }
static struct map_desc standard_io_desc[] __initdata = {
	MAP_DESC(IO_BASE,
		 IO_START,
		 IO_SIZE,
		 DOMAIN_IO,
		 0,1,0,0),
	MAP_DESC(OMAP1510_DSP_BASE,
		 OMAP1510_DSP_START,
		 OMAP1510_DSP_SIZE,
		 DOMAIN_IO,
		 0,1,0,0),
	MAP_DESC(OMAP1510_DSPREG_BASE,
		 OMAP1510_DSPREG_START,
		 OMAP1510_DSPREG_SIZE,
		 DOMAIN_IO,
		 0,1,0,0),
	MAP_DESC(OMAP1510_SRAM_BASE,
		 OMAP1510_SRAM_START,
		 OMAP1510_SRAM_SIZE,
		 DOMAIN_IO,
		 0,1,0,0),
	LAST_DESC
};

void __init omap1510_map_io(void)
{
	iotable_init(standard_io_desc);

	/* REVISIT: Refer to OMAP5910 Errata, Advisory SYS_1: "Timeout Abort
	 * on a Posted Write in the TIPB Bridge".
	 */
	outw(0x0, MPU_PUBLIC_TIPB_CNTL_REG);
	outw(0x0, MPU_PRIVATE_TIPB_CNTL_REG);

	/* Must init clocks early to assure that timer interrupt works
	 */
	init_ck();

	return;
}

EXPORT_SYMBOL(cpu_type);
EXPORT_SYMBOL(omap1510_dsp_idle);
