#ifndef __MAPLE_H
#define __MAPLE_H
/*
 *	$Id: maple.h,v 1.2 2002/01/23 06:56:48 mrbrown Exp $
 *	MAPLE Bus device drivers
 *
 * 	Register definitions are taken from Marcus Comstedt's
 * 	Dreamcast software (See http://mc.pp.se/dc/).
 *	Copyright (C) 2000 Marcus Comstedt 
 */

#include <linux/config.h>
#include <linux/list.h>
#include <linux/wait.h>


/* Maple Bus registers */

#define MAPLE_BASE	0xa05f6c00
#define MAPLE_DMAADDR	(MAPLE_BASE+0x04)
#define MAPLE_RESET2	(MAPLE_BASE+0x10)
#define MAPLE_ENABLE	(MAPLE_BASE+0x14)
#define MAPLE_STATE	(MAPLE_BASE+0x18)
#define MAPLE_SPEED	(MAPLE_BASE+0x80)
#define MAPLE_RESET	(MAPLE_BASE+0x8c)

#define MAPLE_MAGIC	0x6155404f
#define MAPLE_2MBPS	0
#define	MAPLE_TIMEOUT(n) ((n)<<16)


/* Maple Bus command and response codes */

#define MAPLE_RESPONSE_FILEERR -5
#define MAPLE_RESPONSE_AGAIN   -4  /* request should be retransmitted */
#define MAPLE_RESPONSE_BADCMD  -3
#define MAPLE_RESPONSE_BADFUNC -2
#define MAPLE_RESPONSE_NONE    -1  /* unit didn't respond at all */
#define MAPLE_COMMAND_DEVINFO  1
#define MAPLE_COMMAND_ALLINFO  2
#define MAPLE_COMMAND_RESET    3
#define MAPLE_COMMAND_KILL     4
#define MAPLE_RESPONSE_DEVINFO 5
#define MAPLE_RESPONSE_ALLINFO 6
#define MAPLE_RESPONSE_OK      7
#define MAPLE_RESPONSE_DATATRF 8
#define MAPLE_COMMAND_GETCOND  9
#define MAPLE_COMMAND_GETMINFO 10
#define MAPLE_COMMAND_BREAD    11
#define MAPLE_COMMAND_BWRITE   12
#define MAPLE_COMMAND_SETCOND  14


/* Function codes */

#define MAPLE_FUNC_CONTROLLER 0x001
#define MAPLE_FUNC_MEMCARD    0x002
#define MAPLE_FUNC_LCD        0x004
#define MAPLE_FUNC_CLOCK      0x008
#define MAPLE_FUNC_MICROPHONE 0x010
#define MAPLE_FUNC_ARGUN      0x020
#define MAPLE_FUNC_KEYBOARD   0x040
#define MAPLE_FUNC_LIGHTGUN   0x080
#define MAPLE_FUNC_PURUPURU   0x100
#define MAPLE_FUNC_MOUSE      0x200

struct maple_devinfo {
	unsigned long function;	/* big endian! */
	unsigned long function_data[3];	/* big endian! */
	unsigned char area_code;
	unsigned char connector_direction;
	char product_name[30];
	char product_license[60];
	unsigned short standby_power; /* little endian */
	unsigned short max_power; /* little endian */
};


struct maple_driver;
struct maple_device;


#define MAPLEQ_LENGTH_GUN  (1<<9)

struct mapleq {
	struct list_head list;
	void *mq_privdata;
	void *sendbuf;
	char *recvbuf; /* points to ->buf, but 32-byte aligned */

	unsigned char port, unit;
	unsigned char command;
	unsigned char done;
	int length;

	void (*callback)(struct mapleq *mq);
	unsigned char buf[1024+32];
};


struct maple_device {
	unsigned char port;
	unsigned char unit;
	short lock;

	unsigned long function; /* Little endian */
	struct maple_devinfo devinfo;

	char product_name[32];
	char product_license[64];
};

/* Max devices per port */
#define MAPLE_MAX_UNITS 6

struct maple_port {
	unsigned char port;
	unsigned char known_units; /* connected units */
	unsigned char units; /* units to connect/disconnect */
	void (*gunmode)(void *,int,int);
	void *gunmode_data;
	struct maple_device *dev[MAPLE_MAX_UNITS];
};

struct maple_driver_data {
	struct list_head list;
	struct maple_driver *driver;
	struct maple_device *dev;
	void *private_data;
	unsigned long function_data;
	struct mapleq mq;
};

struct maple_driver {
	struct list_head list;
	unsigned long function; /* One or more bits */
	const char *name;

	void (*vblank)(struct maple_driver_data *dev);
	void (*reply)(struct maple_driver_data *dev);
	int (*connect)(struct maple_driver_data *dev);
	void (*disconnect)(struct maple_driver_data *dev);
};


/* Prototypes begin here */
void maple_send(void);
int maple_add_packet(struct mapleq *mq);
int maple_del_packet(struct mapleq *mq);
void maple_register_driver(struct maple_driver *driver);
void maple_unregister_driver(struct maple_driver *driver);
void maple_getcond_vblank_callback(struct maple_driver_data *data);
void maple_set_gunmode(int, void (*)(void *,int,int), void *);
void maple_init_mq(struct mapleq *mq);
/* Prototypes end here */


/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
#endif
