/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "slide-list-proxy.h"

#include <inttypes.h>

#include <glib.h>
#include <glib-object.h>

#include "debug.h"
#include "presentation-view.h"

enum {
	PROP_0,
};

enum {
	N_SIGNALS
};

enum {
	COLUMN_PREVIEW,
	COLUMN_TITLE,
	COLUMN_SLIDE_PTR,
	N_COLUMNS
};

static GType *column_types = NULL;

struct _CriaSlideListProxyPrivate {
	CriaPresentation	* presentation;
};

static void     cria_slide_list_proxy_get_property     (GObject			* object,
							guint			  prop_id,
							GValue			* value,
							GParamSpec		* param_spec);
static void     cria_slide_list_proxy_init	       (CriaSlideListProxy	* self);
static void     cria_slide_list_proxy_init_tree_model_iface
						       (gpointer		  iface,
							gpointer		  data);
static gboolean cria_slide_list_proxy_iter_has_child   (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
static gboolean cria_slide_list_proxy_iter_next	       (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
static gboolean cria_slide_list_proxy_iter_nth_child   (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter,
							GtkTreeIter		* parent,
							gint			  n);
static void     cria_slide_list_proxy_set_property     (GObject			* object,
							guint			  prop_id,
							const GValue		* value,
							GParamSpec		* param_spec);
#if 0
/* enable these to add support for signals */
static	guint	cria_slide_list_proxy_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_list_proxy_signal	       (CriaSlideListProxy	* self,
						const	gchar	* string);
#endif

static void
cria_slide_list_proxy_class_init(CriaSlideListProxyClass* cria_slide_list_proxy_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_list_proxy_class);
#if 0
	/* setting up signal system */
	cria_slide_list_proxy_class->signal = cria_slide_list_proxy_signal;

	cria_slide_list_proxy_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE_LIST_PROXY,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideListProxyClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_list_proxy_set_property;
	g_object_class->get_property = cria_slide_list_proxy_get_property;

	/* setting up the PresentationView interface */
	_cria_presentation_view_install_properties(g_object_class);

	/* setting up the slide list proxy class */
#warning "SlideListProxy::classInit(): FIXME: initialize this a bit more dynamically"
	column_types = g_new0(GType, N_COLUMNS);
	column_types[0] = GDK_TYPE_PIXBUF;
	column_types[1] = G_TYPE_STRING;
	column_types[2] = CRIA_TYPE_SLIDE;
}

/**
 * cria_slide_list_proxy_for_presentation:
 * @presentation: a #CriaPresentation
 *
 * Get a #CriaSlideListProxy for @presentation.
 *
 * Returns a #CriaSlideListProxy that provides a #GtkTreeModel for a #CriaPresentation.
 */
CriaSlideListProxy*
cria_slide_list_proxy_for_presentation(CriaPresentation* presentation) {
#warning "SlideListProxy::forPresentation(): FIXME: build a hash tree and do not always create a new proxy"
	return g_object_new(CRIA_TYPE_SLIDE_LIST_PROXY, "presentation", presentation, NULL);
}

static GType
cria_slide_lis_proxy_get_column_type(GtkTreeModel* tree_model, gint index) {
	g_assert(0 <= index && index < N_COLUMNS);

	return column_types[index];
}

static gboolean
cria_slide_list_proxy_get_iter(GtkTreeModel* tree_model, GtkTreeIter* iter, GtkTreePath* path) {
	gint		    i;
	CriaSlideListProxy* self;
	
	g_return_val_if_fail(CRIA_IS_SLIDE_LIST_PROXY(tree_model), FALSE);
	g_return_val_if_fail(gtk_tree_path_get_depth(path) > 0, FALSE);

	self = (CriaSlideListProxy*)tree_model;

	i = gtk_tree_path_get_indices(path)[0];

	if(i >= cria_presentation_n_slides(self->priv->presentation)) {
		return FALSE;
	}
	
	iter->user_data = cria_presentation_get_slide(self->priv->presentation, i);

	if(iter->user_data) {
		return TRUE;
	}

	return FALSE;
}

static gint
cria_slide_list_proxy_get_n_columns(GtkTreeModel *tree_model) {
	g_return_val_if_fail(CRIA_IS_SLIDE_LIST_PROXY(tree_model), 0);

	return N_COLUMNS;
}

CriaPresentation*
cria_slide_list_proxy_get_presentation(CriaSlideListProxy* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE_LIST_PROXY(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_slide_list_proxy_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideListProxy	* self;

	self = CRIA_SLIDE_LIST_PROXY(object);

	switch(prop_id) {
	case CRIA_PRESENTATION_VIEW_PROP_PRESENTATION:
		g_value_set_object(value, self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

/**
 * cria_slide_list_proxy_get_slide:
 * @self: a #CriaSlideListProxy
 * @iter: a #GtkTreeIter
 *
 * Get the slide represented by an Iterator.
 *
 * Returns the #CriaSlide that's being referred to by the iterator.
 */
CriaSlide*
cria_slide_list_proxy_get_slide(CriaSlideListProxy* self, GtkTreeIter* iter) {
	return iter->user_data;
}

/**
 * cria_slide_list_proxy_get_title_column:
 * 
 * Get the column that should be used by a #GtkTreeView to get the title of a
 * slide.
 *
 * Returns the title column index.
 */
gint
cria_slide_list_proxy_get_title_column(void) {
	return COLUMN_TITLE;
}

GType
cria_slide_list_proxy_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof(CriaSlideListProxyClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_list_proxy_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof(CriaSlideListProxy),
			0,
			(GInstanceInitFunc)cria_slide_list_proxy_init,
			0
		};

		static const GInterfaceInfo slide_view_info = {
			NULL, /* init */
			NULL, /* finalize */
			NULL  /* data */
		};

		static const GInterfaceInfo tree_model_info = {
			cria_slide_list_proxy_init_tree_model_iface, /* init */
			NULL, /* finalize */
			NULL  /* data */
		};

		type = g_type_register_static(G_TYPE_OBJECT,
					      "CriaSlideListProxy",
					      &info,
					      0);
		g_type_add_interface_static(type,
					    CRIA_TYPE_PRESENTATION_VIEW,
					    &slide_view_info);
		g_type_add_interface_static(type,
					    GTK_TYPE_TREE_MODEL,
					    &tree_model_info);
	}

	return type;
}

static void
cria_slide_list_proxy_get_value(GtkTreeModel* tree_model, GtkTreeIter* iter, gint column, GValue* value) {
	g_return_if_fail(0 <= column && column < N_COLUMNS);

	g_value_init(value, column_types[column]);

	switch(column) {
	case COLUMN_PREVIEW:
#warning "SlideListProxy::getValue(): add support for preview pixmaps"
		g_value_set_object(value, NULL);
		break;
	case COLUMN_TITLE:
		g_value_set_string(value, cria_slide_get_title(CRIA_SLIDE(iter->user_data)));
		break;
	case COLUMN_SLIDE_PTR:
		g_value_set_object(value, CRIA_SLIDE(iter->user_data));
		break;
	default:
		g_debug("SlideListProxy::getValue(): unknown column '%i'", column);
		g_assert_not_reached();
	}
}

static void
cria_slide_list_proxy_init(CriaSlideListProxy* self) {
	g_return_if_fail(CRIA_IS_SLIDE_LIST_PROXY(self));

	self->priv = g_new0(CriaSlideListProxyPrivate,1);
}

static void
cria_slide_list_proxy_init_tree_model_iface(gpointer iface, gpointer data) {
	GtkTreeModelIface* tree_model_class = iface;
	g_assert(G_TYPE_FROM_INTERFACE(tree_model_class) == GTK_TYPE_TREE_MODEL);

	tree_model_class->get_column_type = cria_slide_lis_proxy_get_column_type;
	tree_model_class->get_iter        = cria_slide_list_proxy_get_iter;
	tree_model_class->get_n_columns   = cria_slide_list_proxy_get_n_columns;
	tree_model_class->get_value       = cria_slide_list_proxy_get_value;
	tree_model_class->iter_has_child  = cria_slide_list_proxy_iter_has_child;
	tree_model_class->iter_next       = cria_slide_list_proxy_iter_next;
	tree_model_class->iter_nth_child  = cria_slide_list_proxy_iter_nth_child;
}

static gboolean
cria_slide_list_proxy_iter_has_child(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	return FALSE;
}

static gboolean
cria_slide_list_proxy_iter_next(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	gint             n;
	gboolean         retval = FALSE;
	CriaPresentation*presentation;
	CriaSlide       *slide;

	g_return_val_if_fail(CRIA_IS_SLIDE_LIST_PROXY(tree_model), retval);
	
	presentation = CRIA_SLIDE_LIST_PROXY(tree_model)->priv->presentation;
	slide        = CRIA_SLIDE(iter->user_data);
	
	n = cria_presentation_get_slide_index(presentation, slide);

	if((n + 1) < cria_presentation_n_slides(presentation)) {
		iter->user_data = cria_presentation_get_slide(presentation, n + 1);
		retval = TRUE;
	}

	return retval;
}

static gboolean
cria_slide_list_proxy_iter_nth_child(GtkTreeModel* tree_model, GtkTreeIter* iter, GtkTreeIter* parent, gint n) {
	gboolean retval = FALSE;
	g_return_val_if_fail(CRIA_IS_SLIDE_LIST_PROXY(tree_model), retval);
	g_debug("SlideListProxy::iterNthChild(): iter(0x%x), parent(0x%x), n(%i)", (uintptr_t)iter, (uintptr_t)parent, n);

	if(parent == NULL) {
		GtkTreePath* path = gtk_tree_path_new();
		gtk_tree_path_append_index(path, n);
		retval = cria_slide_list_proxy_get_iter(tree_model, iter, path);
		gtk_tree_path_free(path);
	} else {
		g_assert_not_reached();
	}
	
	return retval;
}

static void
cb_presentation_inserted_slide(CriaSlideListProxy* self, gint new_position, CriaSlide* slide, CriaPresentation* presentation) {
	GtkTreeIter	  iter;
	GtkTreePath	* path;
	
	g_return_if_fail(CRIA_IS_SLIDE_LIST_PROXY(self));
	
	path = gtk_tree_path_new();
	gtk_tree_path_append_index(path, new_position);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(self), &iter, path);
	gtk_tree_model_row_inserted(GTK_TREE_MODEL(self), path, &iter);
	gtk_tree_path_free(path);
}

static void
cria_slide_list_proxy_set_presentation(CriaSlideListProxy* self, CriaPresentation* presentation) {
	g_return_if_fail(CRIA_IS_SLIDE_LIST_PROXY(self));
	g_return_if_fail(CRIA_IS_PRESENTATION(presentation));

	if(self->priv->presentation != NULL) {
		g_free(self->priv->presentation);
	}

	self->priv->presentation = g_object_ref(presentation);
	g_signal_connect_swapped(self->priv->presentation, "inserted-slide",
				 G_CALLBACK(cb_presentation_inserted_slide), self);

	g_object_notify(G_OBJECT(self), "presentation");
}

static void
cria_slide_list_proxy_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlideListProxy	* self;
	
	self = CRIA_SLIDE_LIST_PROXY(object);
	
	switch(prop_id) {
	case CRIA_PRESENTATION_VIEW_PROP_PRESENTATION:
		cria_slide_list_proxy_set_presentation(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

