/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "display.h"

#include <glib.h>
#include <glib-object.h>

#include <gnome-canvas-helpers.h>
#include "debug.h"
/* this is for SLIDE_EDITOR_PADDING, etc */
#include "slide-editor.h"

enum {
	PROP_0,
	PROP_PADDING,
	PROP_ZOOM
};

enum {
	N_SIGNALS
};

struct _CriaDisplayPrivate {
	GoPoint			* display,
				* slide;
	GnomeCanvasGroup	* top_group;
	gdouble			  zoom;
	go_unit_t		  padding;
};

static	void	cria_display_get_property      (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_display_init		       (CriaDisplay	* self);
static	void	cria_display_set_property      (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
static void cria_display_update_zoom	       (CriaDisplay     * self);
static void cria_display_update_extents	       (CriaDisplay     * self);
#if 0
/* enable these to add support for signals */
static	guint	cria_display_signals[N_SIGNALS] = { 0 };

static	void	cria_display_signal	       (CriaDisplay	* self,
						const	gchar	* string);
#endif

static void
cria_display_class_init (CriaDisplayClass	* cria_display_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_display_class);
#if 0
	/* setting up signal system */
	cria_display_class->signal = cria_display_signal;

	cria_display_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_DISPLAY,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaDisplayClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_display_set_property;
	g_object_class->get_property = cria_display_get_property;

	g_object_class_install_property(g_object_class,
					PROP_PADDING,
					g_param_spec_uint64("padding",
							    "Padding",
							    "The padding between the slide and the widget edges.",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_ZOOM,
					g_param_spec_double("zoom",
							    "Zoom",
							    "The zoom factor of this display: 1.0 means \"presentation size\"",
							    0.01,
							    G_MAXDOUBLE,
							    0.65,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

go_unit_t
cria_display_get_padding(CriaDisplay* self) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(self->priv != NULL);

	return self->priv->padding;
}

static void
cria_display_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaDisplay	* self;

	self = CRIA_DISPLAY(object);

	switch (prop_id) {
	case PROP_PADDING:
		g_value_set_uint64(value, cria_display_get_padding(self));
		break;
	case PROP_ZOOM:
		g_value_set_double(value, cria_display_get_zoom(self));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

gdouble
cria_display_get_zoom(CriaDisplay* self) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(self->priv != NULL);
	
	return self->priv->zoom;
}

GnomeCanvasGroup*
cria_display_get_top_group(CriaDisplay* self) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->top_group != NULL && GNOME_IS_CANVAS_GROUP(self->priv->top_group));
	
	return self->priv->top_group;
}

GType
cria_display_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof(CriaDisplayClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_display_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof(CriaDisplay),
			0,
			(GInstanceInitFunc)cria_display_init,
			0
		};

		type = g_type_register_static(GNOME_TYPE_CANVAS,
					      "CriaDisplay",
					      &info,
					      0);
	}

	return type;
}

static void
cria_display_init(CriaDisplay* self) {
	self->priv = g_new0(CriaDisplayPrivate,1);

	self->priv->display = g_new0(GoPoint,1);
	self->priv->slide = g_new0(GoPoint,1);
	self->priv->top_group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(self)),
									 gnome_canvas_group_get_type(),
									 "x", 0.0,
									 "y", 0.0,
									 NULL));
	self->priv->zoom = 0.01;
}

CriaDisplay*
cria_display_new(void) {
	CriaDisplay * self = g_object_new(CRIA_TYPE_DISPLAY, "aa", TRUE, NULL);
	return self;
}

void
cria_display_set_padding(CriaDisplay* self, go_unit_t padding) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(self->priv != NULL);

	if(padding != self->priv->padding) {
		self->priv->padding = padding;

		cria_display_update_extents(self);
		
		g_object_notify(G_OBJECT(self), "padding");
	}
}

static void
cria_display_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaDisplay	* self;
	
	self = CRIA_DISPLAY (object);
	
	switch(prop_id) {
	case PROP_PADDING:
		cria_display_set_padding(self, g_value_get_uint64(value));
		break;
	case PROP_ZOOM:
		cria_display_set_zoom(self, g_value_get_double(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

void
cria_display_set_sizes(CriaDisplay* self, const GoPoint* slide, const GoPoint* display) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->slide != NULL && self->priv->display != NULL);

	g_debug("Display::setSizes(): Slide (%lli; %lli)", slide->x, slide->y);
	g_debug("Display::setSizes(): Display (%lli; %lli)", display->x, display->y);

	self->priv->slide->x = slide->x;
	self->priv->slide->y = slide->y;
	self->priv->display->x = display->x;
	self->priv->display->y = display->y;

	cria_display_update_extents(self);

	cria_display_update_zoom(self);
}

void
cria_display_set_zoom(CriaDisplay* self, gdouble zoom) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(self->priv != NULL);

	if(zoom != self->priv->zoom) {
		self->priv->zoom = zoom;

		cria_display_update_zoom(self);

		g_object_notify(G_OBJECT(self), "zoom");
	}
}

static void
cria_display_update_extents(CriaDisplay* self) {
	gnome_canvas_set_scroll_region(GNOME_CANVAS(self),
				       0.0 - self->priv->padding,
				       0.0 - self->priv->padding,
				       0.0 + self->priv->slide->x + SLIDE_EDITOR_SHADOW_OFFSET + self->priv->padding,
				       0.0 + self->priv->slide->y + SLIDE_EDITOR_SHADOW_OFFSET + self->priv->padding);
	gnome_canvas_set_center_scroll_region(GNOME_CANVAS(self), TRUE);
}

static void
cria_display_update_zoom(CriaDisplay* self) {
	g_assert(self != NULL && CRIA_IS_DISPLAY(self));
	g_assert(GNOME_IS_CANVAS(self));
	g_assert(self->priv != NULL);

	g_debug("Display::updateZoom(): self: %f; display: %lli; slide: %lli", self->priv->zoom, self->priv->display->x, self->priv->slide->x);

	if(self->priv->slide->x != 0) {
		gnome_canvas_set_zoom(GNOME_CANVAS(self), self->priv->zoom * self->priv->display->x / self->priv->slide->x);
	}
}

