/****************************************************************************
**
** Copyright (C) 1992-2004 Trolltech AS. All rights reserved.
**
** This file is part of the network module of the Qt Toolkit.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QABSTRACTSOCKET_H
#define QABSTRACTSOCKET_H

namespace Qt {
    enum SocketType {
        TcpSocket,
        UdpSocket,
        UnknownSocketType = -1
    };
    enum NetworkLayerProtocol {
        IPv4Protocol,
        IPv6Protocol,
        UnknownNetworkLayerProtocol = -1
    };
    enum SocketError {
        ConnectionRefusedError,
        RemoteHostClosedError,
        HostNotFoundError,
        SocketAccessError,
        SocketResourceError,
        SocketTimeoutError,
        DatagramTooLargeError,
        NetworkError,
        AddressInUseError,
        SocketAddressNotAvailableError,
        UnsupportedSocketOperationError,
        UnknownSocketError = -1
    };
    enum SocketState {
        UnconnectedState,
        HostLookupState,
        ConnectingState,
        ConnectedState,
        BoundState,
        ListeningState,
        ClosingState
    };
}

#include <qiodevice.h>
#include <qobject.h>

class QHostAddress;
class QAbstractSocketPrivate;

class Q_NETWORK_EXPORT QAbstractSocket : public QIODevice
{
    Q_OBJECT
public:
    QAbstractSocket(Qt::SocketType socketType, QObject *parent);
    virtual ~QAbstractSocket();

    void connectToHost(const QString &hostName, Q_UINT16 port, OpenMode mode = ReadWrite);
    void connectToHost(const QHostAddress &address, Q_UINT16 port, OpenMode mode = ReadWrite);

    bool isValid() const;

    Q_LONGLONG bytesAvailable() const;
    Q_LONGLONG bytesToWrite() const;

    bool canReadLine() const;

    Q_UINT16 localPort() const;
    QHostAddress localAddress() const;
    Q_UINT16 peerPort() const;
    QHostAddress peerAddress() const;
    QString peerName() const;

    Q_LONGLONG readBufferSize() const;
    void setReadBufferSize(Q_LONGLONG size);

    void abort();

    int socketDescriptor() const;
    bool setSocketDescriptor(int socketDescriptor, Qt::SocketState state = Qt::ConnectedState,
                             OpenMode openMode = ReadWrite);

    Qt::SocketType socketType() const;
    Qt::SocketState socketState() const;
    Qt::SocketError socketError() const;

    // from QIODevice
    void close();
    bool flush();
    bool isSequential() const;

    // for synchronous access
    bool waitForConnected(int msecs = 30000);
    bool waitForReadyRead(int msecs = 30000);
    bool waitForBytesWritten(int msecs = 30000);
    bool waitForClosed(int msecs = 30000);

signals:
    void hostFound();
    void connected();
    void closing();
    void closed();
    void stateChanged(int);
    void error(int);

protected:
    Q_LONGLONG readData(char *data, Q_LONGLONG maxlen);
    Q_LONGLONG writeData(const char *data, Q_LONGLONG len);

    void setSocketState(Qt::SocketState state);
    void setSocketError(Qt::SocketError socketError);

protected:
    QAbstractSocket(Qt::SocketType socketType, QAbstractSocketPrivate &p, QObject *parent);

private:
    Q_DECLARE_PRIVATE(QAbstractSocket)
    Q_DISABLE_COPY(QAbstractSocket)

    Q_PRIVATE_SLOT(d, void connectToNextAddress())
    Q_PRIVATE_SLOT(d, void startConnecting(const QDnsHostInfo &))
    Q_PRIVATE_SLOT(d, void abortConnectionAttempt())
    Q_PRIVATE_SLOT(d, void testConnection())
    Q_PRIVATE_SLOT(d, void canReadNotification(int))
    Q_PRIVATE_SLOT(d, void canWriteNotification(int))

#ifdef QT_COMPAT
public:
    enum Error {
        ErrConnectionRefused = Qt::ConnectionRefusedError,
        ErrHostNotFound = Qt::HostNotFoundError,
        ErrSocketRead = Qt::UnknownSocketError
    };
    enum State {
        Idle = Qt::UnconnectedState,
        HostLookup = Qt::HostLookupState,
        Connecting = Qt::ConnectingState,
        Connected = Qt::ConnectedState,
        Closing = Qt::ClosingState,
        Connection = Qt::ConnectedState
    };
    inline QT_COMPAT State state() const { return State(socketState()); }
    inline QT_COMPAT int socket() const { return socketDescriptor(); }
    inline QT_COMPAT void setSocket(int socket) { setSocketDescriptor(socket); }
    inline QT_COMPAT Q_ULONG waitForMore(int msecs, bool *timeout = 0) const
    {
        QAbstractSocket *that = const_cast<QAbstractSocket *>(this);
        if (that->waitForReadyRead(msecs))
            return Q_ULONG(bytesAvailable());
        if (socketError() == Qt::SocketTimeoutError && timeout)
            *timeout = true;
        return 0;
    }
signals:
    QT_MOC_COMPAT void connectionClosed(); // same as closing()
    QT_MOC_COMPAT void delayedCloseFinished(); // same as closed()
#endif


};

#endif
