/*
 * Philips UDA1380 mixer device driver
 *
 * Copyright (c) 2002 Hewlett-Packard Company
 *
 * Copyright (c) 2000 Nicolas Pitre <nico@cam.org>
 *
 * Portions are Copyright (C) 2000 Lernout & Hauspie Speech Products, N.V.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * History:
 *
 * 2000-05-21	Nicolas Pitre	Initial release. of UDA1341.c
 *
 * 2000-08-19	Erik Bunce	More inline w/ OSS API and UDA1341 docs
 * 				including fixed AGC and audio source handling
 *
 * 2000-11-30	Nicolas Pitre	- More mixer functionalities.
 *
 * 2001-06-03	Nicolas Pitre	Made this file a separate module, based on
 * 				the former sa1100-uda1341.c driver.
 *
 * 2001-08-13	Russell King	Re-written as part of the L3 interface
 *
 * 2002-12-18   Jamey Hicks     Rewrote for UDA1380 support.  Uses I2C instead of L3.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/ioctl.h>
#include <linux/soundcard.h>
#include <linux/i2c.h>
#include "uda1380.h"

#include <asm/uaccess.h>

#define DEF_VOLUME	65

/*
 * UDA1380 I2C address and command types
 */
#define UDA1380_I2CADDR		1
#define UDA1380_DATA0		(UDA1380_I2CADDR << 2 | 0)
#if !defined(CONFIG_UDA1344) && !defined(CONFIG_UDA1345)
#define UDA1380_DATA1		(UDA1380_I2CADDR << 2 | 1)
#endif
#define UDA1380_STATUS		(UDA1380_I2CADDR << 2 | 2)

#define REG_MAX 0x24

#define REG0				0x00

#define REG0_EN_ADC			(1 << 11)
#define REG0_EN_DEC			(1 << 10)
#define REG0_EN_DAC			(1 <<  9)
#define REG0_EN_INT			(1 <<  8)
#define REG0_ADC_USE_WSPLL		(1 <<  5)
#define REG0_ADC_USE_SYSCLK		(0 <<  5)
#define REG0_DAC_USE_WSPLL		(1 <<  4)
#define REG0_DAC_USE_SYSCLK		(0 <<  4)
#define REG0_SC_MASK			(3 << 2)
#define REG0_SC_768FS			(3 << 2)
#define REG0_SC_512FS			(2 << 2)
#define REG0_SC_384FS			(1 << 2)
#define REG0_SC_256FS			(0 << 2)
#define REG0_PLL_MASK			(3 << 0)
#define REG0_PLL_6TO12			(0 << 0)
#define REG0_PLL_12TO25			(1 << 0)
#define REG0_PLL_25TO50			(2 << 0)
#define REG0_PLL_50TO100		(3 << 0)

#define I2S_REG				0x01

#define I2S_REG_SFORO_MASK		(7 << 0)
#define I2S_REG_SFORO_SHIFT		0
#define I2S_REG_SFORI_MASK		(7 << 8)
#define I2S_REG_SFORI_SHIFT		8
#define I2S_REG_SEL_DECIMATOR		(1 << 6)
#define I2S_REG_SEL_MIXER		(1 << 6)
#define I2S_REG_SIM_SLAVE		(0 << 4)
#define I2S_REG_SIM_MASTER		(1 << 4)

#define PWR_REG				0x02

#define PWR_REG_PON_PLL			(1 << 15)
#define PWR_REG_PON_HP			(1 << 13)
#define PWR_REG_PON_DAC			(1 << 10)
#define PWR_REG_PON_BIAS		(1 << 8)
#define PWR_REG_EN_AVC			(1 << 7)
#define PWR_REG_PON_AVC			(1 << 6)
#define PWR_REG_PON_LNA			(1 << 4)
#define PWR_REG_PON_PGAL		(1 << 3)
#define PWR_REG_PON_ADCL		(1 << 2)
#define PWR_REG_PON_PGAR		(1 << 1)
#define PWR_REG_PON_ADCR		(1 << 0)

/* analog mixer */
#define AMIX_REG			0x03

/*
 * the analog mixer enables line-input to be played on speaker or
 * headphones without running it through the digital section.
 * We do not use it on the iPAQ.
 *
 * iPAQ h3900:
 *   - line-input is not connected
 *   - built-in microphone connected to mic-in.  No microphone jack.
 *   - line-output mixed together at speaker amplifier.
 *   - headphone amp goes to head-phone jack.  earphone detect is on asic2.
 */

#define AMIX_AVCL(x)	(((x) & 0x1f) << 8)
#define AMIX_AVCL_MASK	(0x1f << 8)
#define AMIX_AVCR(x)	(((x) & 0x1f) << 0)
#define AMIX_AVCR_MASK	(0x1f << 0)

/* digital master volume control register */
#define MASTER_VOL_REG			0x10

#define MASTER_VCL(x)	(((x) & 0xff) << 8)
#define MASTER_VCL_MASK	(0xff << 8)
#define MASTER_VCR(x)	(((x) & 0xff) << 0)
#define MASTER_VCR_MASK	(0xff << 0)

/* digital mixer control register */
#define MIXER_VOL_REG			0x11

#define MIXER_VCL(x)	(((x) & 0xff) << 8)
#define MIXER_VCL_MASK	(0xff << 8)
#define MIXER_VCR(x)	(((x) & 0xff) << 0)
#define MIXER_VCR_MASK	(0xff << 0)

/* mode, bass boost, treble */
#define MBT_REG				0x12
#define MBT_MODE_MASK			(3 << 14)
#define MBT_MODE_FLAT			(0 << 14)
#define MBT_MODE_MIN			(1 << 14)
#define MBT_MODE_MAX			(3 << 14)
#define MBT_TRL_WIDTH			2
#define MBT_TRL_MASK			(3 << 12)
#define MBT_TRL(x)			((x) << 12)
#define MBT_TRR_MASK			(3 << 4)
#define MBT_TRR(x)			((x) << 4)
#define MBT_BBL_WIDTH			4
#define MBT_BBL_MASK			(0xf << 8)
#define MBT_BBL(x)			((x) << 8)
#define MBT_BBR_MASK			(0xf << 0)
#define MBT_BBR(x)			((x) << 0)

/* channel2 is from decimation filter */
/* channel1 is from digital data from I2S */

#define MMCDM_REG			0x13
#define MMCDM_MASTER_MUTE		(1 << 14)
#define MMCDM_CHANNEL2_MUTE		(1 << 11)
#define MMCDM_CHANNEL2_DEEMPHASIS_OFF	(0 << 8)
#define MMCDM_CHANNEL2_DEEMPHASIS_32KHZ	(1 << 8)
#define MMCDM_CHANNEL2_DEEMPHASIS_44KHZ	(2 << 8)
#define MMCDM_CHANNEL2_DEEMPHASIS_48KHZ	(3 << 8)
#define MMCDM_CHANNEL2_DEEMPHASIS_96KHZ	(4 << 8)
#define MMCDM_CHANNEL1_MUTE		(1 << 3)
#define MMCDM_CHANNEL1_DEEMPHASIS_32KHZ	(1 << 8)
#define MMCDM_CHANNEL1_DEEMPHASIS_44KHZ	(2 << 8)
#define MMCDM_CHANNEL1_DEEMPHASIS_48KHZ	(3 << 8)
#define MMCDM_CHANNEL1_DEEMPHASIS_96KHZ	(4 << 8)

#define MIXER_CTL_REG			0x14

#define MIXER_CTL_DAC_POLARITY_INVERT	(1 << 15)
#define MIXER_CTL_SEL_5TH_ORDER_NS	(1 << 14)
#define MIXER_CTL_MIX_POS	        (1 << 13)
#define MIXER_CTL_MIX		        (1 << 12)
#define MIXER_CTL_SILENCE	        (1 << 7)
#define MIXER_CTL_SDET_ON	        (1 << 6)
#define MIXER_CTL_SDET_VALUE_MASK       (3 << 4)
#define MIXER_CTL_SDET_VALUE_SHIFT	4
#define MIXER_CTL_OVERSAMPLING_MASK     (3 << 0)
#define MIXER_CTL_OVERSAMPLING_SHIFT    0

#define DEC_VOL_REG			0x20

#define DEC_VCL(x)	(((x) & 0xff) << 8)
#define DEC_VCL_MASK	(0xff << 8)
#define DEC_VCR(x)	(((x) & 0xff) << 0)
#define DEC_VCR_MASK	(0xff << 0)

#define DPM_REG				0x21

#define DPM_MUTE_ADC	(1 << 15)
#define DPM_GAINL(x)	(((x) & 0xf) << 8)
#define DPM_GAINL_MASK	(0xf << 8)
#define DPM_GAINR(x)	(((x) & 0xf) << 0)
#define DPM_GAINR_MASK	(0xf << 0)

#define DEC_ADC_REG			0x22

#define DEC_ADC_POL_INV			(1 << 12)
#define DEC_ADC_VGA_CTRL(x)		((x) << 8)
#define DEC_ADC_VGA_CTRL_MASK		(0xf << 8)
#define DEC_ADC_SEL_LNA			(1 << 3)
#define DEC_ADC_SEL_MIC			(1 << 2)
#define DEC_ADC_SKIP_DCFIL		(1 << 1)
#define DEC_ADC_EN_DCFIL		(1 << 0)

#define DEC_AGC_REG			0x23

#define DEC_AGC_EN			(1 << 0)


#define REC_MASK	(SOUND_MASK_LINE | SOUND_MASK_MIC)
#define DEV_MASK	(REC_MASK | SOUND_MASK_VOLUME | SOUND_MASK_BASS | SOUND_MASK_TREBLE)

static struct uda1380_reg_info {
	unsigned short num;
	unsigned short default_value;
} uda1380_reg_info[] = {
	{ REG0, (REG0_SC_256FS|REG0_DAC_USE_WSPLL|REG0_ADC_USE_WSPLL|REG0_PLL_25TO50
		 |REG0_EN_ADC|REG0_EN_DEC|REG0_EN_DAC|REG0_EN_INT) },
	{ I2S_REG, (FMT_I2S << I2S_REG_SFORO_SHIFT) | (FMT_I2S << I2S_REG_SFORI_SHIFT) },
	{ PWR_REG, ( PWR_REG_PON_PLL|PWR_REG_PON_HP|PWR_REG_PON_DAC|PWR_REG_PON_BIAS|
		     PWR_REG_PON_LNA | PWR_REG_PON_ADCL) },
	{ AMIX_REG, 0 },
	{ MASTER_VOL_REG, 0 },
	{ MIXER_VOL_REG, 0 },
	{ MBT_REG, 0 },
	{ MMCDM_REG, 0x0000 },
	{ MIXER_CTL_REG, 0 },
	{ DEC_VOL_REG, 0 },
	{ DPM_REG, 0x0000 },
	{ DEC_ADC_REG, DEC_ADC_VGA_CTRL(15) | DEC_ADC_SEL_LNA | DEC_ADC_SEL_MIC },
	{ DEC_AGC_REG, DEC_AGC_EN } 
}; 

struct uda1380 {
	unsigned short regs[REG_MAX];
	int		active;
	unsigned short	volume;
	unsigned short	bass;
	unsigned short	treble;
	unsigned short	line;
	unsigned short	mic;
	int             mic_connected;
	int             line_connected;
	int		mod_cnt;
};

static void uda1380_write_reg(struct i2c_client *clnt, struct uda1380 *uda, int regaddr)
{
	char buffer[3];
	int r;

	buffer[0] = regaddr;
	buffer[1] = uda->regs[regaddr] >> 8;
	buffer[2] = uda->regs[regaddr] & 0xff;

	r = i2c_master_send(clnt, buffer, 3);
	if (r != 3) {
		printk(KERN_ERR "uda1380: write failed, status %d\n", r);
	}
}

static void uda1380_sync(struct i2c_client *clnt)
{
	struct uda1380 *uda = clnt->data;
	int i;

	for (i = 0; i < ARRAY_SIZE(uda1380_reg_info); i++)
		uda1380_write_reg(clnt, uda, uda1380_reg_info[i].num);

}

static void uda1380_cmd_init(struct i2c_client *clnt)
{
	struct uda1380 *uda = clnt->data;

	uda->active = 1;

	/* resend all parameters */
	uda1380_sync(clnt);
}

static int uda1380_configure(struct i2c_client *clnt, struct uda1380_cfg *conf)
{
	struct uda1380 *uda = clnt->data;
	int ret = 0;

	uda->regs[REG0] &= ~(REG0_SC_MASK);

	printk("uda1380_configure: mic_connected=%d\n", conf->mic_connected);

	uda->mic_connected = 1;
	uda->line_connected = 0;

	switch (conf->fs) {
	case 768: uda->regs[REG0] |= REG0_SC_768FS;	break;
	case 512: uda->regs[REG0] |= REG0_SC_512FS;	break;
	case 384: uda->regs[REG0] |= REG0_SC_384FS;	break;
	case 256: uda->regs[REG0] |= REG0_SC_256FS;	break;
	default:  ret = -EINVAL;			break;
	}

	if (ret == 0 && uda->active) {
		uda1380_write_reg(clnt, uda, REG0);
	}

	switch (conf->format) {
	case FMT_I2S:
	case FMT_LSB16:
	case FMT_LSB18:
	case FMT_LSB20:
	case FMT_MSB:
		uda->regs[I2S_REG] &= ~(I2S_REG_SFORO_MASK|I2S_REG_SFORI_MASK); 
		uda->regs[I2S_REG] |= (conf->format << I2S_REG_SFORO_SHIFT);
		uda->regs[I2S_REG] |= (conf->format << I2S_REG_SFORI_SHIFT);
		break;
	default:  
		ret = -EINVAL;
		break;
	}
	if (ret == 0 && uda->active) {
		uda1380_write_reg(clnt, uda, I2S_REG);
	}

	return ret;
}

static int uda1380_update(struct i2c_client *clnt, int cmd, void *arg)
{
	struct uda1380 *uda = clnt->data;
	struct i2c_gain *v = arg;
	struct i2c_gain *gain = arg;
	char newregnum = 0;
	int val;

	switch (cmd) {
	case I2C_SET_VOLUME: /* set volume.  val =  0 to 100 => 255 to 0 */
		uda->regs[MASTER_VOL_REG] = 
			(MASTER_VCL(255 - ((v->left * 255) / 100))
			 |MASTER_VCR(255 - ((v->right * 255) / 100)));
		newregnum = MASTER_VOL_REG;
		break;

	case I2C_SET_BASS:   /* set bass.    val = 50 to 100 => 0 to 15 */
		val = v->left - 50;
		if (val < 0)
			val = 0;
		val = (val * 15) / 50;
		uda->regs[MBT_REG] &= ~(MBT_BBL_MASK|MBT_BBR_MASK);
		uda->regs[MBT_REG] |= (MBT_BBL(val)|MBT_BBR(val));
		newregnum = MBT_REG;
		break;

	case I2C_SET_TREBLE: /* set treble.  val = 50 to 100 => 0 to 3 */
		val = v->left - 50;
		if (val < 0)
			val = 0;
		uda->regs[MBT_REG] &= ~(MBT_TRL_MASK|MBT_TRR_MASK);
		uda->regs[MBT_REG] |= (MBT_TRL(val)|MBT_TRR(val));
		newregnum = MBT_REG;
		break;

	case I2C_SET_LINE_GAIN:
		val = 255 - (gain->left * 255 / 100);
		uda->regs[MIXER_VOL_REG] &= ~(MBT_TRL_MASK|MBT_TRR_MASK);
		uda->regs[MIXER_VOL_REG] |= (MBT_TRL(val)|MBT_TRR(val));
		newregnum = MIXER_VOL_REG;
		break;

	case I2C_SET_MIC_GAIN:
		val = 255 - (gain->left * 255 / 100);
		uda->regs[MIXER_VOL_REG] &= ~(MBT_TRL_MASK|MBT_TRR_MASK);
		uda->regs[MIXER_VOL_REG] |= (MBT_TRL(val)|MBT_TRR(val));
		newregnum = MIXER_VOL_REG;
		break;

	case I2C_SET_AGC:
		break;

	default:
		return -EINVAL;
	}		

	if (uda->active)
		uda1380_write_reg(clnt, uda, newregnum);
	return 0;
}

static int uda1380_mixer_ioctl(struct i2c_client *clnt, int cmd, void *arg)
{
	struct uda1380 *uda = clnt->data;
	struct i2c_gain gain;
	int val, nr = _IOC_NR(cmd), ret = 0;

	if (cmd == SOUND_MIXER_INFO) {
		struct mixer_info mi;

		strncpy(mi.id, "UDA1380", sizeof(mi.id));
		strncpy(mi.name, "Philips UDA1380", sizeof(mi.name));
		mi.modify_counter = uda->mod_cnt;
		return copy_to_user(arg, &mi, sizeof(mi));
	}

	if (_IOC_DIR(cmd) & _IOC_WRITE) {
		ret = get_user(val, (int *)arg);
		if (ret)
			goto out;

		gain.left    = val & 255;
		gain.right   = val >> 8;

		switch (nr) {
		case SOUND_MIXER_VOLUME:
			uda->volume = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_VOLUME, &gain);
			break;

		case SOUND_MIXER_BASS:
			uda->bass = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_BASS, &gain);
			break;

		case SOUND_MIXER_TREBLE:
			uda->treble = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_TREBLE, &gain);
			break;

		case SOUND_MIXER_LINE:
			if (!uda->line_connected)
				return -EINVAL;
			uda->line = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_LINE_GAIN, &gain);
			break;

		case SOUND_MIXER_MIC:
			if (!uda->mic_connected)
				return -EINVAL;
			uda->mic = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_MIC_GAIN, &gain);
			break;

		case SOUND_MIXER_RECSRC:
			break;

		case SOUND_MIXER_AGC:
			uda1380_update(clnt, I2C_SET_AGC, val);
			break;

		default:
			ret = -EINVAL;
		}
	}

	if (ret == 0 && _IOC_DIR(cmd) & _IOC_READ) {
		int nr = _IOC_NR(cmd);
		ret = 0;

		switch (nr) {
		case SOUND_MIXER_VOLUME:     val = uda->volume;	break;
		case SOUND_MIXER_BASS:       val = uda->bass;	break;
		case SOUND_MIXER_TREBLE:     val = uda->treble;	break;
		case SOUND_MIXER_LINE:       val = uda->line;	break;
		case SOUND_MIXER_MIC:        val = uda->mic;	break;
		case SOUND_MIXER_RECSRC:     val = REC_MASK;	goto devmask;
		case SOUND_MIXER_RECMASK:    val = REC_MASK;	goto devmask;
		case SOUND_MIXER_DEVMASK:    val = DEV_MASK;	goto devmask;
		devmask:
			if (!uda->mic_connected) val &= ~SOUND_MASK_MIC;
			if (!uda->line_connected) val &= ~SOUND_MASK_LINE;
			break;
		case SOUND_MIXER_CAPS:       val = 0;		break;
		case SOUND_MIXER_STEREODEVS: val = 0;		break;
		default:	val = 0;     ret = -EINVAL;	break;
		}

		if (ret == 0)
			ret = put_user(val, (int *)arg);
	}
out:
	return ret;
}

static struct i2c_driver uda1380;

static struct i2c_client client_template = {
	name: "(unset)",
	flags:  I2C_CLIENT_ALLOW_USE,
	driver: &uda1380
};

static int uda1380_attach(struct i2c_adapter *adap, int addr, unsigned short flags, int kind)		
{
	struct uda1380 *uda;
	struct i2c_client *clnt;
	int i;

	clnt = kmalloc(sizeof(*clnt), GFP_KERNEL);
	memcpy(clnt, &client_template, sizeof(*clnt));
	clnt->adapter = adap;
	clnt->addr = addr;
	strcpy(clnt->name, "uda1380");

	uda = kmalloc(sizeof(*uda), GFP_KERNEL);
	if (!uda)
		return -ENOMEM;

	memset(uda, 0, sizeof(*uda));

	/* set default values of registers */
	for (i = 0; i < ARRAY_SIZE(uda1380_reg_info); i++) {
		uda->regs[uda1380_reg_info[i].num] = uda1380_reg_info[i].default_value;
	}
	uda->mic_connected = 1;

	clnt->data = uda;

	i2c_attach_client(clnt);

	return 0;
}

static int uda1380_detach_client(struct i2c_client *clnt)
{
	i2c_detach_client(clnt);

	kfree(clnt->data);
	kfree(clnt);

	return 0;
}

/* Addresses to scan */
static unsigned short normal_i2c[] = {0x18,I2C_CLIENT_END};
static unsigned short normal_i2c_range[] = {I2C_CLIENT_END};
static unsigned short probe[]        = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short probe_range[]  = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore[]       = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore_range[] = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short force[]        = { ANY_I2C_BUS, 0x18, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, normal_i2c_range, 
	probe, probe_range, 
	ignore, ignore_range, 
	force
};

static int uda1380_attach_adapter(struct i2c_adapter *adap)
{
	return i2c_probe(adap, &addr_data, uda1380_attach);
}

static int uda1380_open(struct i2c_client *clnt)
{
	uda1380_cmd_init(clnt);
	return 0;
}

static void uda1380_close(struct i2c_client *clnt)
{
	struct uda1380 *uda = clnt->data;
	uda->active = 0;
}

static int uda1380_command(struct i2c_client *clnt, int cmd, void *arg)
{
	int ret = -EINVAL;

	if (_IOC_TYPE(cmd) == 'M')
		ret = uda1380_mixer_ioctl(clnt, cmd, arg);
	else if (cmd == I2C_UDA1380_CONFIGURE)
		ret = uda1380_configure(clnt, arg);
	else if (cmd == I2C_UDA1380_OPEN)
		ret = uda1380_open(clnt);
	else if (cmd == I2C_UDA1380_CLOSE)
		(ret = 0), uda1380_close(clnt);

	return ret;
}

static void uda1380_inc_use(struct i2c_client *clnt)
{
	MOD_INC_USE_COUNT;
}

static void uda1380_dec_use(struct i2c_client *clnt)
{
	MOD_DEC_USE_COUNT;
}

static struct i2c_driver uda1380 = {
	name:		UDA1380_NAME,
	id:		I2C_DRIVERID_UDA1380,
	flags:		I2C_DF_NOTIFY,
	attach_adapter:	uda1380_attach_adapter,
	detach_client:	uda1380_detach_client,
	command:        uda1380_command,
	inc_use:	uda1380_inc_use,
	dec_use:	uda1380_dec_use
};

static int __init uda1380_init(void)
{
	return i2c_add_driver(&uda1380);
}

static void __exit uda1380_exit(void)
{
	i2c_del_driver(&uda1380);
}

module_init(uda1380_init);
module_exit(uda1380_exit);

MODULE_AUTHOR("Jamey Hicks");
MODULE_DESCRIPTION("Philips UDA1380 CODEC driver");
