#include "ZdbManager.h"

ZdbManager::ZdbManager()
{
	setCodec("eucJP");
	m_pZdb = NULL;
	m_addCategory = false;
	m_pCategories = new SlCategories;
}

ZdbManager::~ZdbManager()
{
	if(m_pZdb) delete m_pZdb;
	delete m_pCategories;
}

bool ZdbManager::init(const QString& dbtype, const QString& basedir,
	bool isReadOnly=true)
{
	QString file;
	QString type = dbtype.lower();
	QDir dir(basedir);

	if(!dir.exists()){
		qWarning("Directory Not Exists");
		return false;
	}
	if(m_pZdb) delete m_pZdb;

	if(type == TYPE_DATEBOOK){
		if(basedir == QString::null){
			file = SlZDataBase::datebookFileName();
		} else {
			file = dir.filePath(SlZDataBase::datebookFileName(true));
		}
		m_pZdb = new SlZDataBase(file, SlZDataBase::datebookItems(),
			NULL, isReadOnly);
	} else if(type == TYPE_ADDRESS){
		if(basedir == QString::null){
			file = SlZDataBase::addressbookFileName();
		} else {
			file = dir.filePath(SlZDataBase::addressbookFileName(true));
		}
		m_pZdb = new SlZDataBase(file, SlZDataBase::addressbookItems(),
			NULL, isReadOnly);
	} else if(type == TYPE_TODO){
		if(basedir == QString::null){
			file = SlZDataBase::todolistFileName();
		} else {
			file = dir.filePath(SlZDataBase::todolistFileName(true));
		}
		m_pZdb = new SlZDataBase(file, SlZDataBase::todolistItems(),
			NULL, isReadOnly);
	} else if(type == TYPE_MAILINBOX){
		if(basedir == QString::null){
			file = SlZDataBase::mailInboxFileName();
		} else {
			file = dir.filePath(SlZDataBase::mailInboxFileName(true));
		}
		m_pZdb = new SlZDataBase(file, SlZDataBase::maillistItems(),
			NULL, isReadOnly);
	} else if(type == TYPE_MAILOUTBOX){
		if(basedir == QString::null){
			file = SlZDataBase::mailOutboxFileName();
		} else {
			file = dir.filePath(SlZDataBase::mailOutboxFileName(true));
		}
		m_pZdb = new SlZDataBase(file, SlZDataBase::maillistItems(),
			NULL, isReadOnly);
	} else if(type == TYPE_MEMO){
		if(basedir == QString::null){
			file = "F0:SLMEMO.BOX";
		} else {
			file = dir.filePath("SLMEMO.BOX");
		}
		SlZDataBaseItemInfo items;
		items.addItemInfo(1, "CLAS", SlZDataBaseItemInfo::TEXT, "title");
		items.addItemInfo(2, "MEM1", SlZDataBaseItemInfo::TEXT, "contents");
		items.addItemInfo(3, "DATE", SlZDataBaseItemInfo::TIME, "modified time");
		m_pZdb = new SlZDataBase(file, items, NULL, isReadOnly);
	} else {
		qWarning("Unknown type [%s]", type.latin1());
		return false;
	}
	if(m_pZdb->isOpened()){
		readInfo();
		return true;
	} else {
		return false;
	}
}

void ZdbManager::setFields(const QStringList& fields)
{
	QStringList list = fields;
	for(QStringList::Iterator it=list.begin();
			it!=list.end(); ){
		if(!m_dbData.info.contains(*it)){
			it = list.remove(it);
		} else {
			++it;
		}
	}
	if(!list.isEmpty()){
		m_dbData.fields = list;
	}
}

void ZdbManager::outputData()
{
	QTextOStream stream(stdout);
	stream.setCodec(QTextCodec::codecForName(m_codec));
	QStringList header;
	QStringList list;
	header.append("CARDID");
	header.append("CATEGORY");
	//list += header;
	header += m_dbData.sysfields;
	header += m_dbData.fields;
	QString text = CSVParser::build(header);
	stream << text;
	stream << "\n";
	for(QValueList<RecordData>::Iterator it=m_dbData.data.begin();
			it!=m_dbData.data.end(); ++it){
		list.clear();
		//list.append((*it)["CARDID"]);
		//list.append((*it)["CATEGORY"]);
		for(QStringList::ConstIterator it2=header.begin();
				it2!=header.end(); ++it2){
			list.append((*it)[*it2]);
		}
		text = CSVParser::build(list);
		stream << text;
		stream << "\n";
	}
}

void ZdbManager::outputInfo()
{
	QTextOStream stream(stdout);
	stream.setCodec(QTextCodec::codecForName(m_codec));
	QStringList header;
	//header.append("CARDID");
	//header.append("CATEGORY");
	header = m_dbData.sysfields;
	header += m_dbData.fields;
	QString text = CSVParser::build(header);
	stream << text;
	stream << "\n";
	QStringList type;
	//type.append("ULONG");
	//type.append("ULONG(CSV)");
	/* system fields */
	for(QStringList::Iterator it=m_dbData.sysfields.begin();
			it!=m_dbData.sysfields.end(); ++it){
		type.append(typeName(m_dbData.info[*it].type));
	}
	/* normal fields */
	for(QStringList::Iterator it=m_dbData.fields.begin();
			it!=m_dbData.fields.end(); ++it){
		type.append(typeName(m_dbData.info[*it].type));
	}
	text = CSVParser::build(type);
	stream << text;
	stream << "\n";
}

void ZdbManager::readInfo()
{
	/* get info */
	m_dbData.info.clear();
	QStringList fields;
	for(unsigned int n=0;
			n<=m_pZdb->items().count(); n++){
		if(m_pZdb->nameToFieldId(n) > 0){
			const QString& name = m_pZdb->itemName(n).left(4);
			m_dbData.info[name].type = m_pZdb->readItemType(name);
			m_dbData.info[name].field = n;
			fields.append(name);
		}
	}
	m_dbData.fields = fields;
	fields.clear();
	/* system fields */
	for(QStringList::ConstIterator it=m_sysFields.begin();
			it!=m_sysFields.end(); ++it){
		if(!m_dbData.info.contains(*it) && m_pZdb->nameToItemId(*it) > 0){
			m_dbData.info[*it].type = m_pZdb->readItemType(*it);
			m_dbData.info[*it].field = 0;
			fields.append(*it);
		}
	}
	m_dbData.sysfields = fields;
}

void ZdbManager::readData()
{
	CardId id;
	if((id = m_pZdb->firstCard())){
		/* get data */
		do {
			readRecord(id);
		}while((id = m_pZdb->nextCard(id)));
	}
}

void ZdbManager::readRecord(CardId id)
{
	RecordData record;
	record["CARDID"] = QString::number(id);
	record["CATEGORY"] = readCategory(id);
	/* system records */
	for(QStringList::ConstIterator it=m_dbData.sysfields.begin();
			it!=m_dbData.sysfields.end(); ++it){
		QString s = readItem(*it, id, m_dbData.info[*it].type);
		record[*it] = s;
	}
	/* normal records */
	for(QStringList::ConstIterator it=m_dbData.fields.begin();
			it!=m_dbData.fields.end(); ++it){
		QString s = readItem(*it, id, m_dbData.info[*it].type);
		record[*it] = s;
	}
	m_dbData.data.append(record);
}

int ZdbManager::doInfo()
{
	outputInfo();
	return 0;
}

int ZdbManager::doRead(CardId id=0)
{
	m_dbData.data.clear();
	if(id > 0){
		readRecord(id);
	} else {
		readData();
	}
	outputData();
	return 0;
}

int ZdbManager::doWrite()
{
	QTextIStream stream(stdin);
	stream.setCodec(QTextCodec::codecForName(m_codec));
	int errcnt = 0;

	m_dbData.data.clear();

	QStringList fields = CSVParser::parse(stream);
	if(!fields.isEmpty()){
		setFields(fields);
		QStringList data;
		while(data = CSVParser::parse(stream), !data.isEmpty()){
			RecordData record;
			if(!makeRecord(fields, data, record)){
				errcnt++;
				qWarning("need CARDID");
				continue;
			}
			CardId id;
			if((id = writeRecord(record)) == 0){
				errcnt++;
				record["CARDID"] = "-1";
				m_dbData.data.append(record);
			} else {
				readRecord(id);
			}
		}
		outputData();
	}
	return errcnt;
}

int ZdbManager::doDelete()
{
	QTextIStream stream(stdin);
	stream.setCodec(QTextCodec::codecForName(m_codec));
	int errcnt = 0;

	m_dbData.data.clear();

	QStringList fields = CSVParser::parse(stream);
	if(!fields.isEmpty()){
		QStringList data;
		while(data = CSVParser::parse(stream), !data.isEmpty()){
			RecordData record;
			if(!makeRecord(fields, data, record)){
				errcnt++;
				qWarning("need CARDID");
				continue;
			}
			if(record["CARDID"].toULong() == 0){
				errcnt++;
				m_dbData.data.remove(m_dbData.data.fromLast());
				continue;
			}
			if(!deleteRecord(record)){
				errcnt++;
				qWarning("delete Record error");
				record["CARDID"] = "-1";
			} else {
				record["CARDID"] = "0";
			}
			m_dbData.data.append(record);
		}
		outputData();
	}
	return errcnt;
}

bool ZdbManager::makeRecord(
	const QStringList& fields,
	const QStringList& data,
	RecordData& record)
{
	QStringList::ConstIterator it, it2;
	for(it=fields.begin(),it2=data.begin();
			it!=fields.end() && it2!=data.end(); ++it, ++it2){
		record[*it] = *it2;
	}
	if(!record.contains("CARDID")){
		return false;
	}
	CardId id = record["CARDID"].toULong();
	if(id == 0){
		m_dbData.data.append(record);
	} else {
		readRecord(id);
	}
	return true;
}

bool ZdbManager::deleteRecord(const RecordData& record)
{
	CardId id = record["CARDID"].toULong();

	for(RecordData::ConstIterator it=record.begin();
			it!=record.end(); ++it){
		if(it.key() == "CARDID") continue;
		if(*it != readItem(it.key(), id)){
			qWarning("different record data");
			return false;
		}
	}
	return m_pZdb->deleteCard(&id);
}

QString ZdbManager::readCategory(CardId id)
{
	QArray<int> data = m_pZdb->readCategories(id);
	QStringList categories;
	for(QArray<int>::ConstIterator it=data.begin();
			it!=data.end(); ++it){
		const QString& name = m_pCategories->label(*it);
		if(name.isEmpty()){
			categories.append(QString::number(*it));
		} else {
			categories.append(name);
		}
	}
	return categories.join(",");
}

bool ZdbManager::updateCategories(CardId id, const QString& categories)
{
	if(categories == QString::null) return true;
	QStringList list = CSVParser::parse(categories);
	QArray<int> ids(list.count());
	uint index = 0;
	for(uint i=0; i<ids.size(); i++){
#if 0
		bool ok;
		ids[index] = list[i].toInt(&ok);
		if(ok){
			index++;
		} else {
			if(m_pCategories->exists(list[i])){
				ids[index] = m_pCategories->id(list[i]);
				index++;
			}
		}
#else
		if(m_pCategories->exists(list[i])){
			ids[index] = m_pCategories->id(list[i]);
			index++;
		} else if(m_addCategory){
			ids[index] = m_pCategories->addCategory(list[i]);
			index++;
		}
#endif
	}
	if(ids.size() > 0 && index == 0){
		return true;
	} else if(index != ids.size()){
		ids.resize(index);
	}
	if(!m_pZdb->updateCategories(id, ids)){
		return false;
	}
	return true;
}

CardId ZdbManager::writeRecord(const RecordData& record)
{
	CardId id = record["CARDID"].toULong();

	bool isNewRecord = (id == 0);
	const QString& newcategories = record["CATEGORY"];

	if(!isNewRecord){
		if(!updateCategories(id, newcategories)){
			qWarning("updateCategories error");
			return 0;
		}
	}

	if(!editRecord(id, record)){
		if(!isNewRecord){
			const QString& oldcategories = m_dbData.data.last()["CATEGORY"];
			if(!updateCategories(id, oldcategories)){
				qWarning("updateCategories error");
			}
		}
		return 0;
	}

	if(isNewRecord){
		if(!updateCategories(id, newcategories)){
			qWarning("updateCategories error");
			if(!m_pZdb->deleteCard(&id)){
				qWarning("deleteCard error");
			}
			return 0;
		}
	}

	return id;
}

bool ZdbManager::editRecord(CardId& id, const RecordData& record)
{
	if(!m_pZdb->startEditCard(id)){
		qWarning("startEditCard error");
		return false;
	}
	for(QStringList::ConstIterator it=m_dbData.fields.begin()
			; it!=m_dbData.fields.end(); ++it){
		if(!writeItem(*it, record[*it], m_dbData.info[*it].type)){
			qWarning("writeItem error[%s][%s]", (*it).latin1(), record[*it].latin1());
			m_pZdb->cancelEditCard();
			return false;
		}
	}
	if(!m_pZdb->finishEditCard(&id)){
		qWarning("finishEditCard error");
		return false;
	}
	return true;
}

QString ZdbManager::readItem(const QString& name, CardId id, int type=0)
{
	if(name == "CATEGORY"){
		return readCategory(id);
	}
	if(type == 0){
		type=m_pZdb->readItemType(name);
	}
	QString s;
	int idata;
	unsigned long uldata;
	QDateTime dtdata;
	switch(type){
	case DATATYPE_TEXT:
		s = m_pZdb->readItem(name, id);
		break;
	case DATATYPE_UCHAR:
		idata = m_pZdb->readUcharItem(name, id);
		if(idata != -1){
			s = QString::number(idata);
		}
		break;
	case DATATYPE_USHORT:
		idata = m_pZdb->readUshortItem(name, id);
		if(idata != -1){
			s = QString::number(idata);
		}
		break;
	case DATATYPE_ULONG:
		if(m_pZdb->readUlongItem(name, &uldata, id)){
			s = QString::number(uldata);
		}
		break;
	case DATATYPE_COLOR:
		qWarning("DATATYPE_COLOR[%s]", name.latin1());
		break;
	case DATATYPE_TIME:
		dtdata = m_pZdb->readTimeItem(name, id);
		if(dtdata.date().isValid()){
			s = toISO8601(dtdata);
		}
		break;
	case DATATYPE_BINARY:
		qWarning("DATATYPE_BINARY[%s]", name.latin1());
		break;
	case DATATYPE_CATEGORY:
		qWarning("DATATYPE_CATEGORY[%s]", name.latin1());
		//s = QString::number(m_pZdb->readUlongItem(name, id));
		break;
	default:
		qWarning("DATATYPE_UNKNOWN(%d)[%s]", type, name.latin1());
		break;
	}
	return s;
}

bool ZdbManager::writeItem(const QString& name, QString value, int type=0)
{
	bool success = false;
	QDateTime dtvalue;
	if(value == QString::null){
		return m_pZdb->clearItem(name);
	}
	if(type == 0){
		type=m_pZdb->readItemType(name);
	}
	switch(type){
	case DATATYPE_TEXT:
		success = m_pZdb->writeItem(name, value);
		break;
	case DATATYPE_UCHAR:
		success = m_pZdb->writeItem(name, (uchar)value.toULong());
		break;
	case DATATYPE_USHORT:
		success = m_pZdb->writeItem(name, (ushort)value.toULong());
		break;
	case DATATYPE_ULONG:
		success = m_pZdb->writeItem(name, value.toULong());
		break;
	case DATATYPE_COLOR:
		qWarning("DATATYPE_COLOR[%s]", name.latin1());
		break;
	case DATATYPE_TIME:
		dtvalue = fromISO8601(value.local8Bit());
		if(dtvalue.time().isValid()){
			success = m_pZdb->writeItem(name, dtvalue);
		} else {
			success = m_pZdb->writeItem(name, dtvalue.date());
		}
		break;
	case DATATYPE_BINARY:
		qWarning("DATATYPE_BINARY[%s]", name.latin1());
		break;
	case DATATYPE_CATEGORY:
		qWarning("DATATYPE_CATEGORY[%s]", name.latin1());
		//s = QString::number(m_pZdb->readUlongItem(name, id));
		break;
	default:
		qWarning("DATATYPE_UNKNOWN[%s]", name.latin1());
		break;
	}
	return success;
}

QString ZdbManager::typeName(int type)
{
	QString name;
	switch(type){
	case DATATYPE_TEXT:
		name = "text";
		break;
	case DATATYPE_UCHAR:
		name = "uchar";
		break;
	case DATATYPE_USHORT:
		name = "ushort";
		break;
	case DATATYPE_ULONG:
		name = "ulong";
		break;
	case DATATYPE_COLOR:
		name = "color";
		break;
	case DATATYPE_TIME:
		name = "time";
		break;
	case DATATYPE_BINARY:
		name = "binary";
		break;
	case DATATYPE_CATEGORY:
		name = "category";
		break;
	default:
		name = "unknown";
		break;
	}
	return name;
}

QCString ZdbManager::toISO8601(const QDateTime &dt)
{
	QCString datestr;
	QCString timestr;
	if(dt.date().isValid()){
		const QDate& date = dt.date();
		datestr.sprintf("%04d%02d%02d",
    		date.year(), date.month(), date.day());
	}
	if(dt.time().isValid()){
		const QTime& time = dt.time();
		timestr.sprintf("T%02d%02d%02d",
    		time.hour(), time.minute(), time.second());
	}
	return datestr + timestr;
}

QDateTime ZdbManager::fromISO8601(const QCString& s)
{
#if defined(_OS_WIN32) || defined (Q_OS_WIN32) || defined (Q_OS_WIN64)
    _tzset();
#else
    tzset();
#endif
	QDateTime datetime;
	
	QCString str = s.copy();
	str.stripWhiteSpace();
	str = str.lower();

	int i = str.find('t');
	QCString datestr;
	QCString timestr;
	if(i != -1){
		datestr = str.left(i);
		timestr = str.mid(i+1);
	} else {
		datestr = str;
	}
	datestr.replace(QRegExp("-"), "" );
	timestr.replace(QRegExp(":"), "" );

	// get the date
	if(!datestr.isEmpty()){
		QDate date;
		int y=0,m=1,d=1;
		switch(datestr.length()){
		case 8:
			d = datestr.right(2).toInt();
		case 6:
			m = datestr.mid(4,2).toInt();
		case 4:
			y = datestr.left(4).toInt();
			break;
		default:
			break;
		}
		date.setYMD(y, m, d);
		datetime.setDate(date);
	}

	int tzoff = 0;
	int tzloc = 0;
	if(timestr.find('z') == (int)timestr.length() - 1){
		// UTC
		timestr = timestr.left(timestr.length() - 1);
		time_t tmp = time(0);
		struct tm* lt = localtime(&tmp);
		tzloc = mktime(lt);
		struct tm* ut = gmtime(&tmp);
		tzloc -= mktime(ut);
	} else {
		int plus = timestr.find('+');
		int minus = timestr.find('-');
		QCString off;
		if(plus != -1){
			off = timestr.mid(plus);
			timestr = timestr.left(plus);
		} else if(minus != -1){
			off = timestr.mid(minus);
			timestr = timestr.left(minus);
		}
		if(!off.isNull()){
			int tzoffhour = 0;
			int tzoffmin = 0;
			switch(off.length()){
			case 5:
				tzoffmin = off.mid(3).toInt();
			case 3:
				tzoffhour = off.mid(1,2).toInt();
			default:
				break;
			}
			tzoff = 60*tzoffhour + tzoffmin;
			if(minus != -1){
				tzoff *= -1;
			}
		}
	}

	// get the time
	QTime time;
	if(!timestr.isEmpty()){
		int h=0,m=0,s=0;
		switch(timestr.length()){
		case 6:
			s = timestr.mid(4).toInt();
		case 4:
			m = timestr.mid(2, 2).toInt();
		case 2:
			h = timestr.left(2).toInt();
		default:
			break;
		}
		time.setHMS(h, m, s);
	} else {
		time.setHMS(31,63,63);
	}
	datetime.setTime(time);

	if(datetime.time().isValid()){
		return datetime.addSecs(60 * (-tzloc + tzoff));
	} else {
		return datetime;
	}
}
