/******************************************************************************
//               INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2000 Intel Corporation. All Rights Reserved.
//
//  VSS:
//		$Workfile: ANCdemo.c $
//      $Revision: 1 $
//      $Date: 05-02-28 16:03 $
//      $Archive: /ipp/branch/5_0/beta/install/bulverde/primitive/arm_linux/sample/AdaptiveNoiseCanc/ANCdemo.c $
//
//  Description:
//      LMS adaptive noise cancellation demo 
//
******************************************************************************/

/************
Header files
*************/

/* Linux audio output device interface */
#include	<fcntl.h>					
#include	<unistd.h>
#include	<sys/ioctl.h>
#include	<linux/soundcard.h>
#include	<sched.h>
#include	<signal.h>

/* Console, file, and string I/O */
#include	<stdio.h>
#include	<string.h>

/* Standard IPP definitions */
#include	"ippdefs.h"
#include	"ippSP.h"

/* Interface to IPP-based examples */
#include	"anc.h"

/* Constants */
#define		FS				8000		/* Sample rate */
#define		FRAMES			5			/* Number of GSM-AMR frames in each audio buffer */
#define		M				160			/* Frame size, in samples */
#define		N				FRAMES*M	/* Audio buffer size, in samples */

/* Resampling constants for SA-1110 development module */
#define		DS				6			/* Default audio sample rate = 48 kHz, must downsample to 8 kHz */
#define		US				6			/* Default audio sample rate = 48 kHz, must upsample from 8 kHz */

/* User interface constants */
#define		UI_STACK_SIZE	2500		/* User interface stack size, in 32-bit words */					
#define		MSGBUF_LEN      1000		/* User interface keyboard buffer size, in bytes */
#define		PBUF_LEN		100			/* User interface parsing buffer */
#define		MAX_STR			256			/* Maximum string length, in characters */
#define		MAX_FN			256			/* Maximum filename length, in characters */

/*************
 Prototypes 
**************/
void			Usage							(void);
void			DisplayConsoleStartupBanner		(void);
void			RenderSound						(int *, Ipp16s *, int);
void			GetAudioInput					(int *dev, Ipp16s *sig, int len);
int				OpenAudioDevice					(int *, int, int);
void			CloseFile						(FILE *);
void			CloseAudioDevice				(int);
void			WriteOutputFile					(FILE *, Ipp16s *, int);
int				ReadInputFile					(FILE *fpi, Ipp16s *pcm, int len);
int				UserInterface					(void);
void			ProcessCommandLineArguments		(int argc, char **argv, FILE **fpi, FILE **fpo, 
												 int *AudioDev);

/*************
 Global data 
**************/
int g_Quit=0;					/* Quit flag, set asynchronously by the user interface */
int g_OutputSelect=0;			/* Select s+n input */

/*************************************
 Console-mode application entry point 
**************************************/
/******************************************************************************************
// Name:			 main
//
// Description:		 Adaptive noise cancellation example
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//
*******************************************************************************************/
int main(int argc, char **argv)
{
	FILE *fpi;					/* Input file */							
	FILE *fpo;					/* Output file */							
	Ipp16s s[N];				/* Input PCM speech buffer */
	Ipp16s shat[N];				/* Output PCM speech buffer */
	Ipp16s noise[N];			/* Working noise buffer for ANC */
	Ipp16s signoise[N];			/* Working signal+noise buffer for ANC */
	int AudioDevice;			/* Pointer to the (Linux) audio output device */
	int AudioDevEnable;			/* Audio device enable flag */
	int StackUI[UI_STACK_SIZE];	/* User interface stack (child thread) */
	int status=-1;				/* Audio device status */
	int UIpid;					/* User interface PID */
	int frame=1;				/* Frame counter for file I/O */

	/****************************************************
	Startup banner, parse command line, open I/O devices  
	*****************************************************/
	DisplayConsoleStartupBanner();	
	ProcessCommandLineArguments(argc,argv,&fpi,&fpo,&AudioDevEnable);

	#ifdef PERFORMANCE_TIMER
		/* Initialize performance timer */
		InitTimer();
	#endif

	/* Establish initial noise level */
	SetNoiseVariance(5000);

	/* Spawn user interface thread */
	UIpid=__clone(UserInterface,&(StackUI[UI_STACK_SIZE]),CLONE_VM|CLONE_FS);

	/* Open audio device */
	/* Current driver opens at 48 kHz regardless of requested rate */
	if (AudioDevEnable)
	{
		status=OpenAudioDevice(&AudioDevice,2,48000);
		if (status<0)
		{
			printf("Unable to open audio device\n");
			exit(0);
		}
	}

	printf("\n");

	/* Apply simulated dual-microphone noise cancellation on audio buffer until user aborts */
	while(!g_Quit)
	{
		/* Fill audio input buffer */
		if (status>=0)
			GetAudioInput(&AudioDevice,s,N);
		else
		{
			ippsZero_16s(s,N);
			if (ReadInputFile(fpi,s,N)<N)
				g_Quit=1;
		}

		/* Apply ANC:  shat, noise, and signoise are internal ANC buffers */
		/* Block ANC as written is an inplace operation on the buffer s */
		BlockANC(s,shat,noise,signoise,N,g_OutputSelect);

		/* Render current audio buffer */
		if (status>=0)
			RenderSound(&AudioDevice,s,N);
		else 
		{
			WriteOutputFile(fpo,s,N);
			printf("Frame %d\r",frame++);
			fflush(stdout);
		}
	}

	/* Close I/O devices */	
	CloseFile(fpi);
	CloseFile(fpo);
	if (status>=0)
		CloseAudioDevice(AudioDevice);

	#ifdef PERFORMANCE_TIMER
		/* Close performance timer device driver */
		ShutdownTimer();
	#endif

	/* Shutdown message */
	printf("\n\nDone.\n\n");

	/* End UI process */
	kill(UIpid,SIGKILL);

} /* End of main() */

/********************************
 Console management and file I/O 
*********************************/

/*********************************************************************************************
// Name:			 DisplayConsoleStartupBanner
//
// Description:		 Display startup message on the text console
//
// Input Arguments:  None.
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void DisplayConsoleStartupBanner()
{
	int i;
	for(i=0;i<50;i++) printf("\n");
	printf("\n\nIntel(R) Integrated Performance Primitives (IPP)\nfor Intel(R) XScale(TM) Microarchitecture.\n\n");
	printf("Adaptive Noise Cancellation Example, Version 1.00\n\n");
	printf("Copyright (C) 2000-01, Intel Corporation.  All Rights Reserved.\n\n");
}

/*********************************************************************************************
// Name:			 ProcessCommandLineArguments
//
// Description:		 Extracts run-time parameters from the command line
//					 Opens I/O files
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//					 fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None
**********************************************************************************************/
void ProcessCommandLineArguments(int argc, char **argv, FILE **fpi, FILE **fpo, 
								 int *AudioDev)
{
	char ifile[MAX_FN],ofile[MAX_FN];
	char *arg;
	int i;

	/* Establish defaults */
	strcpy(ifile,"speech.in");
	strcpy(ofile,"speech.out");
	*AudioDev=0;
	
	/* Usage */
	if ((argc>1)&&(strcmp(argv[1],"--help")==0))
		Usage();

    /* Parse arguments */
	for(i=1;i<argc;i++)
	{
		arg=argv[i];
		switch(arg[1])
		{
			case 'a':
				*AudioDev=1;
				break;
			case 'i':
				strcpy(ifile,&(arg[2]));
				break;
			case 'o':
				strcpy(ofile,&(arg[2]));
				break;
			default:
				Usage();
				break;
		}; 
	}

	if (!(*AudioDev))
	{
		/* Open input file */
		if ((*fpi=fopen(ifile,"rb"))==NULL)
		{
			printf("Unable to open %s\n",ifile);
			exit(1);
		}
		else
			printf("Input file:\t%s\n",ifile);	
	
		/* Open output file */
		if ((*fpo=fopen(ofile,"wb"))==NULL)
		{
			printf("Unable to create %s\n",ofile);
			exit(1);
		}
		else
			printf("Output file:\t%s\n",ofile);
	}
	else
	{
		printf("Speech I/O:\tfull-duplex audio device\n");
	}

	g_OutputSelect=0;
	/* Indicate initially selected output signal */
	switch(g_OutputSelect)
	{
		case 0:
			printf("Output Select:  s+n (signal+noise input)\n");
			break;
		case 1:
			printf("Output Select:  s (clean signal)\n");
			break;
		case 2:
			printf("Output Select:  anc out (noise-cancelled output)\n");
			break;
	}

	printf("\nPress h <Return> for a list of commands\n");
	printf("Press s <Return> to select the clean signal\n");
	printf("Press n <Return> to select the signal+noise\n");
	printf("Press a <Return> to select the ANC output\n");
	printf("Press v<value> <Return> to change the noise variance\n");
}

/*********************************************************************************************
// Name:			 Usage
//
// Description:		 Display usage banner
//
// Returns:			 None
**********************************************************************************************/
void Usage(void)
{
	printf("Usage: ANCdemo -i<input file> -o<output file> -a [--help], \n\nwhere\n\n");
	printf("<input file> is the desired binary pcm input file\n");
	printf("<output file> is the desired binary pcm output file\n");
	printf("-a forces the use of the full-duplex audio device instead of the specified files\n");
	printf("No input file forces use of the audio device for input/output.\n\n");
	printf("--help displays usage\n\n");
	printf("\n");
	exit(0);
}

/*********************************************************************************************
// Name:			 CloseFile
//
// Description:		 Close PCM binary output file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseFile(FILE *fp)
{
	if (fp!=NULL) fclose(fp);
}

/*****************************
 Linux Audio Device Interface
******************************/

/***************************************************************************************
// Name:			 RenderSound
//
// Description:		 Write pcm audio samples to the audio output device. 
//					 Assumes 48 kHz sample rate; upsamples from 8->48 kHz
//
// Input Arguments:  fp					- Pointer to the audio output device. 
//					 pcm				- Pointer to a buffer of interleaved pcm audio.
//					 channels			- Number of channels contained in the buffer.
//					 len				- PCM buffer length, in samples (not bytes)
//
// Returns:			 None.
//
// Notes:			 Stereophonic data is interleaved as follows:  R0L0R1L1 ... R575L575; 
//					 For channels=1, right channel data is replicated on to the left.
****************************************************************************************/
void RenderSound(int *fp, Ipp16s *pcm, int len)
{
	static Ipp16s obuf[US*2*N];		/* audio output buffer */
	static Ipp16s ubuf[US*N];
	int i,j;

	/* Filter parameters */
	int RS_TAPS=38;
	static Ipp16s DelayLineRS[80];
	static int DelayLineIndexRS;
	static int init=0;
	Ipp16s rstaps[38]=
	{ -105, 670, 1235, 1980, 2575, 2715, 2130, 735, -1290, -3475, -5095, -5370, -3670,
	   270, 6180, 13250, 20285, 25950, 29110, 29110, 25950, 20285, 13250, 6180, 270, -3670,
	  -5370, -5095, -3475, -1290, 735, 2130, 2715, 2575, 1980, 1235, 670, -105 };

	if (!init)
	{
		init=1;
		ippsZero_16s(DelayLineRS,80);
		DelayLineIndexRS=0;
	}

	/* Clear upsample buffer */
	ippsZero_16s(ubuf,US*len);

	/* Upsample from 12 to 48 kHz */
	for(i=j=0;i<US*len;i+=US,j++)
		ubuf[i]=pcm[j];

	/* Interpolate */
	ippsFIR_Direct_16s_I(ubuf,US*len,rstaps,RS_TAPS,DelayLineRS,&DelayLineIndexRS);		

	/* Replicate signal on both left and right channels */
	for(i=j=0;i<US*2*len;i+=2,j++)
		obuf[i]=obuf[i+1]=ubuf[j];

	/* Pass audio output buffer to the audio output device */
	write(*fp,obuf,sizeof(Ipp16s)*len*2*US);
}

/***************************************************************************************
// Name:			 OpenAudioDevice
//
// Description:		 Initialize Linux audio output device. 
//
// Input Arguments:  dev				- Pointer to the audio output device. 
//					 channels			- Number of audio channels.
//
// Returns:			 Status				 0	 = success, device ready
//										-100 = unable to open device
//										-101 = unable to select 16-bit stereo PCM
//										-102 = unable to select 44100 Hz sample rate
//										-103 = unable to select specified # of channels 
//
****************************************************************************************/
int OpenAudioDevice(int *dev, int channels, int SampleRate)
{
	long param;
	int status=0;

	/* Open audio device (UDA1341 on the Linux SA-1110 Microprocessor Evaluation Module) */
	if ((*dev=open("/dev/dsp",O_RDWR))==-1) status=-100;

	/* Request 16-bit, little-endian, PCM data format */
	param = AFMT_S16_LE;
	if ((ioctl(*dev,SNDCTL_DSP_SETFMT,&param))!=0) { close (*dev); status=-101; }
  
	/* Request sample rate */
	param=SampleRate;
	if ((ioctl (*dev,SNDCTL_DSP_SPEED,&param))!=0) { close (*dev); status=-102; }

	/* Request number of channels */
	param = channels;
	if ((ioctl(*dev,SNDCTL_DSP_CHANNELS,&param))!=0) { close (*dev); status=-103; }

	/* Trap unsuccessful attempts to open audio device */
	if (status<0) { fprintf(stderr,"\nUnable to open audio device.\n"); fflush(stderr); return(status); }
	else return 0;
}

/***************************************************************************************
// Name:			 GetAudioInput
//
// Description:		 Read pcm audio from the audio input device. 
//					 Assumes 48 kHz input stream, generates 8 kHz output stream
//
// Input Arguments:  dev				- Pointer to the audio input device. 
//					 sig				- Pointer to a buffer of interleaved pcm audio.
//					 len				- PCM buffer length, in samples (not bytes)
//
// Returns:			 None.
//
// Notes:			 Stereophonic data is interleaved as follows:  R0L0R1L1 ... R575L575; 
//					 For channels=1, right channel data is replicated on to the left.
****************************************************************************************/
void GetAudioInput(int *dev, Ipp16s *sig, int len)
{
	int i,j;					/* Buffer indices */
	static Ipp16s pcm[DS*2*N];

	/* Downsample filter */
	int TAPS_DS=38;
	static Ipp16s fmem[76];
	static int fptr;
	static int init=0;
	static Ipp16s taps_ds[38]=
	{
	  -42, 268, 494, 792, 1030, 1086, 852, 294, -516, -1390, -2038, -2148, -1468, 108,
       2472, 5300, 8114, 10380, 11644, 11644, 10380, 8114, 5300, 2472, 108, -1468,
	  -2148, -2038, -1390, -516, 294, 852, 1086, 1030, 792, 494, 268, -42
	};

	if (!init)
	{
		init=1;
		ippsZero_16s(fmem,2*TAPS_DS);
		fptr=0;
	}

	/* Fill pcm audio input buffer */
	read(*dev,pcm,DS*2*len*sizeof(Ipp16s));

	/* Extract right channel from the L/R interleaved data since
	   resampler requires contiguous (non-interleaved) samples */
	for(i=0,j=1;i<(DS*len);i++,j+=2)
		pcm[i]=pcm[j];
	 
	/* Resample 48 kHz -> 12 kHz */
	ippsFIR_Direct_16s_I(pcm,DS*len,taps_ds,TAPS_DS,fmem,&fptr);
	for(i=j=0;i<len;i++,j+=DS)
		sig[i]=pcm[j];
}


/*********************************************************************************************
// Name:			 CloseAudioDevice
//
// Description:		 Close audio output device
//
// Input Arguments:  AudioDevice		- Pointer to the audio output device
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseAudioDevice(int AudioDevice)
{
	close(AudioDevice);
}

/*********************************************************************************************
// Name:			 ReadInputFile
//
// Description:		 Read binary file contents to a pcm buffer
//
// Input Arguments:  fpi				- Pointer to the input file stream
//					 pcm				- PCM buffer pointer
//					 N					- Number of samples to read
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
int ReadInputFile(FILE *fpi, Ipp16s *pcm, int len)
{
	if (fpi!=NULL) 
		return(fread(pcm,sizeof(Ipp16s),len,fpi));
	else
		return(0);
}


/*********************************************************************************************
// Name:			 WriteOutputFile
//
// Description:		 Write PCM buffer contents to a binary file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//					 pcm				- PCM output buffer pointer
//					 N					- Number of samples to write
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void WriteOutputFile(FILE *fpo, Ipp16s *pcm, int len)
{
	if (fpo!=NULL) 
		fwrite(pcm,sizeof(Ipp16s),len,fpo);
}

/*********************************************************************************************
// Name:			 User interface thread
//
// Description:		 Keyboard character-based user interface
//					 Thread terminates with the parent
//
// Returns:			 None				
**********************************************************************************************/
int UserInterface()
{
	char msg[MSGBUF_LEN], tstr[PBUF_LEN];	/* Keyboard message and parsing buffer */
	int i,ri,p,n;
	char name[MAX_STR];
	int stddev,snr;

	/* UI main loop */
	while(1)
	{
		/* Get user input from the keyboard */
		gets(msg);

		/* Parse and act on user input */
		switch(msg[0])
		{
			/* a = select ANC output */
			case 'a':
			case 'A':
				g_OutputSelect=2;
				printf("ANC output selected\n");
				break;

			/* h = help */
			case 'h':
			case 'H':
				printf("\n\nCommands:\n");
				printf("a          - Listen to ANC output\n");
				printf("d          - Display SNR at the s+n input microphone\n");
				printf("n          - Listen to s+n input\n");
				printf("s          - Listen to s only (clean input)\n");
				#ifdef PERFORMANCE_TIMER
				printf("p          - display performance measurements\n");
				#endif
				printf("v<stddev>  - Adjust Gaussian sigma to <stddev>\n\n");
				printf("q  - QUIT\n");
				printf("\n\n");
				break;

			/* n = select signal+noise for monitor output */
			case 'n':
			case 'N':
				g_OutputSelect=0;
				printf("signal+noise input selected\n");
				break;

			#ifdef PERFORMANCE_TIMER
			/* p = display performance data */
			case 'p':
			case 'P':
				n=GetPerformanceTimer(-1,0,1,NULL);
				printf("\n\nOSCR Performance Measurements\n\n");
				printf("\nOperation\tCPU Utilization\n\n");
				for(i=0;i<n;i++)
				{
					p=GetPerformanceTimer(i,N,FS,name);
					printf("%s\t%d %%\n",name,p);
				}
				printf("\n\n");
				break;
			#endif
			/* q = Quit */
			case 'q':
			case 'Q':
				g_Quit=1;
				break;

			/* d = select clean input signal for output */
			case 'd':
			case 'D':
				GetSNR(&snr);
				printf("SNR = %d dB (at the s+n input microphone)\n",snr);
				break;

			/* s = select clean input signal for output */
			case 's':
			case 'S':
				g_OutputSelect=1;
				printf("signal only (clean signal) selected\n");
				break;

			/* v = adjust input noise variance */
			case 'v':
			case 'V':
				strncpy(tstr,&(msg[1]),5);
				stddev=atoi(tstr);
				SetNoiseVariance(stddev);
				printf("StdDev: %d\n",stddev);
				break;

			default:
				break;
		}

		/* Clear keyboard message buffer */
		for(i=0;i<MSGBUF_LEN;i++)
			msg[i]='\0';
		/* Clear parse buffer */
		for(i=0;i<PBUF_LEN;i++)
			tstr[i]='\0';
	}
}
