/******************************************************************************
//             INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2001 Intel Corporation. All Rights Reserved.
//
//  VSS: 
//		$Workfile: schroeder.c $
//		$Revision: 1 $
//		$Date: 03-09-10 13:21 $
//
//  Description:
//		Modified Schroeder/Moorer artificial reverberator 
//
******************************************************************************/

#include <math.h>
#include <malloc.h>
#include <ippdefs.h>
#include <ippSP.h>

/* External interface */
#include "schroeder.h"

#ifdef PERFORMANCE_TIMER
#include "oscr.h"
	
/* OSCR timer controls */
#define		START_TIMER		{ixs_get_OSCR_val(&t1);}
#define		STOP_TIMER(x)	{ixs_get_OSCR_val(&t2); g_TimerOSCR[x]+=(t2-t1); g_NumBlocks[x]++;}
#define		RESET_TIMER(x)	{g_TimerOSCR[x]=0; g_NumBlocks[x]=0;}

/* OSCR timer constants */
#define     FC				3250000				/* OSCR clock frequency */
#define		NUM_TIMERS		2					/* Number of timers */
#define		T_COMB			0					/* Comb filters */
#define		T_AP			1					/* Allpass filters */

/* OSCR timer tick and block count vectors */
int g_TimerOSCR[NUM_TIMERS]={0,0};
int g_NumBlocks[NUM_TIMERS]={0,0};

/* OSCR timer display labels */
char *g_TimerDesc[NUM_TIMERS]=
{
	"Parallel comb filters",
	"Series allpass filters"
};
#else

#define		START_TIMER   
#define		STOP_TIMER(x) 

#endif


/* 
   Below are sample parameters that work well 
   for a typical Shcroeder reverberator
   These are used in the sample
   external interface 
*/

/* Number of parallel comb filters */
#define		NC				4						

/* Number of series allpass filters */
#define		NA				2						

/* Comb delays, in milliseconds */
#define		CDLY1			50					
#define		CDLY2			67
#define		CDLY3			74
#define		CDLY4			75

/* Allpass delays,in milliseconds */
#define		APDLY1			10 
#define		APDLY2			16

/* Sparse IIR coefficient quantization parameters */
#define		SF				14						/* Coefficient scalefactor */
#define		SF2				16384					/* 2^SF */
#define		FLOAT2FIX(x)	(((x)*SF2)+0.5)			/* fp->int parameter conversion */

/* Reverb time established by first call via external demo interface */
#define		IRT				0.1						/* in seconds */

/* Shcroeder state variable */
typedef struct  
{
	Ipp32s *CombDelayLine;		/* Combined delay lines for parallel comb filters */
	Ipp32s *AllpassDelayLine;	/* Combined delay lines for series allpass filter */
	Ipp16s *CombOutputBuffer;	/* Parallel comb filter output buffer */
	Ipp32s *CombDelayPtr;		/* Base addresses for each comb filter delay line */
	Ipp32s *CombDelayIndex;		/* Comb filter delay line ring buffer pointers (w(n)) */
	Ipp16s *CombTaps;			/* Combined comb filter taps, all combs */
	Ipp32s *CombLags;			/* Combined comb filter lags, all combs */
	Ipp32s *AllpassDelayPtr;	/* Base addresses for each allpass delay line */
	Ipp32s *AllpassDelayIndex;	/* Allpass filter delay line ring buffer pointers (w(n)) */
	Ipp16s *AllpassTaps;		/* Combined allpass filter taps, all sections */
	Ipp32s *AllpassLags;		/* Combined allpass filter lags, all sections */
	int *CombDelays;			/* Comb filter delays, in samples */
	int *AllpassDelays;			/* Allpass filter delays, in samples */
	int SampleRate;				/* Sample frequency, in Hz */
	int	Nc;						/* Number of comb filters */
	int Na;						/* Number of allpass filters */
} SchroederState;

/* Internal prototypes */
void SchroederInit(float, int, int *, int, int *, int, int, SchroederState *);
void SchroederDestroy(SchroederState *);
void SchroederReverbTime(float, SchroederState *);
int Schroeder(Ipp16s *, Ipp16s *, int, SchroederState *);
IppStatus ippsIIR_DirectSparse_16s(Ipp16s *, Ipp16s *, int, const Ipp16s *, 
								   const Ipp32s *, int *, Ipp32s *);

/* Create one reverberator for the external demo interface */
SchroederState ReverberatorState;

/* Establish new Rt */
void SetReverbTime(float ReverbTime)
{
	SchroederReverbTime(ReverbTime,&ReverberatorState);
}

/* Apply reverberator to one data block */
void ApplyReverb(Ipp16s *in, Ipp16s *out, int len, int SampleRate)
{
	static initialized=0;
	int CombDelays[NC]={CDLY1,CDLY2,CDLY3,CDLY4};
	int AllpassDelays[NA]={APDLY1,APDLY2};

	/* Init Schroeder state variables on first invocation */
	if (!initialized)
	{
		SchroederInit(IRT,SampleRate,CombDelays,NC,AllpassDelays,NA,len,&ReverberatorState);
		initialized=1;
	}
	
	/* Apply reverberator to audio block */
	Schroeder(in,out,len,&ReverberatorState);
}

/* Free all memory allocated for Shroeder state */
void ShutdownReverb()
{
	SchroederDestroy(&ReverberatorState);
}
/************************************************************************
 *
 * End external interface  
 *
 ************************************************************************

/************************************************************************
 *
 * Shroeder Functions  
 * 
 * 1. SchroederInit	      - Allocate and init internal buffers
 * 2. SchroederDestroy	  - Free state variables
 * 3. SchroederReverbTime - Change Rt on the fly
 * 4. Schroeder           - Apply reverberator to one block
 *
 ************************************************************************/

/*********************************************************************************** 
	SchroederInit() 

	ReverbTime			-		Desired reverberation time, in seconds
	Fs					-       Sample rate, in Hz
	CombDelays			-		Comb filter delays, in milliseconds
	Nc					-       Number of comb filters
	AllpassDelays		-		Allpass filter delays, in milliseconds
	Na					-       Number of allpass filters
	BlockSize			-		Audio input/output block size
	s					-		Shroeder state variable 
************************************************************************************/
void SchroederInit(float ReverbTime, 
				   int   Fs, 
				   int	*CombDelays, 
				   int   Nc, 
				   int  *AllpassDelays, 
				   int   Na,
				   int	 BlockSize, 
				   SchroederState *s)
{
	int	i,length;
 	Ipp16s *tap;
	Ipp32s *lag;
	float g_allpass=0.5;
	
	/* Allocate state buffers */
	s->CombDelayPtr=(Ipp32s *)malloc(Nc*sizeof(Ipp32s));
	s->CombDelayIndex=(Ipp32s *)malloc(Nc*sizeof(Ipp32s));
	s->CombTaps=(Ipp16s *)malloc(5*Nc*sizeof(Ipp16s));	
	s->CombLags=(Ipp32s *)malloc(6*Nc*sizeof(Ipp32s));	
	s->AllpassDelayPtr=(Ipp32s *)malloc(Na*sizeof(Ipp32s));
	s->AllpassDelayIndex=(Ipp32s *)malloc(Na*sizeof(Ipp32s));
	s->AllpassTaps=(Ipp16s *)malloc(4*Na*sizeof(Ipp16s));	
	s->AllpassLags=(Ipp32s *)malloc(5*Na*sizeof(Ipp32s));	
	s->CombDelays=(int *)malloc(Nc*sizeof(int));
	s->AllpassDelays=(int *)malloc(Na*sizeof(int));

	/* Initialize filter counts */
	s->Nc=Nc;
	s->Na=Na;

	/* Initialize sample rate */
	s->SampleRate=Fs;

	/* Initialize comb delays */
	for(i=0;i<Nc;i++)
		s->CombDelays[i]=(float)Fs*(float)CombDelays[i]/1000.0+0.5;

	/* Initialize allpass delays */
	for(i=0;i<Na;i++)
		s->AllpassDelays[i]=(float)Fs*(float)AllpassDelays[i]/1000.0+0.5;
	
	/* Initialize parallel comb filter taps given Rt specification */
	SchroederReverbTime(ReverbTime,s);

	/* Initialize comb delay lines */
	for(length=i=0;i<Nc;i++)
		length+=((s->CombDelays)[i]+2);
	s->CombDelayLine=(Ipp32s *)malloc(length*sizeof(Ipp32s));
	ippsZero_16s((Ipp16s*)s->CombDelayLine,2*length);

	/* Initialize parallel comb filter lags */
	for(i=0;i<Nc;i++)
	{
		/* Longest lag = delay+1 */
		lag=&((s->CombLags[6*i]));
		lag[0]=2;
		lag[1]=(s->CombDelays)[i];
		lag[2]=(s->CombDelays)[i]+1;
		lag[3]=2;
		lag[4]=(s->CombDelays)[i];
		lag[5]=(s->CombDelays)[i]+1;

		/* Initialize delay line base addresses */
		if (i==0)
			(s->CombDelayPtr)[0]=(Ipp32s)&((s->CombDelayLine)[0]);
		else
			(s->CombDelayPtr)[i]=(Ipp32s)(s->CombDelayPtr)[i-1]+(((s->CombDelays)[i-1]+2)*sizeof(Ipp32s));
	}

	/* Initialize comb delay line indices */
	for(i=0;i<Nc;i++)
		(s->CombDelayIndex)[i]=0;

	/* Initialize comb output buffer */
	s->CombOutputBuffer=(Ipp16s *)malloc(BlockSize*sizeof(Ipp16s));
	for(i=0;i<BlockSize;i++)
		(s->CombOutputBuffer)[i]=0;

	/* Initialize allpass delay lines */
	for(length=i=0;i<Na;i++)
		length+=((s->AllpassDelays[i])+1);
	s->AllpassDelayLine=(Ipp32s *)malloc(length*sizeof(Ipp32s));
	ippsZero_16s((Ipp16s*)s->AllpassDelayLine,2*length);

	/* Initialize series allpass filters */
	for(i=0;i<Na;i++)
	{
		/* Initialize sparse IIR */
		/* Tap weights */
		tap=&((s->AllpassTaps[4*i]));
		tap[0]=FLOAT2FIX(-g_allpass);
		tap[1]=FLOAT2FIX(1.0);
		tap[2]=FLOAT2FIX(-g_allpass);
		tap[3]=SF;

		/* Lags */		
		/* Longest lag = delay+1 */
		lag=&((s->AllpassLags[5*i]));
		lag[0]=2;
		lag[1]=0;
		lag[2]=s->AllpassDelays[i];
		lag[3]=1;
		lag[4]=s->AllpassDelays[i];

		/* Initialize delay line base addresses */
		if (i==0)
			(s->AllpassDelayPtr)[0]=(Ipp32s)&((s->AllpassDelayLine)[0]);
		else
			(s->AllpassDelayPtr)[i]=(Ipp32s)(s->AllpassDelayPtr)[i-1]+(((s->AllpassDelays)[i-1]+1)*sizeof(Ipp32s));
	}

	/* Initialize allpass delay line indices */
	for(i=0;i<Na;i++)
		(s->AllpassDelayIndex)[i]=0;
}

/*********************************************************************************** 
	SchroederDestroy() 

	s					-		Shroeder state variable 
************************************************************************************/
void SchroederDestroy(SchroederState *s)
{
	free(s->CombDelayLine);		/* Combined delay lines for parallel comb filters */
	free(s->AllpassDelayLine);	/* Combined delay lines for series allpass filter */
	free(s->CombOutputBuffer);	/* Parallel comb filter output buffer */
	free(s->CombDelayPtr);		/* Base addresses for each comb filter delay line */
	free(s->CombDelayIndex);	/* Comb filter delay line ring buffer pointers (w(n)) */
	free(s->CombTaps);			/* Combined comb filter taps, all combs */
	free(s->CombLags);			/* Combined comb filter lags, all combs */
	free(s->AllpassDelayPtr);	/* Base addresses for each allpass delay line */
	free(s->AllpassDelayIndex);	/* Allpass filter delay line ring buffer pointers (w(n)) */
	free(s->AllpassTaps);		/* Combined allpass filter taps, all sections */
	free(s->AllpassLags);		/* Combined allpass filter lags, all sections */
	free(s->CombDelays);		/* Comb filter delays, in samples */
	free(s->AllpassDelays);		/* Allpass filter delays, in samples */
}

/*********************************************************************************** 
	SchroederReverbTime() 

	ReverbTime			-       Desired reverb time
	s					-		Shroeder state variable 
************************************************************************************/
void SchroederReverbTime(float ReverbTime, SchroederState *s)
{
 	Ipp16s *tap;
	float alpha,g;
	float g1,g2,t1;
	int i;

	alpha=0.25;
	for(i=0;i<s->Nc;i++)
	{
		g=pow(10.0,((-3.0*(float)(s->CombDelays)[i])/(ReverbTime*(float)(s->SampleRate))));
		t1=pow(g,1.0-1.0/alpha);
		g1=1.0-(2.0/(1.0+t1));
		g2=g*(1.0-g1);

		/* Initialize comb (sparse IIR) tap weights */
		tap=&((s->CombTaps[5*i]));
		tap[0]=FLOAT2FIX(g2);
		tap[1]=FLOAT2FIX(g1*g2);
		tap[2]=FLOAT2FIX(-g2);
		tap[3]=FLOAT2FIX(-g1*g2);
		tap[4]=SF;
	}
}

/*********************************************************************************** 
	Schroeder() 

	x					-       Input vector
	y					-       Output vector
	len					-       Vector lengths
	s					-		Shroeder state variable 
************************************************************************************/
int Schroeder(Ipp16s *x, Ipp16s *y, int len, SchroederState *s)
{
	int i;
	Ipp16s *CombOutputBuffer=s->CombOutputBuffer;
	unsigned long t1,t2;							/* Timer temps */

	/* Clear output buffer */
	ippsZero_16s(y,len);

	START_TIMER
	/* Apply parallel comb filters */
	for(i=0;i<s->Nc;i++)
	{
		ippsIIR_DirectSparse_16s(x,CombOutputBuffer,len,
			                     &((s->CombTaps)[5*i]),
								 &((s->CombLags)[6*i]),
								 &((s->CombDelayIndex)[i]),
								 (Ipp32s *)(s->CombDelayPtr)[i] ); 

		/* Combine comb outputs */
		ippsAdd_16s_I(CombOutputBuffer,y,len);
	}		
	STOP_TIMER(T_COMB)

	START_TIMER
	/* Apply series allpass filters */
	for(i=0;i<s->Na;i++)
		ippsIIR_DirectSparse_16s(y,y,len,
			                     &((s->AllpassTaps)[4*i]),
								 &((s->AllpassLags)[5*i]),
								 &((s->AllpassDelayIndex)[i]),
								 (Ipp32s *)(s->AllpassDelayPtr)[i] ); 
	STOP_TIMER(T_AP)

	/* Add reverberant sound to the input */
	ippsAdd_16s_I(x,y,len);
	return 0;
}

#ifdef PERFORMANCE_TIMER
/* 
    Compute CPU cost associated with a particular timer   

	n				- timer index (n<0 gives number of timers available)
	N				- block size
	SampleRate		- sample rate
	Desc			- timer label (w_str)
*/
int GetPerformanceTimer(int n, int N, int SampleRate, char *Desc)
{
	Ipp64s PercentageCPU;
	int val;

	/* Return number of timers, otherwise %CPU associated with a particular timer */
	if (n<0)
		return(NUM_TIMERS);
	else
	{
		wcscpy(Desc,g_TimerDesc[n]);
		PercentageCPU=(Ipp64s)g_TimerOSCR[n]*(Ipp64s)SampleRate*100/(Ipp64s)(g_NumBlocks[n]*N)/FC;
		val=(int)PercentageCPU;
		return(val);
	}
}

/* Clear performance timers */
void ResetPerformanceTimers(void)
{
	int i;
	for(i=0;i<NUM_TIMERS;i++)
		RESET_TIMER(i);
}

/* Open OSCR timer device */
void InitTimer()
{
	ixs_open_OSCR();
}

/* Close OSCR timer device */
void ShutdownTimer()
{
	ixs_close_OSCR();
}

#endif


