/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "fullscreen.h"

#include "image.h"

#ifdef HAVE_X11_EXTENSIONS_XINERAMA_H
  #include <X11/Intrinsic.h>
  #include <X11/extensions/Xinerama.h>
  #include <gdk/gdkx.h>
#endif


/*
 *----------------------------------------------------------------------------
 * full screen functions
 *----------------------------------------------------------------------------
 */

static void clear_mouse_cursor(GtkWidget *widget, gint clear)
{
	if (clear)
		{
		GdkCursor *cursor;
		GdkPixmap *p;

		p = gdk_bitmap_create_from_data(widget->window, "\0\0\0", 1, 1);

		cursor = gdk_cursor_new_from_pixmap(p, p,
						    &widget->style->fg[GTK_STATE_ACTIVE],
						    &widget->style->bg[GTK_STATE_ACTIVE],
						    0, 0);

		gdk_window_set_cursor (widget->window, cursor);

		gdk_cursor_unref(cursor);
		gdk_pixmap_unref(p);
		}
	else
		{
		gdk_window_set_cursor (widget->window, NULL);
		}
}

static gint fullscreen_hide_mouse_cb(gpointer data)
{
	FullScreenData *fs = data;

	if (fs->hide_mouse_id == -1) return FALSE;

	clear_mouse_cursor(fs->window, TRUE);

	fs->hide_mouse_id = -1;
	return FALSE;
}

static void fullscreen_hide_mouse_disable(FullScreenData *fs)
{
	if (fs->hide_mouse_id != -1)
		{
		gtk_timeout_remove(fs->hide_mouse_id);
		fs->hide_mouse_id = -1;
		}
}

static void fullscreen_hide_mouse_reset(FullScreenData *fs)
{
	fullscreen_hide_mouse_disable(fs);
	fs->hide_mouse_id = gtk_timeout_add(FULL_SCREEN_HIDE_MOUSE_DELAY, fullscreen_hide_mouse_cb, fs);
}

static gint fullscreen_mouse_moved(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	FullScreenData *fs = data;

	clear_mouse_cursor(fs->window, FALSE);
	fullscreen_hide_mouse_reset(fs);

	return FALSE;
}

static gint fullscreen_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	FullScreenData *fs = data;

	fullscreen_stop(fs);
	return TRUE;
}

FullScreenData *fullscreen_start(GtkWidget *window, ImageWindow *imd,
				 void (*stop_func)(FullScreenData *, gpointer), gpointer stop_data)
{
	FullScreenData *fs;
	gint x;
	gint y;
	gint w;
	gint h;
#ifdef HAVE_X11_EXTENSIONS_XINERAMA_H
	int foo, bar;
	Display *display;
#endif

	if (!window || !imd) return NULL;

	fs = g_new0(FullScreenData, 1);

	fs->hide_mouse_id = -1;
	fs->normal_window = window;
	fs->normal_imd = imd;

	fs->stop_func = stop_func;
	fs->stop_data = stop_data;

	x = 0;
	y = 0;
	w = gdk_screen_width();
	h = gdk_screen_height();

#ifdef HAVE_X11_EXTENSIONS_XINERAMA_H
	display = GDK_DISPLAY();

	if (XineramaIsActive(display) &&
	    XineramaQueryExtension(display, &foo, &bar))
		{
		XineramaScreenInfo *xinerama;
		int xinerama_count;
		int i;

		gdk_window_get_position(window->window, &x, &y);

		xinerama = XineramaQueryScreens(display, &xinerama_count);

		/* check which physical screen we are visible on */
		for (i = 0; i < xinerama_count; i++)
			{
			if (x >= xinerama[i].x_org &&
			    y >= xinerama[i].y_org &&
			    x <  xinerama[i].x_org + xinerama[i].width &&
			    y <  xinerama[i].y_org + xinerama[i].height)
				{
				x = xinerama[i].x_org;
				y = xinerama[i].y_org;
				w = xinerama[i].width;
				h = xinerama[i].height;
				if (debug)
					{
					printf("Xinerama screen %d, at %d,%d (%d x %d)\n", i, x, y, w, h);
					}
				break;
				}
			}
		}
#endif

	fs->window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_container_set_border_width(GTK_CONTAINER(fs->window), 0);
	g_signal_connect(G_OBJECT(fs->window), "delete_event",
			 G_CALLBACK(fullscreen_delete_cb), fs);

	gtk_window_set_title(GTK_WINDOW(fs->window), _("GQview full screen"));
	gtk_window_set_default_size(GTK_WINDOW(fs->window), w, h);
	gtk_window_move(GTK_WINDOW(fs->window), x, y);

	fs->imd = image_new(FALSE);

	gtk_container_add(GTK_CONTAINER(fs->window), fs->imd->widget);

	/* set background to black */
	if (BLACK_BACKGROUND)
		{
		image_background_set_black(fs->imd, TRUE);
		}

	image_auto_refresh(fs->imd, fs->normal_imd->auto_refresh_interval);

	gtk_widget_show(fs->imd->widget);

	image_change_from_image(fs->imd, fs->normal_imd);

	gtk_window_set_modal(GTK_WINDOW(fs->window), TRUE);
	gtk_widget_show(fs->window);

	gdk_keyboard_grab(fs->window->window, TRUE, GDK_CURRENT_TIME);
	gtk_grab_add(fs->window);
	gtk_widget_grab_focus(fs->window);

	/* for hiding the mouse */
	g_signal_connect(G_OBJECT(fs->imd->image), "motion_notify_event",
			   G_CALLBACK(fullscreen_mouse_moved), fs);
	clear_mouse_cursor(fs->window, TRUE);
#if 0
	fullscreen_hide_mouse_reset(fs);
#endif

	/* hide normal window
	 * FIXME: properly restore this window on show
	 */
#ifdef HIDE_WINDOW_IN_FULLSCREEN
	gtk_widget_hide(fs->normal_window);
#endif
	image_change_path(fs->normal_imd, NULL, image_zoom_get(fs->normal_imd));

	return fs;
}

void fullscreen_stop(FullScreenData *fs)
{
	if (!fs) return;

	fullscreen_hide_mouse_disable(fs);
	gdk_keyboard_ungrab(GDK_CURRENT_TIME);

	image_change_from_image(fs->normal_imd, fs->imd);
#ifdef HIDE_WINDOW_IN_FULLSCREEN
	gtk_widget_show(fs->normal_window);
#endif
	if (fs->stop_func) fs->stop_func(fs, fs->stop_data);

	gtk_grab_remove(fs->window);

	gtk_widget_destroy(fs->window);

	g_free(fs);
}

