/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2002  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2002  Haavard Kvaalen
 *
 *  Copyright (C) 2003	     David Lau
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#define G_LOG_DOMAIN "skin.c"

/* TODO: enforce default sizes! */

#include "beep.h"
#include <ctype.h>

#define DEFAULT_SKIN_PATH DATA_DIR "/Skins/Default"

#ifndef HAVE_MKDTEMP
char *mkdtemp(char *path);
#endif

Skin *skin;

static int skin_current_num;

typedef struct {
    const gchar *name;
    gint id;
    const gchar *alt_img;
    gint width, height;
} SMID_Mapping;

typedef struct {
    gint width, height;
    gchar *inistr;
} SkinMaskInfo;

SkinMaskInfo MaskInfo[] = {
    {275, 116, "Normal"},
    {275, 116, "Equalizer"},
    {275, 16, "WindowShade"},
    {275, 16, "EqualizerWS"}
};

SMID_Mapping Mappings[] = {
    {"main", SKIN_MAIN, NULL},
    {"cbuttons", SKIN_CBUTTONS, NULL},
    {"shufrep", SKIN_SHUFREP, NULL},
    {"text", SKIN_TEXT, NULL},
    {"titlebar", SKIN_TITLEBAR, NULL},
    {"volume", SKIN_VOLUME, NULL},
    {"balance", SKIN_BALANCE, "volume"},
    {"monoster", SKIN_MONOSTEREO, NULL},
    {"playpaus", SKIN_PLAYPAUSE, NULL},
    {"nums_ex", SKIN_NUMBERS, "numbers"},
    {"posbar", SKIN_POSBAR, NULL},
    {"eqmain", SKIN_EQMAIN, NULL},
    {"pledit", SKIN_PLEDIT, NULL},
    {"eq_ex", SKIN_EQ_EX, NULL}
};

static const guchar skin_default_viscolor[24][3] = {
    {9, 34, 53},
    {10, 18, 26},
    {0, 54, 108},
    {0, 58, 116},
    {0, 62, 124},
    {0, 66, 132},
    {0, 70, 140},
    {0, 74, 148},
    {0, 78, 156},
    {0, 82, 164},
    {0, 86, 172},
    {0, 92, 184},
    {0, 98, 196},
    {0, 104, 208},
    {0, 110, 220},
    {0, 116, 232},
    {0, 122, 244},
    {0, 128, 255},
    {0, 128, 255},
    {0, 104, 208},
    {0, 80, 160},
    {0, 56, 112},
    {0, 32, 64},
    {200, 200, 200}
};
GdkBitmap *skin_create_transparent_mask(const gchar *, const gchar *,
					const gchar *, GdkWindow *, gint,
					gint, gboolean);


Skin *skin_new(void)
{
    Skin *retval;
    retval = (Skin *) g_malloc0(sizeof(Skin));
    retval->lock = g_mutex_new();
    return retval;
}

void free_skin(void);

void skin_destroy(Skin * sk)
{
    free_skin();
    g_mutex_free(sk->lock);
    g_free(sk);
}

const SMID_Mapping *smid_lookup(gint id)
{
    gint len = sizeof(Mappings) / sizeof(SMID_Mapping);
    gint indx;
    SMID_Mapping *retval = NULL;
    for (indx = 0; indx < len; indx++) {
	if (id == Mappings[indx].id) {
	    retval = &Mappings[indx];
	    break;
	}
    }

    return retval;
}

const gchar *smid_to_name(gint id)
{
    int indx;
    gint len = sizeof(Mappings) / sizeof(SMID_Mapping);

    for (indx = 0; indx < len; indx++) {
	if (id == Mappings[indx].id)
	    return Mappings[indx].name;
    }
    return NULL;
}

/* 
static void skin_set_def_vis_color(Skin* sk)
{
	gint i;
	for (i = 0; i < 24; i++)
	{
		sk->vis_color[i][0] = skin_default_viscolor[i][0];
		sk->vis_color[i][1] = skin_default_viscolor[i][1];
		sk->vis_color[i][2] = skin_default_viscolor[i][2];
	}
	return;
}
*/
static gboolean skin_load_pixmap_id(Skin * sk, gint id, const gchar * pth)
{
    const gchar *name, *path;
    gchar *tmpstr, *filename;
    gboolean trying_alternate = FALSE;
    guint w, h;
    const SMID_Mapping *sm_inf;
    GdkPixmap *gpm;
    gboolean retval = FALSE;

    if (id >= SKIN_PMID_LAST) {
	g_warning("skin pixmap id out of range (%u).", id);
	return retval;
    }

    path = pth ? pth : sk->path;
    sm_inf = smid_lookup(id);
    if (!sm_inf)
	return retval;

    name = sm_inf->name;

  TRY_ALTERNATE:
    tmpstr = g_strdup_printf("%s.bmp", name);

    filename = find_file_recursively(path, tmpstr);
    if (!filename) {

	g_warning("couldn't locate %s in directory %s", tmpstr, path);
	g_free(tmpstr);

	if (!trying_alternate) {
	    if (sm_inf->alt_img != NULL) {
		name = sm_inf->alt_img;
		trying_alternate = TRUE;
		goto TRY_ALTERNATE;
	    }
	}
	return retval;
    }
    g_free(tmpstr);

    SkinPixmap *pm = &sk->pixmaps[id];
    gpm = read_bmp(filename);

    if (!gpm) {
	g_warning("loading of %s failed", filename);
    }

    g_free(filename);
    if (!gpm)
	return retval;

    pm->pixmap = gpm;
    gdk_window_get_size(pm->pixmap, &w, &h);
    pm->width = w;
    pm->height = h;
    pm->current_width = w;	//MIN(w, pm->width);
    pm->current_height = h;	//MIN(h, pm->height);]

    retval = TRUE;
    return retval;
}

void skin_mask_create(Skin * sk, const gchar * path, int id,
		      GdkWindow * win)
{
    sk->masks[id] =
	skin_create_transparent_mask(path, "region.txt",
				     MaskInfo[id].inistr, win,
				     MaskInfo[id].width,
				     MaskInfo[id].height, FALSE);

    sk->ds_masks[id] =
	skin_create_transparent_mask(path, "region.txt",
				     MaskInfo[id].inistr, win,
				     MaskInfo[id].width * 2,
				     MaskInfo[id].height * 2, TRUE);
    return;
}

static gboolean load_skin_pixmaps2(Skin * sk, const gchar * path)
{

    gboolean retval = TRUE;
    int cur_pmid;
    for (cur_pmid = 0; cur_pmid < SKIN_PMID_LAST; cur_pmid++) {
	skin_load_pixmap_id(sk, cur_pmid, path);
    }

    return retval;
}

static void setup_skin_masks(void)
{
    if (cfg.show_wm_decorations)
	return;
    if (cfg.player_visible) {
	gtk_widget_shape_combine_mask(mainwin,
				      skin_get_mask(SKIN_MASK_MAIN,
						    cfg.doublesize,
						    cfg.player_shaded), 0,
				      0);
    }

    gtk_widget_shape_combine_mask(equalizerwin,
				  skin_get_mask(SKIN_MASK_EQ,
						EQUALIZER_DOUBLESIZE,
						cfg.equalizer_shaded), 0,
				  0);
}

static GdkBitmap *create_default_mask(GdkWindow * parent, gint w, gint h)
{
    GdkBitmap *ret;
    GdkGC *gc;
    GdkColor pattern;

    ret = gdk_pixmap_new(parent, w, h, 1);
    gc = gdk_gc_new(ret);
    pattern.pixel = 1;
    gdk_gc_set_foreground(gc, &pattern);
    gdk_draw_rectangle(ret, gc, TRUE, 0, 0, w, h);
    gdk_gc_destroy(gc);

    return ret;
}

static void skin_query_color(GdkColormap * cm, GdkColor * c)
{
    XColor xc = { 0 };

    xc.pixel = c->pixel;
    XQueryColor(GDK_COLORMAP_XDISPLAY(cm), GDK_COLORMAP_XCOLORMAP(cm),
		&xc);
    c->red = xc.red;
    c->green = xc.green;
    c->blue = xc.blue;
}

static glong skin_calc_luminance(GdkColor * c)
{
    return (0.212671 * c->red + 0.715160 * c->green + 0.072169 * c->blue);
}

static void skin_get_textcolors(GdkPixmap * text, GdkColor * bgc,
				GdkColor * fgc)
{
    /*
     * Try to extract reasonable background and foreground colors
     * from the font pixmap
     */

    GdkImage *gi;
    GdkColormap *cm;
    int i;

    if (text == NULL)
	return;

    /* Get the first line of text */
    gi = gdk_image_get(text, 0, 0, 152, 6);
    cm = gdk_window_get_colormap(playlistwin->window);
    g_assert(GDK_IS_WINDOW(playlistwin->window));
    for (i = 0; i < 6; i++) {
	GdkColor c;
	gint x;
	glong d, max_d;

	/* Get a pixel from the middle of the space character */
	bgc[i].pixel = gdk_image_get_pixel(gi, 151, i);
	skin_query_color(cm, &bgc[i]);

	max_d = 0;
	for (x = 1; x < 150; x++) {
	    c.pixel = gdk_image_get_pixel(gi, x, i);
	    skin_query_color(cm, &c);

	    d = labs(skin_calc_luminance(&c) -
		     skin_calc_luminance(&bgc[i]));
	    if (d > max_d) {
		memcpy(&fgc[i], &c, sizeof(GdkColor));
		max_d = d;
	    }
	}
    }
    gdk_image_destroy(gi);
}


void init_skins(const gchar * fskin)
{
    skin = skin_new();
    //skin_get_textcolors(skin->pixmaps[SKIN_TEXT].def_pixmap, skin->def_textbg, skin->def_textfg);

    if (fskin == NULL)
	fskin = DEFAULT_SKIN_PATH;

    load_skin(fskin);
    setup_skin_masks();
    create_skin_window();
}

static guint hex_chars_to_int(gchar c, gchar d)
{
    /*
     * Converts a value in the range 0x00-0xFF
     * to a integer in the range 0-65535
     */
    gchar str[3];

    str[0] = c;
    str[1] = d;
    str[2] = '\0';

    return (CLAMP(strtol(str, NULL, 16), 0, 255) * 256);
}

GdkColor *load_skin_color(const gchar * path, const gchar * file,
			  const gchar * section, const gchar * key)
{
    gchar *filename, *value;
    GdkColor *color = NULL;

    filename = find_file_recursively(path, file);
    if (filename) {
	value = read_ini_string(filename, section, key);
	if (value) {
	    gchar *ptr = value;
	    gint len;
	    color = g_malloc0(sizeof(GdkColor));
	    g_strchug(g_strchomp(value));
	    if (value[0] == '#')
		ptr++;
	    len = strlen(ptr);

	    /*
	     * The handling of incomplete values is done this way
	     * to maximize winamp compatibility
	     */
	    if (len >= 6) {
		color->red = hex_chars_to_int(*ptr, *(ptr + 1));
		ptr += 2;
	    }
	    if (len >= 4) {
		color->green = hex_chars_to_int(*ptr, *(ptr + 1));
		ptr += 2;
	    }
	    if (len >= 2)
		color->blue = hex_chars_to_int(*ptr, *(ptr + 1));


	    gdk_color_alloc(gdk_window_get_colormap(playlistwin->window),
			    color);
	    g_free(value);
	}
	g_free(filename);
    }
    return color;
}



GdkBitmap *skin_create_transparent_mask(const gchar * path,
					const gchar * file,
					const gchar * section,
					GdkWindow * window, gint width,
					gint height, gboolean doublesize)
{
    GdkBitmap *mask = NULL;
    GdkGC *gc = NULL;
    GdkColor pattern;
    GdkPoint *gpoints;

    gchar *filename = NULL;
    gboolean created_mask = FALSE;
    GArray *num, *point;
    gint i, j, k;

    if (path)
	filename = find_file_recursively(path, file);
    /* filename will be null if path wasn't set */
    if (!filename) {
	return create_default_mask(window, width, height);
    }

    if ((num = read_ini_array(filename, section, "NumPoints")) == NULL) {
	g_free(filename);
	return NULL;
    }

    if ((point = read_ini_array(filename, section, "PointList")) == NULL) {
	g_array_free(num, TRUE);
	g_free(filename);
	return NULL;
    }

    mask = gdk_pixmap_new(window, width, height, 1);
    gc = gdk_gc_new(mask);

    pattern.pixel = 0;
    gdk_gc_set_foreground(gc, &pattern);
    gdk_draw_rectangle(mask, gc, TRUE, 0, 0, width, height);
    pattern.pixel = 1;
    gdk_gc_set_foreground(gc, &pattern);

    j = 0;
    for (i = 0; i < num->len; i++) {
	if ((point->len - j) >= (g_array_index(num, gint, i) * 2)) {
	    created_mask = TRUE;
	    gpoints =
		g_malloc(g_array_index(num, gint, i) * sizeof(GdkPoint));
	    for (k = 0; k < g_array_index(num, gint, i); k++) {
		gpoints[k].x =
		    g_array_index(point, gint,
				  j + k * 2) * (1 + doublesize);
		gpoints[k].y =
		    g_array_index(point, gint,
				  j + k * 2 + 1) * (1 + doublesize);
	    }
	    j += k * 2;
	    gdk_draw_polygon(mask, gc, TRUE, gpoints,
			     g_array_index(num, gint, i));
	    g_free(gpoints);
	}
    }
    g_array_free(num, TRUE);
    g_array_free(point, TRUE);
    g_free(filename);

    if (!created_mask)
	gdk_draw_rectangle(mask, gc, TRUE, 0, 0, width, height);

    gdk_gc_destroy(gc);

    return mask;
}

void load_skin_viscolor(const gchar * path, const gchar * file)
{
    FILE *f;
    gint i, c;
    gchar line[256], *filename;
    GArray *a;

    for (i = 0; i < 24; i++) {
	skin->vis_color[i][0] = skin_default_viscolor[i][0];
	skin->vis_color[i][1] = skin_default_viscolor[i][1];
	skin->vis_color[i][2] = skin_default_viscolor[i][2];
    }

    filename = find_file_recursively(path, file);
    if (!filename)
	return;

    if ((f = fopen(filename, "r")) == NULL) {
	g_free(filename);
	return;
    }

    for (i = 0; i < 24; i++) {
	if (fgets(line, 255, f)) {
	    a = string_to_garray(line);
	    if (a->len > 2) {
		for (c = 0; c < 3; c++)
		    skin->vis_color[i][c] = g_array_index(a, gint, c);
	    }
	    g_array_free(a, TRUE);
	} else
	    break;

    }
    fclose(f);
    g_free(filename);
}

/*
static void skin_numbers_generate_dash(SkinPixmap *numbers)
{
	GdkGC *gc;
	GdkPixmap *pixmap;

	if (numbers->pixmap == NULL ||
	    numbers->current_width < 99)
		return;

	gc = gdk_gc_new(numbers->pixmap);
	pixmap = gdk_pixmap_new(mainwin->window, 108,
				numbers->current_height,
				gdk_rgb_get_visual()->depth);
	skin_draw_pixmap(pixmap, gc, SKIN_NUMBERS,
			 0, 0, 0, 0, 99, 13);
	skin_draw_pixmap(pixmap, gc, SKIN_NUMBERS,
			 90, 0, 99, 0, 9, 13);
	skin_draw_pixmap(pixmap, gc, SKIN_NUMBERS,
			 20, 6, 101, 6, 5, 1);
	gdk_gc_unref(gc);
	gdk_pixmap_unref(numbers->pixmap);
	numbers->pixmap = pixmap;
	numbers->current_width = 108;
}
*/

static void skin_free_pixmap(SkinPixmap * p)
{
    if (p->pixmap) {
	gdk_pixmap_unref(p->pixmap);
    }
    p->pixmap = NULL;
}

void free_skin(void)
{
    gint i;
    return;
    for (i = 0; i < SKIN_PMID_LAST; i++) {
	skin_free_pixmap(&skin->pixmaps[i]);
    }

    for (i = 0; i < SKIN_MID_LAST; i++) {
	if (skin->masks[i])
	    gdk_bitmap_unref(skin->masks[i]);
	if (skin->ds_masks[i])
	    gdk_bitmap_unref(skin->ds_masks[i]);

	skin->masks[i] = NULL;
	skin->ds_masks[i] = NULL;
    }

    for (i = 0; i < 24; i++) {
	skin->vis_color[i][0] = skin_default_viscolor[i][0];
	skin->vis_color[i][1] = skin_default_viscolor[i][1];
	skin->vis_color[i][2] = skin_default_viscolor[i][2];
    }
}

static void skin_load_pixmaps(const char *path)
{
    GdkPixmap *text_pm;
    //gchar *tmp_path = skin->path;
    //skin->path = path; /* watch out!! skin->path is not const */
    load_skin_pixmaps2(skin, path);
    //skin->path=tmp_path;


/*	if (skin->pixmaps[SKIN_TEXT].current_width < 152 || 
	    skin->pixmaps[SKIN_TEXT].current_height<6)
		text_pm = skin->pixmaps[SKIN_TEXT].def_pixmap;
	else*/
    text_pm = skin->pixmaps[SKIN_TEXT].pixmap;

    if (text_pm)
	skin_get_textcolors(text_pm, skin->textbg, skin->textfg);


    if (skin->pixmaps[SKIN_NUMBERS].pixmap == NULL)
	g_print("toublblb~!\n");
    skin->colors[SKIN_PLEDIT_NORMAL] =
	load_skin_color(path, "pledit.txt", "text", "normal");
    skin->colors[SKIN_PLEDIT_CURRENT] =
	load_skin_color(path, "pledit.txt", "text", "current");
    skin->colors[SKIN_PLEDIT_NORMALBG] =
	load_skin_color(path, "pledit.txt", "text", "normalbg");
    skin->colors[SKIN_PLEDIT_SELECTEDBG] =
	load_skin_color(path, "pledit.txt", "text", "selectedbg");


    skin_mask_create(skin, path, SKIN_MASK_MAIN, mainwin->window);
    skin_mask_create(skin, path, SKIN_MASK_SHADE, mainwin->window);

    skin_mask_create(skin, path, SKIN_MASK_EQ, equalizerwin->window);
    skin_mask_create(skin, path, SKIN_MASK_EQ_SHADE, equalizerwin->window);

    load_skin_viscolor(path, "viscolor.txt");
    return;
}

static gboolean _load_skin(Skin * skin, const gchar * path, gboolean force)
{
    gboolean retval = FALSE;
    g_assert(skin != NULL);
    if (!force) {
	if (skin->path && path)
	    if (!strcmp(skin->path, path))
		return retval;
	if (!skin->path && !path)
	    return retval;
	free_skin();
    }

    skin_current_num++;
    if (path) {
	skin->path = g_realloc(skin->path, strlen(path) + 1);
	strcpy(skin->path, path);

	if (file_is_archive(path)) {
	    char *cpath = decompress_archive(path);
	    if (cpath != NULL) {
		skin_load_pixmaps(cpath);
		del_directory(cpath);
		g_free(cpath);
	    }
	} else
	    skin_load_pixmaps(path);
    } else {
	if (skin->path)
	    g_free(skin->path);
	skin->path = NULL;
    }

    setup_skin_masks();

    draw_main_window(TRUE);
    draw_playlist_window(TRUE);
    draw_equalizer_window(TRUE);

    playlistwin_update_list();
    retval = TRUE;
    return retval;
}

void skin_install_skin(const gchar * path)
{
    char *tmp, *dest;

    dest = g_build_filename(g_get_home_dir(), BMP_RCPATH, "Skins", NULL);
    tmp = g_strdup_printf("cp %s %s", path, dest);
    if (system(tmp)) {
	g_warning("Unable to install \"%s\" to \"%s\"", path, dest);
    }
    g_free(tmp);
    g_free(dest);
}


void load_skin(const gchar * path)
{

    g_mutex_lock(skin->lock);
    _load_skin(skin, path, FALSE);
    g_mutex_unlock(skin->lock);
}

void reload_skin(void)
{
    _load_skin(skin, skin->path, TRUE);
}

/*
void cleanup_skins(void)
{
	free_skin();
}
*/

void skins_cleanup()
{
    if (!skin)
	return;
    skin_destroy(skin);
}

static SkinPixmap *get_skin_pixmap(SkinIndex si)
{

    if (si < SKIN_PMID_LAST)
	return &skin->pixmaps[si];
    g_error("Unable to find skin pixmap");
    return NULL;
}

GdkBitmap *skin_get_mask(MaskIndex mi, gboolean doublesize,
			 gboolean shaded)
{
    g_assert(skin != NULL);

    GdkBitmap **masks;

    g_assert(mi < SKIN_MID_LAST);
    masks = (doublesize ? skin->ds_masks : skin->masks);
    return masks[mi];
}

GdkColor *get_skin_color(SkinColorIndex si)
{
    GdkColor *ret = NULL;

    switch (si) {
    case SKIN_TEXTBG:
	if (skin->pixmaps[SKIN_TEXT].pixmap)
	    ret = skin->textbg;
	else
	    ret = skin->def_textbg;
	break;
    case SKIN_TEXTFG:
	if (skin->pixmaps[SKIN_TEXT].pixmap)
	    ret = skin->textfg;
	else
	    ret = skin->def_textfg;
	break;
    default:
	if (si < SKIN_COLORID_LAST)
	    ret = skin->colors[si];
	break;
    }
    return ret;
}

void get_skin_viscolor(guchar vis_color[24][3])
{
    gint i;

    for (i = 0; i < 24; i++) {
	vis_color[i][0] = skin->vis_color[i][0];
	vis_color[i][1] = skin->vis_color[i][1];
	vis_color[i][2] = skin->vis_color[i][2];
    }
}

int skin_get_id(void)
{
    return skin_current_num;
}

void skin_draw_pixmap(GdkDrawable * drawable, GdkGC * gc, SkinIndex si,
		      gint xsrc, gint ysrc, gint xdest, gint ydest,
		      gint width, gint height)
{
    SkinPixmap *pixmap = get_skin_pixmap(si);
    GdkPixmap *tmp = pixmap->pixmap;
    if (tmp == NULL)
	return;

    if (pixmap->pixmap != NULL) {
	if (xsrc > pixmap->width || ysrc > pixmap->height)
	    return;

	tmp = pixmap->pixmap;
	width = MIN(width, pixmap->width - xsrc);
	height = MIN(height, pixmap->height - ysrc);
    } else
	return;
    if (tmp != NULL) {
	gdk_draw_pixmap(drawable, gc, tmp, xsrc, ysrc, xdest, ydest, width,
			height);
    }
    return;
}

void skin_get_eq_spline_colors(guint32(*colors)[19])
{
    gint i;
    GdkPixmap *pixmap;
    GdkImage *img;
    g_assert(skin != NULL);
    SkinPixmap *eqmainpm = &skin->pixmaps[SKIN_EQMAIN];
    if (eqmainpm->pixmap != NULL &&
	eqmainpm->current_width >= 116 && eqmainpm->current_height >= 313)
	pixmap = eqmainpm->pixmap;
    else
	return;
    if (!GDK_IS_DRAWABLE(pixmap))
	return;
    img = gdk_image_get(pixmap, 115, 294, 1, 19);
    if (!img)
	return;

    for (i = 0; i < 19; i++)
	(*colors)[i] = gdk_image_get_pixel(img, 0, i);

    gdk_image_destroy(img);
    return;
}
