#include <stdio.h>
#include <stdlib.h>
#include "DZasm.h"

extern enum truefalse	debug;
extern long		Codesize;
extern FILE		*logfile;
extern DZarea		*gExtern;		/* list	of extern areas	*/
extern DZarea		*gAreas;		/* list	of areas currently examined */

char			*gAreaTypes[] =	{ "void", "program", "addrtable", "string", "data", "nop", "mth" };

DZarea			*InitArea(long	startaddr, long	 endaddr, enum atype  t);
DZarea			*NewArea(void);
void			JoinAreas(DZarea  *currarea);
enum atype		SearchArea(DZarea  *currarea, unsigned short  pc);
int			InsertArea(struct area **arealist, long	 startrange, long  endrange, enum atype	 t);
void			DispAreas(FILE *out, DZarea *arealist);
void			ValidateAreas(DZarea  *currarea);
void			DispVoidAreas(FILE  *out, DZarea *arealist);
float			ResolvedAreas(void);


void	DispAreas(FILE	*out, DZarea *arealist)
{
	DZarea		*currarea;
	int		counter;

	currarea = arealist;		  /* point at first area */
	counter	= 0;

	while(currarea != NULL)	{
		if (counter++ %	3 == 0)	fputc('\n',out);
		fprintf(out, "%04Xh-%04Xh [%s]\t",
			currarea->start, currarea->end,	gAreaTypes[currarea->areatype]);
		currarea = currarea->nextarea;		/* next	area...	*/
	}
}


float	ResolvedAreas(void)
{
	float		totalarea = 0;
	DZarea		*currarea;

	currarea = gAreas;		/* pointer to first area */
	while(currarea != NULL)	{
		if (currarea->areatype != vacuum) totalarea += currarea->end - currarea->start + 1;
		currarea = currarea->nextarea;
	}

	return totalarea*100/Codesize;
}


void	DispVoidAreas(FILE  *out, DZarea *arealist)
{
	DZarea		*currarea;
	int		counter;

	currarea = arealist;		/* point at first area */
	counter	= 0;

	while(currarea != NULL)	{
		if (currarea->areatype == vacuum) {
			if (counter++ %	3 == 0)	putchar('\n');
			fprintf(out, "%04Xh-%04Xh [%s]\t",
				currarea->start, currarea->end,	gAreaTypes[currarea->areatype]);
		}
		currarea = currarea->nextarea;		/* area	not found, check next */
	}
	fputc('\n', out);
}



enum atype	SearchArea(DZarea  *currarea, unsigned short  pc)
{
	while(currarea != NULL)	{
		if (pc<=currarea->end) {
			if (pc>=currarea->start)
				return currarea->areatype;
		}
		currarea = currarea->nextarea;		/* area	not found, check next */
	}

	return notfound;       /* area not found */
}


/* insert area (program	or data) into an area list */
int	InsertArea(DZarea **arealist, long  startrange,	long  endrange,	enum atype	t)
{
	DZarea	   *newarea, *currarea,	*tmparea;

	newarea	= InitArea(startrange, endrange, t);
	if (newarea==NULL) return 0;

	if (debug == true)
		fprintf(logfile, "Inserting [%04xh - %04xh] area\n", startrange, endrange);

	if (*arealist==NULL)
		*arealist = newarea;		/* first area in list */
	else {
		currarea = *arealist;		/* point at first subarea */

		for(;;)				/* parse list for entry	of new area */
		{
			if (newarea->start > currarea->end) {
				if (currarea->nextarea == NULL)	{
					if (debug == true) fputs("new area appended to end of list", logfile);
					/* append newarea to end of list */
					currarea->nextarea = newarea;
					newarea->prevarea = currarea;
					break;	/* exit	search loop */
				}
				else {
					currarea = currarea->nextarea;	/* examine next	sub-area in list */
				}
			}
			else {
				if (newarea->start > currarea->start) {
					if (newarea->end < currarea->end) {
						if (debug == true) fputs("split current area and insert newarea in between\n", logfile);
						tmparea	= InitArea(newarea->end+1, currarea->end, currarea->areatype);
						if (tmparea == NULL) {
							free(newarea);	/* Ups - no more room */
							return 0;
						}

						if (currarea->nextarea != NULL)
							currarea->nextarea->prevarea = tmparea;

						tmparea->nextarea = currarea->nextarea;
						tmparea->prevarea = newarea;		/* new upper bound */

						newarea->nextarea = tmparea;
						newarea->prevarea = currarea;		/* middle inserted */

						currarea->end =	newarea->start-1;
						currarea->nextarea = newarea;		/* lower bound adjusted	*/
					}
					else {
						/* New area end	> current area end */
						if (newarea->end > currarea->end) {
							if (newarea->end == currarea->nextarea->end) {
								if (debug == true) fputs("overlap from current 'middle' to end of next\n", logfile);
								currarea->end =	startrange - 1;
								currarea->nextarea->start = startrange;
								free(newarea);		/* remove redundant area */
							}
							else {
								if (debug == true) fputs("overlap from current 'middle' to 'middle' of next\n",	logfile);
								currarea->end =	startrange - 1;
								currarea->nextarea->start = endrange + 1;

								newarea->nextarea = currarea->nextarea;
								currarea->nextarea->prevarea = newarea;

								newarea->prevarea = currarea;
								currarea->nextarea = newarea;
							}
						}
						else {
							if (debug == true) fputs("New area start from 'middle' and fill rest of current\n", logfile);
							newarea->nextarea = currarea->nextarea;
							if (currarea->nextarea != NULL)
								currarea->nextarea->prevarea = newarea;

							currarea->nextarea = newarea;
							newarea->prevarea = currarea;

							currarea->end =	newarea->start - 1;	/* adjust area intervals */
						}
					}

					break;
				}
				else {
					if (newarea->start == currarea->start) {
						if (newarea->end == currarea->end) {
							/* area	size matches, newarea redundant	*/
							if (debug == true) fputs("New area fits current area\n", logfile);
							free(newarea);
							currarea->areatype = t;
						}
						else {
							if (newarea->end < currarea->end) {
								if (debug == true) fputs("new area from start of current to end before current\n", logfile);
								newarea->nextarea = currarea;	/* insert newarea before current */
								newarea->prevarea = currarea->prevarea;

								if (currarea->prevarea != NULL)
									currarea->prevarea->nextarea = newarea;
								else {
									if (debug == true) fputs("Inserted as first in list\n",	logfile);
									*arealist = newarea;	/* newarea inserted first in list */
								}
								currarea->prevarea = newarea;	/* newarea now inserted	properly */
								currarea->start	= newarea->end+1;
							}
							else {
								if (newarea->end == currarea->nextarea->end) {
									if (debug == true) fputs("overlap from current start to end of next\n",	logfile);

									currarea->nextarea->start = currarea->start;
									currarea->nextarea->areatype = t;

									currarea->nextarea->prevarea = currarea->prevarea;
									if (currarea->prevarea != NULL)
										currarea->prevarea->nextarea = currarea->nextarea;
									else
										*arealist = currarea->nextarea;
									free(currarea);
									free(newarea);		/* remove redundant area */
								}
								else {
									if (debug == true) fputs("overlap from current start to 'middle' of next\n", logfile);
									currarea->end =	endrange;
									currarea->areatype = t;
									currarea->nextarea->start = endrange+1;
									free(newarea);
								}
							}
						}
					}
					else {
						if (debug == true) fputs("new area inserted before current\n", logfile);
						newarea->nextarea = currarea;	/* insert newarea before current */
						newarea->prevarea = currarea->prevarea;

						if (currarea->prevarea != NULL)
							currarea->prevarea->nextarea = newarea;
						else {
							if (debug == true) fputs("Inserted as first in list\n",	logfile);
							*arealist = newarea;	/* newarea inserted first in list */
						}
						currarea->prevarea = newarea;	/* newarea now inserted	properly */

						if (newarea->end > currarea->start)
							currarea->start	= newarea->end+1;	/* adjust for overlap */
					}

					break;
				}
			}
		} /* for */
	}

	if (debug == true) {
		DispAreas(logfile, *arealist);
		ValidateAreas(*arealist);
	}

	return 1;			/* newarea inserted successfully */
}


/* Join	two equal type areas into a single area	*/
void	JoinAreas(DZarea    *currarea)
{
	DZarea		*tmparea;

	while(currarea != NULL)	{
		while (currarea->nextarea != NULL) {
			tmparea	= currarea->nextarea;
			if (currarea->areatype == tmparea->areatype) {
				/* extend end range next into current */
				/* delete next and adjust pointers in list */
				
				currarea->end =	tmparea->end;		/* range extended */
				currarea->nextarea = tmparea->nextarea;	/* new nextarea	*/
				if (tmparea->nextarea != NULL)
					tmparea->nextarea->prevarea = currarea;
				free(tmparea);				/* tmparea now redundant */
			}
			else
				break;		/* two areas not equal,	move to	next area */
		}
		currarea = currarea->nextarea;
	}
}


void	ValidateAreas(DZarea  *currarea)
{
	unsigned short	pc;

	pc = currarea->start;
	while(currarea != NULL)	{
		if ((pc	> currarea->start) || (pc > currarea->end)) {
			printf("Area range out of order: [%04xh - %04xh]\n", currarea->start, currarea->end);
			fprintf(logfile, "Area range out of order: [%04xh - %04xh]\n", currarea->start,	currarea->end);
			fclose(logfile);
			return;
		}
		if (currarea->start > currarea->end) {
			printf("Illegal range found: [%04xh - %04xh]\n", currarea->start, currarea->end);
			fprintf(logfile, "Illegal range found: [%04xh - %04xh]\n", currarea->start, currarea->end);
			fclose(logfile);
			return;
		}
		if (currarea->prevarea != NULL)	{
			if ((currarea->start - currarea->prevarea->end)	> 1) {
				printf("Illegal gap found between: [%04xh - %04xh] and [%04xh - %04xh]\n",
					currarea->prevarea->start, currarea->prevarea->end, currarea->start, currarea->end);
				fprintf(logfile, "Illegal gap found between: [%04xh - %04xh] and [%04xh - %04xh]\n",
					currarea->prevarea->start, currarea->prevarea->end, currarea->start, currarea->end);
				fclose(logfile);
				return;
			}
		}
		pc = currarea->end;
		currarea = currarea->nextarea;		/* area	not found, check next */
	}
}


/* create and initialize an area */
DZarea	*InitArea(long	startaddr, long	 endaddr, enum atype  t)
{
	DZarea	   *narea;

	narea =	NewArea();
	if (narea == NULL) return NULL;

	narea->start = startaddr;
	narea->end = endaddr;
	narea->areatype	= t;
	narea->parsed =	false;
	narea->prevarea	= NULL;
	narea->nextarea	= NULL;

	return narea;
}


/* create an area */
DZarea	*NewArea(void)
{
	return (DZarea *) malloc(sizeof(DZarea));
}
