/*
 *   MediaMVP Server Widget Library
 *
 *   (C) 2004 Dave Kelly
 *
 *   $Id: box.c,v 1.4 2004/08/30 08:55:24 ravelox Exp $
 *   $Date: 2004/08/30 08:55:24 $
 *
 *   Window handling
 *
 */

#include "mvpwindow.h"

static void box_set_fill_col(window_t *window, uint32_t fill_col);
static void box_set_border_col(window_t *window, uint32_t border_col);
static void box_set_border_size(window_t *window, int border_size);
static uint32_t box_get_fill_col(window_t *box);
static uint32_t box_get_border_col(window_t *box);
static int box_get_border_size(window_t *box);

static attrib_fns_t box_fns = {
	0,
	box_set_fill_col,
	box_set_border_col,
	box_set_border_size,
	NULL,
	NULL,
	NULL,
	box_get_fill_col,
	box_get_border_col,
	box_get_border_size,
	NULL,
	NULL,
	NULL
};

typedef struct {
	int flags;
	int border_size;
	uint32_t border_col;
	uint32_t fill_col;
} box_t;

static int box_get_border_size(window_t *box)
{
	box_t *data;

	if( window_get_type(box)  != WINDOW_BOX ) {
		return 0;
	}

	if(! (data = window_get_data(box)) ) {
		return 0;
	}

	return data->border_size;
}

static uint32_t box_get_border_col(window_t *box)
{
	box_t *data;

	if( window_get_type(box)  != WINDOW_BOX ) {
		return 0;
	}

	if(! (data = window_get_data(box)) ) {
		return 0;
	}

	return data->border_col;
}

static uint32_t box_get_fill_col(window_t *box)
{
	box_t *data;

	if( window_get_type(box)  != WINDOW_BOX ) {
		return 0;
	}

	if(! (data = window_get_data(box)) ) {
		return 0;
	}

	return data->fill_col;
}

static void draw_box(window_t *window)
{
	int x, y;
	int width, height;
	int offset_x, offset_y;
	box_t *box;

	if(! window) {
		return;
	}

	if( window_get_type(window) != WINDOW_BOX || ! window_get_contents(window) ) {
		return;
	}

	box = (box_t *)window_get_data(window);

	if(! box) {
		return;
	}

	width = window_get_width(window);
	height = window_get_height(window);
	offset_x = window_get_xoffset(window);
	offset_y = window_get_yoffset(window);

	/* Fill the box */
	window_clear(window, box->fill_col);

	/* Draw the border */
	if(box->border_size > 0) {

		for( y = 0 ; y < box->border_size ; y++ ) {
			for( x = 0 ; x < width - offset_x ; x++ ) {
				window_set_pixel(window, x + offset_x, y + offset_y, box->border_col);
				window_set_pixel(window, x + offset_x, height - offset_y - y, box->border_col);
			}
		}

		for( x = 0 ; x < box->border_size ; x++ ) {
			for( y = 0 ; y < height - offset_y; y++ ) {
				window_set_pixel(window, x + offset_x, y + offset_y, box->border_col);
				window_set_pixel(window, width - offset_x - x, y + offset_y, box->border_col);
			}
		}

	}
}

static void box_set_border_size(window_t *window, int border_size)
{
	box_t *box;

	if(! window) {
		return;
	}

	if(window_get_type(window) != WINDOW_BOX) {
		return;
	}

	box = (box_t *)window_get_data(window);
	box->border_size = border_size;
	window_mark_dirty(window);
}

static void box_set_border_col(window_t *window, uint32_t border_col)
{
	box_t *box;

	if(! window ) {
		return;
	}

	if(window_get_type(window) != WINDOW_BOX) {
		return;
	}

	box = (box_t *)window_get_data(window);
	box->border_col = border_col;
	window_mark_dirty(window);
}

static void box_set_fill_col(window_t *window, uint32_t fill_col)
{
	box_t *box;

	if(! window ) {
		return;
	}

	if(window_get_type(window) != WINDOW_BOX) {
		return;
	}

	box = (box_t *)window_get_data(window);
	box->fill_col = fill_col;
	window_mark_dirty(window);
}

window_t *new_box(window_t *parent, int x, int y, int width, int height)
{
	window_t *window;
	box_t *box;


	window = new_window(parent, WINDOW_BOX, x, y, width, height);
	window_set_attrib_fns(window, &box_fns);

	box = (box_t *)malloc(sizeof(box_t));

	box->flags = 0;
	box->border_size = 0;
	box->border_col = 0;
	box->fill_col = 0;

	window_set_data(window, box);

	window_set_draw_func(window, draw_box);

	window_set_name(window, "%internal% box");
	return window;
}

