/*
 *   MediaMVP Server Library
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: url_file.c,v 1.8 2004/05/11 19:26:06 dom Exp $
 *   $Date: 2004/05/11 19:26:06 $
 *
 *
 *   Deals with opening a file and fobbing off playlist detection
 *   to a lower level
 */

#include "libmvp_internal.h"

static void           *file_open(char *name, int *type, fops_t **fops_ptr, readready_cb cb, void *cb_ptr );
static void            file_close(void *ptr);
static off_t           file_seek(void *ptr, off_t offset, int whence);
static int             file_read(void *ptr, unsigned char *buf, size_t buflen);
static int             file_info(void *ptr, int cmd, void *dest);

typedef struct {
    int            fd;
    char          *name;
#ifdef HAVE_LIBID3TAG
    mp3info_t     *info;
#endif

} file_t;

static fops_t          file_fops = {
    NULL,
    file_open,
    file_close,
    file_seek,
    file_read,
    file_info,
};

void urlfile_init()
{
    urlhandler_add("file://",&file_fops);
}


static void *file_open(char *name, int *type, fops_t **fops_ptr, readready_cb cb, void *cb_ptr )
{
    file_t  *file;
    char    *filename;
    char    *ptr;
    int      fd;


    /* This shouldn't happen... */
    if ( strncmp(name,"file://",7) ) {
        return NULL;
    }

    filename = name + 7;

    if ( ( ptr = strrchr(filename,'.') ) != NULL ) {
        if ( strcasecmp(ptr,".mp3") == 0 ) {
            *type = MEDIA_MP3;
        } else if ( strcasecmp(ptr,".mpeg") == 0 || strcasecmp(ptr,".vdr") == 0 ||
                    strcasecmp(ptr,".mpg") == 0 || strcasecmp(ptr,".nvu") == 0 ) {
            *type = MEDIA_MPEG;
        }
    } else {
        /* Try and guess the filetype */
        *type = file_get_type(filename);
    }

    if ( *type > 0 ) {
        if ( (fd = open(filename,O_RDONLY) ) < 0 ) {
            return NULL;
        }
        file = (file_t*)calloc(1,sizeof(*file));
        file->fd = fd;

#ifdef HAVE_LIBID3TAG
        file->info = NULL;

        if ( *type == MEDIA_MP3 ) {
            file->info = mp3_get_info_fd(file->fd);
        }

        if ( mp3_get_title(file->info) == NULL ) {
            file->name = strdup(filename);
        } else {
            file->name = strdup(mp3_get_title(file->info));
        }
#else            
        file->name = strdup(filename);
#endif

        return file;
    }

    return NULL;
}

static void file_close(void *ptr)
{
    file_t   *file = (file_t*)ptr;
    close(file->fd);
    if ( file->name ) {
        free(file->name);
        file->name = NULL;
    }
#ifdef HAVE_LIBID3TAG
    mp3_info_delete(file->info);
    file->info = NULL;
#endif
    file->fd = -1;
    free(file);
}

static off_t file_seek(void *ptr, off_t off, int whence)
{
    file_t *file = (file_t*)ptr;

    return lseek(file->fd,off,whence);
}

static int file_read(void *ptr, unsigned char *buf, size_t buflen)
{
    file_t  *file = (file_t*)ptr;

    return read(file->fd,buf,buflen);
}

static int file_info(void *ptr, int cmd, void *dest)
{
    file_t      *file = (file_t*)ptr;
    mpeginfo_t  *info = NULL;

    struct stat  sb;

    switch ( cmd ) {
    case URLFILE_SIZE:
        *((off_t *)dest) = 0;
        if ( fstat(file->fd,&sb) == 0 ) {
            if ( S_ISREG(sb.st_mode) ) {
                *((off_t *)dest) = sb.st_size;
                return 0;
            }
        }
        return -1;
    case URLFILE_NAME:
        *(char **)dest = file->name;
        break;
#ifdef HAVE_LIBID3TAG
    case URLFILE_MP3TAG:
        dest = file->info;
        break;
#endif     
    case URLFILE_MPEG_FRAME_RATE:
        info = mpeg_get_info_fd(file->fd);
        *((int *)dest) = mpeg_get_rate_index(info);
        delete_mpeg_info(info);
        break;
    default:
        return -1;
    }
    return 0;
}



