/*
 *   MediaMVP Plugin
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: vdrurl.c,v 1.7 2004/01/19 21:52:00 dom Exp $
 *   $Date: 2004/01/19 21:52:00 $
 *
 *
 *   Handle the vdr:// URL type
 */

#include "libmvp.h"

#include "vdrurl.h"
#include "misc.h"

static fops_t vdrurl_fops = {
    NULL,
    cMediamvpVdrURL::Open,
    cMediamvpVdrURL::Close,
    cMediamvpVdrURL::Seek,
    cMediamvpVdrURL::Read
};


cMediamvpVdrURL::cMediamvpVdrURL()
{
    m_Transceiver = NULL;
}

cMediamvpVdrURL::~cMediamvpVdrURL()
{

}

void cMediamvpVdrURL::Init()
{
    urlhandler_add("vdr://",&vdrurl_fops);
}

cDevice *cMediamvpVdrURL::GetDevice(const cChannel *Channel, int Priority)
{
    cDevice    *device = NULL;
    int         i;

    for (i = 0; i < cDevice::NumDevices(); ++i) {
        device = cDevice::GetDevice(i);
        if (device == cDevice::PrimaryDevice())
            // Prefer secondary devices over Primary
            continue;
        
        if (!device->ProvidesChannel(Channel, Priority))
            continue;

        if (device != cDevice::ActualDevice())
            // Should be ok if the current channel doesn't run there
            break;

        cChannel *current   = Channels.GetByNumber(cDevice::PrimaryDevice()->CurrentChannel());
        if (current->Frequency() == Channel->Frequency() )
            // Should be ok if the tuned Frequency is equal to the requested one
            break;
    }

    if (i == cDevice::NumDevices()) {
        device = cDevice::PrimaryDevice();
        if (device == cDevice::ActualDevice()) {
            cChannel *current = Channels.GetByNumber(cDevice::PrimaryDevice()->CurrentChannel());
            if (current->Frequency() != Channel->Frequency() )
#ifdef SINGLE_DEVICE
                device = cDevice::PrimaryDevice();
#else
                device = NULL;
#endif
        }
    }

    Dprintf(MOO,"Selected device @%p (primary = %p)\n",device,cDevice::PrimaryDevice());
        
    return device;
}

void *cMediamvpVdrURL::Open(char *name, int *type, fops_t **fops, readready_cb cb, void *cb_ptr)
{
    cMediamvpVdrURL *vdrurl = new cMediamvpVdrURL;
    char            *realname;
    const cChannel  *Channel;
    cDevice         *device;

    vdrurl->cb = cb;
    vdrurl->cb_ptr = cb_ptr;

    if ( strncmp(name,"vdr://",strlen("vdr://") ) ) {
        return NULL;
    }
    
    realname = name + strlen("vdr://");

    vdrurl->name = strdup(name);

    if ( ( Channel = ChannelFromString(realname) ) == NULL ) {
        return NULL;
    }

    
    if ( (device = vdrurl->GetDevice(Channel,0) ) == NULL ) {
        return NULL;
    }

    if ( Channel->Vpid() == 0 || Channel->Vpid() == 1 || Channel->Vpid() == 0x1fff ) {
        *type = MEDIA_MP3 | MEDIA_SOCKET | MEDIA_LIVE;
    } else {
        *type = MEDIA_MPEG | MEDIA_SOCKET | MEDIA_LIVE;
    }

    if ( pipe(&vdrurl->transfer_pipe[0]) == -1 ) {
        return NULL;
    }

    /* Create an event */
    vdrurl->event = (struct event *)malloc(sizeof(*vdrurl->event));
    event_set(vdrurl->event,vdrurl->transfer_pipe[0],EV_READ,cMediamvpVdrURL::ReadEvent,vdrurl);
    event_add(vdrurl->event,NULL);


    /* Create a new transceiver */
    vdrurl->m_Transceiver = new cMediamvpTransceiver(Channel,0,vdrurl->transfer_pipe[1],device);
    
    Dprintf(MOO,"Opened channel %s vdrurl %p transceiver %p\n",name,vdrurl,vdrurl->m_Transceiver);

    vdrurl->m_Transceiver->Activate(TRUE);

    return vdrurl;
}

off_t cMediamvpVdrURL::Seek(void *ptr, off_t offset, int whence)
{
    return (off_t) -1;
}

void cMediamvpVdrURL::Close(void *ptr)
{
    cMediamvpVdrURL *vdrurl = ( cMediamvpVdrURL *) ptr;

    if ( vdrurl != NULL ) {
        Dprintf(MOO,"Closing channel %s vdrurl %p transceiver %p\n",vdrurl->name,ptr,vdrurl->m_Transceiver);
        if ( vdrurl->m_Transceiver != NULL ) {
            DELETENULL(vdrurl->m_Transceiver);
        }

        if ( vdrurl->event ) {
            event_del(vdrurl->event);
            free(vdrurl->event);
            vdrurl->event = NULL;
        }

        close(vdrurl->transfer_pipe[0]);
        close(vdrurl->transfer_pipe[1]);

        free(vdrurl->name);
        delete vdrurl;
    }

    return;
}

int cMediamvpVdrURL::Read(void *ptr, unsigned char *buf, size_t buflen)
{
    cMediamvpVdrURL *vdrurl = ( cMediamvpVdrURL *)ptr;

    if ( vdrurl != NULL ) {
        return read(vdrurl->transfer_pipe[0],buf,buflen);       
    }
    return 0;
}

void cMediamvpVdrURL::ReadEvent(int fd, short event, void *arg)
{
    cMediamvpVdrURL *vdrurl = ( cMediamvpVdrURL *)arg;

    if ( vdrurl != NULL ) {
        event_add(vdrurl->event,NULL);
        vdrurl->cb(vdrurl->cb_ptr);
    }
}
