/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: app.c,v 1.34 2003/11/30 23:06:20 dom Exp $
 *   $Date: 2003/11/30 23:06:20 $
 *
 *
 *   Implements the application
 */

#include "mediamvp.h"

/* Need to know some GUI constants */
#include "gui.h"

/* Now get some message types */
#include "msgs.h"

#include <dirent.h>
#include <sys/stat.h>



typedef struct {
    char   *name;
    int     flags;
} filelist_t;


struct _app {
    dongle_t  *dongle;
    render_t  *render;
    gui_t     *gui;
    menu_t    *menu;
    char      *current_media;
    int        num_next_media;
    char     **next_media;
    char       paused;
};


typedef struct {
    app_t      *app;
    char       *path;
    /* For handling scanning of directories */
    int         filelist_num;
    filelist_t *filelist;
} dirlist_t;



typedef struct {
    char     *name;
    char     *url;
} channel_t;

static char         *c_livetv_file = NULL;
static char         *c_liveradio_file = NULL;
static char         *c_mp3_dir     = ".";
static char         *c_mpeg_dir    = ".";

static int           livetv_num    = 0;
static channel_t    *livetv        = NULL;
static time_t        livetv_time   = 0;

static int           liveradio_num    = 0;
static channel_t    *liveradio        = NULL;
static time_t        liveradio_time   = 0;

static int           app_main(menu_t *menu,void *iapp, int sel);
static int           app_live_tv(menu_t *menu,void *iapp, int sel);
static int           process_livetv(menu_t *menu,void *iapp, int sel);
static int           livetv_keys(void *ptr, int code);
static int           livetv_ack(int acktype, void *param, unsigned char *buf, int len);

static int           app_live_radio(menu_t *menu,void *iapp, int sel);
static int           process_liveradio(menu_t *menu,void *iapp, int sel);
static int           liveradio_keys(void *ptr, int code);
static int           liveradio_ack(int acktype, void *param, unsigned char *buf, int len);


static int           app_mpeg(menu_t *menu,void *iapp, int sel);
static int           process_mpeg(menu_t *menu,void *iapp, int sel);
static int           mpeg_keys(void *ptr, int code);
static int           mpeg_ack(int acktype, void *param, unsigned char *buf, int len);
static int           mpeg_colour_keys(menu_t *menu, void *iapp, int key, int sel);

static int           app_music(menu_t *menu,void *iapp, int sel);
static int           process_music(menu_t *menu,void *iapp, int sel);
static int           music_keys(void *ptr, int code);
static int           music_ack(int acktype, void *param, unsigned char *buf, int len);
static int           music_colour_keys(menu_t *menu, void *iapp, int key, int sel);


static int           app_settings(menu_t *menu,void *iapp, int sel);
static int           process_settings(menu_t *menu,void *iapp, int sel);

static filelist_t   *directory_scan(char *path, int *scanned);
static void          directory_sort(int num,filelist_t *list);

static dirlist_t    *new_dirlist(app_t *app);
static void          delete_dirlist(void *dir);    /* Ahem */


static void          next_media_clear(app_t *app);
static void          next_media_pop(app_t *app);
static void          next_media_push(app_t *app, char *url);



static int           read_channel_file(char *filename, int *num, channel_t **list, time_t *update_time);

void app_config()
{
    iniparse_add("media:livetv",OPT_STR,&c_livetv_file);
    iniparse_add("media:liveradio",OPT_STR,&c_liveradio_file);
    iniparse_add("media:mp3",OPT_STR,&c_mp3_dir);
    iniparse_add("media:mpeg",OPT_STR,&c_mpeg_dir);
}


void app_init()
{
    if ( c_livetv_file != NULL ) {
        if ( read_channel_file(c_livetv_file,&livetv_num,&livetv,&livetv_time)  < 0 ) {
            printf("Malformed file %s\n",c_livetv_file);
            exit(1);
        }
    }   
    if ( c_liveradio_file != NULL ) {
        if ( read_channel_file(c_liveradio_file,&liveradio_num,&liveradio,&liveradio_time)  < 0 ) {
            printf("Malformed file %s\n",c_liveradio_file);
            exit(1);
        }
    }   

}

app_t *new_app(dongle_t *dongle, render_t *render, gui_t *gui)
{
    app_t      *app = malloc(sizeof(*app));

    app->render = render;
    app->gui    = gui;
    app->dongle = dongle;
    app->current_media = NULL;
    app->num_next_media = 0;
    app->next_media = NULL;
    app->paused = FALSE;


    app_main(NULL,app,0);    

    return app;
}


void delete_app(app_t *app)
{

    delete_menu_tree(app->menu);

    if ( app->current_media ) {
        free(app->current_media);
        app->current_media = NULL;
    }

    if ( app->next_media ) {
        next_media_clear(app);
    }

    free(app);
}




static int app_main(menu_t *menu,void *iapp, int sel)
{   
    app_t   *app = iapp;

    app->menu = new_menu(app->dongle,app->render,NULL,app,MENU_NUMBERS);
    menu_set_title(app->menu,"MediaMVP");
    menu_add(app->menu,"Live TV",app_live_tv);
    menu_add(app->menu,"Live Radio",app_live_radio);
    menu_add(app->menu,"Recordings",app_mpeg);
    menu_add(app->menu,"Music",app_music);
    menu_add(app->menu,"Settings",app_settings);
    menu_display(app->menu);
    return 1;
}


/***********************
 *
 *   Handling of the Live TV Menus
 *
 ***********************/

static int app_live_tv(menu_t *menu,void *iapp, int sel)
{
    app_t     *app = iapp;
    channel_t *chan;
    menu_t    *child;
    int        i;

    /* Force a re-read of the channel file */
    read_channel_file(c_livetv_file,&livetv_num,&livetv,&livetv_time);

    child = new_menu(app->dongle,app->render,menu,app,0);
    menu_set_title(child,"Live TV"); 

    for ( i = 0; i < livetv_num; i++ ) {
        chan = &livetv[i];
        menu_add(child,chan->name,process_livetv);
    }
    menu_display(child);
    return 1;
}

static int process_livetv(menu_t *menu,void *iapp, int sel)
{
    app_t     *app = iapp;

    channel_t *chan = &livetv[sel];
    gui_send_play(app->gui,chan->url);
    gui_register_keys(app->gui,REGISTER_MEDIA,livetv_keys,app);
    gui_register_ack(app->gui,REGISTER_MEDIA,livetv_ack,app);
    return 1;
}

static int livetv_ack(int acktype, void *param, unsigned char *buf, int len)
{
    app_t     *app = param;

    switch ( acktype ) {
    case RDC_PLAY:
        if ( (stream_get_type(app->dongle->stream) & MEDIA_MASK) == MEDIA_MPEG ) {
            gui_send_message(app->gui,&rfb_media_menu);
            gui_send_message(app->gui,&rfb_display_off);
        }
        break;
    case RDC_STOP:             
        /* The client has sent a stop message, this could be response to "Stop" key press or
           us sending out a stop so we can switch to the net media file */ 
        if ( app->num_next_media ) {
            next_media_pop(app);           
            app->paused = FALSE;
            gui_send_play(app->gui,app->current_media);
        } else {
            if ( ( stream_get_lasttype(app->dongle->stream) & MEDIA_MASK ) == MEDIA_MPEG ) {
                gui_send_message(app->gui,&rfb_media_menu);
                gui_send_message(app->gui,&rfb_display_on);
            }
            if ( app->current_media ) {
                free(app->current_media);
                app->current_media = NULL;
            }
            app->paused = FALSE;
            /* Remove registered handlers */
            gui_register_keys(app->gui,REGISTER_MEDIA,NULL,app);
            gui_register_ack(app->gui,REGISTER_MEDIA,NULL,app);
        }


        break;
    default:
        return 0;    /* Kick upstairs to menu ack handler */
    }

    return 1;       /* No need to pass back to menu handler */
}


static int livetv_keys(void *ptr, int code)
{
    app_t    *app = ptr;

    switch ( code ) {
    case kMenu:
    case kStop:
        gui_send_message(app->gui,&rfb_media_stop);
        menu_display(app->menu);
        break;
    case kVolUp:
        gui_send_message(app->gui,&rfb_media_volup);
        break;
    case kVolDn:
        gui_send_message(app->gui,&rfb_media_voldown);
        break;
    case kMute:
        gui_send_message(app->gui,&rfb_media_mute);
        break;
    default:
        return 0;  /* Do more keys */
    }
    return 1;
}


/***********************
 *
 *   Handling of the Live Radio Menus
 *
 ***********************/

static int app_live_radio(menu_t *menu,void *iapp, int sel)
{
    app_t     *app = iapp;
    channel_t *chan;
    menu_t    *child;
    int        i;

    /* Force a re-read of the channel file */
    read_channel_file(c_liveradio_file,&liveradio_num,&liveradio,&liveradio_time);
    child = new_menu(app->dongle,app->render,menu,app,0);
    menu_set_title(child,"Live Radio Stations");  
    for ( i = 0; i < liveradio_num; i++ ) {
        chan = &liveradio[i];
        menu_add(child,chan->name,process_liveradio);
    } 
    menu_display(child);
    return 1;
}

static int process_liveradio(menu_t *menu,void *iapp, int sel)
{
    app_t     *app = iapp;
    channel_t *chan = &liveradio[sel];

    printf("Sending media do request %s\n",chan->url);
    if ( app->current_media == NULL ) {        
        app->current_media = strdup(chan->url);
        app->paused = FALSE;
        gui_send_play(app->gui,chan->url);
    } else {
        next_media_clear(app);
        next_media_push(app,chan->url);       
        gui_send_message(app->gui,&rfb_media_stop);
    }
    gui_register_keys(app->gui,REGISTER_MEDIA,liveradio_keys,app);
    gui_register_ack(app->gui,REGISTER_MEDIA,liveradio_ack,app);
    return 1;
}


static int liveradio_keys(void *ptr, int code)
{
    app_t     *app = ptr;

    switch ( code ) { 
    case kStop:
        gui_send_message(app->gui,&rfb_media_stop);
        next_media_clear(app);
        menu_display(app->menu);
        break;
    case kVolUp:
        gui_send_message(app->gui,&rfb_media_volup);
        break;
    case kVolDn:
        gui_send_message(app->gui,&rfb_media_voldown);
        break;
    case kMute:
        gui_send_message(app->gui,&rfb_media_mute);
        break;
    default:
        return 0;      /* Carry on processing menu keys */
    }   
    return 1;
}

static int liveradio_ack(int acktype, void *param, unsigned char *buf, int len)
{
    app_t     *app = param;

    switch ( acktype ) {
    case RDC_STOP:     /* The client has sent a stop message */  
        if ( app->num_next_media ) {
            next_media_pop(app);
            app->paused = FALSE;
            gui_send_play(app->gui,app->current_media);
        } else {
            /* Remove registered handlers */
            if ( app->current_media ) {
                free(app->current_media);
                app->current_media = NULL;
            }
            app->paused = FALSE;
            gui_register_keys(app->gui,REGISTER_MEDIA,NULL,app);
            gui_register_ack(app->gui,REGISTER_MEDIA,NULL,app);
        }
        break;
    default:
        return 0;    /* Kick it up a level */
    }

    return 1;        /* Don't process anymore */
}


/*
 * Handle menus for mpeg files 
 */
static int app_mpeg(menu_t *menu,void *iapp, int sel)
{
    app_t      *app = iapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[256];
    int         i;

    dirlist = new_dirlist(app);

    dirlist->path = strdup(c_mpeg_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan(c_mpeg_dir,&dirlist->filelist_num);

    child = new_menu(app->dongle,app->render,menu,dirlist,0);
    menu_set_title(child,"Video Recordings");   
    menu_set_app_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
            snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,process_mpeg);
    }
    menu_set_colour_actions(child,"Back",NULL,NULL,NULL,mpeg_colour_keys);

    menu_display(child);
    return 1;  
}

static int process_mpeg(menu_t *menu,void *iapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = iapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;


    entry = &dirlist->filelist[sel];

    if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->app);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->app->dongle,dirlist2->app->render,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Video - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_app_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
                snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,process_mpeg);
        }
        menu_set_colour_actions(child,"Back",NULL,NULL,NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        if ( dirlist->app->current_media == NULL ) {        
            dirlist->app->current_media = strdup(buf);
            dirlist->app->paused = FALSE;
//            next_media_push(dirlist->app,buf);
            gui_send_play(dirlist->app->gui,buf);
        } else {
            next_media_clear(dirlist->app);
            next_media_push(dirlist->app,buf);           
            dirlist->app->paused = FALSE;
            gui_send_message(dirlist->app->gui,&rfb_media_stop);    
        }
        gui_register_keys(dirlist->app->gui,REGISTER_MEDIA,mpeg_keys,dirlist->app);
        gui_register_ack(dirlist->app->gui,REGISTER_MEDIA,mpeg_ack,dirlist->app);
    }  

    return 1;     /* Carry on displaying this menu */
}

static int mpeg_colour_keys(menu_t *menu, void *iapp, int key, int sel)
{
    switch ( key ) {
    case kRed:
        return 0;   
    default:
    }

    return 1;     /* Carry on displaying this menu */
}

static int mpeg_keys(void *ptr, int code)
{
    app_t     *app = ptr;

    switch ( code ) { 
    case kStop:
        next_media_clear(app);
        gui_send_message(app->gui,&rfb_media_stop);
        menu_display(app->menu);
        break;
    case kPlay:
        gui_send_play(app->gui,app->current_media);
        app->paused = FALSE;
        break;
    case kPause:
        if ( app->paused ) {
            app->paused = FALSE;
            gui_send_play(app->gui,app->current_media);
        } else {
            app->paused = TRUE;
            gui_send_message(app->gui,&rfb_media_pause);
        }
        break;
    case kVolUp:
        gui_send_message(app->gui,&rfb_media_volup);
        break;
    case kVolDn:
        gui_send_message(app->gui,&rfb_media_voldown);
        break;
    case kMute:
        gui_send_message(app->gui,&rfb_media_mute);
        break;
#if 0
        /* Forward/reverse don't work...erk! */
    case kFastFwd:
        gui_send_message(app->gui,&rfb_media_forward);
        break;
    case kFastRew:
        gui_send_message(app->gui,&rfb_media_rewind);
        break;
#endif
    case kSkip:
        gui_send_message(app->gui,&rfb_media_stop);
        break;
    default:
        return 0;      /* Carry on processing menu keys */
    }   
    return 1;
}


static int mpeg_ack(int acktype, void *param, unsigned char *buf, int len)
{
    app_t         *app = param;

    switch ( acktype ) {
    case RDC_PLAY:
        if ( (stream_get_type(app->dongle->stream) & MEDIA_MASK) == MEDIA_MPEG ) {
            gui_send_message(app->gui,&rfb_media_menu);
            gui_send_message(app->gui,&rfb_display_off);
        }
        break;
    case RDC_STOP:     /* The client has sent a stop message */ 
        if ( app->num_next_media ) {
            next_media_pop(app);
            app->paused = FALSE;
            gui_send_play(app->gui,app->current_media);          
        } else {
          if ( ( stream_get_lasttype(app->dongle->stream) & MEDIA_MASK ) == MEDIA_MPEG ) {
                gui_send_message(app->gui,&rfb_media_menu);
                gui_send_message(app->gui,&rfb_display_on);
            }
            if ( app->current_media ) {
                free(app->current_media);
                app->current_media = NULL;
            }
            app->paused = FALSE;
            /* Remove registered handlers */
            gui_register_keys(app->gui,REGISTER_MEDIA,NULL,app);
            gui_register_ack(app->gui,REGISTER_MEDIA,NULL,app);            
        }
        break;
    case RDC_FORWARD:
    case RDC_REWIND:
        gui_send_play(app->gui,app->current_media);
        break;
    default:
        return 0;    /* Kick it up a level */
    }

    return 1;        /* Don't process anymore */
}




/*
 *   Handle the root menu for Music menu
 */
static int app_music(menu_t *menu,void *iapp, int sel)
{
    app_t      *app = iapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[256];
    int         i;

    dirlist = new_dirlist(app);

    dirlist->path = strdup(c_mp3_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan(c_mp3_dir,&dirlist->filelist_num);

    child = new_menu(app->dongle,app->render,menu,dirlist,0);
    menu_set_title(child,"Music");   
    menu_set_app_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
            snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,process_music);
    }
    menu_set_colour_actions(child,"Back","Play dir",NULL,NULL,music_colour_keys);

    menu_display(child);
    return 1;
}


static int process_music(menu_t *menu,void *iapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = iapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;


    entry = &dirlist->filelist[sel];

    if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->app);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->app->dongle,dirlist2->app->render,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Music - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_app_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
                snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,process_music);
        }
        menu_set_colour_actions(child,"Back","Play dir",NULL,NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        if ( dirlist->app->current_media == NULL ) {        
            dirlist->app->current_media = strdup(buf);
            dirlist->app->paused = FALSE;
            gui_send_play(dirlist->app->gui,buf);
        } else {
            next_media_clear(dirlist->app);
            next_media_push(dirlist->app,buf);           
            dirlist->app->paused = FALSE;
            gui_send_message(dirlist->app->gui,&rfb_media_stop);    
        }
        gui_register_keys(dirlist->app->gui,REGISTER_MEDIA,music_keys,dirlist->app);
        gui_register_ack(dirlist->app->gui,REGISTER_MEDIA,music_ack,dirlist->app);
    }  

    return 1;     /* Carry on displaying this menu */
}

static int music_colour_keys(menu_t *menu, void *iapp, int key, int sel)
{
    char           buf[FILENAME_MAX+1];
    dirlist_t     *dir = iapp;
    int            i;
    char           play = TRUE;

    switch ( key ) {
    case kRed:
        return 0;
    case kGreen:       /* Add all contents of the directory */
        if ( dir->app->current_media ) {
            play = FALSE;
        }
        next_media_clear(dir->app);

        for ( i = 0; i < dir->filelist_num; i++ ) {
            if ( dir->filelist[i].flags == 0 ) {
                snprintf(buf,sizeof(buf),"%s/%s",dir->path,dir->filelist[i].name);
                next_media_push(dir->app,buf);
            }
        }
        if ( dir->app->num_next_media ) {
            if ( play ) {
                next_media_pop(dir->app);
                gui_send_play(dir->app->gui,dir->app->current_media);
                gui_register_keys(dir->app->gui,REGISTER_MEDIA,music_keys,dir->app);
                gui_register_ack(dir->app->gui,REGISTER_MEDIA,music_ack,dir->app);
            } else {
                gui_send_message(dir->app->gui,&rfb_media_stop);
            }
        }
        break;
    default:
    }

    return 1;     /* Carry on displaying this menu */
}

static int music_keys(void *ptr, int code)
{
    app_t     *app = ptr;

    switch ( code ) { 
    case kStop:
        next_media_clear(app);
        gui_send_message(app->gui,&rfb_media_stop);
        menu_display(app->menu);
        break;
    case kPlay:
        gui_send_play(app->gui,app->current_media);
        app->paused = FALSE;
        break;
    case kPause:
        if ( app->paused ) {
            app->paused = FALSE;
            gui_send_play(app->gui,app->current_media);
        } else {
            app->paused = TRUE;
            gui_send_message(app->gui,&rfb_media_pause);
        }
        break;
    case kVolUp:
        gui_send_message(app->gui,&rfb_media_volup);
        break;
    case kVolDn:
        gui_send_message(app->gui,&rfb_media_voldown);
        break;
    case kMute:
        gui_send_message(app->gui,&rfb_media_mute);
        break;
    case kFastFwd:
        gui_send_message(app->gui,&rfb_media_forward);
        break;
    case kFastRew:
        gui_send_message(app->gui,&rfb_media_rewind);
        break;
    case kSkip:
        gui_send_message(app->gui,&rfb_media_stop);
        break;
    default:
        return 0;      /* Carry on processing menu keys */
    }   
    return 1;
}

static int music_ack(int acktype, void *param, unsigned char *buf, int len)
{
    app_t         *app = param;

    switch ( acktype ) {
    case RDC_STOP:     /* The client has sent a stop message */  
        if ( app->num_next_media ) {
            next_media_pop(app);
            app->paused = FALSE;
            gui_send_play(app->gui,app->current_media);          
        } else {
            /* Remove registered handlers */
            if ( app->current_media ) {
                free(app->current_media);
                app->current_media = NULL;
            }
            app->paused = FALSE;
            gui_register_keys(app->gui,REGISTER_MEDIA,NULL,app);
            gui_register_ack(app->gui,REGISTER_MEDIA,NULL,app);
        }
        break;
    case RDC_FORWARD:
    case RDC_REWIND:
        gui_send_play(app->gui,app->current_media);
        break;
    default:
        return 0;    /* Kick it up a level */
    }

    return 1;        /* Don't process anymore */
}




/***********************
 *
 *   Handling of the settings Menu
 *
 ***********************/
static int app_settings(menu_t *menu,void *iapp, int sel)
{
    app_t           *app = iapp;
    menu_t          *child;
    static char     *tvmode[]  = { "NTSC", "PAL" };
    static char     *flicker[] = { "None", "Low", "Medium", "High" };
    static char     *output[]  = { "RGB Detected", "Svideo", "Composite", "RGB/Composite" };
    static char     *aspect[]  = { "4:3", "16:9" };

    child = new_menu(app->dongle,app->render,menu,app,0);
    menu_set_title(child,"Settings");

 

    /* Add in all possible options - we'll validate things on the callback */
    menu_add_option(child,"TV Mode:",process_settings,
                    MENU_INT,2,tvmode,&app->dongle->tvmode);

    /* A little bit of fiddling - see rules in proces_settings */
    if ( app->dongle->videooutput != 0 ) {
        menu_add_option(child,"Output:",process_settings,
                        MENU_INT,4,output,&app->dongle->videooutput);
    } else {
        menu_add(child,"Output: RGB Detected",process_settings);
    }

    menu_add_option(child,"Flicker Control:",process_settings,
                    MENU_INT,4,flicker,&app->dongle->flickermode);

    menu_add_option(child,"Aspect Ratio:",process_settings,
                    MENU_INT,2,aspect,&app->dongle->aspectratio);

    menu_add(child,"Cancel changes",process_settings);
    menu_add(child,"Save settings (may take a while)",process_settings);

    menu_display(child);
    return 1;
}

static int process_settings(menu_t *menu,void *iapp, int sel)
{
    app_t   *app = iapp;
    /* Some rules preened from the configuration.html:

    1). For NTSC unit -- in response to GUI query, video output value 
    is always '3', so you show "Video Output" as "Composite/Svideo" no changes
    allowed !!

    2). For SCART Unit --
    a). '0' : then you show "Video Output" as "RGB Detected", No changes 
    allowed !!!
    (b). '1' : then you show "Video Output" as "SVIDEO", it can only be
    changed to RGB/COMPOSITE (value == 2)
    (c). '2': you show "Video Output" as "RGB/COMPOSITE", it can only be 
    changed to SVIDEO (value == 1)

    We try to obey them [but probably fail miserable]!
    */

    /* If we come through with -1 then we should test the settings */
    switch ( sel ) {
    case -1:  /* If we come through with -1 then we should test the settings */
        if ( app->dongle->tvmode == 0 ) {
            app->dongle->videooutput = 3;
        }
        gui_send_settings(app->gui,RDC_SETTINGS_TEST);
        return 1;  /* Carry on displaying current menu */
    case 4:
        gui_send_settings(app->gui,RDC_SETTINGS_CANCEL);
        return 0;
    case 5:
        gui_send_settings(app->gui,RDC_SETTINGS_SAVE);
        return 0;
    default:
        return 0;   /* Carry on displaying */
    } 
}



static dirlist_t    *new_dirlist(app_t *app)
{
    dirlist_t   *dir = malloc(sizeof(*dir));

    dir->filelist_num = 0;
    dir->filelist = NULL;
    dir->app = app;

    return dir;
}

static void          delete_dirlist(void *idir)
{
    dirlist_t *dir = idir;
    int        i;

    for ( i = 0; i < dir->filelist_num; i++ ) {
        free(dir->filelist[i].name);
    }
    if ( dir->filelist ) {
        free(dir->filelist);
    }

    free(dir->path);

    free(dir);
}

static filelist_t *directory_scan(char *path, int *scanned)
{
    DIR    *dir;
    struct dirent *dirent;
    struct stat sb;
    char   filename[FILENAME_MAX+1];
    filelist_t *list = NULL;
    int         num  = 0;
    int         flags;
    int         i;


    *scanned = 0;

    if ( ( dir = opendir(path) ) == NULL ) {
        perror("opendir");
        return NULL;
    }

    while ( (dirent = readdir(dir)) != NULL ) {
        if ( dirent->d_name[0] == '.' ) {
            continue;
        }

        snprintf(filename,sizeof(filename),"%s/%s",path,dirent->d_name);
        if ( stat(filename,&sb) < 0 ) {
            continue;
        }

        if ( S_ISDIR(sb.st_mode) ) {
            flags = 1;
        } else {
            flags = 0;
        }

        i = num++;
        list = realloc(list, num * sizeof(filelist_t));

        list[i].name = strdup(dirent->d_name);
        list[i].flags = flags;
            
    }

    closedir(dir);

    directory_sort(num,list);

    *scanned = num;
    return list;
}

static void directory_sort(int num,filelist_t *list)
{
    int    i,j;
    int    t_flags;
    char  *t_name;

    for ( i = (num-1) ; i >= 0; i-- ) {
        t_name = list[i].name;
        t_flags = list[i].flags;
        j = i + 1;

        while ( j < num && strcasecmp(list[j].name,t_name) < 0 ) {
            list[j-1].name = list[j].name;
            list[j-1].flags = list[j].flags;

            j++;
        }

        list[j-1].name = t_name;
        list[j-1].flags = t_flags;

    }
}



static int read_channel_file(char *filename, int *num, channel_t **list, 
                             time_t *update_time)
{
    char         buf[500];
    int          channum;
    channel_t   *chan = *list;
    char        *ptr;
    FILE        *fp;
    struct stat  sb;
    int          i;

    if ( stat(filename,&sb) < 0 ) {
        return 0;
    }

    if ( *update_time < sb.st_mtime ) {
        *update_time = sb.st_mtime;
        for ( i = 0; i < *num; i++ ) {
            free(chan[i].name);
            free(chan[i].url);
        }


        if ( ( fp = fopen(filename,"r") ) == NULL ) {
            return -1;
        }

        if ( chan != NULL ) {
            free(chan);
        }
        chan = NULL;
        channum = 0;

        *list = NULL;
        *num = 0;

        while ( !feof(fp) && fgets(buf,sizeof(buf),fp) ) {
            if ( (ptr = strchr(buf,';') ) == NULL ) {
                continue;
            }
            *ptr = 0;
            ptr++;

            i = channum++;

            chan = realloc(chan,channum * sizeof(channel_t));
            chan[i].name = strdup(buf);
            chan[i].url = strdup(ptr);            
        }
        *list = chan;
        *num = channum;
    }

    return 0;
}

static void next_media_clear(app_t *app)
{
    int         i;

    if ( app->num_next_media ) {
        for ( i = 0; i > app->num_next_media; i++ ) {
            free(app->next_media[i]);
        }
        free(app->next_media);
        app->next_media = NULL;
        app->num_next_media = 0;
    }
}

static void next_media_pop(app_t *app)
{
    int       i;

    if ( app->num_next_media ) {     /* Which it already is... */
        if ( app->current_media ) {
            free(app->current_media);
        }

        app->current_media = app->next_media[0];

        for ( i = 1; i < app->num_next_media; i++ ) {
            app->next_media[i-1] = app->next_media[i];
        }
        if ( --app->num_next_media == 0 ) {
            free(app->next_media);
            app->next_media = NULL;
        }
    } 
}

static void next_media_push(app_t *app, char *url)
{
    int       i = app->num_next_media++;

    app->next_media = realloc(app->next_media,app->num_next_media * sizeof(char *));
    app->next_media[i] = strdup(url);
}



/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
