/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: app.c,v 1.20 2003/11/24 21:14:16 dom Exp $
 *   $Date: 2003/11/24 21:14:16 $
 *
 *
 *   Implements the application
 */

#include "mediamvp.h"

/* Need to know some GUI constants */
#include "gui.h"

/* Now get some message types */
#include "msgs.h"


struct _app {
    dongle_t  *dongle;
    render_t  *render;
    gui_t     *gui;
    menu_t    *menu;
    char      *current_media;
    char      *next_media;
};

typedef struct {
    char     *name;
    char     *url;
} channel_t;

static char         *c_livetv_file = NULL;
static char         *c_liveradio_file = NULL;
static char         *c_mp3_dir     = NULL;
static char         *c_mpeg_dir    = NULL;

static int           livetv_num    = 0;
static channel_t    *livetv        = NULL;
static time_t        livetv_time   = 0;

static int           liveradio_num    = 0;
static channel_t    *liveradio        = NULL;
static time_t        liveradio_time   = 0;

static int           app_main(menu_t *menu,app_t *app, int sel);
static int           app_live_tv(menu_t *menu,app_t *app, int sel);
static int           process_livetv(menu_t *menu,app_t *app, int sel);
static int           livetv_keys(void *ptr, int code);
static int           livetv_ack(int acktype, void *param, unsigned char *buf, int len);

static int           app_live_radio(menu_t *menu,app_t *app, int sel);
static int           process_liveradio(menu_t *menu,app_t *app, int sel);
static int           liveradio_keys(void *ptr, int code);
static int           liveradio_ack(int acktype, void *param, unsigned char *buf, int len);


static int           app_recordings(menu_t *menu,app_t *app, int sel);
static int           app_music(menu_t *menu,app_t *app, int sel);

static int           app_settings(menu_t *menu,app_t *app, int sel);
static int           process_settings(menu_t *menu,app_t *app, int sel);



static int           read_channel_file(char *filename, int *num, channel_t **list, time_t *update_time);

void app_config()
{
    iniparse_add("media:livetv",OPT_STR,&c_livetv_file);
    iniparse_add("media:liveradio",OPT_STR,&c_liveradio_file);
    iniparse_add("media:mp3",OPT_STR,&c_mp3_dir);
    iniparse_add("media:mpeg",OPT_STR,&c_mpeg_dir);
}


void app_init()
{
    if ( c_livetv_file != NULL ) {
        if ( read_channel_file(c_livetv_file,&livetv_num,&livetv,&livetv_time)  < 0 ) {
            printf("Malformed file %s\n",c_livetv_file);
            exit(1);
        }
    }   
    if ( c_liveradio_file != NULL ) {
        if ( read_channel_file(c_liveradio_file,&liveradio_num,&liveradio,&liveradio_time)  < 0 ) {
            printf("Malformed file %s\n",c_liveradio_file);
            exit(1);
        }
    }   

}

app_t *new_app(dongle_t *dongle, render_t *render, gui_t *gui)
{
    app_t      *app = malloc(sizeof(*app));

    app->render = render;
    app->gui    = gui;
    app->dongle = dongle;
    app->current_media = NULL;
    app->next_media = NULL;

    app_main(NULL,app,0);    

    return app;
}


void delete_app(app_t *app)
{

    delete_menu_tree(app->menu);

    if ( app->current_media ) {
	free(app->current_media);
	app->current_media = NULL;
    }

    if ( app->next_media ) {
	free(app->next_media);
	app->next_media = NULL;
    }

    free(app);
}




static int app_main(menu_t *menu,app_t *app, int sel)
{   
    app->menu = new_menu(app->render,NULL,app,app->gui,MENU_NUMBERS);
    menu_set_title(app->menu,"MediaMVP");
    menu_add(app->menu,"Live TV",app_live_tv);
    menu_add(app->menu,"Live Radio",app_live_radio);
    menu_add(app->menu,"Recordings",app_recordings);
    menu_add(app->menu,"Music",app_music);
    menu_add(app->menu,"Settings",app_settings);
    menu_display(app->menu);
    return 1;
}


/***********************
 *
 *   Handling of the Live TV Menus
 *
 ***********************/

static int app_live_tv(menu_t *menu,app_t *app, int sel)
{
    channel_t *chan;
    menu_t    *child;
    int        i;

    /* Force a re-read of the channel file */
    read_channel_file(c_livetv_file,&livetv_num,&livetv,&livetv_time);

    child = new_menu(app->render,menu,app,app->gui,0);
    menu_set_title(child,"Live TV"); 

    for ( i = 0; i < livetv_num; i++ ) {
        chan = &livetv[i];
        menu_add(child,chan->name,process_livetv);
    }
    menu_display(child);
    return 1;
}

static int process_livetv(menu_t *menu,app_t *app, int sel)
{
    channel_t *chan = &livetv[sel];
    gui_send_play(app->gui,chan->url);
    gui_register_keys(app->gui,REGISTER_MEDIA,livetv_keys,app);
    gui_register_ack(app->gui,REGISTER_MEDIA,livetv_ack,app);
    return 1;
}

static int livetv_ack(int acktype, void *param, unsigned char *buf, int len)
{
    app_t     *app = param;

    switch ( acktype ) {
    case RDC_PLAY:
        if ( (stream_get_type(app->dongle->stream) & MEDIA_MASK) == MEDIA_MPEG ) {
            gui_send_message(app->gui,&rfb_media_menu);
            gui_send_message(app->gui,&rfb_display_off);
        }
        break;
    case RDC_STOP:             
        /* The client has sent a stop message, this could be response to "Stop" key press or
           us sending out a stop so we can switch to the net media file */ 
        if ( app->next_media ) {
	    if ( app->current_media ) {
		free(app->current_media);
		app->current_media = NULL;
	    }
            gui_send_play(app->gui,app->next_media);
	    app->current_media = app->next_media;
            free(app->next_media);
        } else {
	    if ( ( stream_get_lasttype(app->dongle->stream) & MEDIA_MASK ) == MEDIA_MPEG ) {
		gui_send_message(app->gui,&rfb_media_menu);
		gui_send_message(app->gui,&rfb_display_on);
	    }
	    if ( app->current_media ) {
		free(app->current_media);
		app->current_media = NULL;
	    }
	    /* Remove registered handlers */
	    gui_register_keys(app->gui,REGISTER_MEDIA,NULL,app);
	    gui_register_ack(app->gui,REGISTER_MEDIA,NULL,app);
        }


        break;
    default:
        return 0;    /* Kick upstairs to menu ack handler */
    }

    return 1;       /* No need to pass back to menu handler */
}


static int livetv_keys(void *ptr, int code)
{
    app_t    *app = ptr;

    switch ( code ) {
    case kMenu:
    case kStop:
        gui_send_message(app->gui,&rfb_media_stop);
        menu_display(app->menu);
        break;
    case kVolUp:
        gui_send_message(app->gui,&rfb_media_volup);
        break;
    case kVolDn:
        gui_send_message(app->gui,&rfb_media_voldown);
        break;
    case kMute:
        gui_send_message(app->gui,&rfb_media_mute);
        break;
    default:
	return 0;  /* Do more keys */
    }
    return 1;
}


/***********************
 *
 *   Handling of the Live TV Menus
 *
 ***********************/

static int app_live_radio(menu_t *menu,app_t *app, int sel)
{
    channel_t *chan;
    menu_t    *child;
    int        i;

    /* Force a re-read of the channel file */
    read_channel_file(c_liveradio_file,&liveradio_num,&liveradio,&liveradio_time);
    child = new_menu(app->render,menu,app,app->gui,0);
    menu_set_title(child,"Live Radio Stations");  
    for ( i = 0; i < liveradio_num; i++ ) {
        chan = &liveradio[i];
        menu_add(child,chan->name,process_liveradio);
    } 
    menu_display(child);
    return 1;
}

static int process_liveradio(menu_t *menu,app_t *app, int sel)
{
    channel_t *chan = &liveradio[sel];
    printf("Sending media do request %s\n",chan->url);
    if ( app->current_media == NULL ) {	
	app->current_media = strdup(chan->url);
	gui_send_play(app->gui,chan->url);
    } else {
	if ( app->next_media ) {
	    free(app->next_media);
	}
	app->next_media = strdup(chan->url);
	gui_send_message(app->gui,&rfb_media_stop);
    }
    gui_register_keys(app->gui,REGISTER_MEDIA,liveradio_keys,app);
    gui_register_ack(app->gui,REGISTER_MEDIA,liveradio_ack,app);
    return 1;
}


static int liveradio_keys(void *ptr, int code)
{
    app_t     *app = ptr;

    switch ( code ) { 
    case kStop:
        gui_send_message(app->gui,&rfb_media_stop);
        menu_display(app->menu);
        break;
    case kVolUp:
        gui_send_message(app->gui,&rfb_media_volup);
        break;
    case kVolDn:
        gui_send_message(app->gui,&rfb_media_voldown);
        break;
    case kMute:
        gui_send_message(app->gui,&rfb_media_mute);
        break;
    default:
        return 0;      /* Carry on processing menu keys */
    }   
    return 1;
}

static int liveradio_ack(int acktype, void *param, unsigned char *buf, int len)
{
    app_t     *app = param;

    switch ( acktype ) {
    case RDC_STOP:     /* The client has sent a stop message */  
        if ( app->next_media ) {
	    if ( app->current_media ) {
		free(app->current_media);
	    }
            gui_send_play(app->gui,app->next_media);
            app->current_media = app->next_media;
            app->next_media = NULL;
        } else {
	    /* Remove registered handlers */
	    if ( app->current_media ) {
		free(app->current_media);
		app->current_media = NULL;
	    }
	    gui_register_keys(app->gui,REGISTER_MEDIA,NULL,app);
	    gui_register_ack(app->gui,REGISTER_MEDIA,NULL,app);
	}
        break;
    default:
        return 0;    /* Kick it up a level */
    }

    return 1;        /* Don't process anymore */
}






static int app_recordings(menu_t *menu,app_t *app, int sel)
{
    menu_t   *child;
    child = new_menu(app->render,menu,app,app->gui,0);
    menu_set_title(child,"Video Recordings");  


    menu_display(child);
    return 1;
}

static int app_music(menu_t *menu,app_t *app, int sel)
{
    menu_t   *child;
    child = new_menu(app->render,menu,app,app->gui,0);
    menu_set_title(child,"Music");   
    menu_display(child);
    return 1;
}


/***********************
 *
 *   Handling of the Live TV Menus
 *
 ***********************/
static int app_settings(menu_t *menu,app_t *app, int sel)
{
    menu_t          *child;
    static char     *tvmode[]  = { "NTSC", "PAL" };
    static char     *flicker[] = { "None", "Low", "Medium", "High" };
    static char     *output[]  = { "RGB Detected", "Svideo", "Composite", "RGB/Composite" };
    static char     *aspect[]  = { "16:9", "4:3" };

    child = new_menu(app->render,menu,app,app->gui,0);
    menu_set_title(child,"Settings");

 

    /* Add in all possible options - we'll validate things on the callback */
    menu_add_option(child,"TV Mode:",process_settings,
                    MENU_INT,2,tvmode,&app->dongle->tvmode);

    /* A little bit of fiddling - see rules in proces_settings */
    if ( app->dongle->videooutput != 0 ) {
	menu_add_option(child,"Output:",process_settings,
			MENU_INT,4,output,&app->dongle->videooutput);
    } else {
	menu_add(child,"Output: RGB Detected",process_settings);
    }

    menu_add_option(child,"Flicker Control:",process_settings,
                    MENU_INT,3,flicker,&app->dongle->flickermode);

    menu_add_option(child,"Aspect Ratio:",process_settings,
                    MENU_INT,2,aspect,&app->dongle->aspectratio);

    menu_add(child,"Cancel changes",process_settings);
    menu_add(child,"Save settings (may take a while)",process_settings);

    menu_display(child);
    return 1;
}

static int process_settings(menu_t *menu,app_t *app, int sel)
{
   /* Some rules preened from the configuration.html:

    1). For NTSC unit -- in response to GUI query, video output value 
    is always '3', so you show "Video Output" as "Composite/Svideo" no changes
    allowed !!

    2). For SCART Unit --
    a). '0' : then you show "Video Output" as "RGB Detected", No changes 
    allowed !!!
    (b). '1' : then you show "Video Output" as "SVIDEO", it can only be
    changed to RGB/COMPOSITE (value == 2)
    (c). '2': you show "Video Output" as "RGB/COMPOSITE", it can only be 
    changed to SVIDEO (value == 1)

    We try to obey them [but probably fail miserable]!
    */

    /* If we come through with -1 then we should test the settings */
    switch ( sel ) {
    case -1:  /* If we come through with -1 then we should test the settings */
	if ( app->dongle->tvmode == 0 ) {
	    app->dongle->videooutput = 3;
	}
        gui_send_settings(app->gui,RDC_SETTINGS_TEST);
        return 1;  /* Carry on displaying current menu */
    case 4:
	gui_send_settings(app->gui,RDC_SETTINGS_CANCEL);
	return 0;
    case 5:
	gui_send_settings(app->gui,RDC_SETTINGS_SAVE);
	return 0;
    default:
	return 1;   /* Carry on displaying */
    } 
}






static int read_channel_file(char *filename, int *num, channel_t **list, 
			     time_t *update_time)
{
    char         buf[500];
    int          channum;
    channel_t   *chan = *list;
    char        *ptr;
    FILE        *fp;
    struct stat  sb;
    int          i;

    if ( stat(filename,&sb) < 0 ) {
        return 0;
    }

    if ( *update_time < sb.st_mtime ) {
        *update_time = sb.st_mtime;
        for ( i = 0; i < *num; i++ ) {
            free(chan[i].name);
            free(chan[i].url);
        }


        if ( ( fp = fopen(filename,"r") ) == NULL ) {
            return -1;
        }

        if ( chan != NULL ) {
            free(chan);
        }
        chan = NULL;
        channum = 0;

        *list = NULL;
        *num = 0;

        while ( !feof(fp) && fgets(buf,sizeof(buf),fp) ) {
            if ( (ptr = strchr(buf,';') ) == NULL ) {
                continue;
            }
            *ptr = 0;
            ptr++;

            i = channum++;

            chan = realloc(chan,channum * sizeof(channel_t));
            chan[i].name = strdup(buf);
            chan[i].url = strdup(ptr);	    
        }
        *list = chan;
        *num = channum;
    }

    return 0;
}



/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
