#!/usr/bin/python3
#/------------+----------+----------------------------------------------------\
#|  |\  /|    | Filename | mComm.py                                           |
#|  | \/ |    |  Version | 1.2                                                |
#|  |    |    |   Author | Kurt McCullum                                      |
#|  |  /-+-\  +----------+-------------- DESCRIPTION -------------------------+
#|  |  | |    | mComm for Python.                                             |
#|     \---\  |                                                               |
#|         |  | Version 1.0 supports a single folder along with the Sardine   |
#|     \---/  | dictionary for spell checking. Included with this code are    |
#| ---------- | the TS-DOS files (DOS100.CO, DOS200.CO and DOSNEC.CO) for use |
#|  McCullum  | with the UR-II or Sardine ROMs. Also included are the RAM     |
#|  Software  | versions of Sardine (SAR100.CO and SAR200.CO)                 |
#+------------+--------------------------- HISTORY ---------------------------+
#| 03/27/2020 | v1.0 Original file                                            |
#| 03/30/2020 | v1.1 Added support for multiple folders and a default mode    |
#| 03/31/2020 | v1.2 Added the command line argument processor                |
#\------------+---------------------------------------------------------------/

#/---------\
#| Modules |
#\----------------------------------------------------------------------------/
import sys						#System-specific parameters and functions
import serial					#Functions for working with serial ports
import serial.tools.list_ports	#Functions for listing all serial ports
import io						#Core tools for working with steams
import argparse					#Command line argument processor
import os						#for home folder

# Change to the directory of the script before doing anything
os.chdir(os.path.dirname(os.path.realpath(__file__)))
import tpdd 					#TPDD functions

#/-----------------------\
#| Constants & Variables |
#\-----------------------+----------------------------------------------------/
VERSION = "1.20"

#/----------------\
#| OpenSerialPort |
#+----------------+-----------------------------------------------------------\
#| Open the serial port and set it to 19200,N,8,1                             |
#\----------------------------------------------------------------------------/
def OpenSerialPort(PortName):
	ser = serial.Serial()
	ser.port = PortName
	ser.baudrate = 19200
	ser.bytesize = 8
	ser.parity = 'N'
	ser.stopbits = 1
	ser.timeout = 0
	ser.xonxoff = 0
	ser.rtscts = 1
	ser.dtrdsr = 1
	ser.open()
	ser.reset_input_buffer()
	return ser

#/------------\
#| SerialLoop |
#+------------+---------------------------------------------------------------\
#| Main serial port loop that will send commands to the proper functions      |
#\----------------------------------------------------------------------------/
def SerialLoop(myTPDD, inPort):		# This is the main TPDD command parser loop
	CancelService = 0				# Keep looping while this is 0
	CommandLength = 0				# Integer to hold the length of the TPDD command
	TPDD_Command = bytearray()		# Byte array to hold the TPDD requests
	TPDD_Response = bytearray()		# Byte array to hold a standard response to the request
	SarTrack = 0					# Integer to hold the selected Sardine track
	SarSector = 0					# Integer to hold the selected Sardine sector
	ResendFeedback = 0				# Send the CRFeedback a second time

	while CancelService == 0:
		if inPort.in_waiting > 0:
			baInput = bytearray(inPort.read(1))
			if len(TPDD_Command) == 0:
				if baInput[0] == bytearray('Z', 'ASCII')[0]:			# TPDD mode command
					CommandLength = 4						# Set the initial length to the shortest possible command
					TPDD_Command.append(baInput[0])			# Set the first byte of the command

				elif baInput[0] == bytearray('R', 'ASCII')[0]:		# Sardine dictionary request
					CommandLength = 7						# Set command length
					TPDD_Command.append(baInput[0])			# Set the first byte of the command

				elif baInput[0] == bytearray('M', 'ASCII')[0]:		# Change mode request
					CommandLength = 2						# Set command length
					TPDD_Command.append(baInput[0])			# Set the first byte of the command

				elif baInput[0] == bytearray('\r', 'ASCII')[0]:		# CR values are handled in two ways
					if len(myTPDD.CRFeedback) > 0:			# If there is a bytearray set
						inPort.write(myTPDD.CRFeedback)		# send it. Otherwise ignore it
						if ResendFeedback == 0:				# Sardine gets picky about it's feedback
							ResendFeedback = 1				# Sometimes the feedback is sent too fast
															# So this allows it to be sent twice
						else:
							ResendFeedback = 0				# Clear the Resend
							myTPDD.CRFeedback = bytearray()	# Clear the bytearray
							CommandLength = 0				# Set CommandLength back to 0

				else:
					# Invalid command so ignore the byte
					myTPDD.LogMessage("Invalid " + str(baInput[0]), True)

			else:
				TPDD_Command.append(baInput[0])
				if len(TPDD_Command) >= CommandLength:
					if TPDD_Command[0] == bytearray('Z','ASCII')[0] and CommandLength == 4:
						# Resize the command length based on byte 3
						CommandLength = TPDD_Command[3] + 5	# Change the CommandLength to the actual length

					else:
						# This is a valid command so process it
						if TPDD_Command[0] == bytearray('Z', 'ASCII')[0]:
							# Standard mode command
							TPDD_Response = myTPDD.ProcessModeCommand(TPDD_Command)

						elif TPDD_Command[0] == bytearray('R', 'ASCII')[0]:
							# Sardine dictionary request
							SarTrack = ((TPDD_Command[1] - 48) * 10) + (TPDD_Command[2] - 48)
							SarSector = (TPDD_Command[5] - 48)
							TPDD_Response = myTPDD.GetSardineTrack(SarTrack, SarSector)

						elif TPDD_Command[0] == bytearray('M', 'ASCII')[0]:
							# Change mode command
							myTPDD.ReceivedM1 = 1

						if(TPDD_Response != None):
							if len(TPDD_Response) > 0: # If there was a response, send it
								inPort.write(TPDD_Response)

						# Clean the command byte array
						TPDD_Command = bytearray()
						CommandLength = 0

#/------\
#| main |
#+------+---------------------------------------------------------------------\
#| Main entry point of the program                                            |
#\----------------------------------------------------------------------------/
def main():
	parser = argparse.ArgumentParser()
	parser.add_argument("-l", "--list", help="List all available ports", action="store_true")
	parser.add_argument("-s", "--silent", help="Run in silent mode", action="store_true")
	group = parser.add_argument_group('Serial Port')
	group.add_argument('--port', help="The seirial port to open", action="store")
	group = parser.add_argument_group('TPDD Path')
	group.add_argument('--path', help="The base directory for all files (Default = /home/{user}/TPDD)", action="store")
	args = parser.parse_args()

	if args.list:
		print("Listing Ports")
		lPorts = serial.tools.list_ports.comports()
		if len(lPorts) > 0:
			print("Available serial ports")
			for tPort in lPorts:
				print(tPort.device + "\t" + tPort.description)
		else:
			print("No serial ports available")

		exit()

	if(args.port == None):
		lPorts = serial.tools.list_ports.comports()
		if len(lPorts) > 0:
			args.port = lPorts[0].device

		else:
			print("No serial ports available")
			exit()

	if(args.path == None):
		args.path = os.environ['HOME'] + "/TPDD"		# Creat the TPDD directory in the users home directory
		if(os.path.isdir(args.path) == False):
			os.mkdir(args.path)

	myTPDD = tpdd.TPDD(args.path)
	if(args.silent):
		myTPDD.SilentMode = True

	else:
		print("mComm " + VERSION)
		print("----------")
		print("Serial port=" + args.port)
		print("Base folder=" + args.path)
		print("----------")
		print("CTRL-C to exit")

	SerPort = OpenSerialPort(args.port)
	SerialLoop(myTPDD, SerPort)
	exit()


#/----------\
#| __main__ |
#+----------+-----------------------------------------------------------------\
#| Call the Main entry point of the program                                   |
#\----------------------------------------------------------------------------/
if __name__ == '__main__':
    main()
