/*
 *  Abuse - dark 2D side-scrolling platform game
 *  Copyright (c) 2001 Anthony Kruize <trandor@labyrinth.net.au>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 */

#include "config.h"

#include <SDL/SDL.h>
#include <sys/dirent.h>

#include "sound.hpp"
#include "jmalloc.hpp"
#include "readwav.hpp"
#include "specs.hpp"
#include "setup.h"

class effect_handle
{
public:
	effect_handle	*prev;		// Handle to the previous effect
	effect_handle	*next;		// Handle to the next effect
	Uint8			*data;		// Audio data
	Uint8			*pos;		// current playing position in the data
	Uint32			length;		// Length of the data
	Uint32			volume;		// Volume for this effect.

	effect_handle( effect_handle *Next )
	{
		next = Next;
		if( next )
		{
			next->prev = this;
		}
		prev = NULL;
		data = NULL;
		pos = NULL;
	}

	~effect_handle()
	{
		if( next )
		{
			next->prev = prev;
		}
		if( prev )
		{
			prev->next = next;
		}
	}
};

static effect_handle * fx_list = NULL;
extern flags_struct flags;
static int sound_enabled = 0;
static SDL_AudioSpec audioObtained;

// PSP: Limit the amount of sounds that are played at once to keep
// the framerate stable. There are scenes with more than 40 sounds
// that are playing at once, which is a mess anyway.
static int max_active_sounds = 20;
static int active_sounds = 0;

//
// mix_audio
//
// Do the actual playing of the audio by looping through our effects list
// and mixing each sample.
//
void mix_audio(void *udata, Uint8 *stream, int len)
{
	effect_handle *handle = fx_list;

	while( handle )
	{
		if( handle->length > 0 && handle->pos )
		{
			len = ( len > (int)handle->length ? handle->length : len );
			SDL_MixAudio( stream, handle->pos, len, handle->volume);
			handle->pos += len;
			handle->length -= len;
			handle = handle->next;
		}
		else
		{
			// update the list pointer if we're deleting the first node
			if( fx_list == handle )
			{
				fx_list = handle->next;
			}
			effect_handle *tmp = handle->next;
			delete handle;
			handle = tmp;
      active_sounds--;
		}
	}
}

//
// sound_init()
// Initialise audio
//
int sound_init( int argc, char **argv )
{
	SDL_AudioSpec audioWanted;
	char *sfxdir, *datadir;
	DIR *fd = NULL;

	// Disable sound if requested.
	if( flags.nosound )
	{
		// User requested that sound be disabled
		pspDebugScreenPrintf( "Sound : Disabled (-nosound)\n" );
		return 0;
	}

	// Check for the sfx directory, disable sound if we can't find it.
  // PSP: fopen fails on a directory, use opendir instead
  datadir = get_filename_prefix();
	sfxdir = (char *)jmalloc( strlen( datadir ) + 5 + 1, "sfxdir" );
	sprintf( sfxdir, "%s/sfx", datadir );
  if( (fd = opendir(sfxdir)) == NULL )
	{
		// Didn't find the directory, so disable sound.
		pspDebugScreenPrintf( "Sound : Disabled (couldn't find the sfx directory)\n" );
		return 0;
	}
  closedir(fd);
	jfree( sfxdir );

	audioWanted.freq = 11025;
	audioWanted.format = AUDIO_U8;
	audioWanted.channels = 2 - flags.mono;
	audioWanted.samples = 128*4; // PSP: larger buffer
	audioWanted.callback = mix_audio;
	audioWanted.userdata = NULL;

	// Now open the audio device
	if( SDL_OpenAudio( &audioWanted, &audioObtained ) < 0 )
	{
		pspDebugScreenPrintf( "Sound : Unable to open audio - %s\nSound : Disabled (error)\n", SDL_GetError() );
		return 0;
	}

	sound_enabled = 1;
	pspDebugScreenPrintf( "Sound : Enabled\n" );

	SDL_PauseAudio( 0 );

	// It's all good
	return sound_enabled;
}

//
// sound_uninit
//
// Shutdown audio and release any memory left over.
//
void sound_uninit()
{
	if( sound_enabled )
	{
		SDL_PauseAudio( 1 );
		while( fx_list )
		{
			effect_handle *last = fx_list;
			fx_list = fx_list->next;
			jfree( last );
		}
		SDL_CloseAudio();
	}
}

//
// sound_effect constructor
//
// Read in the requested .wav file.
//
sound_effect::sound_effect( char * filename )
{
	if( sound_enabled )
	{
		long sample_speed;
		void* temp_data = (void *)read_wav( filename, sample_speed, size );

    // PSP: Do the audio conversion here instead of in ::play
	  Uint32 cvtBufferSize;
	  SDL_AudioCVT audiocvt;

		SDL_BuildAudioCVT( &audiocvt, AUDIO_U8, 1, 11025, audioObtained.format, audioObtained.channels, audioObtained.freq );
    data = malloc( size * audiocvt.len_mult );
    memcpy( data, temp_data, size );
		audiocvt.buf = (Uint8*)data;
		audiocvt.len = size;
		SDL_ConvertAudio( &audiocvt );
		size = (Uint32)((double)size * audiocvt.len_ratio);

    jfree(temp_data);
	}
}

//
// sound_effect destructor
//
// Release the audio data.
//
sound_effect::~sound_effect()
{
	if( sound_enabled )
	{
		if( data )
		{
			jfree( data );
		}
	}
}

//
// sound_effect::play
//
// Insert a new effect_handle into the list and modify the audio
// if we're doing stereo.
// panpot defines the pan position for the sound effect.
//   0   - Completely to the right.
//   128 - Centered.
//   255 - Completely to the left.
//
void sound_effect::play( int volume, int pitch, int panpot )
{

	if( sound_enabled )
	{
    if (active_sounds > max_active_sounds)
      return;

    SDL_LockAudio();

    active_sounds++;

		fx_list = new effect_handle( fx_list );
		if( fx_list == NULL )
		{
			printf( "Sound : ERROR - Failed to create new effect.\n" );
			SDL_UnlockAudio();
			return;
		}

    // PSP: Removed panning code because the effect is not very pronounced
    // and is in fact completely lost through the internal speakers.
    // It could be added in with SDL_mixer, but it might not be worth the effort.

	  fx_list->data = (Uint8*)this->data;
	  fx_list->pos = (Uint8*)this->data;
	  fx_list->length = this->size;
	  fx_list->volume = volume / 4; // PSP: Default volume is too high

//    printf("play sound %d %d %d\n", (unsigned int)fx_list->data, (unsigned int)fx_list->length, (unsigned int)fx_list->volume);

	  SDL_UnlockAudio();
  }
}


//
// We don't handle songs.  These are just stubs to basically do nothing.
// I tried using SDL_mixer to do this, but with no success.
//

song::song( char const * filename )
{
	data = NULL;
	Name = strcpy((char *)jmalloc( strlen( filename ) + 1, "song name" ), filename );
	song_id = 0;
}

song::~song()
{
	if( playing() )
	{
		stop();
	}
	if( data )
	{
		jfree( data );
	}
	jfree( Name );
}

void song::play( unsigned char volume )
{
	song_id = 1;
}

void song::stop( long fadeout_time )
{
	song_id = 0;
}

int song::playing()
{
	return song_id;
}

void song::set_volume( int volume )
{
//	do nothing...
}
