/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *
 * The contents of this file are subject to the Netscape Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/NPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is Mozilla Communicator client code.
 *
 * The Initial Developer of the Original Code is Netscape Communications
 * Corporation.  Portions created by Netscape are
 * Copyright (C) 1998 Netscape Communications Corporation. All
 * Rights Reserved.
 *
 * Contributor(s): 
 */
#include "nsIDOMHTMLObjectElement.h"
#include "nsIScriptObjectOwner.h"
#include "nsIDOMEventReceiver.h"
#include "nsIHTMLContent.h"
#include "nsGenericHTMLElement.h"
#include "nsHTMLAtoms.h"
#include "nsHTMLIIDs.h"
#include "nsIStyleContext.h"
#include "nsIMutableStyleContext.h"
#include "nsStyleConsts.h"
#include "nsIPresContext.h"
#include "nsDOMError.h"
#include "nsIPresShell.h"
#include "nsIDocument.h"
#include "nsIFrame.h"
#include "nsObjectFrame.h"
#include "nsLayoutAtoms.h"
#include "nsIServiceManager.h"
#include "nsIPluginInstance.h"
#include "nsIJVMPluginInstance.h"
#include "nsIJVMManager.h"
#include "nsILiveconnectManager.h"


class nsHTMLObjectElement : public nsIDOMHTMLObjectElement,
                            public nsIJSScriptObject,
                            public nsIHTMLContent
{
public:
  nsHTMLObjectElement(nsINodeInfo *aNodeInfo);
  virtual ~nsHTMLObjectElement();

  // nsISupports
  NS_DECL_ISUPPORTS

  // nsIDOMNode
  NS_IMPL_IDOMNODE_USING_GENERIC(mInner)

  // nsIDOMElement
  NS_IMPL_IDOMELEMENT_USING_GENERIC(mInner)

  // nsIDOMHTMLElement
  NS_IMPL_IDOMHTMLELEMENT_USING_GENERIC(mInner)

  // nsIDOMHTMLObjectElement
  NS_DECL_IDOMHTMLOBJECTELEMENT

  // nsIJSScriptObject
  NS_IMETHOD GetScriptObject(nsIScriptContext* aContext,
                             void** aScriptObject);
  NS_IMETHOD SetScriptObject(void *aScriptObject);

  virtual PRBool    AddProperty(JSContext *aContext, JSObject *aObj, 
                        jsval aID, jsval *aVp);
  virtual PRBool    DeleteProperty(JSContext *aContext, JSObject *aObj, 
                        jsval aID, jsval *aVp);
  virtual PRBool    GetProperty(JSContext *aContext, JSObject *aObj, 
                        jsval aID, jsval *aVp);
  virtual PRBool    SetProperty(JSContext *aContext, JSObject *aObj, 
                        jsval aID, jsval *aVp);
  virtual PRBool    EnumerateProperty(JSContext *aContext, JSObject *aObj);
  virtual PRBool    Resolve(JSContext *aContext, JSObject *aObj, jsval aID,
                            PRBool* aDidDefineProperty);
  virtual PRBool    Convert(JSContext *aContext, JSObject *aObj, jsval aID);
  virtual void      Finalize(JSContext *aContext, JSObject *aObj);


  // nsIContent
  NS_IMPL_ICONTENT_USING_GENERIC(mInner)

  // nsIHTMLContent
  NS_IMPL_IHTMLCONTENT_USING_GENERIC(mInner)

protected:
  nsresult GetPluginInstance(nsIPluginInstance** aPluginInstance);

protected:
  nsGenericHTMLContainerElement mInner;
  PRBool                        mReflectedApplet;
};

nsresult
NS_NewHTMLObjectElement(nsIHTMLContent** aInstancePtrResult,
                        nsINodeInfo *aNodeInfo)
{
  NS_ENSURE_ARG_POINTER(aInstancePtrResult);
  NS_ENSURE_ARG_POINTER(aNodeInfo);

  nsIHTMLContent* it = new nsHTMLObjectElement(aNodeInfo);
  if (nsnull == it) {
    return NS_ERROR_OUT_OF_MEMORY;
  }
  return it->QueryInterface(NS_GET_IID(nsIHTMLContent), (void**) aInstancePtrResult);
}


nsHTMLObjectElement::nsHTMLObjectElement(nsINodeInfo *aNodeInfo)
{
  NS_INIT_REFCNT();
  mInner.Init(this, aNodeInfo);
  mReflectedApplet = PR_FALSE;
}

nsHTMLObjectElement::~nsHTMLObjectElement()
{
}

NS_IMPL_ADDREF(nsHTMLObjectElement)

NS_IMPL_RELEASE(nsHTMLObjectElement)

nsresult
nsHTMLObjectElement::QueryInterface(REFNSIID aIID, void** aInstancePtr)
{
  NS_IMPL_HTML_CONTENT_QUERY_INTERFACE(aIID, aInstancePtr, this)
  if (aIID.Equals(NS_GET_IID(nsIDOMHTMLObjectElement))) {
    nsIDOMHTMLObjectElement* tmp = this;
    *aInstancePtr = (void*) tmp;
    NS_ADDREF_THIS();
    return NS_OK;
  }
  return NS_NOINTERFACE;
}

nsresult
nsHTMLObjectElement::CloneNode(PRBool aDeep, nsIDOMNode** aReturn)
{
  nsHTMLObjectElement* it = new nsHTMLObjectElement(mInner.mNodeInfo);
  if (nsnull == it) {
    return NS_ERROR_OUT_OF_MEMORY;
  }
  nsCOMPtr<nsIDOMNode> kungFuDeathGrip(it);
  mInner.CopyInnerTo(this, &it->mInner, aDeep);
  return it->QueryInterface(NS_GET_IID(nsIDOMNode), (void**) aReturn);
}

NS_IMETHODIMP
nsHTMLObjectElement::GetForm(nsIDOMHTMLFormElement** aForm)
{
  *aForm = nsnull;/* XXX */
  return NS_OK;
}

NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Code, code)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Align, align)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Archive, archive)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Border, border)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, CodeBase, codebase)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, CodeType, codetype)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Data, data)
NS_IMPL_BOOL_ATTR(nsHTMLObjectElement, Declare, declare)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Height, height)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Hspace, hspace)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Name, name)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Standby, standby)
NS_IMPL_INT_ATTR(nsHTMLObjectElement, TabIndex, tabindex)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Type, type)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, UseMap, usemap)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Vspace, vspace)
NS_IMPL_STRING_ATTR(nsHTMLObjectElement, Width, width)

NS_IMETHODIMP
nsHTMLObjectElement::GetContentDocument(nsIDOMDocument** aContentDocument)
{
  NS_ENSURE_ARG_POINTER(aContentDocument);
  *aContentDocument = nsnull;

  return NS_OK;
}

NS_IMETHODIMP
nsHTMLObjectElement::SetContentDocument(nsIDOMDocument* aContentDocument)
{
  return NS_ERROR_DOM_INVALID_MODIFICATION_ERR;
}

NS_IMETHODIMP
nsHTMLObjectElement::StringToAttribute(nsIAtom* aAttribute,
                                       const nsAReadableString& aValue,
                                       nsHTMLValue& aResult)
{
  if (aAttribute == nsHTMLAtoms::align) {
    if (nsGenericHTMLElement::ParseAlignValue(aValue, aResult)) {
      return NS_CONTENT_ATTR_HAS_VALUE;
    }
  }
  else if (nsGenericHTMLElement::ParseImageAttribute(aAttribute,
                                                     aValue, aResult)) {
    return NS_CONTENT_ATTR_HAS_VALUE;
  }
  return NS_CONTENT_ATTR_NOT_THERE;
}

NS_IMETHODIMP
nsHTMLObjectElement::AttributeToString(nsIAtom* aAttribute,
                                       const nsHTMLValue& aValue,
                                       nsAWritableString& aResult) const
{
  if (aAttribute == nsHTMLAtoms::align) {
    if (eHTMLUnit_Enumerated == aValue.GetUnit()) {
      nsGenericHTMLElement::AlignValueToString(aValue, aResult);
      return NS_CONTENT_ATTR_HAS_VALUE;
    }
  }
  else if (nsGenericHTMLElement::ImageAttributeToString(aAttribute,
                                                        aValue, aResult)) {
    return NS_CONTENT_ATTR_HAS_VALUE;
  }
  return mInner.AttributeToString(aAttribute, aValue, aResult);
}

static void
MapAttributesInto(const nsIHTMLMappedAttributes* aAttributes,
                  nsIMutableStyleContext* aContext,
                  nsIPresContext* aPresContext)
{
  nsGenericHTMLElement::MapImageAlignAttributeInto(aAttributes, aContext, aPresContext);
  nsGenericHTMLElement::MapImageAttributesInto(aAttributes, aContext, aPresContext);
  nsGenericHTMLElement::MapImageBorderAttributeInto(aAttributes, aContext, aPresContext, nsnull);
  nsGenericHTMLElement::MapCommonAttributesInto(aAttributes, aContext, aPresContext);
}

NS_IMETHODIMP
nsHTMLObjectElement::GetMappedAttributeImpact(const nsIAtom* aAttribute,
                                              PRInt32& aHint) const
{
  if (! nsGenericHTMLElement::GetCommonMappedAttributesImpact(aAttribute, aHint)) {
    if (! nsGenericHTMLElement::GetImageBorderAttributeImpact(aAttribute, aHint)) {
      if (! nsGenericHTMLElement::GetImageMappedAttributesImpact(aAttribute, aHint)) {
        if (! nsGenericHTMLElement::GetImageAlignAttributeImpact(aAttribute, aHint)) {
          aHint = NS_STYLE_HINT_CONTENT;
        }
      }
    }
  }

  return NS_OK;
}


NS_IMETHODIMP
nsHTMLObjectElement::GetAttributeMappingFunctions(nsMapAttributesFunc& aFontMapFunc,
                                                  nsMapAttributesFunc& aMapFunc) const
{
  aFontMapFunc = nsnull;
  aMapFunc = &MapAttributesInto;
  return NS_OK;
}


NS_IMETHODIMP
nsHTMLObjectElement::HandleDOMEvent(nsIPresContext* aPresContext,
                                    nsEvent* aEvent,
                                    nsIDOMEvent** aDOMEvent,
                                    PRUint32 aFlags,
                                    nsEventStatus* aEventStatus)
{
  return mInner.HandleDOMEvent(aPresContext, aEvent, aDOMEvent,
                               aFlags, aEventStatus);
}


NS_IMETHODIMP
nsHTMLObjectElement::SizeOf(nsISizeOfHandler* aSizer, PRUint32* aResult) const
{
  return mInner.SizeOf(aSizer, aResult, sizeof(*this));
}

NS_METHOD
nsHTMLObjectElement::GetPluginInstance(nsIPluginInstance** aPluginInstance)
{
  NS_ENSURE_ARG_POINTER(aPluginInstance);
  *aPluginInstance = nsnull;

  nsresult result;
  nsCOMPtr<nsIPresContext> context;
  nsCOMPtr<nsIPresShell> shell;
  
  if (mInner.mDocument) {
    // Make sure the presentation is up-to-date
    result = mInner.mDocument->FlushPendingNotifications();
    if (NS_FAILED(result)) {
      return result;
    }
  }
  
  result = nsGenericHTMLElement::GetPresContext(this, 
                                                getter_AddRefs(context));
  if (NS_FAILED(result)) {
    return result;
  }
  
  result = context->GetShell(getter_AddRefs(shell));
  if (NS_FAILED(result)) {
    return result;
  }
  
  nsIFrame* frame;
  result = shell->GetPrimaryFrameFor(this, &frame);
  if (NS_FAILED(result)) {
    return result;
  }
  
  if (frame) {
    nsCOMPtr<nsIAtom> type;
    
    frame->GetFrameType(getter_AddRefs(type));
    
    if (type.get() == nsLayoutAtoms::objectFrame) {
      // XXX We could have created an interface for this, but Troy
      // preferred the ugliness of a static cast to the weight of
      // a new interface.
      
      nsObjectFrame* objectFrame = NS_STATIC_CAST(nsObjectFrame*, frame);
      
      return objectFrame->GetPluginInstance(*aPluginInstance);
    }

    return NS_OK;
  }

  return NS_ERROR_FAILURE;
}

// nsIJSScriptObject

NS_IMETHODIMP
nsHTMLObjectElement::GetScriptObject(nsIScriptContext* aContext,
                                     void** aScriptObject)
{
  if (mInner.mDOMSlots && mInner.mDOMSlots->mScriptObject)
    return mInner.GetScriptObject(aContext, aScriptObject);

  nsresult rv;
  *aScriptObject = nsnull;

  nsCOMPtr<nsIPluginInstance> pi;
  rv = GetPluginInstance(getter_AddRefs(pi));

  // If GetPluginInstance() fails it means there's no frame for this element
  // yet, in that case we return the script object for the element but we
  // don't cache it so that the next call can get the correct script object
  // if the plugin instance is available at the next call.
  if (NS_FAILED(rv)) {
    mInner.GetScriptObject(aContext, aScriptObject);

    if (mInner.mDocument) {
      // Since we're resetting the script object to null we'll remove the
      // reference to it so that we won't add the same named reference
      // again the next time someone requests the script object.
      aContext->RemoveReference((void *)&mInner.mDOMSlots->mScriptObject,
                                mInner.mDOMSlots->mScriptObject);
    }

    mInner.SetScriptObject(nsnull);

    return NS_OK;
  }

  // If it is not a plugin, then no special processing required
  if (!pi) {
    return mInner.GetScriptObject(aContext, aScriptObject);
  }

  // Check if this is a Java applet first
  nsCOMPtr<nsIJVMPluginInstance>  pJVMPluginInstance(do_QueryInterface(pi, &rv));
  if (NS_SUCCEEDED(rv) && pJVMPluginInstance) {
    NS_WITH_SERVICE(nsIJVMManager, jvm, nsIJVMManager::GetCID(), &rv);
    if (NS_SUCCEEDED(rv)) {
      // Get the JS object corresponding to this dom node.  This will become
      // the javascript prototype object of the object we eventually reflect to the
      // DOM.
      JSObject* elementObject = nsnull;
      rv = mInner.GetScriptObject(aContext, (void**)&elementObject);
      if (NS_FAILED(rv) || !elementObject)
        return rv;

      // Flush pending reflows to ensure the plugin is instansiated, assuming
      // it's visible
      if (mInner.mDocument) {
        mInner.mDocument->FlushPendingNotifications();
      }

      if (!mReflectedApplet) {
        // Get the Java object corresponding to this applet, and reflect it into
        // JavaScript using the LiveConnect manager.
        JSContext* context = (JSContext*)aContext->GetNativeContext();
        JSObject* wrappedAppletObject = nsnull;
        jobject appletObject = nsnull;
        rv = pJVMPluginInstance->GetJavaObject(&appletObject);
        if (NS_OK == rv) {
          nsILiveConnectManager* manager = NULL;
          rv = nsServiceManager::GetService(nsIJVMManager::GetCID(),
                                            NS_GET_IID(nsILiveConnectManager),
                           (nsISupports **)&manager);
          if (rv == NS_OK) {
            rv = manager->WrapJavaObject(context, appletObject, &wrappedAppletObject);
            nsServiceManager::ReleaseService(nsIJVMManager::GetCID(), manager);
          }
        }

        // Set the __proto__ field of the applet object to be the element script object.
        if (nsnull != wrappedAppletObject) {
          JS_SetPrototype(context, wrappedAppletObject, elementObject);
          mInner.SetScriptObject(wrappedAppletObject);
          mReflectedApplet = PR_TRUE;
        }
        *aScriptObject = wrappedAppletObject;
      }
      else {
        rv = mInner.GetScriptObject(aContext, aScriptObject);
      }
      return rv;
    }
    else {
      return mInner.GetScriptObject(aContext, aScriptObject);
    }
  }
  else {
    return mInner.GetScriptObject(aContext, aScriptObject);
  }
}

NS_IMETHODIMP
nsHTMLObjectElement::SetScriptObject(void *aScriptObject)
{
  return mInner.SetScriptObject(aScriptObject);
}

PRBool    
nsHTMLObjectElement::AddProperty(JSContext *aContext, JSObject *aObj, jsval aID, jsval *aVp)
{
  return mInner.AddProperty(aContext, aObj, aID, aVp);
}

PRBool    
nsHTMLObjectElement::DeleteProperty(JSContext *aContext, JSObject *aObj, jsval aID, jsval *aVp)
{
  return mInner.DeleteProperty(aContext, aObj, aID, aVp);
}

PRBool    
nsHTMLObjectElement::GetProperty(JSContext *aContext, JSObject *aObj, jsval aID, jsval *aVp)
{
  return mInner.GetProperty(aContext, aObj, aID, aVp);
}

PRBool    
nsHTMLObjectElement::SetProperty(JSContext *aContext, JSObject *aObj, jsval aID, jsval *aVp)
{
  return mInner.SetProperty(aContext, aObj, aID, aVp);
}

PRBool    
nsHTMLObjectElement::EnumerateProperty(JSContext *aContext, JSObject *aObj)
{
  return mInner.EnumerateProperty(aContext, aObj);
}

PRBool    
nsHTMLObjectElement::Resolve(JSContext *aContext, JSObject *aObj, jsval aID,
                            PRBool *aDidDefineProperty)
{
  return mInner.Resolve(aContext, aObj, aID, aDidDefineProperty);
}

PRBool    
nsHTMLObjectElement::Convert(JSContext *aContext, JSObject *aObj, jsval aID)
{
  return mInner.Convert(aContext, aObj, aID);
}

void      
nsHTMLObjectElement::Finalize(JSContext *aContext, JSObject *aObj)
{
  mInner.Finalize(aContext, aObj);
}

