/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TransforMiiX XSLT processor.
 *
 * The Initial Developer of the Original Code is
 * IBM Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2002
 * IBM Corporation. All Rights Reserved.
 *
 * Contributor(s):
 *   IBM Corporation
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "Expr.h"
#include "NodeSet.h"
#include "txIXPathContext.h"
#include "txAXPathWalkCallback.h"

txNamedAttributeStep::txNamedAttributeStep(Type aType, PRInt32 aNsID,
                                           nsIAtom* aPrefix,
                                           nsIAtom* aLocalName)
    : mType(aType),
      mNsID(aNsID),
      mPrefix(aPrefix),
      mLocalName(aLocalName)
{
}

nsresult
txNamedAttributeStep::evaluate(txIEvalContext* aContext,
                               txAExprResult** aResult)
{
    *aResult = nsnull;

    nsresult rv = NS_OK;
    Node* contextNode = aContext->getContextNode();
    Element* elem = nsnull;
    if (contextNode->getNodeType() == Node::ELEMENT_NODE) {
        elem = NS_STATIC_CAST(Element*, contextNode);
    }

    switch (mType) {
        case NODESET:
        {
            nsRefPtr<NodeSet> nodes;
            rv = aContext->recycler()->getNodeSet(getter_AddRefs(nodes));
            NS_ENSURE_SUCCESS(rv, rv);

            if (elem) {
                Node* node = elem->getAttrNode(mLocalName, mNsID);
                if (node) {
                    nodes->append(node);
                }
            }
            *aResult = nodes;
            NS_ADDREF(*aResult);

            return NS_OK;
        }
        case HAS_ATTR:
        {
            aContext->recycler()->getBoolResult(elem &&
                                                elem->hasAttr(mLocalName,
                                                              mNsID),
                                                aResult);

            return NS_OK;
        }
        case ATTR_VALUE:
        {
            nsRefPtr<StringResult> strRes;
            rv = aContext->recycler()->getStringResult(getter_AddRefs(strRes));
            NS_ENSURE_SUCCESS(rv, rv);

            if (elem) {
                elem->getAttr(mLocalName, mNsID, strRes->mValue);
            }
            *aResult = strRes;
            NS_ADDREF(*aResult);

            return NS_OK;
        }
        case VALUE_COMPARISON:
        {
            nsAutoString attrValue;
            aContext->recycler()->getBoolResult(elem &&
                                                elem->getAttr(mLocalName,
                                                              mNsID,
                                                              attrValue) &&
                                                attrValue.Equals(*mString),
                                                aResult);
            return NS_OK;
        }
    }

    return NS_ERROR_UNEXPECTED;
}

void
txNamedAttributeStep::toString(nsAString& aDest)
{
    aDest.Append(PRUnichar('@'));
    if (mPrefix) {
        nsAutoString prefix;
        mPrefix->ToString(prefix);
        aDest.Append(prefix);
        aDest.Append(PRUnichar(':'));
    }
    nsAutoString localName;
    mLocalName->ToString(localName);
    aDest.Append(localName);

    if (mType == VALUE_COMPARISON) {
        aDest.Append(NS_LITERAL_STRING(" = "));

        PRUnichar ch = '\'';
        if (mString->FindChar(ch) != kNotFound) {
            ch = '\"';
        }
        aDest.Append(ch);
        aDest.Append(*mString);
        aDest.Append(ch);
    }
}

nsresult
txNamedAttributeStep::iterateSubItems(txAXPathWalkCallback* aCallback)
{
    return NS_OK;
}

Expr::ExprType
txNamedAttributeStep::getType()
{
    return NAMEDATTRIBUTESTEP_EXPR;
}

Expr::ResultType
txNamedAttributeStep::getReturnType()
{
    switch (mType) {
        case NODESET:
            return NODESET_RESULT;

        case HAS_ATTR:
        case VALUE_COMPARISON:
            return BOOLEAN_RESULT;

        case ATTR_VALUE:
            return STRING_RESULT;
    }

    NS_NOTREACHED("how'd we get here?");
    return ANY_RESULT;
}

Expr::ContextSensitivity
txNamedAttributeStep::getContextSensitivity()
{
    return NODE_CONTEXT;
}
