package com.ibm.ulc.ui.lists;

/*
 * Instances of this class hold a table of UiRows. They are the items of a <code>IListModel</code>
 * that have been loaded into the UI
 */
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.ui.UIProxy;
import com.ibm.ulc.ui.base.IEnableListener;
import com.ibm.ulc.ui.base.IEnableListenerTarget;
import com.ibm.ulc.util.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.EventListener;
import javax.swing.ListModel;
import javax.swing.table.TableModel;
import javax.swing.event.ListDataListener;
import javax.swing.event.TableModelListener;
public  class UIItemList extends UIProxy implements IItemCacheListener {
	protected Vector fIndexToRow = new Vector();
	protected UlcHashtable fOidToIndex = new UlcHashtable();
	protected Vector fColumns = new Vector();
	protected UiItemCache fItemCache = null;
	protected Vector fItemListListeners = new Vector();
	private int fPageSize = 20;
	protected int fTotalNumberOfRows = 0;
	protected int fRequestsStarted = 0;
	protected int fRequestFrom = -1;
	protected int fRequestTo = -1;
public void add(UIColumn uiColumn) {
	if (uiColumn != null) {
		uiColumn.setIndex(fColumns.size());
		fColumns.addElement((Object) uiColumn);
	}
}
/**
 * Add the given component as listener, who will be informed
 * when the state of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * @param component : The Component which will act as listener.
 */
public void addListener(IItemListListener listener) {
	fItemListListeners.addElement(listener);
}
/**
 * Add the given listener (instances of the various swing list/table/etc model classes)
 * as listener. The listener will be notified before any normal listeners are notified.
 * this is because the swing widgets should be updated before the UI proxies are
 *
 * @param listener : The Component which will act as listener.
 */
public void addModelListener(IItemListListener listener) {
	fItemListListeners.insertElementAt(listener, 0);
}
protected void contentsChanged(ORBConnection conn, Anything args) {
	fIndexToRow = new Vector();
	fOidToIndex = new UlcHashtable();
	startRequest();
	setIndices(args);
	stopRequest();
	notify(TABLE_MODEL_CONTENTS_CHANGED, new String[0], 0, fTotalNumberOfRows);
}
/**
 * Returns the value at the specified index.  
 */
protected void ensureCapacity(Vector vector, int newCapacity, Object filler) {
	vector.ensureCapacity(newCapacity);
	for (int i = vector.size(); i < newCapacity; i++) {
		vector.addElement(filler);
	}
}
/** 
 * Returns the length of the list.
 */
protected Anything extractRowIds(Anything args) {
	return fItemCache.extractRowIds(args, "rowids");
}
/**
 * make sure to request the number of rows specified by the receiver's item cache.
 * the added rows are added to the receiver's itemList, so as to actually instantiate
 * them when the requested rowIds are handled 
 */
private void fixupForPageSize() {
	
	int requestedPageSize = fRequestTo - fRequestFrom + 1;
	int toBeRequested = Math.min(fPageSize - requestedPageSize, fTotalNumberOfRows - 1 - fRequestTo);
	if (toBeRequested > 0) {
		int offset = fRequestTo; // cache it, because fRequestTo will be modified in the loop
		for (int i = 0; i <= toBeRequested; i++) {
			getRowAt(offset + i);
		}
	}
}
/** 
 * Returns the length of the list.
 */
protected String[] getColumnNames() {
	String[] columns = new String[fColumns.size()];
	for (int i = 0; i < columns.length; i++) {
		columns[i] = ((UIColumn) fColumns.elementAt(i)).getColumnId();
	}
	return columns;
}
/** 
 * Returns the length of the list.
 */
private int getElementCountFromRangeAnything(Anything ranges) {
	int answer = 0;
	for (int i = 0; i < ranges.size(); i++) {
		Anything rangeAny = ranges.get(i);
		answer = answer + (rangeAny.get("e", -1)) - (rangeAny.get("s", -1)) + 1;
	}
	return answer;
}
/**
 * return the first index of the row within the specified range that has not been requested.
 * return -1 if the entire range has been requested.
 */
private int getFirstNonNullRowStartingFrom(int startIndex, int endIndex) {
	for (int i = startIndex; i <= endIndex; i++) {
		Integer oid = ((Integer) fIndexToRow.elementAt(i));
		if (oid.intValue() == -2) {
			return i;
		}
	}
	return -1;
}
/** 
 * Returns the length of the list.
 */
public UiItemCache getItemCache() {
	return fItemCache;
}
/**
 * Answer the receiver's current itemList.
 *
 * The receiver's <code>UIItemList</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 * @see UIItemList
 *
 */
public UIItemList getItemList() {
	return this;
}
/**
 * Returns the value at the specified index.  
 */
public IRow getRow(int oid) {
	return fItemCache.getRow(oid);
}
/**
 * Returns the value at the specified index.  
 */
protected IRow getRowAt(int index) {
	int oid = -1;
	try {
		oid = ((Integer) fIndexToRow.elementAt(index)).intValue();
	}
	catch (ArrayIndexOutOfBoundsException e) {
		oid = -1;
	}
	if (oid < 0) {
		oid = ((IRow) handleMissingRow(index)).getOid();
	}
	return getRow(oid);
}
/**
 * Returns the number of records managed by the data source object. A
 * <B>JTable</B> uses this method to determine how many rows it
 * should create and display.  This method should be quick, as it
 * is call by <B>JTable</B> quite frequently.
 *
 * @return the number or rows in the model
 * @see #getColumnCount()
 */
public int getRowCount() {
	return getSize();
}
/**
 * Returns the value at the specified index.  
 */
protected int getRowIdAt(int index) {
	int oid = -1;
	try {
		oid = ((Integer) fIndexToRow.elementAt(index)).intValue();
	}
	catch (ArrayIndexOutOfBoundsException e) {
		oid = -1;
	}
	return oid;
}
/** 
 * Returns the length of the list.
 */
public int getSize() {
	return fTotalNumberOfRows;
}
/**
 * Returns an attribute value for the cell at <I>columnIndex</I>
 * and <I>rowIndex</I>.
 *
 * @param	rowIndex	the row whose value is to be looked up
 * @param	columnIndex 	the column whose value is to be looked up
 * @return	the value Object at the specified cell
 */
public Object getValueAt(int rowIndex, String attributeName) {
	getRowAt(rowIndex);
	return getValueAtOid(indexToOid(rowIndex), attributeName);
}
/**
 * Returns an attribute value for the cell at <I>columnIndex</I>
 * and <I>oid</I>.
 *
 * @param	rowIndex	the row whose value is to be looked up
 * @param	columnIndex 	the column whose value is to be looked up
 * @return	the value Object at the specified cell
 */
protected Object getValueAtOid(int oid, String attributeName) {
	if (getItemCache() != null)
		return getItemCache().getValue(oid, attributeName);
	return null;
}
/**
 * Returns the value at the specified index.  
 */
protected IRow handleMissingRow(int rowIndex) {
	IRow row = UiInvalidRow.getInstance();
	if (getRowCount() > 0) {
		requestOids(rowIndex, rowIndex);
		ensureCapacity(fIndexToRow, rowIndex + 1, new Integer(-2));
	}
	return row;
}
/*
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setData")) {
		startRequest();
		IRow[] newRows = setData(args);
		getItemCache().setData(args, false);
		notifyCellsChanged(newRows, getColumnNames());
		stopRequest();
		return;
	}
	if (request.equals("setIndices")) {
		startRequest();
		IRow[] rows = setIndices(args);
		notify(TABLE_MODEL_ROWS_CHANGED, null, rows);
		stopRequest();
		return;
	}
	if (request.equals("insertRows")) {
		insertRows(args);
		return;
	}
	if (request.equals("contentsChanged")) {
		contentsChanged(conn, args);
		return;
	}
	// The command below seems to be dead i.e. no longer sent - but we leave it for now	
	if (request.equals("setOids")) {
		setOids(args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/** 
 * Returns the length of the list.
 */
protected int indexToOid(int index) {
	Integer answer = new Integer(-1);
	try {
		answer = (Integer) fIndexToRow.elementAt(index);
	}
	catch (ArrayIndexOutOfBoundsException e) {
	};
	return answer.intValue();
}
/** 
 * Returns the length of the list.
 */
public void initialize(ORBConnection connection, int identifier) {
	fConnection = connection;
	fId = identifier;
}
/**
 * Set the index-to-oid mappings specified in args. The format expected is
 *
 *	'if' (Index From)
 *	'it' (Index To)
 *	'rowids' (the oids for the index range)
 *
 * 'rowids' may be omitted if the oids are identical to their indices.
 * a range is extracted from the anything in #extractRowIds
 */
protected void insertData(Anything range, Vector addedRows) {
	if (range == null)
		return;
	int from = range.get("if", 0);
	int to = range.get("it", 0);
	Anything rowIds = extractRowIds(range);
	if (from >= fIndexToRow.size())
		ensureCapacity(fIndexToRow, to, new Integer(-2));
	for (int i = 0; i < rowIds.size(); i++) {
		int oid = rowIds.get(i).asInt(-1);
		fIndexToRow.insertElementAt(new Integer(oid), from + i);
		IRow row = getRow(oid);
		addedRows.addElement(row);
	}
	rebuildOidToIndex();
}
/**
 * Returns the value at the specified index.  
 */
protected void insertRows(Anything args) {
	startRequest();
	fTotalNumberOfRows = args.get("rows", fTotalNumberOfRows);
	int counter = 0;
	Anything ranges = args.get("ranges");
	Vector addedRows = new Vector();
	for (int i = 0; i < ranges.size(); i++) {
		Anything rangeAny = ranges.get(i);
		insertData(rangeAny, addedRows);
	}
	IRow[] rows = new IRow[addedRows.size()];
	addedRows.copyInto(rows);
	notify(TABLE_MODEL_ROWS_ADDED, null, rows);
	stopRequest();
}
void notify(int changeType, String[] attributeNames, int[] changedIndices) {
	if (changedIndices.length == 0)
		return;
	int[][] intervals = new UlcSorter().sortToIntervals(changedIndices);
	for (int i = 0; i < intervals.length; i++) {
		notify(changeType, attributeNames, intervals[i][0], intervals[i][1]);
	}
}
/**
 * handle the case of the missing row
 */
protected void notify(int changeType, String[] attributeNames, IRow[] changedRows) {
	if (changedRows.length == 0)
		return;
	Vector indexVector = new Vector();
	for (int i = 0; i < changedRows.length; i++) {
		int oid = oidToIndex(changedRows[i].getOid());
		if (oid != -1)
			indexVector.addElement(new Integer(oid));
	}
	int[] indices = new int[indexVector.size()];
	for (int i = 0; i < indices.length; i++) {
		indices[i] = ((Integer) indexVector.elementAt(i)).intValue();
	}
	notify(changeType, attributeNames, indices);
}
protected void notify(int type, String[] attributeNames, int start, int end) {
	startRequest();
	for (int i = 0; i < fItemListListeners.size(); i++) {
		((IItemListListener) fItemListListeners.elementAt(i)).notify(type, attributeNames, start, end);
	}
	stopRequest();
}
/**
 * handle the case of the missing row
 */
public void notifyCellChanged(IRow changedRow, String attributeName) {
	int index = oidToIndex(changedRow.getOid());
	notify(TABLE_MODEL_CELL_CHANGED, new String[] {attributeName}, index, index);
}
/**
 * handle the case of the missing row
 */
public void notifyCellsChanged(IRow[] changedRows, String[] attributeNames) {
	notify(TABLE_MODEL_CELL_CHANGED, attributeNames, changedRows);
}
/**
 * handle the case of the missing row
 */
public void notifyRowsChanged(IRow[] changedRows) {
	notify(TABLE_MODEL_ROWS_CHANGED, new String[0], changedRows);
}
/**
 * handle the case of the missing row
 */
public void notifyRowsRemoved(IRow[] removedRows) {
	Vector toBeRemoved = new Vector();
	for (int i = 0; i < removedRows.length; i++) {
		fTotalNumberOfRows--;
		if (oidToIndex(removedRows[i].getOid()) > -1) {
			toBeRemoved.addElement(removedRows[i]);
		}
	}
	IRow[] reallyRemoved = new IRow[toBeRemoved.size()];
	toBeRemoved.copyInto(reallyRemoved);
	int[] indices = new int[reallyRemoved.length];
	for (int i = 0; i < reallyRemoved.length; i++) {
		indices[i] = oidToIndex(reallyRemoved[i].getOid());
	}
	for (int i = 0; i < reallyRemoved.length; i++) {
		fIndexToRow.removeElementAt(oidToIndex(reallyRemoved[i].getOid()));
		rebuildOidToIndex();
	}
	notify(TABLE_MODEL_ROWS_REMOVED, new String[0], indices);
}
/**
 * handle the case of the missing row
 */
public void notifyRowsSet(IRow[] setRows) {
	notifyRowsChanged(setRows);
}
/** 
 * Returns the length of the list.
 */
protected int oidToIndex(int oid) {
	Integer answer = (Integer) fOidToIndex.get(new Integer(oid), new Integer(-1));
	return answer.intValue();
}
protected void rebuildOidToIndex() {
	fOidToIndex.clear();
	for (int i = 0; i < fIndexToRow.size(); i++) {
		Integer oid = (Integer) fIndexToRow.elementAt(i);
		fOidToIndex.put(oid, new Integer(i));
	}
}
/**
 * remove the given component as listener
 *
 * @param component : The Component removed from listeners.
 */
public void removeListener(IItemListListener listener) {
	fItemListListeners.removeElement(listener);
}
/**
 * Request the indices for the specified oids. This is needed only when
 * the selected items of the receiver have no mapping for the receiver.
 * The answer for this request is handled in #setIndices.
 *
 */
protected void requestOidIndices(int[] oids) {
	Anything anything = new Anything();
	Anything oidAny = new Anything();
	for (int i = 0; i < oids.length; i++) {
		oidAny.append(new Anything(oids[i]));
	}
	anything.put("rowids", oidAny);
	sendULC("requestIndices", anything);
}
/**
 * Request the ULC for the oids of indices previously requested by a widget
 * of the receiver. Make sure to request as full a page as possible and needed.
 * The answer for this request is handled in #setData.
 */
protected void requestOids() {
	if (fRequestFrom > -1) {
		Anything anything = new Anything();
		fixupForPageSize();
		int firstIndex = getFirstNonNullRowStartingFrom(fRequestFrom, fRequestTo);
		fRequestFrom = firstIndex;
		if (firstIndex == -1) {
			fRequestFrom = -1;
			fRequestTo = -1;
			return;
		}
		fixupForPageSize();
		for (int i = fRequestFrom; i < fRequestTo; i++)
			fIndexToRow.setElementAt(new Integer(-1), i);
		anything.put("s", fRequestFrom);
		anything.put("e", fRequestTo);
		fRequestFrom = -1;
		fRequestTo = -1;
		sendULC("getData", anything);
	}
}
/**
 * The indices from to to have been touched by a widget. The oids for these
 * indices are still missing. Adjust the appropriate slots so that the existing
 * range is extended.
 */
protected void requestOids(int from, int to) {
	if (fRequestFrom == -1) {
		fRequestFrom = from;
		fRequestTo = to;
	} else {
		fRequestFrom = java.lang.Math.min(fRequestFrom, from);
		fRequestTo = java.lang.Math.max(fRequestTo, to);
	}
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	super.restoreState(conn, args);
	UITableModel model = (UITableModel) getManaged(UITableModel.class, conn, args.get("model"));
	if (model == null) {
		if (getItemCache() == null)
			trouble("restoreState", "no TableModel");
	} else {
		setCache(model.getItemCache());
	}
	fTotalNumberOfRows = args.get("rows", fTotalNumberOfRows);
	int oid = args.get("oid", 0);
	if ((conn.find(oid)) == null) {
		super.restoreState(conn, args);
	} else {
		fId = args.get("oid", 0);
		if (args.isDefined("list"))
			addMany(conn, args.get("list"));
	}
	fPageSize = args.get("prefetchCount", fPageSize);
	if (fItemCache != null) {
		startRequest();
		IRow[] rows = setData(args.get("rowids"));
		notifyCellsChanged(rows, getColumnNames());
		stopRequest();
	}
	//notifyContentsChanged();
}
/** 
 * Returns the length of the list.
 */
public void setCache(UiItemCache cache) {
	if (fItemCache != null) {
		fItemCache.removeListener(this);
	}
	fItemCache = cache;
	fItemCache.addListener(this);
}
public void setConnection(ORBConnection connection) {
	fConnection = connection;
}
/**
 * Set the index-to-oid mappings specified in args. The format expected is
 *
 *	'if' (Index From)
 *	'it' (Index To)
 *	'rowids' (the oids for the index range)
 *
 * 'rowids' may be omitted if the oids are identical to their indices or
 * if the rowids are sent as a range defined by 'of' (from) and 'ot' (to).
 *
 * a range is extracted from the anything in #extractRowIds
 */
protected IRow[] setData(Anything range) {
	if (range == null)
		return new IRow[0];
	Vector newRows = new Vector();
	int from = range.get("if", 0);
	int to = range.get("it", 0);
	Anything rowIds = extractRowIds(range);
	ensureCapacity(fIndexToRow, to + 1, new Integer(-2));
	for (int i = 0; i < rowIds.size(); i++) {
		int oid = rowIds.get(i).asInt(-1);
		fIndexToRow.setElementAt(new Integer(oid), from + i);
		fOidToIndex.put(new Integer(oid), new Integer(from + i));
		IRow row = getRow(oid);
		newRows.addElement(row);
	}
	IRow[] rows = new IRow[newRows.size()];
	newRows.copyInto(rows);
	return rows;
}
/**
 * Returns the value at the specified index.  
 */
protected IRow[] setIndices(Anything args) {
	fTotalNumberOfRows = args.get("rows", fTotalNumberOfRows);
	Anything ranges = args.get("r"); // getting the ranges
	if (ranges == null)
		return new IRow[0];
	Vector newRows = new Vector();
	for (int i = 0; i < ranges.size(); i++) {
		Anything range = ranges.get(i);
		IRow[] addedRows = setData(range);
		for (int ai = 0; ai < addedRows.length; ai++)
			newRows.addElement(addedRows[ai]);
	}
	IRow[] rows = new IRow[newRows.size()];
	newRows.copyInto(rows);
	return rows;
}
/**
 * Returns the value at the specified index.  
 */
protected void setOids(Anything args) {
	startRequest();
	if (args.isDefined("rows"))
		fTotalNumberOfRows = args.get("rows", 0);
	int start = args.get("from", 0);
	int end = args.get("to", 0);
	Anything rowIds = extractRowIds(args);
	IRow[] newRows = new IRow[rowIds.size()];
	ensureCapacity(fIndexToRow, getRowCount() + newRows.length, new Integer(-2));
	for (int i = 0; i < rowIds.size(); i++) {
		Anything entry = rowIds.get(i);
		int oid = entry.get("r", -1);
		int index = entry.get("i", -1);
		if (oid <= -1 || index <= -1) {
			trouble("setOid", "item not fully specified");
		}
		else {
			ensureCapacity(fIndexToRow, index + 1, new Integer(-1));
			fIndexToRow.setElementAt(new Integer(oid), index);
			fOidToIndex.put(new Integer(oid), new Integer(index));
			IRow row = getRow(oid);
			newRows[i] = row;
			Enumeration preloadedAttributes = fItemCache.fPreloadAttributes.elements();
			while (preloadedAttributes.hasMoreElements()) {
				row.getValueAt((String) preloadedAttributes.nextElement(), fItemCache);
			}
		}
	}
	notifyCellsChanged(newRows, getColumnNames());
	stopRequest();
}
public void setRowToBeEdited(int index) {
	getItemCache().setRowToBeEdited(indexToOid(index));
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public void setValue(int index, String attributeName, Object value, int notificationPolicy) {
	getItemCache().setValue(indexToOid(index), attributeName, value, notificationPolicy);
}
/**
 * Returns the value at the specified index.  
 */
public void startRequest() {
	if (fItemCache == null)
		return;
	if (fRequestsStarted == 0)
		fItemCache.startRequest();
	fRequestsStarted++;
}
/**
 * Returns the value at the specified index.  
 */
public void stopRequest() {
	fRequestsStarted--;
	if (fRequestsStarted == 0) {
		requestOids();
		fItemCache.stopRequest();
	}
}
}
