package com.ibm.ulc.ui.lists;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;

import com.ibm.ulc.comm.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.base.*;
import com.ibm.ulc.ui.*;

/**
 * Implements the UI side of ULCContainerFlowedIconList
 */
public class UIContainer extends UIComponent implements IDefaults {

	// The internal basic component used to display the container
	transient private JPanel	fIconContainer= createJPanel();
	
	transient private JScrollPane	fScrollPane= new JScrollPane() {
				public void doLayout() {
					this.getViewport().getView().invalidate();
					super.doLayout();
				}
			};
	
	// The selected items
	transient private Vector fSelectedItems= new Vector();

	// Hold the size of the biggest icon
	transient private Dimension fMaxIconSize= new Dimension(0, 0);

	// Needed for multi-selection allowed
	transient private boolean fMultiSelectionAllowed= true;

	// Needed for handling the icon dragging
	transient private boolean fDragIconsAllowed= false; // Depends on fAutoArranging
	transient private boolean fDragging= false;
	transient private int fDragStartX= 0;
	transient private int fDragStartY= 0;
	transient private FlowLayout fLayoutManager= null;

	// Needed for label position rel. to icon
	transient private int fLabelPosition= BOX_BOTTOM;

	// Tells whether icons are auto-arranged or not
	transient private boolean fAutoArranging= true;  // Depends on fDragIconsAllowed

	/**
	 * There are 3 ways in which a scroll bar can be displayed.
	 * This integer will represent one of these 3 displays -
	 * (SCROLLBARS_ALWAYS, SCROLLBARS_AS_NEEDED, SCROLLBARS_NEVER)
	 * @see #setScrollbarDisplayPolicy
	 */
	transient private int fScrollbarDisplayPolicy= SCROLLBARS_AS_NEEDED;
/**
 * Add the  <code>UIComponent</code> to the receiver's list of children.
 *
 * @param component : A <code>UIComponent</code>
 */
public void add(UIComponent component) {
	if (component != null && component instanceof UILabel) {
		UiJContainerIcon comp= convertToIcon((UILabel) component);
		if (comp != null)
			addItem(comp);
	}
}
/**
 * Add and display a new item
 *
 * @params item the new item
 */
public void addItem(UiJContainerIcon item) {
	if (item == null)
		return;
	int oldMaxIconHeight= fMaxIconSize.height;
	int oldMaxIconWidth= fMaxIconSize.width;
	fMaxIconSize.setSize(
		Math.max(fMaxIconSize.width, (int) item.getUI().getPreferredSize(item).width),
		Math.max(fMaxIconSize.height, (int) item.getUI().getPreferredSize(item).height));
	if (fMaxIconSize.width != oldMaxIconWidth | fMaxIconSize.height != oldMaxIconHeight) {
		Component[] items=  fIconContainer.getComponents();
		for (int i= 0; i < items.length; i++) 
			((UiJContainerIcon)  items[i]).setPreferredSize(fMaxIconSize);
	}
	item.setPreferredSize(fMaxIconSize);
	item.setSize(fMaxIconSize);
	if (!fAutoArranging)
		computeAndSetLocation(item);
	fIconContainer.add(item);

	// Adjust the scrollbar increment
	fScrollPane.getHorizontalScrollBar().setUnitIncrement(fMaxIconSize.width + fLayoutManager.getHgap());
	fScrollPane.getVerticalScrollBar().setUnitIncrement(fMaxIconSize.height + fLayoutManager.getVgap());

	fScrollPane.revalidate();
	fScrollPane.repaint();
}
/**
 * Generic method to add a collection of child widgets to this widget.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param list		the <code>Anything</code> containing the list of children
 */
public void addMany(ORBConnection conn, Anything list) {
	Vector items= new Vector();
	if (list != null)
		for (int i = 0; i < list.size(); i++) {
			UIProxy m = getManaged(UIProxy.class, conn, list.get(i));
			if (m != null)
				items.addElement(m);
//				internalAddToChildren(m);
		}
	setItems(items);
}
/**
 * Deselect all items
 * This includes the GUI update.
 */
public void clearSelection() {
	for (Enumeration enum= fSelectedItems.elements(); enum.hasMoreElements();)
		((UiJContainerIcon) enum.nextElement()).setSelected(false);
	fSelectedItems= new Vector();
}
/**
 * Compute and set the location of the given <code>item</code>
 *
 * @params items the list with the icons
 */
private void computeAndSetLocation(UiJContainerIcon item) {
	UiJContainerIcon loopItem= null;
	Point maxPosition= new Point(
		fIconContainer.getInsets().left + fLayoutManager.getHgap(),
		fIconContainer.getInsets().top + fLayoutManager.getVgap());
	for (int i= 0; i < fIconContainer.getComponents().length; i++) {
		loopItem= (UiJContainerIcon) fIconContainer.getComponent(i);
		maxPosition.setLocation(
			Math.max(maxPosition.x, loopItem.getX()),
			Math.max(maxPosition.y, loopItem.getY()));
	}
	if (fIconContainer.getComponentCount() > 0)
		maxPosition.x += fMaxIconSize.width + fLayoutManager.getHgap() * 2;
	item.setLocation(maxPosition);

	maxPosition.x += fMaxIconSize.width + fLayoutManager.getHgap() + fIconContainer.getInsets().right;
	maxPosition.y += fMaxIconSize.height + fLayoutManager.getVgap() + fIconContainer.getInsets().bottom;

	Dimension panelSize= new Dimension();
	panelSize.width=  Math.max(fIconContainer.getWidth(), maxPosition.x);
	panelSize.height= Math.max(fIconContainer.getHeight(), maxPosition.y);

	fIconContainer.setPreferredSize(panelSize);
	fIconContainer.setSize(panelSize);
}
/**
 * Compute the size of the largest icon
 */
private void computeLargestIconSize() {
	fMaxIconSize.setSize(0, 0);
	UiJContainerIcon item= null;
	for (int i= 0; i < fIconContainer.getComponentCount(); i++) {
		item= (UiJContainerIcon)  fIconContainer.getComponent(i);
		fMaxIconSize.setSize(
			Math.max(fMaxIconSize.width, (int) item.getUI().getPreferredSize(item).width),
			Math.max(fMaxIconSize.height, (int) item.getUI().getPreferredSize(item).height));
	}
}
	/**
	 * Checks whether the rectangle <code>mainR</code> entirely contains 
	 * the rectangle<code>searchR</code>.
	 *
	 * This method can be replaced by Rectangle::contains(Rectangle) as
	 * soon as we give up JDK1.1 support.
	 *
	 * @param     mainR	the <code>Rectangle</code> in which we search
	 * @param     searchR the <code>Rectangle</code> we search inside <code>mainR</code>
	 * @param     H   the height of the <code>Rectangle</code>
	 * @return    <code>true</code> if the <code>Rectangle</code> specified by
	 *            (<i>searchR</i>
	 *            is entirely enclosed inside <code>mainR</code>; 
	 *            <code>false</code> otherwise.
	 * @since     ULC 3.0
	 */
	private boolean contains(Rectangle mainR, Rectangle searchR) {
	if (mainR.width <= 0 || mainR.height <= 0 || searchR.width <= 0 || searchR.height <= 0) {
	    return false;
	}
	return (searchR.x >= mainR.x &&
		searchR.y >= mainR.y &&
		searchR.x + searchR.width <= mainR.x + mainR.width &&
		searchR.y + searchR.height <= mainR.y + mainR.height);
	}
protected UiJContainerIcon convertToIcon(UILabel label) {
	UiJContainerIcon icon= new UiJContainerIcon();
	icon.setText(label.getText());
	icon.setIcon(label.getIcon());

	// Retrieve the toolTip and font from the UILabel component's Swing component
	// ==> should we also copy all other attributes like enabled
	//     or get focus?
	icon.setToolTipText(((JComponent) label.getComponent()).getToolTipText());
	icon.setFont(label.getComponent().getFont());

	// Set the label relative to the icon
	if (fLabelPosition == BOX_TOP) {
		icon.setHorizontalTextPosition(SwingConstants.CENTER);
		icon.setVerticalTextPosition(SwingConstants.TOP);
	}
	else if (fLabelPosition == BOX_BOTTOM) {
		icon.setHorizontalTextPosition(SwingConstants.CENTER);
		icon.setVerticalTextPosition(SwingConstants.BOTTOM);
	}
	else if (fLabelPosition == BOX_LEFT) {
		icon.setHorizontalTextPosition(SwingConstants.LEFT);
		icon.setVerticalTextPosition(SwingConstants.CENTER);
	}
	else if (fLabelPosition == BOX_RIGHT) {
		icon.setHorizontalTextPosition(SwingConstants.RIGHT);
		icon.setVerticalTextPosition(SwingConstants.CENTER);
	}
	// unregister listener
	label.getComponent().removeMouseListener(label);

	// replace the internal widget of UILabel
	label.setComponent(icon);
	return icon;
}
/**
 * Resize the icon container (JPanel) because
 * the ScrollPane is being resized/relayouted
 */

protected JPanel createJPanel() {
	return new JPanel() {
		public void doLayout() {
			JScrollPane scrollPane = (JScrollPane) this.getParent().getParent();
			Dimension viewportSize = scrollPane.getViewport().getSize();
			Dimension virtualContainerSize = scrollPane.getViewport().getSize();
			boolean xIsComputed = false;
			boolean yIsComputed = false;
			if (getLayout() == null || (scrollPane.getHorizontalScrollBarPolicy() == ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER && scrollPane.getVerticalScrollBarPolicy() == ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER))
				
				// Do nothing if there are no scroll bars at all
				return;
			if (scrollPane.getHorizontalScrollBarPolicy() == ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS && scrollPane.getVerticalScrollBarPolicy() == ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS) {
				// I assume user wants to scroll triple the width if both scroll bars must be shown :-)
				virtualContainerSize.width *= 3;
				virtualContainerSize.height = Integer.MAX_VALUE;
				xIsComputed = true;
				yIsComputed = true;
			} else
				if (scrollPane.getHorizontalScrollBarPolicy() == javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS || (scrollPane.getVerticalScrollBarPolicy() == ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER && scrollPane.getHorizontalScrollBarPolicy() == ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED)) {
					// Only use horizontal scroll bar ==> vertical one not used (even if specified AS_NEEDED)
					int ny = virtualContainerSize.height / (fMaxIconSize.height + ((FlowLayout) getLayout()).getVgap());
					if (ny == 0)
						ny = 1;
					int nx = getComponentCount() / ny;
					if (getComponentCount() % ny > 0)
						nx += 1;
					virtualContainerSize.width = nx * (fMaxIconSize.width + ((FlowLayout) getLayout()).getHgap());
					xIsComputed = true;
				} else {
					// only use vertical scroll bar ==> horizontal one not used (even if specified AS_NEEDED)
					virtualContainerSize.height = Integer.MAX_VALUE;
					yIsComputed = true;
				}

				// Set the a new (preferred) size to the icon container (JPanel)
				// which is too big. Then let the container layout it's children
			setPreferredSize(virtualContainerSize);
			setSize(virtualContainerSize);
			super.doLayout();

			// Compute the size which the container would
			// need to show all its icons
			if (xIsComputed) {
				int xMax = viewportSize.width;
				for (int i = 0; i < getComponents().length; i++)
					xMax = Math.max(xMax, this.getComponent(i).getBounds().x + this.getComponent(i).getSize().width);
				virtualContainerSize.width = 5 + xMax; // Gap for cosmetic reasons
			}
			if (yIsComputed) {
				int yMax = viewportSize.height;
				for (int i = 0; i < getComponents().length; i++)
					yMax = Math.max(yMax, this.getComponent(i).getBounds().y + this.getComponent(i).getSize().height);
				virtualContainerSize.height = yMax;
			}
			// Finally set the containers size to be exactly the
			// the size all the icons will need. This sice will be
			// bigger or equal the viewport size
			setPreferredSize(virtualContainerSize);
			setSize(virtualContainerSize);
			scrollPane.doLayout();
		}
	};
}
/**
 * Handle the dragging of the items
 *
 */
protected void dragging(java.awt.event.MouseEvent e) {
	if (fDragIconsAllowed && fDragging && e.getComponent() != fIconContainer) {
		// move item
		Component item= e.getComponent();
		int x= item.getLocation().x + e.getX();
		int y= item.getLocation().y + e.getY();
		item.setLocation(x - fDragStartX, y- fDragStartY);
	}
}
/**
 * Find the component which is nearest below
 * the currently focused component
 */
protected UiJContainerIcon findClosestToDown() {
	// Get item with focus
	Component focusedComponent = (((Window) fIconContainer.getTopLevelAncestor()).getFocusOwner());
	if (focusedComponent == null || !(focusedComponent instanceof UiJContainerIcon))
		return null;

	// Set search area
	Component closestComponent = null;
	Rectangle searchR = new Rectangle();
	searchR.setLocation((int) Math.max(0, 0.5 + focusedComponent.getBounds().x - 0.5 * focusedComponent.getBounds().width), focusedComponent.getBounds().y);
	searchR.setSize(2 * focusedComponent.getBounds().width, fIconContainer.getBounds().height - focusedComponent.getBounds().y);
	//		fIconContainer.getPreferredSize().height - focusedComponent.getBounds().y);

	// Find right-most icon that is inside the search area
	for (int i = 0; i < fIconContainer.getComponentCount(); i++)
		if (fIconContainer.getComponent(i).isEnabled() && contains(searchR, fIconContainer.getComponent(i).getBounds()) && fIconContainer.getComponent(i) != focusedComponent && (closestComponent == null || fIconContainer.getComponent(i).getBounds().y < closestComponent.getBounds().y))
			closestComponent = fIconContainer.getComponent(i);

		// If the found component is outside
		// the current viewport then force scrolling
	if (closestComponent != null) {
		if (fScrollPane.getViewport().getViewPosition().y + fScrollPane.getViewport().getExtentSize().height < closestComponent.getBounds().y + closestComponent.getBounds().height) {
			fScrollPane.getViewport().setViewPosition(new Point(fScrollPane.getViewport().getViewPosition().x, fScrollPane.getViewport().getViewPosition().y + fMaxIconSize.height + fLayoutManager.getVgap()));
			fScrollPane.revalidate();
			fScrollPane.repaint();
		}
	}
	return (UiJContainerIcon) closestComponent;
}
/**
 * Find the component which is nearest to
 * the left of the currently focused component
 */
protected UiJContainerIcon findClosestToLeft() {
	// Get item with focus
	Component focusedComponent = (((Window) fIconContainer.getTopLevelAncestor()).getFocusOwner());
	if (focusedComponent == null || !(focusedComponent instanceof UiJContainerIcon))
		return null;

	// Set search area
	Component closestComponent = null;
	Rectangle searchR = new Rectangle();
	searchR.setLocation(0, (int) Math.max(0, 0.5 + focusedComponent.getBounds().y - 0.5 * focusedComponent.getBounds().height));
	searchR.setSize(focusedComponent.getBounds().x + focusedComponent.getBounds().width, 2 * focusedComponent.getBounds().height);

	// Find right-most icon that is inside the search area
	for (int i = 0; i < fIconContainer.getComponentCount(); i++)
		if (fIconContainer.getComponent(i).isEnabled() && contains(searchR, fIconContainer.getComponent(i).getBounds())
		&& fIconContainer.getComponent(i) != focusedComponent && (closestComponent == null || fIconContainer.getComponent(i).getBounds().x + fIconContainer.getComponent(i).getBounds().width > closestComponent.getBounds().x + closestComponent.getBounds().width))
			closestComponent = fIconContainer.getComponent(i);

		// If the found component is outside
		// the current viewport then force scrolling
	if (closestComponent != null) {
		if (fScrollPane.getViewport().getViewPosition().x > closestComponent.getBounds().x) {
			fScrollPane.getViewport().setViewPosition(new Point(fScrollPane.getViewport().getViewPosition().x - fMaxIconSize.width + fLayoutManager.getHgap(), fScrollPane.getViewport().getViewPosition().y));
			fScrollPane.revalidate();
			fScrollPane.repaint();
		}
	}
	return (UiJContainerIcon) closestComponent;
}
/**
 * Find the component which is nearest to
 * the right of the currently focused component
 */
protected UiJContainerIcon findClosestToRight() {
	// Get item with focus
	Component focusedComponent = (((Window) fIconContainer.getTopLevelAncestor()).getFocusOwner());
	if (focusedComponent == null || !(focusedComponent instanceof UiJContainerIcon))
		return null;

	// Set search area
	Component closestComponent = null;
	Rectangle searchR = new Rectangle();
	searchR.setLocation(focusedComponent.getBounds().x, (int) Math.max(0, 0.5 + focusedComponent.getBounds().y - 0.5 * focusedComponent.getBounds().height));
	searchR.setSize(fIconContainer.getBounds().width - focusedComponent.getBounds().x, 2 * focusedComponent.getBounds().height);

	// Find left-most icon that is inside the search area
	for (int i = 0; i < fIconContainer.getComponentCount(); i++)
		if (fIconContainer.getComponent(i).isEnabled() && contains(searchR, fIconContainer.getComponent(i).getBounds()) && fIconContainer.getComponent(i) != focusedComponent && (closestComponent == null || fIconContainer.getComponent(i).getBounds().x < closestComponent.getBounds().x))
			closestComponent = fIconContainer.getComponent(i);

		// If the found component is outside
		// the current viewport then force scrolling
	if (closestComponent != null) {
		if (fScrollPane.getViewport().getViewPosition().x + fScrollPane.getViewport().getExtentSize().width < closestComponent.getBounds().x + closestComponent.getBounds().width) {
			fScrollPane.getViewport().setViewPosition(new Point(fScrollPane.getViewport().getViewPosition().x + fMaxIconSize.width + fLayoutManager.getHgap(), fScrollPane.getViewport().getViewPosition().y));
			fScrollPane.revalidate();
			fScrollPane.repaint();
		}
	}
	return (UiJContainerIcon) closestComponent;
}
/**
 * Find the component which is nearest above
 * the currently focused component
 */
protected UiJContainerIcon findClosestToUp() {
	// Get item with focus
	Component focusedComponent = (((Window) fIconContainer.getTopLevelAncestor()).getFocusOwner());
	if (focusedComponent == null || !(focusedComponent instanceof UiJContainerIcon))
		return null;

	// Set search area
	Component closestComponent = null;
	Rectangle searchR = new Rectangle();
	searchR.setLocation((int) Math.max(0, 0.5 + focusedComponent.getBounds().x - 0.5 * focusedComponent.getBounds().width), 0);
	searchR.setSize(2 * focusedComponent.getBounds().width, focusedComponent.getBounds().y + focusedComponent.getBounds().height);

	// Find right-most icon that is inside the search area
	for (int i = 0; i < fIconContainer.getComponentCount(); i++)
		if (fIconContainer.getComponent(i).isEnabled() && contains(searchR, fIconContainer.getComponent(i).getBounds())
		  && fIconContainer.getComponent(i) != focusedComponent && (closestComponent == null || fIconContainer.getComponent(i).getBounds().y + fIconContainer.getComponent(i).getBounds().height > closestComponent.getBounds().y + closestComponent.getBounds().height))
			closestComponent = fIconContainer.getComponent(i);

		// If the found component is outside
		// the current viewport then force scrolling
	if (closestComponent != null) {
		if (fScrollPane.getViewport().getViewPosition().y > closestComponent.getBounds().y) {
			fScrollPane.getViewport().setViewPosition(new Point(fScrollPane.getViewport().getViewPosition().x, fScrollPane.getViewport().getViewPosition().y - fMaxIconSize.height - fLayoutManager.getVgap()));
			fScrollPane.revalidate();
			fScrollPane.repaint();
		}
	}
	return (UiJContainerIcon) closestComponent;
}
public void free() {
	if (fIconContainer != null) {
		// remove listeners

		// reset fields
		fIconContainer= null;
		fLayoutManager= null;
	}
	super.free();
}
/**
 * Answer the primary swing component (widget) being
 * wrapped by the receiver.
 *
 * @see #getComponent()
 */
public Component getBasicComponent() {
	return fIconContainer;
}
/**
 * Return the internal component
 */
public java.awt.Component getComponent() {
	return fScrollPane;
}
/**
 * Return the horizontal gap between two icons
 *
 * @return <code>int</code> The horizontal gap between icons in pixels
 */
public int getHorizontalGap() {
	return fLayoutManager.getHgap();
}
private Anything getSelectedOidsAsAnything() {
	Anything oidsAny = new Anything();
	for (Enumeration enum= fSelectedItems.elements(); enum.hasMoreElements();)
		oidsAny.append(new Anything(oidOf(enum.nextElement())));
	return oidsAny;	
}
/**
 * Return the vertical gap between two icons
 *
 * @return <code>int</code> The vertical gap between icons in pixels
 */
public int getVerticalGap() {
	return fLayoutManager.getVgap();
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("allItemsRemoved"))
		setItems(new Vector());

	else if (request.equals("setLabelPosition"))
		setLabelPosition(args.asInt(BOX_BOTTOM));

	else if (request.equals("setVGap"))
		setVerticalGap(args.asInt(5));

	else if (request.equals("setHGap"))
		setHorizontalGap(args.asInt(5));
		
	else if (request.equals("setDragIconsAllowed"))
		setDragIconsAllowed(args.asBoolean(false));

	else if (request.equals("setMultiSelectionAllowed"))
		setMultiSelectionAllowed(args.asBoolean(true));

	else if (request.equals("setAutoArranging"))
		setAutoArranging(args.asBoolean(true));

	else if (request.equals("setScrollbarDisplayPolicy"))
		setScrollbarDisplayPolicy(args.asInt(java.awt.ScrollPane.SCROLLBARS_AS_NEEDED));

	else if (request.equals("setSelectedOids"))
		setSelectedOids(args);
	
	else 
		super.handleRequest(conn, request, args);
}
/**
 * Searches for the first occurence of the given argument.
 * Testing is done via equality using the <code>equals</code>
 * method. 
 *
 * @param   element an object.
 * @return  the index of the first occurrence of the object argument.
 *			Returns <code>-1</code> if the object is not found.
 * @see     Object#equals(Object)
 */
private int indexOf(Object elem) {
	Component[] elements= fIconContainer.getComponents();
	if (elem == null)
		return -1;	// will not be in the list of components
	else
		for (int i= 0; i < elements.length; i++)
			if (elem.equals(elements[i]))
		 	   return i;
	return -1;
}
/**
 * Answer if dragging of icons is allowed or not
 * The default is false
 *
 * @return if dragging of icons is allowed
 */
public boolean isDragIconsAllowed() {
	return fDragIconsAllowed;
}
/**
 * Answer if multi-selection is allowed or not
 * The default is true
 *
 * @return if multi-selection is allowed
 */
public boolean isMultiSelectionAllowed() {
	return fMultiSelectionAllowed;
}
/**
 * Notification that a mouse event happened.
 * See what it is and update ULC side if needed
 * 
 * @param event : The notification event.
 */
protected void mouseClickOnPanel(MouseEvent e) {
	UiJContainerIcon item= null;
	Anything		 oidAny= null;

	if (! isEnabled())
		return;

	if (e.isPopupTrigger()) {
		showPopup(e);
		return;
	}
	fDragging= false;
	if ((e.getModifiers() & e.BUTTON1_MASK) > 0) {
		if (fIconContainer == e.getComponent()) {
			// Clicked on the container
			if (fSelectedItems.size() > 0) {
				clearSelection();
				sendEventULC("selectionCleared");
			}
			return;			
		}
		item= (UiJContainerIcon) e.getComponent();
		if (item == null || !item.isEnabled())
			return;
		item.requestFocus();
		oidAny= new Anything(oidOf(item));
		if (e.getClickCount() == 2) {
			// Double-click
			sendEventULC("action", "cmd", new Anything("double-clicked"));
			return;
		}
		if ((e.getModifiers() & e.CTRL_MASK) > 0) {
			// switch state of selected icon
			String type= null;
			if (item.isSelected())
				type= "unselectedItem";
			else
				type= "selectedItem";
			switchSelectionState(item);
			sendEventULC(type, "oid", oidAny);			
		}	
		else {
			// deselect all selected items and select the current one
			if (!(fSelectedItems.size() == 1 && fSelectedItems.contains(item))) {
				singleSelectItem(item);
				sendEventULC("singleItemSelected", "oid", oidAny);
			}
		}
	}
}
protected void moveFocusDown(ActionEvent e) {
	moveFocusTo(findClosestToDown());
}
protected void moveFocusLeft(ActionEvent e) {
	moveFocusTo(findClosestToLeft());
}
protected void moveFocusRight(ActionEvent e) {
	moveFocusTo(findClosestToRight());
}
/**
 * Move focus to <code>icon</code>
 * @param icon the icon which grabs focus
 */
protected void moveFocusTo(UiJContainerIcon icon) {
	if (icon != null)
		// Move focus to icon
		icon.requestFocus();
}
protected void moveFocusUp(ActionEvent e) {
	moveFocusTo(findClosestToUp());
}
protected void moveMultiSelectionDown(ActionEvent e) {
	moveMultiSelectionTo(findClosestToDown());
}
protected void moveMultiSelectionLeft(ActionEvent e) {
	moveMultiSelectionTo(findClosestToLeft());
}
protected void moveMultiSelectionRight(ActionEvent e) {
	moveMultiSelectionTo(findClosestToRight());
}
/**
 * Move focus to <code>icon</code> and select
 * all icons that are now inside the rectangle
 * spawned by the first selected icon and the
 * current one
 * @param icon the icon which grabs focus
 */
protected void moveMultiSelectionTo(UiJContainerIcon icon) {
	if (icon == null)
		return;
	if (!fMultiSelectionAllowed)
		moveSelectionTo(icon);

	// Move focus to icon
	icon.requestFocus();

	// Compute rectangle for selection
	Rectangle selectR = null;
	if (fSelectedItems.isEmpty())
		selectR = icon.getBounds();
	else {
		UiJContainerIcon firstSelection = (UiJContainerIcon) fSelectedItems.firstElement();
		// Assure this stays the first selected item
		singleSelectItem(firstSelection);
		selectR = firstSelection.getBounds().union(icon.getBounds());
	}
	// Mark all icons that are inside this rectangle
	for (int i = 0; i < fIconContainer.getComponentCount(); i++)
		if (fIconContainer.getComponent(i).isEnabled() && contains(selectR, fIconContainer.getComponent(i).getBounds()))
			selectItem((UiJContainerIcon) fIconContainer.getComponent(i));

		// Inform ULC side of selection changes
		// Note: Here we send ALL selected items to the ULC side. Normally
		//       we only send the change/difference to the ULC side
	sendEventULC("selectedItems", "selectedOids", getSelectedOidsAsAnything());
}
protected void moveMultiSelectionUp(ActionEvent e) {
	moveMultiSelectionTo(findClosestToUp());
}
protected void moveSelectionDown(ActionEvent e) {
	moveSelectionTo(findClosestToDown());
}
protected void moveSelectionLeft(ActionEvent e) {
	moveSelectionTo(findClosestToLeft());
}
protected void moveSelectionRight(ActionEvent e) {
	moveSelectionTo(findClosestToRight());
}
/**
 * Move the selection and the focus to <code>icon</code>
 * @param icon the icon to which the selection has been
 * moved by an arrow key
 */
protected void moveSelectionTo(UiJContainerIcon icon) {
	if (icon != null) {
		// Move focus to closestComponent
		icon.requestFocus();
		singleSelectItem(icon);
		sendEventULC("singleItemSelected", "oid", new Anything(oidOf(icon)));
	}
}
protected void moveSelectionUp(ActionEvent e) {
	moveSelectionTo(findClosestToUp());
}
/**
 * Searches for the first occurence of the given argument.
 * Testing is done via equality using the <code>equals</code>
 * method. 
 *
 * @param   element an object.
 * @return  the index of the first occurrence of the element
 *			Returns <code>-1</code> if the object is not found.
 * @see     Object#equals(Object)
 */
private int oidOf(Object elem) {
	return ((UILabel) fList.elementAt(indexOf(elem))).getId();
}
/**
 * The ENTER key has been pressed.
 * This corresponds to an ULC action.
 *
 * @param event the <code>ActionEvent</code>
 */
protected void pressedEnterKey(ActionEvent e) {
	if (!fSelectedItems.isEmpty())
		sendEventULC("action", "cmd", new Anything("enter"));
}
/**
 * Register for keyboard actions and act appropriately
 * i.e. move selection and/or focus according to the
 * pressed arrow key.
 * If CTRL is down then only the focus is moved
 * If SHIFT is down all icons inside the rectangle
 * [first selected icon, currently selected icon]
 * are marked as selected.
 * If CTRL and SHIFT are down, SHIFT is preferred.
 *
 * @param state tells if dragging icons is allowed
 */
protected void registerKeyboardActions() {
	// ENTER pressed
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {pressedEnterKey(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	
	// ARROW KEYS
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveSelectionDown(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, 0, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveSelectionUp(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_UP, 0, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveSelectionLeft(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, 0, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveSelectionRight(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, 0, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveFocusDown(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, KeyEvent.CTRL_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveFocusUp(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_UP, KeyEvent.CTRL_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveFocusLeft(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, KeyEvent.CTRL_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveFocusRight(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, KeyEvent.CTRL_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveMultiSelectionDown(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, KeyEvent.SHIFT_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveMultiSelectionUp(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_UP, KeyEvent.SHIFT_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveMultiSelectionLeft(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, KeyEvent.SHIFT_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
	fIconContainer.registerKeyboardAction(
		new ActionListener() {
			public void actionPerformed(ActionEvent e) {moveMultiSelectionRight(e);}
		},
		KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, KeyEvent.SHIFT_MASK, false),
		JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT
	);
}
/**
 * Remove the  <code>UIComponent</code> from the receiver's list of children.
 *
 * @param component : A <code>UIComponent</code>
 */
public void remove(UIComponent component) {
	removeItem((UiJContainerIcon) component.getComponent());
}
/**
 * Remove an item
 *
 * @params item the item to be removed
 */
public void removeItem(UiJContainerIcon item) {
	if (item == null)
		return;
	if (item.isSelected())
		fSelectedItems.removeElement(item);
	fIconContainer.remove(item);
//	fIconContainer.revalidate();
//	fIconContainer.repaint();
	fScrollPane.revalidate();
	fScrollPane.repaint();
}
/**
 * Removes an item from the list of selected items.
 * This includes the GUI update.
 *
 * @param item the item to be removed from selection 
 */
public void removeItemFromSelection(UiJContainerIcon item) {
	item.setSelected(false);
	fSelectedItems.removeElement(item);
}
/**
 * Resize each item to the largest icon
 */
private void resizeToLargestIconSize() {
	computeLargestIconSize();
	for (int i= 0; i < fIconContainer.getComponentCount(); i++) 
		((UiJContainerIcon)  fIconContainer.getComponent(i)).setPreferredSize(fMaxIconSize);

	// Adjust the scrollbar increment
	fScrollPane.getHorizontalScrollBar().setUnitIncrement(fMaxIconSize.width + fLayoutManager.getHgap());
	fScrollPane.getVerticalScrollBar().setUnitIncrement(fMaxIconSize.height + fLayoutManager.getVgap());

	fScrollPane.revalidate();
	fScrollPane.repaint();
}
public void restoreState(ORBConnection conn, Anything args) {
	// Prepare widgets
	fLayoutManager= new FlowLayout(FlowLayout.LEFT);
	fIconContainer.setLayout(fLayoutManager);
	fScrollPane.setViewportView(fIconContainer);
	fScrollPane.setBorder(new BevelBorder(BevelBorder.LOWERED));

	// Add listeners
	registerKeyboardActions();
	fIconContainer.addMouseMotionListener(new MouseMotionAdapter() {
		public void mouseDragged(java.awt.event.MouseEvent e) {
			dragging(e);
		}
	});
	fIconContainer.addMouseListener(new MouseAdapter() {
		public void mousePressed(MouseEvent e) {
			startDrag(e);
		};
		public void mouseReleased(MouseEvent e) {
			mouseClickOnPanel(e);
		}
	});

	// Update fields from ULC side
	fLayoutManager.setVgap(args.get("vg", 5));
	fLayoutManager.setHgap(args.get("hg", 5));	
	fScrollPane.setPreferredSize(new Dimension(args.get("w", 200), args.get("h", 200)));
	setMultiSelectionAllowed(args.get("multiSelectionAllowed", true));	
	fLabelPosition= args.get("labelPosition", BOX_BOTTOM);
	setDragIconsAllowed(args.get("dragIconsAllowed", false));
	setScrollbarDisplayPolicy(args.get("scrollbarDisplayPolicy", SCROLLBARS_AS_NEEDED));
	if (!args.get("autoArranging", true)) {
		// Use the layout manager (i.e. auto-arrangement) to
		// place all the items then turn auto-arrangement off
		DeferredRequest r = new DeferredRequest(conn) {
			public void safeDispatch() {
				setAutoArranging(false);
			}
		};
		conn.postRequest(r);
	}
	super.restoreState(conn, args); // this adds the items
	if (args.isDefined("selectedOids"))
		setSelectedOids(args.get("selectedOids"));
}
/**
 * Set the selection to the given item and
 * remove the selection from all other items
 * if multi-selection is turned off.
 * Do nothing if item is already selected.
 *
 * @param item the item to be selected
 */
public void selectItem(UiJContainerIcon item) {
	if (item.isSelected())
		return;
	if (isMultiSelectionAllowed()) {
		item.setSelected(true);
		fSelectedItems.addElement(item);
	}
	else
		singleSelectItem(item);
}
/**
 * Set if auto-arrangement of icons is turned on or off.
 * If we change from off to on then the
 * icons get re-arranged and dragging of icons
 * gets disabled.
 *
 * @params state tells if auto-arrangement of icons is turned on
 */
public void setAutoArranging(boolean state) {
	if (fAutoArranging != state) {
		fAutoArranging= state;
		if (fAutoArranging) {
			// do not use setDragIconsAllowed
			// this would cause a loop
			fDragIconsAllowed= false;
			fIconContainer.setLayout(fLayoutManager);
			fScrollPane.revalidate();
			fScrollPane.repaint();
		}
		else
			fIconContainer.setLayout(null);
	}
}
/**
 * Set if it is allowed to drag the icons or not.
 * If we change from allowed to not-allowed then
 * we re-arrange the icons.
 * Allow dragging automatically turns off the
 * auto-arrangement of icons.
 *
 * @params state tells if dragging icons is allowed
 */
public void setDragIconsAllowed(boolean state) {
	if (fAutoArranging != state) {
		fDragIconsAllowed= state;
		if (fDragIconsAllowed)
			setAutoArranging(false);
	}
}
/**
 * Set the horizontal gap between two icons
 *
 * @param gap <code>int</code> The vertical gap between icons in pixels
 */
public void setHorizontalGap(int gap) {
	fLayoutManager.setHgap(gap);
	if (fAutoArranging) {
		// layout manager is used
		fIconContainer.revalidate();
		fIconContainer.repaint();
	}
}
/**
 * Set and display a new item list
 *
 * WARNING: This method will throw a runtime
 *          exception if the items don't conform
 *          to the class UiJContainerIcon.
 *
 * @params items the new list to be displayed
 */
public void setItems(Vector items) {
	if (items == null)
		items= new Vector();
	clearSelection();
	fIconContainer.removeAll();
	fList.removeAllElements();
	for (Enumeration enum= items.elements(); enum.hasMoreElements(); ) {
		UILabel label= (UILabel)  enum.nextElement();
		if (fAutoArranging)
			fIconContainer.add(convertToIcon(label));
		else
			// this is slower because we compute the position manually
			addItem(convertToIcon(label));
		internalAddToChildren(label);
	}
	if (fAutoArranging)
		resizeToLargestIconSize();
	else {
		fScrollPane.revalidate();
		fScrollPane.repaint();
	}

}
/**
 * Resize each item to the largest icon
 *
 * @params items the list with the icons to be resized
 */
protected void setLabelPosition(int labelPosition) {
	UiJContainerIcon icon= null;
	fLabelPosition= labelPosition;
	for (int i= 0; i < fIconContainer.getComponents().length; i++) {
		icon= (UiJContainerIcon) fIconContainer.getComponents()[i];
		if (fLabelPosition == BOX_TOP) {
			icon.setHorizontalTextPosition(SwingConstants.CENTER);
			icon.setVerticalTextPosition(SwingConstants.TOP);
		}
		else if (fLabelPosition == BOX_BOTTOM) {
			icon.setHorizontalTextPosition(SwingConstants.CENTER);
			icon.setVerticalTextPosition(SwingConstants.BOTTOM);
		}
		else if (fLabelPosition == BOX_LEFT) {
			icon.setHorizontalTextPosition(SwingConstants.LEFT);
			icon.setVerticalTextPosition(SwingConstants.CENTER);
		}
		else if (fLabelPosition == BOX_RIGHT) {
			icon.setHorizontalTextPosition(SwingConstants.RIGHT);
			icon.setVerticalTextPosition(SwingConstants.CENTER);
		}
	}
	fIconContainer.revalidate();
	fIconContainer.repaint();	
}
/**
 * Set if multi-selection is allowed or not
 * If more than one item is selected then
 * the the item that has been first selected
 * gets selected.
 *
 * @params state tells if multiselection is allowed
 */
public void setMultiSelectionAllowed(boolean state) {
	if (fMultiSelectionAllowed != state) {
		fMultiSelectionAllowed= state;
		if (!fMultiSelectionAllowed && !fSelectedItems.isEmpty()) {
			singleSelectItem((UiJContainerIcon) fSelectedItems.firstElement());
		}
	}
}
/**
 * Overwrites the super method because we
 * already register for window events
 */
protected void setPopupMenu(UIMenu menu) {
	fPopupMenu= menu;
}
/**
 * Set the scroll bar display policy.
 *
 * @params policy the scrollbar display policy
 */
protected void setScrollbarDisplayPolicy(int policy) {
	fScrollbarDisplayPolicy= policy;
	if (policy == SCROLLBARS_ALWAYS) {
		fScrollPane.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		fScrollPane.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
	}
	else if (policy == SCROLLBARS_NEVER) {
		fScrollPane.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_NEVER);
		fScrollPane.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
	}
	else if (policy == SCROLLBARS_AS_NEEDED) {
		fScrollPane.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		fScrollPane.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
	}
	fScrollPane.revalidate();
	fScrollPane.repaint();	
}
/**
 * Set list of selected items according to
 * the oids (of the ULCLabels).
 * This includes the GUI update.
 *
 * @param indices the indices of the items to be selected
 */
protected void setSelectedOids(Anything oids) {
	clearSelection();
	for (int i= 0; i < oids.size(); i++)
		selectItem((UiJContainerIcon) ((UILabel) fConnection.getRegistry().find(oids.get(i).asInt(-1))).getComponent());
}
/**
 * Set the vertical gap between two icons
 *
 * @param gap <code>int</code> The vertical gap between icons in pixels
 */
public void setVerticalGap(int gap) {
	fLayoutManager.setVgap(gap);
	if (fAutoArranging) {
		// layout manager is used
		fScrollPane.revalidate();
		fScrollPane.repaint();
	}
}
/**
 * Invoked when a popup has been requested
 */
public void showPopup(MouseEvent event) {
	if (event.getComponent() == fIconContainer)
		super.showPopup(event);
	else {
		UIComponent item= null;
		for (Enumeration enum= getChildren().elements(); enum.hasMoreElements();) {
			item= (UIComponent) enum.nextElement();
			if (item.getBasicComponent() == event.getComponent())
				break;
			item= null;
		}
		if (item == null || !item.isEnabled())
			return;
		if (fSelectedItems.size() == 1 && fSelectedItems.firstElement() == item.getComponent())
			item.showPopup(event);
		else if (fSelectedItems.indexOf(item.getComponent()) >= 0) {
			// DM:	here we would have to compute the
			//		intersection of the menus of all selected items
			//		We don't support that right now hence we single
			// 		select the item under the cursor
			item.getComponent().requestFocus();
			singleSelectItem((UiJContainerIcon) item.getComponent());
			sendEventULC("singleItemSelected", "oid", new Anything(oidOf(item.getComponent())));
			item.showPopup(event);
		}
		else {
			item.getComponent().requestFocus();
			singleSelectItem((UiJContainerIcon) item.getComponent());
			sendEventULC("singleItemSelected", "oid", new Anything(oidOf(item.getComponent())));
			item.showPopup(event);
		}
	}
}
/**
 * Set the selection only to the given item and
 * remove the selection from all other items.
 * This includes the GUI update.
 *
 * @param item the only item to be selected
 */
protected void singleSelectItem(UiJContainerIcon item) {
	clearSelection();
	item.setSelected(true);
	fSelectedItems.addElement(item);
}
/**
 *	Remember where we begin dragging
 */
private void startDrag(MouseEvent e) {
	fDragging= true;
	fDragStartX= e.getX();
	fDragStartY= e.getY();
}
/**
 * Change the selection state of the item
 * This includes the GUI update.
 *
 * Note: This method will no longer be used
 *
 * @param item the item of which the selection state is switched
 */
protected void switchSelectionState(UiJContainerIcon item) {
	if (item.isSelected())
		removeItemFromSelection(item);
	else
		selectItem(item);
}
}
