package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A ULCNoteBook lets the user switch between a group of components (a 'page') by clicking on a tab
 * with a given title.
 *
 * @see ULCPage
 */
public class ULCNotebook extends ULCComponent {
	/**
	 * @serial	 	 
	 */
	protected int fCurrentTab = 0;
	/**
	 * @serial	 	 
	 */
	protected int fWidth = 0;
	/**
	 * @serial	 	 
	 */
	protected int fHeight = 0;
	/**
	 * @serial	 	 
	 */
	protected boolean fVeto = false;
	/**
	 * @serial	 	 
	 */
	protected int fTabPlacement = BOX_TOP;
/**
 * Constructs a new ULCNoteBook using the default width and height.
 * It is not receommened to use this method as the default dimension of the notebook 
 * is likely to be wrong for the page contents.
 */
public ULCNotebook() {
	this(false, 100, 100, BOX_TOP);
}
/**
 * Constructs a new ULCNoteBook with the given width and height.
 * The dimensions have to be given in advance because pages are loaded on demand and their size
 * is not know initially.
 *
 * @param veto 		if true the application has to switch pages explicitely by listening for an selectionChangedEvent
 * 					and sending a corresponding showTab request. If false the UIEngine switches a page automatically.
 * @param width		The width of the notebook.
 * @param height	The height of the notebook.
 */
public ULCNotebook(boolean veto, int width, int height) {
	this(veto,width,height, BOX_TOP);
}
/**
 * Constructs a new ULCNoteBook with the given width and height.
 * The dimensions have to be given in advance because pages are loaded on demand and their size
 * is not know initially.
 *
 * @param veto 			if true the application has to switch pages explicitely by 
 *						listening for an selectionChangedEvent and sending a 
 *						corresponding showTab request. 
 *						If false the UIEngine switches a page automatically.
 * @param width			The width of the notebook.
 * @param height		The height of the notebook.
 * @param tabPlacement	Placement of the tab can be either BOX_TOP, BOX_BOTTOM, BOX_LEFT or BOX_RIGHT
 */
public ULCNotebook(boolean veto, int width, int height, int tabPlacement) {
	fWidth = width;
	fHeight = height;
	fVeto = veto;
	fTabPlacement= tabPlacement;
}
/**
 * Add the specified page to the receiver.
 *
 * @param page	A ULCPage
 */
public void addPage(ULCPage page) {
	internalAdd(page);
}
/**
 * Adds the collection of <code>ULCPage's</code> to the notebook.
 *
 * @param pages A <code>Vector</code> containing <code>ULCPages's</code>
 */
public void addPages(Vector pages) {
	internalAdd(pages);
}
/**
 * Registers the given listener to begin receiving notifications
 * when switching notebook pages.
 *
 * @param listener	The object interested in my selectionChangedEvents.
 */
public void addSelectionChangedListener(ISelectionChangedListener listener) {
	internalAddListener("tabChanged", listener);
}
/**
 * Adds a new page with the specified title and containing the given ULCComponent to the notebook.
 *
 * @param title	The <code>String</code> label of the the new <code>ULCPage</code>
 * @param c		The widget that should be displayed in this new  <code>ULCPage</code>.
 */
public void addTab(String title, ULCComponent c) {
	internalAdd(new ULCPage(title, c));
}
/**
 * Returns the index of the currently active page.
 *
 * @return The index of the <code>ULCPage</code> within the collection of active pages.
 */
public int getActiveTab() {
	return fCurrentTab;
}
/**
 * Gets the current height of the Notebook.
 */
public int getHeight() {
	return fHeight;
}
/**
 * Return the <code>Vector</code> containing all the <code>ULCPage's</code> in this notebbok.
 *
 * @return The <code>Vector</code> of <code>ULCPage's</code>
 */
public Vector getPages() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		Assert.isTrue(next instanceof ULCPage);
		components.addElement((ULCPage) next);
	}
	return components;
}
/**
 * Returns the tabPlacement of the Notebook.
 * The tabPlacement can be BOX_TOP, BOX_BOTTOM, BOX_LEFT or BOX_RIGHT
 */
public int getTabPlacement() {
	return fTabPlacement;
}
/**
 * Determines whether the UIEngine switches pages without involving the application.
 *
 * If veto is true, the UI Engine always sends the request to the application and 
 *				the application must explicitly call <code>showTab</code>.
 */
public boolean getVeto() {
	return fVeto;
}
/**
 * returns the current width of the Notebook
 */
public int getWidth() {
	return fWidth;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("tabChanged")) {
			if (args.get("veto", false)) { // just track state change but don't fire anything
				int tab = args.get("tab", fCurrentTab);
				distributeToListeners("tabChanged", new ULCSelectionChangedEvent(this, "tabChanged", tab, tab));
			} else {
				fCurrentTab = args.get("tab", fCurrentTab);
				distributeToListeners("tabChanged", new ULCSelectionChangedEvent(this, "tabChanged", fCurrentTab, fCurrentTab));
			}
			return;
		}
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Remove the specified ULCpage from my list of pages.
 *
 * @param page	The <code>ULCpage</code> to be removed.
 */
public void removePage(ULCPage page) {
	internalRemove(page);
}
/**
 * Removes the collection of <code>ULCPage's</code> from the notebook.
 *
 * @param pages A <code>Vector</code> containing <code>ULCPages's</code>
 */
public void removePages(Vector pages) {
	internalRemove(pages);
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive events. 
 *
 * @param listener	The object that was registered for my selectionChanged events
 */
public void removeSelectionChangedListener(ISelectionChangedListener listener) {
	internalRemoveListener("tabChanged", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	// turn on preloading on current tab
	if (fCurrentTab >= 0 && fList != null) {
		ULCPage page = (ULCPage) fList.elementAt(fCurrentTab);
		if (page != null)
			page.setPreload(true);
	}
	super.saveState(a);
	if (fWidth != 0)
		a.put("w", fWidth);
	if (fHeight != 0)
		a.put("h", fHeight);
	if (fCurrentTab >= 0)
		a.put("tab", fCurrentTab);
	if (fVeto)
		a.put("veto", fVeto);
	if (fTabPlacement != BOX_TOP)
		a.put("tabPlacement", fTabPlacement);
}
/**
 * Sets page at tabIndex to be the active page.
 *
 * @param The index of the <code>ULCPage</code> within the collection of active pages.
 */
public void setActiveTab(int tabIndex) {
	if (fCurrentTab != tabIndex) {
		fCurrentTab = tabIndex;
		sendUI("showTab", new Anything(fCurrentTab));
		distributeToListeners("tabChanged", new ULCSelectionChangedEvent(this, "tabChanged", fCurrentTab, fCurrentTab));
	}
}
/**
 * Sets the current height of the Notebook.
 * This method must be used before the notebook is uploaded to the UI.
 * If used after the notebook is uploaded it has no effect.
 */
public void setHeight(int height) {
	fHeight = height;
}
/**
 * Sets the tabPlacement of the Notebook.
 * The tabPlacement can be BOX_TOP, BOX_BOTTOM, BOX_LEFT or BOX_RIGHT
 */
public void setTabPlacement(int tabPlacement) {
	if (fTabPlacement != tabPlacement) {
		fTabPlacement = tabPlacement;
		sendUI("setTabPlacement", new Anything(fTabPlacement));
	}
}
/**
 * Determines whether the UIEngine switches pages without involving the application.
 *
 * @param veto 	If true, the UI Engine always sends the request to the application and 
 *				the application must explicitly call <code>showTab</code>.
 */
public void setVeto(boolean veto) {
	if (fVeto != veto) {
		fVeto = veto;
		sendUI("setVeto", new Anything(fVeto));
	}
}
/**
 * Sets the current width of the Notebook.
 * This method must be used before the notebook is uploaded to the UI.
 * If used after the notebook is uploaded it has no effect.
 */
public void setWidth(int width) {
	fWidth = width;
}
}
