package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.Vector;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A ULCFormModel is a default implementation for a 'remote' model, that is a model which is split
 * across the UIEngine and the ULC application side. It implements the protocol defined by the UIEngine
 * for synchronizing two model halfs in terms of a simple getValue/setValue API.
 * Clients have to subclass from ULCFormModel to at least implement the <code>getValueAt</code> method.
 * Typically subclasses are not storing the data itself but are adapters which redirect the getValueAt and
 * setValueAt calls to some other model or data structure.
 *
 * @see ULCFormModel#getValueAt(String)
 * @see ULCFormModel#setValueAt(Object, String)
 */
abstract public class ULCFormModel extends ULCAbstractModel implements IEnabler, IForm {
	/**
	 * The current notification policy.
	 * @serial	 
	 */
	protected int fNotificationPolicy = FORM_NOTIFICATION_ON_REQUEST;
	/**
	 * This flag will be set to false after the outstanding synchronous request is complete.
	 * @serial	 
	 */
	protected boolean fGotReply = false;
	/**
	 * If true all changes in the UI will be sent to the application for confirmation.
	 * @serial	 
	 */
	protected boolean fVetoChanges = false;
	/**
	 * @serial	 
	 */
	private static final boolean DEBUG = false;
/**
 * Construct a new ULCFormModel with the default notification policy.
 */
public ULCFormModel() {
	super();
	setUploadAllAttributes(true);
}
/**
 * Construct a new ULCFormModel with the given notification policy.
 *
 * @param policy	The notification policy
 * <pre>
 * The Notification policy can be one of:
 *
 * 	FORM_NOTIFICATION_IMMEDIATE
 *		any changes in the form field are immediately
 *		reported back to the application.
 *
 *	FORM_NOTIFICATION_ON_REQUEST
 *		Changes will be reported back only when the user explicitly calls
 *		<code>saveInput</code> on the FormModel.
 *
 * </pre>
 */
public ULCFormModel(int policy) {
	this();
	fNotificationPolicy = policy;
}
/**
 * Add all attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes	The <code>Vector</code> into which the receiver's
 									preloadAttributes should be added.
 */
protected void addPreloadAttributesInto(Vector vectorOfPreloadAttributes) {
	if (!getUploadAllAttributes())
		return;
	addPreloadFormAttributesInto(vectorOfPreloadAttributes);
}
/**
 * Notify my peer that all values are to be marked as invalid.
 * This will cause the UI to request the values from the application once again.
 * Any changes that have been made in the UI are lost.
 */
public void cancelInput() {
	notify(FORM_MODEL_CHANGED, null);
}
/**
 * Internal method which gathers data in the given Anything by calling getValueAt and
 * sends it back to the UIEngine.
 *
 * @param cols dictionary of attribute names and null value slots.
 */
public void getData(Anything cols) {
	prepareData(cols);
	sendUI("setData", cols);
}
/**
 * Gets the notification policy of this component.
 *
 * @param policy The new notification policy to be set.
 * <pre>
 * The Notification policy can be one of:
 *
 * 	FORM_NOTIFICATION_IMMEDIATE
 *		any changes in the form field are immediately
 *		reported back to the application.
 *
 *	FORM_NOTIFICATION_ON_REQUEST	
 *		Changes will be reported back only when the user explicitly calls
 *		<code>saveInput</code> on the FormModel.
 *
 * </pre>
 */
public int getNotificationPolicy() {
	return fNotificationPolicy;
}
/**
 * Override this method to return the requested value for the given key.
 *
 * @param formAttributeName 	The key at which the object is.
 */
abstract public Object getValueAt(String formAttributeName);
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
synchronized public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("getData")) {
		getData(args);
		return;
	}
	if (request.equals("setData")) {
		Anything data = args.get("data");
		if (data != null) {
			for (int i = 0; i < data.size(); i++) {
				String key = data.slotName(i);
				Anything value = data.get(i);
				if (key != null && value != null)
					setValueAt(convert(value), key);
			}
		}
		fGotReply = true;
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Notify UI FormModel of changes.
 *
 * @param type The kind of change.
 * <pre>
 * The type of change can be one of:
 *		FORM_MODEL_UNSPECIFIED
 *		FORM_MODEL_INVALID
 *		FORM_MODEL_CHANGED
 * </pre>
 *
 * @param formAttributeName 	The key at which the change occurred.
 */
public void notify(int type, String formAttributeName) {
	Anything a = new Anything();
	a.put("type", type);
	if (formAttributeName != null)
		a.put("key", formAttributeName);
	sendUI("change", a);
}
/**
 * Internal method for collecting values in the given Anything.
 *
 * @param cols Anything		The Anything into which the values are added.
 */
public void prepareData(Anything cols) {
	for (int col = 0; col < cols.size(); col++) {
		String colId = cols.slotName(col);
		if (colId != null) {
			Object o = getValueAt(colId);
			cols.put(colId, convert(fContext, o));
		}
	}
	if (DEBUG) {
		System.out.print("prepareData: ");
		cols.dump(System.out);
		System.out.println();
	}
}
/**
 * Internal method for coolecting all the data for all preloadAttributes.
 */
private Anything preparePreloadAttributes() {
	Anything a = new Anything();
	Vector o = new Vector();
	addPreloadFormAttributesInto(o);
	for (int i = 0; i < o.size(); i++)
		a.put((String) o.elementAt(i), new Anything());
	prepareData(a);
	return a;
}
/**
 * Synchroneously request any changes of the FormModel.
 * Updated values are reported via the setValueAt method on this model while blocking in this method.
 *
 * @returns true if the values have been recieved. Returns false if the connection failed
 * 			while waiting for the reply.
 */
public boolean saveInput() {
	sendUI("flush", new Anything());
	fGotReply = false;
	while ((!fGotReply) && !fContext.processNextRequest(0)) {
		try {
			Thread.sleep(100);
		}
		catch (Exception e) {
		}
	};
	return fGotReply;
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fNotificationPolicy != FORM_NOTIFICATION_ON_REQUEST)
		a.put("notificationPolicy", fNotificationPolicy);
	if (fVetoChanges)
		a.put("veto", fVetoChanges);
	if (getUploadAllAttributes())
		a.put("data", preparePreloadAttributes());
}
/**
 * Set the notification policy of this component.
 *
 * @param policy The new notification policy to be set.
 * <pre>
 * The Notification policy can be one of:
 *
 * 	FORM_NOTIFICATION_IMMEDIATE
 *		any changes in the form field are immediately
 *		reported back to the application.
 *
 *	FORM_NOTIFICATION_ON_REQUEST	
 *		Changes will be reported back only when the user explicitly calls
 *		<code>saveInput</code> on the FormModel.
 *
 * </pre>
 */
public void setNotificationPolicy(int policy) {
	if (fNotificationPolicy != policy) {
		fNotificationPolicy = policy;
		sendUI("setNotificationPolicy", new Anything(fNotificationPolicy));
	}
}
/**
 * Override this method to save updated form values.
 * The default implementation prints the given argument on the console.
 *
 * @param formAttributeName 	The key at which the change occurred.
 * @param value				 	The value to be set at the specified key.
 */
public void setValueAt(Object value, String formAttributeName) {
	System.out.println("setValueAt(" + formAttributeName + "): " + value.toString());
}
/**
 * returns the fully qualified class name of my UI class
 */
protected String typeString() {
	return "FormModel";
}
}
