package com.ibm.ulc.application;

/*
 * Copyright (c) 1999 Object Technology International Inc.
 */

import java.io.*;

/**
 * A file input stream is an input stream for reading data from a 
 * <code>ULCFile</code> located on the client. 
 *
 * @see	    ULCFileOutputStream
 */
public class ULCFileInputStream extends InputStream {
	protected BufferedInputStream fDelegate = null;
	protected ULCContext fContext = null;
/**
 * Creates an input file stream to read from the specified 
 * <code>ULCFile</code> object located on the client system. 
 *
 * @param      context the ULCContext upon which the file stream should 
 *             be created.
 * @param      file   the file to be opened for reading.
 * @exception  FileNotFoundException  if the file is not found.
 */
public ULCFileInputStream(com.ibm.ulc.application.ULCContext context, ULCFile file) throws FileNotFoundException {
	this(context, file.getPath());
}
/**
 * Creates an input file stream to read from a file with the 
 * specified name located on the client system. 
 *
 * @param      context the ULCContext upon which the file stream should 
 *             be created.
 * @param      name   the system-dependent file name on the client.
 * @exception  FileNotFoundException  if the file is not found.
 * @see        java.lang.SecurityManager
 */
public ULCFileInputStream(com.ibm.ulc.application.ULCContext context, String name) throws FileNotFoundException {
	fDelegate = new BufferedInputStream(new AFileInputStream(context, name), 64 * 1024);
	fContext = context;
}
/**
 * Returns the number of bytes that can be read from this file input
 * stream without blocking.
 *
 * @return     the number of bytes that can be read from this file input
 *             stream without blocking.
 * @exception  IOException  if an I/O error occurs.
 */
public int available() throws IOException {
	return fDelegate.available();
}
/**
 * Closes this file input stream and releases any system resources 
 * associated with the stream. 
 *
 * @exception  IOException  if an I/O error occurs.
 */
public void close() throws IOException {
	fDelegate.close();
}
/**
 * Marks the current position in this input stream. A subsequent 
 * call to the <code>reset</code> method repositions this stream at 
 * the last marked position so that subsequent reads re-read the same 
 * bytes. 
 * <p>
 * The <code>readlimit</code> arguments tells this input stream to 
 * allow that many bytes to be read before the mark position gets 
 * invalidated. 
 * <p>
 * The <code>mark</code> method of <code>InputStream</code> does nothing.
 *
 * @param   readlimit   the maximum limit of bytes that can be read before
 *                      the mark position becomes invalid.
 * @see     java.io.InputStream#reset()
 */
public synchronized void mark(int readlimit) {
	fDelegate.mark(readlimit);
}
/**
 * Tests if this input stream supports the <code>mark</code> 
 * and <code>reset</code> methods. The <code>markSupported</code> 
 * method of <code>InputStream</code> returns <code>false</code>. 
 *
 * @return  <code>true</code> if this true type supports the mark and reset
 *          method; <code>false</code> otherwise.
 * @see     java.io.InputStream#mark(int)
 * @see     java.io.InputStream#reset()
 */
public boolean markSupported() {
	return fDelegate.markSupported();
}
/**
 * Reads a byte of data from this input stream. This method blocks 
 * if no input is yet available. 
 *
 * @return     the next byte of data, or <code>-1</code> if the end of the
 *             file is reached.
 * @exception  IOException  if an I/O error occurs.
 */
public int read() throws IOException {
	return fDelegate.read();
}
/**
 * Reads up to <code>b.length</code> bytes of data from this input 
 * stream into an array of bytes. This method blocks until some input 
 * is available. 
 *
 * @param      b   the buffer into which the data is read.
 * @return     the total number of bytes read into the buffer, or
 *             <code>-1</code> if there is no more data because the end of
 *             the file has been reached.
 * @exception  IOException  if an I/O error occurs.
 */
public int read(byte b[]) throws IOException {
	return fDelegate.read(b, 0, b.length);
}
/**
 * Reads up to <code>len</code> bytes of data from this input stream 
 * into an array of bytes. This method blocks until some input is 
 * available. 
 *
 * @param      b     the buffer into which the data is read.
 * @param      off   the start offset of the data.
 * @param      len   the maximum number of bytes read.
 * @return     the total number of bytes read into the buffer, or
 *             <code>-1</code> if there is no more data because the end of
 *             the file has been reached.
 * @exception  IOException  if an I/O error occurs.
 */
public int read(byte b[], int off, int len) throws IOException {
	return fDelegate.read(b, off, len);
}
/**
 * Repositions this stream to the position at the time the 
 * <code>mark</code> method was last called on this input stream. 
 * <p>
 * The <code>reset</code> method of <code>InputStream</code> throws 
 * an <code>IOException</code>, because input streams, by default, do 
 * not support <code>mark</code> and <code>reset</code>.
 * <p>
 * Stream marks are intended to be used in
 * situations where you need to read ahead a little to see what's in
 * the stream. Often this is most easily done by invoking some
 * general parser. If the stream is of the type handled by the
 * parser, it just chugs along happily. If the stream is not of
 * that type, the parser should toss an exception when it fails,
 * which, if it happens within readlimit bytes, allows the outer
 * code to reset the stream and try another parser.
 *
 * @exception  IOException  if this stream has not been marked or if the
 *               mark has been invalidated.
 * @see     java.io.InputStream#mark(int)
 * @see     java.io.IOException
 */
public synchronized void reset() throws IOException {
	fDelegate.reset();
}
/**
 * Skips over and discards <code>n</code> bytes of data from the 
 * input stream. The <code>skip</code> method may, for a variety of 
 * reasons, end up skipping over some smaller number of bytes, 
 * possibly <code>0</code>. The actual number of bytes skipped is returned.
 *
 * @param      n   the number of bytes to be skipped.
 * @return     the actual number of bytes skipped.
 * @exception  IOException  if an I/O error occurs.
 */
public long skip(long n) throws IOException {
	return fDelegate.skip(n);
}
}
