package com.ibm.ulc.application;

/*
 * Copyright (c) 1999 Object Technology International Inc.
 */

import com.ibm.ulc.util.*;
import com.ibm.ulc.base.*;
import com.ibm.ulc.comm.*;
import java.util.*;
import java.io.*;
/**
 * The ULCFileChooser class is used to select a file for read or write from the file
 * system of the UIEngine client.
 *
 * @see ULCFile
 */
public class ULCFileChooser extends ULCComponent {
	protected String fDialogTitle = null;
	protected String fApproveButtonText = null;
	protected String fApproveButtonToolTipText = null;
	protected boolean fIsSet_ApproveButtonToolTipText = false;
	protected int fApproveButtonMnemonic = 0;
	protected int fDialogType = FILE_OPEN_DIALOG;
	protected int fReturnValue = FILE_ERROR_OPTION;
	protected boolean fUseFileHiding = true;
	protected int fFileSelectionMode = FILE_FILES_ONLY;
	protected boolean fMultiSelectionEnabled = false;
	protected String fCurrentDirectoryPath = null;
	protected ULCFile fCurrentDirectory = null;
	protected ULCFile fSelectedFile = null;
	protected ULCFile[] fSelectedFiles = null;
	protected UlcFileFilter fFilter = null;
	protected Vector fFilters = new Vector();
	protected Anything fReply = null;
	protected ULCShell fParentShell = null;
/**
 * Creates a ULCFileChooser pointing to the user's home directory.
 */
public ULCFileChooser() {
	fVisible = false;
}
/**
 * Creates a ULCFileChooser pointing to the user's home directory.
 */
public ULCFileChooser(ULCContext context) {
	this(context, (ULCFile) null);
}
/**
 * Creates a ULCFileChooser using the given File as the path. Passing
 * in a null file causes the file chooser to point to the users's
 * home directory.
 *
 * @param directory  a File object specifying the path to a file 
 *                   or directory
 */
public ULCFileChooser(ULCContext context, ULCFile currentDirectory) {
	fContext = context;
	fCurrentDirectory = currentDirectory;
	fVisible= false;
}
/**
 * Creates a ULCFileChooser using the given path. Passing in a null
 * string causes the file chooser to point to the users home directory.
 *
 * @param path  a String giving the path to a file or directory
 */
public ULCFileChooser(ULCContext context, String currentDirectoryPath) {
	fContext = context;
	fCurrentDirectoryPath = currentDirectoryPath;
	fVisible = false;
}
/**
 * Adds a filter to the list of user choosable file filters.
 * 
 * @param filter the ULCFileFilter to add to the choosable file
 *               filter list
 *
 * @see #getChoosableFileFilters
 * @see #removeChoosableFileFilter
 * @see #resetChoosableFileFilters
 */
public void addChoosableFileFilter(UlcFileFilter filter) {
	if (!fFilters.contains(filter) && !filter.equals(getAcceptAllFileFilter())) {
		fFilters.addElement(filter);
	}
}
/**
 * Returns the AcceptAll file filter (e.g. (All Files *.*) on windows).
 */
public UlcFileFilter getAcceptAllFileFilter() {
	return UlcFileFilter.getAcceptAllFileFilter();
}
/**
 * Returns the approve button's mnemonic.
 * @return an int value for the mnemonic key
 *
 * @see #setApproveButtonMnemonic
 */
public int getApproveButtonMnemonic() {
	return fApproveButtonMnemonic;
}
/**
 * Returns the text used in the ApproveButton in the FileChooserUI.
 * If null, the UI object will determine the button's text.
 *
 * Typically, this would be "Open" or "Save".
 *
 * @return the text used in the ApproveButton
 *
 * @see #setApproveButtonText
 * @see #setDialogType
 * @see #showDialog
 */
public String getApproveButtonText() {
	return fApproveButtonText;
}
/**
 * Returns the tooltip text used in the ApproveButton.
 * If null, the UI object will determine the button's text.
 *
 * @return the text used in the ApproveButton
 *
 * @see #setApproveButtonText
 * @see #setDialogType
 * @see #showDialog
 */
public String getApproveButtonToolTipText() {
	return fApproveButtonToolTipText;
}
/**
 * Gets the list of user choosable file filters
 *
 * @return a FileFilter array containing all the choosable
 *         file filters
 *
 * @see #addChoosableFileFilter
 * @see #removeChoosableFileFilter
 * @see #resetChoosableFileFilters
 */
public UlcFileFilter[] getChoosableFileFilters() {
	UlcFileFilter[] filterArray = new UlcFileFilter[fFilters.size()];
	fFilters.copyInto(filterArray);
	return filterArray;
}
/**
 * Returns the current directory. 
 *
 * @return the current directory
 * @see #setCurrentDirectory
 */
public ULCFile getCurrentDirectory() {
	return fCurrentDirectory;
}
/**
 * Gets the string that goes in the FileChooser's titlebar.
 *
 * @see #setDialogTitle
 */
public String getDialogTitle() {
	return fDialogTitle;
}
/**
 * Returns the type of this dialog.
 *
 * @return   the type of dialog to be displayed:
 *           FILE_OPEN_DIALOG, FILE_SAVE_DIALOG, FILE_CUSTOM_DIALOG
 *
 * @see #setDialogType
 */
public int getDialogType() {
	return fDialogType;
}
/**
 * Returns the currently selected file filter.
 *
 * @return the current file filter.
 * @see #setFileFilter
 * @see #addChoosableFileFilter
 */
public UlcFileFilter getFileFilter() {
	return fFilter;
}
/**
 * Returns the current file-selection mode.
 *
 * @param an int indicating the type of dialog to be displayed:
 *                   FILE_FILES_ONLY, FILE_DIRECTORIES_ONLY, FILE_FILES_AND_DIRECTORIES
 * @see #setFileSelectionMode
 */
public int getFileSelectionMode() {
	return fFileSelectionMode;
}
/**
 * Returns the selected file. This can be set either by the
 * programmer via setFile() or by a user action, such as
 * either typing the filename int the UI or selecting the
 * file from a list in the UI.
 * 
 * @see #setSelectedFile
 * @return the selected file
 */
public ULCFile getSelectedFile() {
	return fSelectedFile;
}
/**
 * Returns a list of selected files if the filechooser is
 * set to allow multi-selection.
 */
public ULCFile[] getSelectedFiles() {
	return fSelectedFiles;
}
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("reply")) {
		fReply = args;
		fVisible = false;
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Convenience call that determines if directories are selectable based on the current
 * file selection mode
 *
 * @see #setFileSelectionMode
 * @see #getFileSelectionMode
 */
public boolean isDirectorySelectionEnabled() {
	return ((fFileSelectionMode == FILE_DIRECTORIES_ONLY) || (fFileSelectionMode == FILE_FILES_AND_DIRECTORIES));
}
/**
 * If true, hidden files are not shown in the filechooser
 *
 * @return the status of the file hiding property
 * @see #setFileHidingEnabled
 */
public boolean isFileHidingEnabled() {
	return fUseFileHiding;
}
/**
 * Convenience call that determines if files are selectable based on the current
 * file selection mode
 *
 * @see #setFileSelectionMode
 * @see #getFileSelectionMode
 */
public boolean isFileSelectionEnabled() {
	return ((fFileSelectionMode == FILE_FILES_ONLY) || (fFileSelectionMode == FILE_FILES_AND_DIRECTORIES));
}
/**
 * Returns true if multiple files can be selected.
 * @return true if multiple files can be selected.
 * @see #setMultiSelectionEnabled
 */
public boolean isMultiSelectionEnabled() {
	return fMultiSelectionEnabled;
}
protected void prepareArguments(Anything args) {
	if (fParentShell != null)
		args.put("parentShell", fParentShell.getRef(fContext));
	args.put("dialogType", fDialogType);
	args.put("useFileHiding", fUseFileHiding);
	args.put("fileSelectionMode", fFileSelectionMode);
	args.put("multiSelectionEnabled", fMultiSelectionEnabled);
	if (fDialogTitle != null)
		args.put("dialogTitle", fDialogTitle);
	if (fApproveButtonText != null)
		args.put("approveButtonText", fApproveButtonText);
	if (fIsSet_ApproveButtonToolTipText)
		args.put("approveButtonToolTipText", fApproveButtonToolTipText);
	if (fApproveButtonMnemonic > 0)
		args.put("approveButtonMnemonic", fApproveButtonMnemonic);
	if (fCurrentDirectoryPath != null)
		args.put("currentDirectoryPath", fCurrentDirectoryPath);
	if (fCurrentDirectory != null)
		args.put("currentDirectory", fCurrentDirectory.getAbsolutePath());
	if (fSelectedFile != null)
		args.put("selectedFile", fSelectedFile.getAbsolutePath());
	if (fSelectedFiles != null) {
		String files[] = new String[fSelectedFiles.length];
		for (int i = 0; i < files.length; i++) {
			files[i] = fSelectedFiles[i].getAbsolutePath();
		}
		args.put("selectedFiles", files);
	}
	if (fFilter != null) {
		args.put("filter", fFilter);
		if (fFilters.contains(fFilter)) {
			removeChoosableFileFilter(fFilter);
		}
	}
	if (fFilters != null)
		args.put("filters", getChoosableFileFilters());
}
/**
 * Removes a filter from the list of user choosable file filters. Returns
 * true if the file filter was removed;
 *
 * @see #addChoosableFileFilter
 * @see #getChoosableFileFilters
 * @see #resetChoosableFileFilters
 */
public boolean removeChoosableFileFilter(UlcFileFilter f) {
	if (fFilters.contains(f)) {
		fFilters.removeElement(f);
		return true;
	} else {
		return false;
	}
}
/**
 * Resets the choosable file filter list to it's starting state. Normally,
 * this removes all added file filters while leaving the AcceptAll file filter.
 *
 * @see #addChoosableFileFilter
 * @see #getChoosableFileFilters
 * @see #removeChoosableFileFilter
 */
public void resetChoosableFileFilters() {
	fFilters.removeAllElements();
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	prepareArguments(a);
}
/**
 * Sets the approve button's mnemonic using a character.
 * @param an char value for the mnemonic key
 *
 * @see #getApproveButtonMnemonic
 */
public void setApproveButtonMnemonic(char mnemonic) {
	int vk = (int) mnemonic;
	if (vk >= 'a' && vk <= 'z') {
		vk -= ('a' - 'A');
	}
	setApproveButtonMnemonic(vk);
}
/**
 * Sets the approve button's mnemonic using a numeric keycode.
 * @param an int value for the mnemonic key
 *
 * @see #getApproveButtonMnemonic
 */
public void setApproveButtonMnemonic(int mnemonic) {
	fApproveButtonMnemonic = mnemonic;
}
/**
 * Sets the text used in the ApproveButton in the FileChooserUI.
 *
 * @param approveButtonText the text used in the ApproveButton
 *
 * @see #getApproveButtonText
 * @see #setDialogType
 * @see #showDialog
 */
public void setApproveButtonText(String approveButtonText) {
	fApproveButtonText = approveButtonText;
}
/**
 * Sets the tooltip text used in the ApproveButton.
 * If null, the UI object will determine the button's text.
 *
 * @beaninfo
 *   preferred: true
 *       bound: true
 * description: The tooltip text for the ApproveButton
 *
 * @return the text used in the ApproveButton
 *
 * @see #setApproveButtonText
 * @see #setDialogType
 * @see #showDialog
 */
public void setApproveButtonToolTipText(String toolTipText) {
	fApproveButtonToolTipText = toolTipText;
	fIsSet_ApproveButtonToolTipText = true;
	fDialogType = FILE_CUSTOM_DIALOG;
}
/**
 * Sets the current directory. Passing in null sets the filechooser
 * to point to the users's home directory.
 *
 * If the file passed in as currentDirectory is not a directory, the
 * parent of the file will be used as the currentDirectory. If the
 * parent is not traversable, then it will walk up the parent tree
 * until it finds a traversable direcotry, or hits the root of the
 * file system.
 *     
 * @param currentDirectory the current directory to point to
 * @see #getCurrentDirectory
 */
public void setCurrentDirectory(ULCFile dir) {
	fCurrentDirectoryPath = null;
	fCurrentDirectory = dir;
}
/**
 * Sets the string that goes in the FileChooser window's title bar.
 *
 * @see #getDialogTitle
 *
 */
public void setDialogTitle(String dialogTitle) {
	fDialogTitle = dialogTitle;
}
/**
 * Sets the type of this dialog. Use OPEN_DIALOG when you want to
 * bring up a filechooser that the user can use to open a file. Likewise,
 * use FILE_SAVE_DIALOG for letting the user choose a file for saving.
 *
 * Use FILE_CUSTOM_DIALOG when you want to use the filechooser in a context
 * other than "Open" or "Save". For instance, you might want to bring up
 * a filechooser that allows the user to choose a file to execute. Note that
 * you normally would not need to set the FileChooser to use FILE_CUSTOM_DIALOG
 * since a call to setApproveButtonText does this for you.
 *
 * @param dialogType the type of dialog to be displayed:
 *                   FILE_OPEN_DIALOG, FILE_SAVE_DIALOG, FILE_CUSTOM_DIALOG
 *
 * @see #getDialogType
 * @see #setApproveButtonText
 */
public void setDialogType(int dialogType) {
	fDialogType = dialogType;
}
/**
 * Sets the current File Filter. The file filter is used by the
 * filechooser to filter out files from view from the user.
 *
 * @param filter the new current file filter to use
 * @see #getFileFilter
 */
public void setFileFilter(UlcFileFilter filter) {
	if (filter.equals(getAcceptAllFileFilter())) {
		fFilter = null;
	} else {
		fFilter = filter;
	}
}
/**
 * Sets file hiding on or off. If true, hidden files are not shown
 * in the filechooser. The job of determining which files are
 * show is done by the FileView.
 *
 * @param b the boolean value that determines whether file hiding is
 *          turned on or not.
 * @see #isFileHidingEnabled
 */
public void setFileHidingEnabled(boolean b) {
	fUseFileHiding = b;
}
/**
 * Sets the FileChooser to allow the user to just select files, just select
 * directories, or select both files and directetories.
 *
 * @param dialogType the type of dialog to be displayed:
 *                   FILE_FILES_ONLY, FILE_DIRECTORIES_ONLY, FILE_FILES_AND_DIRECTORIES
 *
 * @see #getFileSelectionMode
 */
public void setFileSelectionMode(int mode) {
	fFileSelectionMode = mode;
}
/**
 * Sets the filechooser to allow multiple file selections.
 * NOTE: this functionality is not yet implemented in the current L&Fs.
 *
 * @see #isMultiSelectionEnabled
 */
public void setMultiSelectionEnabled(boolean b) {
	fMultiSelectionEnabled = b;
}
/**
 * Sets the selected file. If the file's parent directory is
 * not the current directory, it changed the current directory
 * to be the files parent directory.
 *
 * @see #getSelectedFile
 *
 * @param selectedFile the selected file 
 */
public void setSelectedFile(ULCFile selectedFile) {
	fSelectedFile = selectedFile;
}
/**
 * Sets the list of selected files if the filechooser is
 * set to allow multi-selection.
 *
 */
public void setSelectedFiles(ULCFile[] selectedFiles) {
	fSelectedFiles = selectedFiles;
}
/**
 * Pops open the currently configured file chooser dialog.
 *
 * @param   parentShell The ULCShell over which this dialog should be opened.
 * @return  the return state of the filechooser on popdown:
 *             FILE_CANCEL_OPTION, FILE_APPROVE_OPTION
 */
 protected int showDialog(ULCShell parentShell) {
	if ((fContext == null) && (parentShell != null))
		fContext = parentShell.getContext();
	fParentShell = parentShell;
	Anything args = new Anything();
	prepareArguments(args);
	if (fParentShell != null)
		fParentShell.setModalComponent(this);
	Anything res = syncCall("showDialog", args);
	if (fParentShell != null)
		fParentShell.clearModalComponent(this);
	String directory = res.get("currentDirectory", (String) null);
	if (directory != null) {
		fCurrentDirectory = new ULCFile(fContext, directory);
	}
	String file = res.get("selectedFile", (String) null);
	if (file != null) {
		fSelectedFile = new ULCFile(fContext, file);
	}
	String files[] = (String[]) res.get("selectedFiles", (Serializable) null);
	if (files != null) {
		fSelectedFiles = new ULCFile[files.length];
		for (int i = 0; i < files.length; i++) {
			fSelectedFiles[i] = new ULCFile(fContext, files[i]);
		}
	}
	fReturnValue = res.get("returnValue", FILE_ERROR_OPTION);
	return fReturnValue;
}
/**
 * Pops a custom file chooser dialog with a custom ApproveButton.
 *
 * e.g. filechooser.showDialog(parentWindow, "Run Application");
 * would pop up a filechooser with a "Run Application" button
 * (instead of the normal "Save" or "Open" button).
 *
 * Alternatively, the following code will do the same thing:
 *    JFileChooser chooser = new JFileChooser(null);
 *    chooser.setApproveButtonText("Run Application");
 *    chooser.showDialog(this, null);
 * 
 * @param   approveButtonText the text of the ApproveButton
 * @return  the return state of the filechooser on popdown:
 *             FILE_CANCEL_OPTION, FILE_APPROVE_OPTION
 */
public int showDialog(ULCShell parent, String approveButtonText) {
	fApproveButtonText = approveButtonText;
	fDialogType = FILE_CUSTOM_DIALOG;
	return showDialog(parent);
}
/**
 * Pops up an "Open File" file chooser dialog. Note that the
 * text that appears in the approve button is determined by
 * the L&F.
 *
 *
 * @return   the return state of the filechooser on popdown:
 *             FILE_CANCEL_OPTION, FILE_APPROVE_OPTION
 */
public int showOpenDialog(com.ibm.ulc.application.ULCShell parent) {
	fDialogType = FILE_OPEN_DIALOG;
	return showDialog(parent);
}
/**
 * Pops up a "Save File" file chooser dialog. Note that the
 * text that appears in the approve button is determined by
 * the L&F.
 *
 * @return   the return state of the filechooser on popdown:
 *             FILE_CANCEL_OPTION, FILE_APPROVE_OPTION
 */
public int showSaveDialog(com.ibm.ulc.application.ULCShell parent) {
	fDialogType = FILE_SAVE_DIALOG;
	return showDialog(parent);
}
protected synchronized Anything syncCall(String request, Anything args) {
	upload(fContext);
	fReply = null;
	fVisible = true;
	sendUI(request, args);
	while ((fReply == null) && !fContext.processNextRequest(0)) {
		try {
			Thread.sleep(100);
		} catch (Exception e) {
		}
	};
	return fReply;
}
/**
 * returns a logical name for this component.
 * This name is used to locate the other half of this object in the UI Engine.
 * The default implementation extracts the name from the class name by stripping
 * off the package name and an optional prefix "ULC".
 * widgets that are not found in the com.ibm.ulc.ui.swing package should
 * override this method to return the fully qualified class name of the UI class
 * eg: com.ibm.ulc.ui.UIApplication
 *
 * @return The Logical name for this component.
 */
protected String typeString() {
	return "com.ibm.ulc.ui.io.UIFileChooser";
}
}
