-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_bp.db2                                                    --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Collects snapshot data for bufferpools, and writes messages about --
-- potential problems to a message table, db2perf_msg                --
--                                                                   --
-----------------------------------------------------------------------


----------------------------------------------------------------------
-- Drop & then (re)create the db2perf_bufferpool stored procedure.
-- The drop is done via the db2perf_quiet_drop() routine, so that we
-- avoid any error messages if it doesn't exit.
-- Then call db2perf_crmsg() (defined in db2perf_utils.db2) to create
-- the message table we'll use.

CALL db2perf_quiet_drop('PROCEDURE db2perf_bufferpool')@

CALL db2perf_crmsg()@

CREATE PROCEDURE db2perf_bufferpool
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN
   DECLARE current_ts TIMESTAMP;
   DECLARE severity   CHAR(8);
   DECLARE comments   CHAR(40);

   DECLARE data_l_reads            BIGINT;
   DECLARE index_l_reads           BIGINT;
   DECLARE data_and_index_writes   BIGINT;
   DECLARE data_p_reads            BIGINT;
   DECLARE index_p_reads           BIGINT;
   DECLARE async_data_reads        BIGINT;
   DECLARE async_index_reads       BIGINT;
   DECLARE dirty_pg_steals         BIGINT;
   DECLARE dirty_steals_per_10k_tx BIGINT;
   DECLARE transactions            BIGINT;
   DECLARE files_closed            BIGINT;
   DECLARE files_closed_per_10k_tx BIGINT;
   DECLARE data_hr                 DECIMAL(3,1);
   DECLARE idx_hr                  DECIMAL(3,1);
   DECLARE page_clean_ratio        DECIMAL(3,1);
   DECLARE data_prefetch_ratio     DECIMAL(3,1);
   DECLARE index_prefetch_ratio    DECIMAL(3,1);
   DECLARE bp_name 		   CHAR(16);

   DECLARE at_end INT DEFAULT 0;


   ----------------------------------------------------------------------
   -- Declare a cursor against the snapshot_database table function to collect all
   -- the overall bufferpool statistics we're going to want.

   DECLARE overall CURSOR FOR
     SELECT 
        pool_data_l_reads,
        pool_index_l_reads,
	pool_data_writes+pool_index_writes,
        pool_data_p_reads,
        pool_index_p_reads,
        pool_async_data_reads,
        pool_async_index_reads,
	pool_drty_pg_steal_clns,
	commit_sql_stmts,
	files_closed,
	cast(
		(cast(
			(pool_data_l_reads  - pool_data_p_reads) 
		 as double) )*100.0/(pool_data_l_reads+1) 
	as decimal(3,1)) as data_hr,
	cast(
		(cast(
			(pool_index_l_reads - pool_index_p_reads) 
		as double))*100.0/(pool_index_l_reads+1) 
	as decimal(3,1)) as idx_hr,
	cast(
		(cast( 
			(pool_async_data_writes + pool_async_index_writes) 
		as double))*100.0/(pool_data_writes+pool_index_writes+1) 
	as decimal(3,1)) as page_clean_ratio,
	cast(
		(cast(
			pool_async_data_reads 
		as double))*100.0/(pool_data_p_reads+1) 
	as decimal(3,1)) as data_prefetch_ratio,
	cast(
		(cast(
			pool_async_index_reads 
		as double))*100.0/(pool_index_p_reads+1) 
	as decimal(3,1)) as index_prefetch_ratio
	FROM table(snapshot_database(cast (NULL as varchar(256)),-1)) as t;


   ----------------------------------------------------------------------
   -- Declare a cursor against the snapshot_bufferpool table function to collect all
   -- per-bufferpool statistics.

   DECLARE per_bufferpool CURSOR FOR
     SELECT 
 	substr(bp_name,1,16),
        pool_data_l_reads,
        pool_index_l_reads,
	pool_data_writes+pool_index_writes,
        pool_data_p_reads,
        pool_index_p_reads,
        pool_async_data_reads,
        pool_async_index_reads,
	cast(
		(cast(
			pool_data_l_reads  - pool_data_p_reads 
		as double)*100.0)/(pool_data_l_reads+1) 
	as decimal(3,1)) as data_hr,
	cast(
		(cast(
			pool_index_l_reads - pool_index_p_reads 
		as double)*100.0)/(pool_index_l_reads+1) 
	as decimal(3,1)) as idx_hr,
	cast(
		(cast(
			pool_async_data_writes + pool_async_index_writes 
		as double)*100.0)/(pool_data_writes+pool_index_writes+1) 
	as decimal(3,1)) as page_clean_ratio,
	cast(
		(cast(
			pool_async_data_reads 
		as double)*100.0)/(pool_data_p_reads+1) 
 	as decimal(3,1)) as data_prefetch_ratio,
	cast(
		(cast(
			pool_async_index_reads 
		as double)*100.0)/(pool_index_p_reads+1) 
	as decimal(3,1)) as index_prefetch_ratio
	FROM table(snapshot_bp(cast (NULL as varchar(256)),-1)) as t;


   DECLARE msg_cursor CURSOR WITH RETURN TO CALLER FOR
     SELECT * 
     FROM db2perf_msg 
     WHERE ts = current_ts
     ORDER BY ts DESC, severity DESC;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 



   SET current_ts = current timestamp;


   ----------------------------------------------------------------------
   -- First open the cursor with the overall statis.

   OPEN overall;
   FETCH overall INTO data_l_reads, index_l_reads, data_and_index_writes, data_p_reads, index_p_reads, 
		async_data_reads, async_index_reads, dirty_pg_steals, transactions,
		files_closed, data_hr, idx_hr, page_clean_ratio, data_prefetch_ratio, index_prefetch_ratio;
   CLOSE overall;

   ----------------------------------------------------------------------
   -- If there are fewer than 1000 logical reads, then there's not enough activity
   -- in the system to come up with a judgment on whether the hit ratio is ok.
   -- If there is enough activity, then we assign a severity based on the value
   -- of the hit ratio.  For example, if the data hit ratio is under 60%, we 
   -- consider that to be potentially a severe problem with severity 5.

   SET comments = '';
   CASE 
        WHEN (data_l_reads < 1000) THEN
	  SET severity = '';
	  SET comments = 'No data BP activity';
        WHEN (data_hr < 60) THEN
	  SET severity = '5';
        WHEN (data_hr < 75) THEN
	  SET severity = '3';
        WHEN (data_hr < 90) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;

   -- Insert the message with the severity, etc., into our report table.
   -- All rows in the report table are marked with the current timestamp.
   insert into db2perf_msg values 
   	( current_ts, severity, 'Overall BP data hit ratio', cast(data_hr as char(16)), comments );


   ----------------------------------------------------------------------
   -- Repeat the process with the index hit ratio.
   SET comments = '';
   CASE 
        WHEN (index_l_reads < 1000) THEN
	  SET severity = '';
	  SET comments = 'No index BP activity';
        WHEN (idx_hr < 75) THEN
	  SET severity = '5';
        WHEN (idx_hr < 85) THEN
	  SET severity = '3';
        WHEN (idx_hr < 95) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;
   insert into db2perf_msg values 
   	( current_ts, severity, 'Overall BP index hit ratio', cast(idx_hr as char(16)), comments );


   ----------------------------------------------------------------------
   -- And the page clean ratio.  We define this as the ratio of page writes that are
   -- synchronous (written by the agent, or triggered by a steal) to the writes that
   -- are asynchronous (written by a cleaner.)
   SET comments = '';
   CASE 
        WHEN (data_and_index_writes < 1000) THEN
	  SET severity = '';
	  SET comments = 'No BP cleaning activity';
        WHEN (page_clean_ratio < 40) THEN
	  SET severity = '5';
        WHEN (page_clean_ratio < 65) THEN
	  SET severity = '3';
        WHEN (page_clean_ratio < 90) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;
   insert into db2perf_msg values 
   	( current_ts, severity, 'Overall BP page clean ratio', cast(page_clean_ratio as char(16)), comments );


   ----------------------------------------------------------------------
   -- And the data & index prefetch ratios.  We define this as the ratio of page reads that are
   -- synchronous (read by the agent) to the reads that are asynchronous (read by a prefetcher.)
   -- We assume that if there is no prefetch activity at all, this is probably a system doing
   -- only random reads, so a prefetch ratio isn't too helpful to calculate.

   SET comments = '';
   CASE 
        WHEN (async_data_reads < 1000) THEN
	  SET severity = '';
	  SET comments = 'No data prefetching activity';
        WHEN (data_prefetch_ratio < 50) THEN
	  SET severity = '5';
        WHEN (data_prefetch_ratio < 70) THEN
	  SET severity = '3';
        WHEN (data_prefetch_ratio < 90) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;
   insert into db2perf_msg values 
   	( current_ts, severity, 'Overall BP data prefetch ratio', cast(data_prefetch_ratio as char(16)), comments );


   SET comments = '';
   CASE 
        WHEN (async_index_reads < 1000) THEN
	  SET severity = '';
	  SET comments = 'No index prefetching activity';
        WHEN (index_prefetch_ratio < 50) THEN
	  SET severity = '5';
        WHEN (index_prefetch_ratio < 70) THEN
	  SET severity = '3';
        WHEN (index_prefetch_ratio < 90) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;
   insert into db2perf_msg values 
   	( current_ts, severity, 'Overall BP index prefetch ratio', cast(index_prefetch_ratio as char(16)), comments );


   ----------------------------------------------------------------------
   -- A 'dirty steal' is the case where a new bufferpool page is required, and no
   -- free or 'in use but unmodified' ones can be found.  So before the agent can
   -- read the new data into the bufferpool, it has to write out the modified
   -- data first.  This is very expensive, so our severity levels are quite strict.
   -- Here we calculate number of steals per 10,000 transactions - more than 50
   -- is considered a '5' severity.

   SET comments = '';
   SET dirty_steals_per_10k_tx = 10000 * dirty_pg_steals / (transactions+1);
   CASE 
        WHEN (transactions < 100) THEN
	  SET severity = '';
	  SET comments = 'No transaction activity recorded';
        WHEN (dirty_steals_per_10k_tx > 50) THEN
	  SET severity = '5';
        WHEN (dirty_steals_per_10k_tx > 20) THEN
	  SET severity = '3';
        WHEN (dirty_steals_per_10k_tx > 1) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;
   insert into db2perf_msg values 
   	( current_ts, severity, 'Dirty Page Steals / 10k Tx', cast(dirty_steals_per_10k_tx as char(16)), comments );


   ----------------------------------------------------------------------
   -- When DB2 needs to open a new tablespace container file, it uses a file
   -- handle to do it.  The operating system only has a limited supply of these,
   -- so when they're all in use, DB2 has to close another file first, and then
   -- reuse the handle.   This generally isn't too expensive unless the numbers
   -- get quite high.  

   SET comments = '';
   SET files_closed_per_10k_tx = 10000 * files_closed / (transactions+1);
   CASE 
        WHEN (transactions < 100) THEN
	  SET severity = '';
	  SET comments = 'No transaction activity recorded';
        WHEN (files_closed_per_10k_tx > 1000) THEN
	  SET severity = '5';
        WHEN (files_closed_per_10k_tx > 100) THEN
	  SET severity = '3';
        WHEN (files_closed_per_10k_tx > 10) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
   END CASE;
   insert into db2perf_msg values 
   	( current_ts, severity, 'Files closed / 10k Tx', cast(files_closed_per_10k_tx as char(16)), comments );




   ----------------------------------------------------------------------
   -- Now we repeat the above measurements & messages for individual bufferpools.


   SET at_end=0;
   OPEN per_bufferpool;
   FETCH per_bufferpool INTO 
   	bp_name,data_l_reads,index_l_reads,data_and_index_writes,data_p_reads,index_p_reads, 
	async_data_reads,async_index_reads,data_hr,idx_hr,page_clean_ratio,data_prefetch_ratio,
	index_prefetch_ratio;

   WHILE at_end = 0 DO

     
     ----------------------------------------------------------------------
     -- Data hit ratio
     SET comments = '';
     CASE 
        WHEN (data_l_reads < 1000) THEN
	  SET severity = '';
	  SET comments = 'No data BP activity';
        WHEN (data_hr < 60) THEN
	  SET severity = '5';
        WHEN (data_hr < 75) THEN
	  SET severity = '3';
        WHEN (data_hr < 90) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
     END CASE;
     insert into db2perf_msg values 
     	( current_ts, severity, rtrim(bp_name)||' data hit ratio', cast(data_hr as char(16)), comments );


     ----------------------------------------------------------------------
     -- Index hit ratio
     SET comments = '';
     CASE 
        WHEN (index_l_reads < 1000) THEN
	  SET severity = '';
	  SET comments = 'No index BP activity';
        WHEN (idx_hr < 75) THEN
	  SET severity = '5';
        WHEN (idx_hr < 85) THEN
	  SET severity = '3';
        WHEN (idx_hr < 95) THEN
	  SET severity = '1';
	ELSE
   	  SET severity = '0';
     END CASE;
     insert into db2perf_msg values 
     	( current_ts, severity, rtrim(bp_name)||' idx hit ratio', cast(idx_hr as char(16)), comments );


     ----------------------------------------------------------------------
     -- Data prefetch ratio
     SET comments = '';
     CASE 
	  WHEN (async_data_reads < 1000) THEN
	    SET severity = '';
	    SET comments = 'No data prefetching activity';
	  WHEN (data_prefetch_ratio < 50) THEN
	    SET severity = '5';
	  WHEN (data_prefetch_ratio < 70) THEN
	    SET severity = '3';
	  WHEN (data_prefetch_ratio < 90) THEN
	    SET severity = '1';
	  ELSE
	    SET severity = '0';
     END CASE;
     insert into db2perf_msg values 
     	( current_ts, severity, rtrim(bp_name)||' data pftch ratio', cast(data_prefetch_ratio as char(16)), comments );


     ----------------------------------------------------------------------
     -- Index prefetch ratio
     SET comments = '';
     CASE 
	  WHEN (async_index_reads < 1000) THEN
	    SET severity = '';
	    SET comments = 'No index prefetching activity';
	  WHEN (index_prefetch_ratio < 50) THEN
	    SET severity = '5';
	  WHEN (index_prefetch_ratio < 70) THEN
	    SET severity = '3';
	  WHEN (index_prefetch_ratio < 90) THEN
	    SET severity = '1';
	  ELSE
	    SET severity = '0';
     END CASE;
     insert into db2perf_msg values 
     	( current_ts, severity, rtrim(bp_name)||' index pftch ratio', cast(index_prefetch_ratio as char(16)), comments );


     -- Get the data for the next bufferpool
     FETCH per_bufferpool INTO 
     	bp_name,data_l_reads,index_l_reads,data_and_index_writes,data_p_reads,index_p_reads, 
	async_data_reads,async_index_reads,data_hr,idx_hr,page_clean_ratio,data_prefetch_ratio,
	index_prefetch_ratio;

   END WHILE;

   ----------------------------------------------------------------------
   -- Now open the cursor to return a result set to the caller which has the new
   -- rows added to the message table in this run.

   OPEN msg_cursor;

END @
