/*							*/
/*   Licensed Materials - Property of IBM               */
/*							*/
/*   5765-530            				*/
/*   (C) Copyright IBM Corp. 1996 All Rights Reserved.	*/
/*							*/

/*   watchdog.c for ingres has minimal change from previous wm.red      */
/*   version. Added new header files watchdog.h and wdDeclare.h         */
/*   along with a new routine format_reg_msg.                           */
/*   Major change is that watchdogs now longer connect to all the       */
/*   various database during initial setup (routine database_directory  */
/*   has been deleted). Checks are pushed from the InfobaseAgent module */
/*   to each watchdog based on it unique name created by the routine    */
/*   format_reg_msg.                                                    */
/*                                                                      */
/*   Code base used is from wm.red. See wm.red for previous version     */

/******************************************************************************
*				Includes
*****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <memory.h>
#include <syslog.h>
#include <stdlib.h>
#include "watchdog.h"
#include "wdDeclare.h"
#include "ipc.h"
#include "dba-c.h"
#include "trace.h"

#define MIN(a,b)        ((a) < (b) ? (a) : (b))

/******************************************************************************
*				Globals
*****************************************************************************/

EXEC SQL INCLUDE SQLDA;
EXEC SQL INCLUDE SQLCA;

EXEC SQL BEGIN DECLARE SECTION;
        char    idbname[20];
        char    statm[4096];
	char    errbuf[256];
EXEC SQL END DECLARE SECTION;

static IISQLDA *sqldap, *dp;
extern IISQLDA *sqlda = (IISQLDA *)0;
char    *hname;
char    dbdir[255] = "";
char    mserver[30] = "";
int     portno; 
char	dbtype[] = "ing64";
char    dbandInst[250] = "";

static void get_field( IISQLDA *dp, int col, char *fbuf) ;
void db_disconnect();
static IISQLDA *do_fetch();
int watchdog_command( char *command);

/******************************************************************************
*                               Routines
*****************************************************************************/

void format_reg_msg( int messType, char *ipAddress, void *message )
{
        TRACE_FUNC_ENTRY( 1, "format_reg_msg");
        TRACE_FUNC_DATA( 2, STRING, (char *)dbtype);
        sprintf( dbandInst, "%s$%s", dbtype, dbdir );   /* add instance name to message */
        if ( format_message( "db", dbandInst, messType, ipAddress, message ) == 0 )
                TRACE_FUNC_DATA( 2, STRING, "register message formatted" )
        else {
                TRACE_FUNC_DATA( 2, STRING, "error formatting register message" );
                exit (-2);
        }
        TRACE_FUNC_EXIT( 1, "format_reg_msg", NONE, 0 );
}

/* 
 * allocate memory for the sqlda, this is only done once 
 */
void
do_initialize()
{
  static int ready = 0;

  if( ready)
	return;
  else {
	sqldap = (IISQLDA *)calloc( 1, IISQDA_HEAD_SIZE + 
				(SELECT_SIZE * IISQDA_VAR_SIZE));
	sqldap->sqln = SELECT_SIZE;
	sqldap->sqlvar[0].sqldata = malloc(30);
	sqldap->sqlvar[0].sqlind = (short *) malloc(sizeof(short));
	ready = 1;
  }
}

/*
 * Function  : db_connect( char *)
 * Arguments
 *      arg1 :  Name of database to connect to
 * Return Value:
 *      SQL_SUCCESS if connect succeeded, Error otherwise
 * Comments  :
 *      Connects to a database
 * Author    : Marc Gathier
 * History
 *      Created         : Tue Oct  4 15:07:29 CET 1994
 *      Modifications   :
 *
 *
 */
db_connect( char *db)
{
	int	rc=0;

	TRACE_FUNC_ENTRY( 2, "db_connect");
	TRACE_FUNC_DATA( 2, STRING, db);

	do_initialize();

	strcpy( idbname, db);

	EXEC SQL CONNECT :idbname;

	if( sqlca.sqlcode ) {
		rc = sqlca.sqlcode;
		EXEC SQL INQUIRE_SQL ( :errbuf = errortext);
		syslog( LOG_WARNING, "connect to database %s failed "
		"with error %s", idbname, errbuf);
	}

	TRACE_FUNC_EXIT( 2, "db_connect", INT, (char *)&rc);

	return(rc);
}
	
void db_disconnect()
{
	TRACE_FUNC_ENTRY( 2, "db_disconnect");

	EXEC SQL COMMIT;
	EXEC SQL DISCONNECT;

	TRACE_FUNC_EXIT( 2, "db_disconnect", NONE, 0);
}

/*
 * Function  : fetch_value( char *)
 * Arguments
 *      arg1 :  Value fetched
 * Return Value:
 * Comments  :
 *      Fetches the first value from the active query and closes the query.
 *      Value fetched should be an integer or float.
 * Author    : Marc Gathier
 * History
 *      Created         : Tue Oct  4 15:12:51 CET 1994
 *      Modifications   :
 *
 *
 */
fetch_value( char *value)
{
	int ret = -1; 
	short	*indicator;

	TRACE_FUNC_ENTRY( 2, "fetch_value");
	if( (dp = do_fetch() ) != (IISQLDA *) 0) {
		indicator = (short *) dp->sqlvar[0].sqlind;
		if ( dp->sqlvar[0].sqltype < 0 && *indicator < 0)
			syslog( LOG_WARNING, "NULL value encountered");
		else {
			get_field( dp, 0, value);
			ret = 0;
		}
	}


	TRACE_FUNC_EXIT( 2, "fetch_value", INT, (char *)&ret);
	return( ret);
}


/*
 * Function  : watchdog_command( char *)
 * Arguments
 *      arg1 :  Command to execute
 * Return Value:
 *      SQL errorcode if something went wrong, 0 if statement could be
 *      compiled
 * Comments  :
 *      Execute a SQL command and bind the first output column to a float
 *      variable.
 * Author    : Marc Gathier
 * History
 *      Created         : Tue Oct  4 15:18:24 CET 1994
 *      Modifications   :
 *
 *
 */
int watchdog_command( char *command)
{
	int rc = 0;

	TRACE_FUNC_ENTRY( 2, "watchdog_command");
	TRACE_FUNC_DATA( 2, STRING, command);

	strcpy( statm, command);
	EXEC SQL WHENEVER SQLERROR GOTO sql_err;

	EXEC SQL PREPARE S0 FROM :statm;

	EXEC SQL DESCRIBE S0 INTO sqldap;

	EXEC SQL DECLARE C0 CURSOR FOR S0;

	sqldap->sqln = sqldap->sqld;

	EXEC SQL OPEN C0;

	TRACE_FUNC_EXIT( 2, "watchdog_command", NONE, 0);
	return(0);
sql_err:

	rc = sqlca.sqlcode;
	EXEC SQL WHENEVER SQLERROR CONTINUE;

	EXEC SQL INQUIRE_SQL ( :errbuf = errortext);
	syslog( LOG_WARNING, "watchdog command %s failed with error %s", 
		command, errbuf);

	TRACE_FUNC_EXIT( 2, "watchdog_command", NONE, (char *)&(rc));
	return(rc);
}

static IISQLDA *do_fetch()
{
	TRACE_FUNC_ENTRY( 3, "do_fetch");
	EXEC SQL WHENEVER NOT FOUND GOTO not_found2;

	EXEC SQL FETCH C0 USING DESCRIPTOR sqldap;

	TRACE_FUNC_EXIT( 3, "do_fetch", NONE,0);
	return sqldap;

not_found2:

	TRACE_FUNC_EXIT( 3, "do_fetch", NONE,0);
	return (IISQLDA *)0;
}
	
static void get_field( IISQLDA *dp, int col, char *fbuf) 
{
	int first, last, field_len;
	int	length, j, k;

	TRACE_FUNC_ENTRY( 3, "get_field");


	switch( dp->sqlvar[col].sqltype )
	{
		case IISQ_INT_TYPE:
		case -IISQ_INT_TYPE:
			if( dp->sqlvar[col].sqllen == 4) 
				sprintf( fbuf, "%d", *(long *)dp->sqlvar[col].sqldata);
			else
				sprintf( fbuf, "%d", *(short *)dp->sqlvar[col].sqldata);
			break;
		case IISQ_FLT_TYPE:
		case -IISQ_FLT_TYPE:
			sprintf( fbuf, "%g", *(double *)dp->sqlvar[col].sqldata);
			break;
		case IISQ_VCH_TYPE:
		case -IISQ_VCH_TYPE:
			strncpy( fbuf, dp->sqlvar[col].sqldata +2, WD_RET_STRLEN/**(short *)dp->sqlvar[col].sqldata*/);
			fbuf[WD_RET_STRLEN-1/**(short *)dp->sqlvar[col].sqldata*/] = '\0';
			break;
		default:

			length = MIN( strlen( dp->sqlvar[col].sqldata), 32);
	
			/* skip leading spaces*/
			for (k = 0; dp->sqlvar[col].sqldata[k] == ' ' &&
	     				k < length; k ++)
	  		;
			for (j = k; j < length; j++)
	  			fbuf[j-k] = dp->sqlvar[col].sqldata[j];

			j = j-k;
       			while (j > 0 && fbuf[j-1] == ' ')
	  			j--;
	
			fbuf[j] = '\0';
	}

	TRACE_FUNC_EXIT( 3, "get_field", NONE,0);
}

void
wd_init()
{

	if( ! strcmp( dbdir, "") ) {
		if( getenv( "II_SYSTEM") == NULL) {
			syslog( LOG_WARNING, 
				"environment variable II_SYSTEM not set");
			exit(1);
		}
		strcpy( dbdir, getenv( "II_SYSTEM"));
	}else
		Setenv(  "II_SYSTEM", dbdir);

	EXEC SQL WHENEVER NOT FOUND continue;
	EXEC SQL set lockmode session where readlock=nolock;
}

