/* lsample.c                                                                 */
/*****************************************************************************/
/* LUA Sample Program                                                        */
/*                                                                           */
/* (C) Copyright 1990-1997 Data Connection Ltd.                              */
/*                                                                           */
/* This program is a crude 3270 emulator.  It uses the LUA API to access     */
/* both the SSCP and LU sessions.  Outbound data from the host is displayed  */
/* on the screen unformatted.  Both SSCP and LU data are shown.  If the      */
/* outbound data is an RQD request an automatic positive response is sent.   */
/* Inbound data can be entered at the keyboard and is sent on the current    */
/* session.  This current session can be toggled between the SSCP and LU     */
/* sessions by hitting the [ (left square bracket) key followed by <cr>      */
/* (carriage return), although it switches automatically on receipt of BIND  */
/* or UNBIND.                                                                */
/*                                                                           */
/* The program is invoked with a single parameter - the name of the LUA LU   */
/* to use.  This is converted to upper case and must match an LUA LU in the  */
/* configuration file.  This LU should be configured for a 327? on the host. */
/*                                                                           */
/* It works reasonably well with TSO provided that complex formatted logon   */
/* screens are not used.                                                     */
/*                                                                           */
/* To exit the program type ] (right square bracket) followed by <cr>        */
/* (carriage return).                                                        */
/*                                                                           */
/* The CSV CONVERT function is used to convert data between ASCII and        */
/* EBCDIC.  This uses the type G conversion table, so the environment        */
/* variable SNATBLG must be set to point to a suitable file.                 */
/*****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <sys/stropts.h>
#include <lua_c.h>
#include <acssvcc.h>

/*****************************************************************************/
/* Program constants                                                         */
/*****************************************************************************/
#define DATASIZE 4096                  /* Max size of RU                     */
#define SWITKEY  0x5B                  /* '['   to switch sessions           */
#define EXITKEY  0x5D                  /* ']'   to exit program              */

#define BETB     1                     /* Between brackets                   */
#define SEND     2                     /* In bracket and can send            */
#define RECV     3                     /* In bracket, but cannot send        */

/*****************************************************************************/
/* Global variables                                                          */
/*                                                                           */
/* Inbound data is sent as typed on the SSCP session.  On the LU session it  */
/* is prefixed with the <enter> key AID plus a two byte cursor address.      */
/*****************************************************************************/
LUA_VERB_RECORD read_verb;             /* RUI read verb                      */
LUA_VERB_RECORD rbid_verb;             /* RUI read verb                      */
LUA_VERB_RECORD other_verb;            /* RUI init, write or term verb       */
LUA_VERB_RECORD reinit_verb;           /* RUI init, write or term verb       */
LUA_VERB_RECORD bid_verb;              /* RUI bid verb                       */
LUA_VERB_RECORD purge_verb;            /* RUI purge verb                     */

unsigned char read_data[DATASIZE];     /* Outbound RU                        */
unsigned char write_array[DATASIZE + 3] = {0x7d, 0x40, 0x40};
                                       /* Inbound RU                         */
unsigned char *write_data = write_array + 3;
                                       /* Pointer to inbound RU              */

struct convert convert_to_asc;         /* Outbound convert verb              */
struct convert convert_to_ebc;         /* Inbound convert verb               */

int sna_fd = -1;                       /* SNA file descriptor                */
int semaphore;                         /* Semaphore for RUI verbs            */
int terminating = 0;                   /* are we already in closedown() ?    */
unsigned char lu_name[9] = "        "; /* LU name                            */
unsigned char pu_name[9] = "        "; /* PU name                            */
unsigned char lu_number  = 0;          /* LU number                          */

unsigned long sid;                     /* RUI session ID                     */
int lu_session = 0;                    /* LU session inbound flag            */
int send_state = BETB;                 /* LU session send state              */
int initial_bid_issued = 0;            /* Have we issued the first           */
                                       /* bid verb?                          */
int bid_outstanding = 0;               /* Is there a bid verb outstanding?   */

extern char * optarg;                  /* array of additional command line   */
extern int optind, opterr;             /* arguments                          */

int use_bid;                           /* use BID to check for data          */
int use_lu_name;                       /* use sessionID is default           */
int use_purge;                         /* use PURGE verb                     */
int saved_retry_reinit;                /* total # of times to retry REINIT   */
int retry_reinit = 0;                  /* remaining times to retry REINIT    */
int use_tracing = 0;                   /* use tracing                        */
struct define_trace csv_trace;

/*****************************************************************************/
/* Function declarations.                                                    */
/*****************************************************************************/
int  main();                           /* Entry point                        */
void trace_on();                       /* Turn tracing on                    */
void trace_off();                      /* Turn tracing off                   */
void csv_init();                       /* Set up CSV convert verbs           */
int  issue_verb();                     /* Issue INIT or TERM verb            */
void issue_read_or_bid();              /* Issue READ or BID verb             */
void issue_purges();                   /* Issue PURGE verbs                  */
void issue_bid();                      /* Issue READ or BID verb             */
void issue_read();                     /* Issue READ verb                    */
void issue_reinit();                   /* Issue REINIT verb                  */
void issue_rsp();                      /* Send a positive response           */
void reinit_done();                    /* REINIT completion callback         */
void other_done();                     /* INIT, TERM or WRITE callback       */
void read_done();                      /* READ completion callback           */
void rsp_done();                       /* Response completion callback       */
void closedown();                      /* Close down the RUI                 */
int  do_select();                      /* Wait for verb completion or input  */


/**PROC+**********************************************************************/
/* Name:      main                                                           */
/*                                                                           */
/* Purpose:   Entry point                                                    */
/*                                                                           */
/* Returns:   0 on successful execution otherwise an error code              */
/*                                                                           */
/* Params:    IN argc             -  argument count                          */
/*            IN argv             -  argument array                          */
/*                                                                           */
/* Operation: Parse the command line for initialization options.             */
/*            After initialization, loop reading input from the keyboard.    */
/*            When data is entered, a RUI_WRITE call is issued to send data  */
/*            to the host, and on completion, the loop is resumed.           */
/*                                                                           */
/* There are two special command strings which the user may enter, as        */
/* follows:                                                                  */
/*                                                                           */
/* If the user enters the [ symbol, the current inbound session is toggled   */
/* between the SSCP and LU sessions.                                         */
/*                                                                           */
/* If the user enters the ] symbol, the program exits.                       */
/*                                                                           */
/*****************************************************************************/
int main(argc, argv)
int argc;
char *argv[];
{
  int ok;
  int ii;
  unsigned short len;
  int chx;
  int args_left;

  /***************************************************************************/
  /* Parse the command line options.                                         */
  /***************************************************************************/
  while ((chx = getopt(argc,argv,"bnptur:")) != EOF)
  switch (chx)
  {
  case 't':
    use_tracing = 1;
    break;
  case 'b':
    use_bid = 1;
    break;
  case 'p':
    use_purge = 1;
    break;
  case 'n':
    use_lu_name = 1;
    break;
  case 'u':
    retry_reinit = -1;
    break;
  case 'r':
    retry_reinit = atoi(optarg);
    break;
  }

  /***************************************************************************/
  /* Keep track of the total number of times to retry the REINIT so that we  */
  /* can reset the value after a successful REINIT.                          */
  /***************************************************************************/
  saved_retry_reinit = retry_reinit;

  args_left = argc - optind;

  /***************************************************************************/
  /* Validate parameter usage and get LU name.                               */
  /***************************************************************************/
  if ((args_left != 2) && (args_left != 1))
  {
    printf("Usage: lsample <options> puname lunumber\n");
    printf("or:    lsample <options> luname\n");
    exit(1);
  }

  if (strlen(argv[optind]) > 8)
  {
    printf("LU/PU name too long\n");
    exit(1);
  }

  /***************************************************************************/
  /* Convert to upper case                                                   */
  /***************************************************************************/
  ii=0;
  while (argv[optind][ii])
  {
    if (argv[optind][ii] >= 'a' && argv[optind][ii] <= 'z')
    {
      argv[optind][ii] = argv[optind][ii] - 'a' + 'A';
    }
    ii++;
  }

  if (args_left == 1)
  {
    /*************************************************************************/
    /* Using an LU name to identify the LU                                   */
    /*************************************************************************/
    memcpy(lu_name, argv[optind], strlen(argv[optind]));
  }
  else
  {
    /*************************************************************************/
    /* Using a PU name and an LU number to identify the LU                   */
    /*************************************************************************/
    lu_name[0] = 0;
    memcpy(pu_name, argv[optind], strlen(argv[optind]));
    lu_number = atoi(argv[optind + 1]);
  }

  if (use_tracing)
  {
    trace_on();
  }

  /***************************************************************************/
  /* Set up CSV convert verbs                                                */
  /***************************************************************************/
  csv_init();

  /***************************************************************************/
  /* Indicate that we intend to use using application scheduling.            */
  /***************************************************************************/
  SNA_USE_FD_SCHED();

  /***************************************************************************/
  /* Initialize RUI and wait for ACTLU fom host.                             */
  /***************************************************************************/
  if (issue_verb((unsigned int) LUA_OPCODE_RUI_INIT))
  {
    printf("Init failed\n");
    exit(1);
  }

  /***************************************************************************/
  /* Issue the first RUI read.  After this, all read processing is performed */
  /* asynchronously - the callback routine at the end of one RUI_READ starts */
  /* the next one.                                                           */
  /***************************************************************************/
  issue_read_or_bid();

  /***************************************************************************/
  /* Loop reading inbound data from the keyboard and processing verb         */
  /* completions.                                                            */
  /***************************************************************************/
  while (terminating == 0)
  {
    memset(write_data, 0, DATASIZE);
    /*************************************************************************/
    /* do_select can return 0 or 1                                           */
    /*                                                                       */
    /* 0 - if there was an event on the SNA file descriptor due to a         */
    /* completing verb. In this case, the callback has already been made and */
    /* a new verb has been issued. We just go around the main loop again.    */
    /*                                                                       */
    /* 1 - if there was user input. In this case we need to read the input   */
    /* from stdin, issue a WRITE and wait for it to complete before doing    */
    /* anything else.                                                        */
    /*************************************************************************/
    if (do_select(1) == 1)
    {
      if (fgets((char *)write_data, DATASIZE, stdin))
      {
        /*********************************************************************/
        /* Handle the two special keys                                       */
        /*********************************************************************/
        if (write_data[0] == EXITKEY)
        {
          closedown();
        }
        else if (write_data[0] == SWITKEY)
        {
          if (lu_session)
          {
            printf("SSCP session\n");
            lu_session = 0;
          }
          else
          {
            printf("LU session\n");
            lu_session = 1;
          }
        }

        /*********************************************************************/
        /* Write inbound data.                                               */
        /*********************************************************************/
        else
        {
          ok  = 1;
          len = (strlen((char*)write_data) - 1);
          convert_to_ebc.len = len;
          if (convert_to_ebc.len > 0)
          {
            ACSSVC_C((char *)&convert_to_ebc);
          }
          memset(&other_verb, 0, sizeof(other_verb));
          other_verb.common.lua_verb             = LUA_VERB_RUI;
          other_verb.common.lua_verb_length      = sizeof(read_verb);
          other_verb.common.lua_opcode           = LUA_OPCODE_RUI_WRITE;
          other_verb.common.lua_correlator       = __LINE__;
          if (use_lu_name)
          {
            memcpy(other_verb.common.lua_luname, lu_name, 8);
          }
          else
          {
            other_verb.common.lua_sid = sid;
          }
          other_verb.common.lua_data_length      = len;
          other_verb.common.lua_data_ptr         = (char *) write_data;
          other_verb.common.lua_post_handle      = (unsigned long) other_done;
          other_verb.common.lua_rh.bci           = 1;
          other_verb.common.lua_rh.eci           = 1;
          other_verb.common.lua_rh.dr1i          = 1;

          if (lu_session)
          {
            /*****************************************************************/
            /* On the LU session we must add the <enter> key prefix.         */
            /* All inbound requests flow RQE with the BBI and CDI flags set  */
            /* depending on the current session state.                       */
            /*****************************************************************/
            other_verb.common.lua_data_ptr      -= 3;
            other_verb.common.lua_data_length   += 3;
            other_verb.common.lua_flag1.lu_norm  = 1;
            other_verb.common.lua_rh.ri          = 1;
            if (send_state == BETB)
            {
              /***************************************************************/
              /* Between bracket, so open bracket and give direction.        */
              /* Note that we can do this since we will always be contention */
              /* winner.                                                     */
              /***************************************************************/
              other_verb.common.lua_rh.bbi         = 1;
              other_verb.common.lua_rh.cdi         = 1;
              send_state = RECV;
            }
            else if (send_state = SEND)
            {
              /***************************************************************/
              /* In bracket and we have direction, so simply give direction. */
              /***************************************************************/
              other_verb.common.lua_rh.cdi         = 1;
              send_state = RECV;
            }
            else
            {
              /***************************************************************/
              /* In bracket and we do not have direction, so do not send.    */
              /***************************************************************/
              printf("Wait\n");
              ok = 0;
            }
          }
          else
          {
            /*****************************************************************/
            /* On the SSCP session things are straightforward.               */
            /*****************************************************************/
            other_verb.common.lua_flag1.sscp_norm  = 1;
          }

          if (ok)
          {
            /*****************************************************************/
            /* Issue write verb and wait for completion.                     */
            /*****************************************************************/
            semaphore = 1;
            RUI(&other_verb);
            if (other_verb.common.lua_flag2.async)
            {
              /***************************************************************/
              /* We must wait for the verb to complete before proceeding -   */
              /* so we can't take any more keyboard input yet.               */
              /***************************************************************/
              do_select(0);
            }
            else
            {
              /***************************************************************/
              /* The verb completed immediately, so make the callback.       */
              /***************************************************************/
              other_done(&other_verb);
            }
          }
        }
      }
    }
  }

  /***************************************************************************/
  /* Normal termination.                                                     */
  /***************************************************************************/
  exit(0);

} /* main */


/**PROC+**********************************************************************/
/* Name:      trace_on                                                       */
/*                                                                           */
/* Purpose:   Turn RUI tracing on                                            */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void trace_on()
{
  /***************************************************************************/
  /* Fill in the verb control block and issue the verb                       */
  /***************************************************************************/
  memset(&csv_trace,0,sizeof(csv_trace));
  csv_trace.opcode     = SV_DEFINE_TRACE;
  csv_trace.dt_set     = SV_ON;
  csv_trace.appc       = 0;
  csv_trace.comm_serv  = 0;
  csv_trace.ehllapi    = 0;
  csv_trace.rui        = 1<<7;
  csv_trace.reset_trc  = SV_NO;
  csv_trace.trunc      = 0;

  ACSSVC_C((char *)&csv_trace);

} /* trace_on */


/**PROC+**********************************************************************/
/* Name:      trace_off                                                      */
/*                                                                           */
/* Purpose:   Turn RUI tracing off                                           */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void trace_off()
{
  /***************************************************************************/
  /* Fill in the verb control block and issue the verb                       */
  /***************************************************************************/
  memset(&csv_trace,0,sizeof(csv_trace));
  csv_trace.opcode     = SV_DEFINE_TRACE;
  csv_trace.dt_set     = SV_OFF;
  csv_trace.appc       = SV_IGNORE;
  csv_trace.comm_serv  = SV_IGNORE;
  csv_trace.ehllapi    = SV_IGNORE;
  csv_trace.rui        = 0x80;
  csv_trace.reset_trc  = SV_NO;
  csv_trace.trunc      = 0;

  ACSSVC_C((char *)&csv_trace);

} /* trace_off */

/**PROC+**********************************************************************/
/* Name:      csv_init                                                       */
/*                                                                           */
/* Purpose:   Set up outbound and inbound CSV convert verbs                  */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: Set up all the static fields.                                  */
/*                                                                           */
/**PROC-**********************************************************************/
void csv_init()
{
  /***************************************************************************/
  /* The len field is set up when the verb is called                         */
  /***************************************************************************/
  convert_to_asc.opcode       = SV_CONVERT;
  convert_to_asc.direction    = SV_EBCDIC_TO_ASCII;
  convert_to_asc.char_set     = SV_G;
  convert_to_asc.source       = read_data;
  convert_to_asc.target       = read_data;

  convert_to_ebc.opcode       = SV_CONVERT;
  convert_to_ebc.direction    = SV_ASCII_TO_EBCDIC;
  convert_to_ebc.char_set     = SV_G;
  convert_to_ebc.source       = write_data;
  convert_to_ebc.target       = write_data;

} /* csv_init */


/**PROC+**********************************************************************/
/* Name:      issue_verb                                                     */
/*                                                                           */
/* Purpose:   Issues a RUI_INIT or RUI_TERM verb.                            */
/*                                                                           */
/* Returns:   0                   -      If the verb completed successfully  */
/*            1                   -      Otherwise                           */
/*                                                                           */
/* Params:    type                -      Verb LUA opcode                     */
/*                                                                           */
/* Operation: Issue the verb, then wait for it to complete.                  */
/*                                                                           */
/**PROC-**********************************************************************/
int issue_verb(type)
unsigned int  type;
{
  memset(&other_verb, 0, sizeof(other_verb));
  other_verb.common.lua_verb        = LUA_VERB_RUI;
  other_verb.common.lua_verb_length = sizeof(other_verb);
  other_verb.common.lua_opcode      = type;
  other_verb.common.lua_correlator  = __LINE__;

  if (type == LUA_OPCODE_RUI_INIT)
  {
    if (lu_name[0] == '\0')
    {
      memcpy(other_verb.specific.init.lua_puname, pu_name, 8);
      other_verb.specific.init.lua_lunumber = lu_number;
    }
    else
    {
      memcpy(other_verb.common.lua_luname, lu_name, 8);
    }
  }
  else
  {
    if (use_lu_name)
    {
      memcpy(other_verb.common.lua_luname, lu_name, 8);
    }
    else
    {
      other_verb.common.lua_sid = sid;
    }
  }
  other_verb.common.lua_post_handle = (unsigned long) other_done;
  other_verb.common.lua_flag2.async = 1;
  semaphore = 1;

  RUI(&other_verb);

  if (other_verb.common.lua_flag2.async)
  {
    do_select(0);
  }
  else
  {
    other_done(&other_verb);
  }

  return(other_verb.common.lua_prim_rc != LUA_OK);

} /* issue_verb */


/**PROC+**********************************************************************/
/* Name:      issue_read_or_bid                                              */
/*                                                                           */
/* Purpose:   Issue a RUI_READ or RUI_BID verb                               */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void issue_read_or_bid()
{
  if (use_bid)
  {
    issue_bid();
  }
  else
  {
    issue_read();
  }

} /* issue_read_or_bid */


/**PROC+**********************************************************************/
/* Name:      issue_purges                                                   */
/*                                                                           */
/* Purpose:   Issue a RUI_PURGE verb                                         */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void issue_purges()
{
  int ii;
  for (ii = 0; ii < 2; ii++)
  {
    memset(&purge_verb, 0, sizeof(purge_verb));
    purge_verb.common.lua_verb             = LUA_VERB_RUI;
    purge_verb.common.lua_verb_length      = sizeof(purge_verb);
    purge_verb.common.lua_opcode           = LUA_OPCODE_RUI_PURGE;
    purge_verb.common.lua_correlator       = __LINE__;

    if (use_lu_name)
    {
      memcpy(purge_verb.common.lua_luname, lu_name, 8);
    }
    else
    {
      purge_verb.common.lua_sid = sid;
    }

    /*************************************************************************/
    /* First time around, purge the read verb record.                        */
    /* Second time around, purge the bid verb record.                        */
    /*************************************************************************/
    if (ii == 0)
    {
      purge_verb.common.lua_data_ptr = (char*)&read_verb;
    }
    else
    {
      purge_verb.common.lua_data_ptr = (char*)&rbid_verb;
    }

    purge_verb.common.lua_post_handle      = (unsigned long) other_done;

    RUI(&purge_verb);

    /*************************************************************************/
    /* If the verb completed immediately, make the callback.                 */
    /*************************************************************************/
    if (purge_verb.common.lua_flag2.async == 0)
    {
      other_done(&purge_verb);
    }
  }
} /* issue_purges */


/**PROC+**********************************************************************/
/* Name:      issue_bid                                                      */
/*                                                                           */
/* Purpose:   Issue a RUI_BID verb                                           */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: If we haven't issued the first BID, do so.                     */
/*            Otherwise, issue a READ with bid enabled, unless there is a    */
/*            bid verb currently outstanding.                                */
/*                                                                           */
/**PROC-**********************************************************************/
void issue_bid()
{

  if (!initial_bid_issued)
  {
    memset(&bid_verb, 0, sizeof(bid_verb));
    bid_verb.common.lua_verb             = LUA_VERB_RUI;
    bid_verb.common.lua_verb_length      = sizeof(bid_verb);
    bid_verb.common.lua_opcode           = LUA_OPCODE_RUI_BID;
    bid_verb.common.lua_correlator       = __LINE__;
    if (use_lu_name)
    {
      memcpy(bid_verb.common.lua_luname, lu_name, 8);
    }
    else
    {
      bid_verb.common.lua_sid = sid;
    }
    bid_verb.common.lua_post_handle      = (unsigned long) other_done;

    bid_outstanding = 1;

    RUI(&bid_verb);

    /*************************************************************************/
    /* If the verb completed immediately, make the callback                  */
    /*************************************************************************/
    if (bid_verb.common.lua_flag2.async == 0)
    {
      other_done(&bid_verb);
    }
    initial_bid_issued = 1;
  }
  else if (!bid_outstanding)
  {
    /*************************************************************************/
    /* Already issued the first bid, and there are no bids outstanding so    */
    /* issue a read with bid enabled.                                        */
    /*************************************************************************/
    memset(&rbid_verb, 0, sizeof(rbid_verb));
    memset(read_data, 0, DATASIZE);
    rbid_verb.common.lua_verb             = LUA_VERB_RUI;
    rbid_verb.common.lua_verb_length      = sizeof(rbid_verb);
    rbid_verb.common.lua_opcode           = LUA_OPCODE_RUI_READ;
    rbid_verb.common.lua_correlator       = __LINE__;
    if (use_lu_name)
    {
      memcpy(rbid_verb.common.lua_luname, lu_name, 8);
    }
    else
    {
      rbid_verb.common.lua_sid = sid;
    }
    rbid_verb.common.lua_max_length       = DATASIZE;
    rbid_verb.common.lua_data_ptr         = (char *) read_data;
    rbid_verb.common.lua_post_handle      = (unsigned long) other_done;
    rbid_verb.common.lua_flag1.lu_norm    = 1;
    rbid_verb.common.lua_flag1.lu_exp     = 1;
    rbid_verb.common.lua_flag1.sscp_norm  = 1;
    rbid_verb.common.lua_flag1.sscp_exp   = 1;
    rbid_verb.common.lua_flag1.bid_enable = 1;
    rbid_verb.common.lua_flag1.nowait     = 1;

    bid_outstanding = 1;

    RUI(&rbid_verb);

    /*************************************************************************/
    /* If the verb completed immediately, make the callback                  */
    /*************************************************************************/
    if (rbid_verb.common.lua_flag2.async == 0)
    {
      other_done(&rbid_verb);
    }
  }
} /* issue_bid */


/**PROC+**********************************************************************/
/* Name:      issue_read                                                     */
/*                                                                           */
/* Purpose:   Issue a RUI_READ verb                                          */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void issue_read()
{
  memset(&read_verb, 0, sizeof(read_verb));
  memset(read_data, 0, DATASIZE);
  read_verb.common.lua_verb             = LUA_VERB_RUI;
  read_verb.common.lua_verb_length      = sizeof(read_verb);
  read_verb.common.lua_opcode           = LUA_OPCODE_RUI_READ;
  read_verb.common.lua_correlator       = __LINE__;
  if (use_lu_name)
  {
    memcpy(read_verb.common.lua_luname, lu_name, 8);
  }
  else
  {
    read_verb.common.lua_sid = sid;
  }
  read_verb.common.lua_max_length       = DATASIZE;
  read_verb.common.lua_data_ptr         = (char *) read_data;
  read_verb.common.lua_post_handle      = (unsigned long) read_done;
  read_verb.common.lua_flag1.lu_norm    = 1;
  read_verb.common.lua_flag1.lu_exp     = 1;
  read_verb.common.lua_flag1.sscp_norm  = 1;
  read_verb.common.lua_flag1.sscp_exp   = 1;

  RUI(&read_verb);

  /***************************************************************************/
  /* If the verb completed immediately, make the callback                    */
  /***************************************************************************/
  if (read_verb.common.lua_flag2.async == 0)
  {
    read_done(&read_verb);
  }
} /* issue_read */


/**PROC+**********************************************************************/
/* Name:      issue_reinit                                                   */
/*                                                                           */
/* Purpose:   Issue a RUI_REINIT verb                                        */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void issue_reinit()
{
  /***************************************************************************/
  /* Set lu_session back to zero, since the PLU-SLU session has now gone.    */
  /* Also if we are testing BIDs then we will need to issue a RUI_BID,       */
  /* rather than a RUI_READ with bid_enabled.                                */
  /***************************************************************************/
  lu_session = 0;
  initial_bid_issued = 0;

  memset(&read_verb, 0, sizeof(read_verb));
  memset(read_data, 0, DATASIZE);
  reinit_verb.common.lua_verb             = LUA_VERB_RUI;
  reinit_verb.common.lua_verb_length      = sizeof(reinit_verb);
  reinit_verb.common.lua_opcode           = LUA_OPCODE_RUI_REINIT;
  reinit_verb.common.lua_correlator       = __LINE__;
  if (use_lu_name)
  {
    memcpy(reinit_verb.common.lua_luname, lu_name, 8);
  }
  else
  {
    reinit_verb.common.lua_sid = sid;
  }
  reinit_verb.common.lua_post_handle      = (unsigned long) reinit_done;

  /***************************************************************************/
  /* Clear the flag which indicates we have a bid outstanding, since this is */
  /* going to reset everything.                                              */
  /***************************************************************************/
  bid_outstanding = 0;

  RUI(&reinit_verb);

  /***************************************************************************/
  /* If the verb completed immediately, make the callback                    */
  /***************************************************************************/
  if (reinit_verb.common.lua_flag2.async == 0)
  {
    reinit_done(&reinit_verb);
  }
} /* issue_reinit */


/**PROC+**********************************************************************/
/* Name:      issue_rsp                                                      */
/*                                                                           */
/* Purpose:   Respond to requests from the host.                             */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN  sense           -  sense code                              */
/*                verb            -  VCB on which to issue the response      */
/*                                                                           */
/* Operation: Re-use the record on whcih the request was read. This record   */
/* contains much of the necessar information.  This means that further reads */
/* are delayed until the write completes.                                    */
/*                                                                           */
/*****************************************************************************/
void issue_rsp(sense,verb)
unsigned long * sense;
LUA_VERB_RECORD *verb;

{
  verb->common.lua_opcode           = LUA_OPCODE_RUI_WRITE;
  verb->common.lua_correlator       = __LINE__;
  verb->common.lua_max_length       = 0;
  verb->common.lua_post_handle      = (unsigned long) rsp_done;
  verb->common.lua_rh.rri           = 1;
  verb->common.lua_flag1.lu_norm    = 0;
  verb->common.lua_flag1.lu_exp     = 0;
  verb->common.lua_flag1.sscp_norm  = 0;
  verb->common.lua_flag1.sscp_exp   = 0;
  verb->common.lua_flag1.bid_enable = 0;
  verb->common.lua_flag1.nowait     = 0;
  verb->common.lua_flag2.bid_enable = 0;

  if (*sense != 0l)
  {
    /*************************************************************************/
    /* negative response                                                     */
    /*************************************************************************/
    verb->common.lua_data_length      = 4;
    memcpy(read_data, sense, 4);
    verb->common.lua_rh.ri            = 1;
  }
  else
  {
    /*************************************************************************/
    /* positive response                                                     */
    /*************************************************************************/
    verb->common.lua_data_length      = 0;
    verb->common.lua_rh.ri            = 0;
  }
  /***************************************************************************/
  /* Send the response back on the same flow from which we got the request   */
  /***************************************************************************/
  if (verb->common.lua_flag2.lu_norm)
  {
    verb->common.lua_flag1.lu_norm = 1;
  }
  else if (verb->common.lua_flag2.lu_exp)
  {
    verb->common.lua_flag1.lu_exp = 1;
  }
  else if (verb->common.lua_flag2.sscp_norm)
  {
    verb->common.lua_flag1.sscp_norm = 1;
  }
  else if (verb->common.lua_flag2.sscp_exp)
  {
    verb->common.lua_flag1.sscp_exp = 1;
  }

  RUI(verb);

  /***************************************************************************/
  /* If the verb completed immediately, make the callback                   */
  /***************************************************************************/
  if (verb->common.lua_flag2.async == 0)
  {
    rsp_done(&verb);
  }
} /* issue_rsp */

/**PROC+**********************************************************************/
/* Name:      reinit_done                                                    */
/*                                                                           */
/* Purpose:   REINIT callback                                                */
/*                                                                           */
/* Returns:   none                                                           */
/*                                                                           */
/* Params:    IN verb             -   Verb record                            */
/*                                                                           */
/* Operation: If the reinit failed, and we can retry, then do so, otherwise  */
/* exit.  If the reinit succeeded, issue a read or bid.                      */
/*                                                                           */
/*****************************************************************************/
void reinit_done(verb)
LUA_VERB_RECORD *verb;
{
  if (verb->common.lua_prim_rc == LUA_OK)
  {
    retry_reinit = saved_retry_reinit;
    printf("LU RE active\n");
    issue_read_or_bid();
  }
  else
  {
    printf("REINIT failed, (%4.4x, %8.8x)\n",
           verb->common.lua_prim_rc, verb->common.lua_sec_rc);

    if (retry_reinit != 0)
    {
      if (retry_reinit > 0) retry_reinit--;
      sleep(1);
      issue_reinit();
    }
    else
    {
      exit(1);
    }
  }
} /* reinit_done */


/**PROC+**********************************************************************/
/* Name:      other_done                                                     */
/*                                                                           */
/* Purpose:   INIT, WRITE or TERM callback                                   */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN   verb           -   Verb record                            */
/*                                                                           */
/* Operation: Print status, if appropriate, and clear the semaphore to allow */
/*            the main program to continue.                                  */
/*                                                                           */
/**PROC-**********************************************************************/
void other_done(verb)
LUA_VERB_RECORD *verb;
{
  switch (verb->common.lua_opcode)
  {
    case LUA_OPCODE_RUI_INIT:
         if (verb->common.lua_prim_rc == LUA_OK)
         {
           printf("LU active\n");
           sid = other_verb.common.lua_sid;
           /******************************************************************/
           /* preserve actual LU name                                        */
           /******************************************************************/
           memcpy(lu_name, other_verb.common.lua_luname, 8);
         }
         else
         {
           printf("INIT failed, (%4.4x, %8.8x)\n",
                  verb->common.lua_prim_rc, verb->common.lua_sec_rc);
         }
         break;

    case LUA_OPCODE_RUI_PURGE:
         if (verb->common.lua_prim_rc != LUA_OK)
         {
           printf("PURGE failed, (%4.4x, %8.8x)\n",
                  verb->common.lua_prim_rc, verb->common.lua_sec_rc);
         }
         else
         {
           printf("PURGE succeeded, (%4.4x, %8.8x)\n",
                  verb->common.lua_prim_rc, verb->common.lua_sec_rc);
         }
         break;

    case LUA_OPCODE_RUI_WRITE:
         if (verb->common.lua_prim_rc != LUA_OK)
         {
           printf("WRITE failed, (%4.4x, %8.8x)\n",
                  verb->common.lua_prim_rc, verb->common.lua_sec_rc);
         }
         break;

    case LUA_OPCODE_RUI_BID:
         if (verb->common.lua_prim_rc == LUA_OK)
         {
           /******************************************************************/
           /* Clear the flag that indicates we have a bid outstanding        */
           /******************************************************************/
           bid_outstanding = 0;
           issue_read();
         }
         else
         {
           printf("BID failed, (%4.4x, %8.8x)\n",
                  verb->common.lua_prim_rc, verb->common.lua_sec_rc);
           exit(0);
         }
         break;

    case LUA_OPCODE_RUI_READ:
         /********************************************************************/
         /* We only come through here for RUI_READ verbs with the            */
         /* bid_enabled flag set.  If it returns OK then there is data on the*/
         /* VCB, (data which we must process); if it does not return OK then */
         /* that is the same as having re-issued the BID, so we will do      */
         /* nothing.                                                         */
         /********************************************************************/
         if (verb->common.lua_prim_rc == LUA_OK)
         {
           bid_outstanding = 0;
           read_done(verb);
         }
         break;

    case LUA_OPCODE_RUI_TERM:
         printf("Terminated\n");
         break;

  }

  /***************************************************************************/
  /* Allow the main program to continue                                      */
  /***************************************************************************/
  semaphore = 0;

} /* other_done */


/**PROC+**********************************************************************/
/* Name:      read_done                                                      */
/*                                                                           */
/* Purpose:   READ callback                                                  */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN   verb           -   Verb record                            */
/*                                                                           */
/* Operation: Display simple outbound SSCP or LU data.                       */
/*            Respond to RQD requests.                                       */
/*            Check validity of BIND requests.                               */
/*            Discard responses.                                             */
/*            Update session state dependent upon the BBI, EBI and CDI       */
/*            flags.                                                         */
/*            Issue a new READ, or a response, as appropriate.               */
/*                                                                           */
/**PROC-**********************************************************************/
void read_done(verb)
LUA_VERB_RECORD *verb;
{
  int count;
  unsigned char *data;
  int action = 0;
  unsigned long sense = 0l;
  int reinit_issued = 0;

  if (verb->common.lua_prim_rc == LUA_OK)
  {
    if ((verb->common.lua_message_type == LUA_MESSAGE_TYPE_LU_DATA)   ||
        (verb->common.lua_message_type == LUA_MESSAGE_TYPE_SSCP_DATA))
    {
      if (verb->common.lua_data_length > 0)
      {
        /*******************************************************************/
        /* Display outbound SSCP or LU data.                               */
        /*******************************************************************/
        convert_to_asc.len = (unsigned short) verb->common.lua_data_length;
        ACSSVC_C((char *)&convert_to_asc);
        data  = read_data;
        count = strlen((char*)data);
        while (count > 0)
        {
          if (count <= 80)
          {
            printf("%s\n",data);
            count = 0;
          }
          else
          {
            printf("%80.80s\n",data);
            count -= 80;
            data  += 80;
          }
        }
      }

      /*********************************************************************/
      /* Set session state.                                                */
      /*********************************************************************/
      if (verb->common.lua_rh.ebi)
      {
        printf("EB\n");
        send_state = BETB;
      }
      else if (verb->common.lua_rh.cdi)
      {
        printf("CD\n");
        send_state = SEND;
      }
      else
      {
        send_state = RECV;
      }
    }

    /*******************************************************************/
    /* Handle BIND or UNBIND as a special case.                        */
    /*******************************************************************/
    else if (verb->common.lua_message_type == LUA_MESSAGE_TYPE_BIND)
    {
      if (((read_data[6] & 0x20) == 0x20)  &&  /* Brackets used and BETB */
          ((read_data[7] & 0xF0) == 0x80)  &&  /* HDX-FF, sec con winner */
          ( read_data[14]        == 0x02))     /* LU type 2              */
      {
        printf("BIND\n");
        send_state = BETB;
        lu_session = 1;
      }
      else
      {
        printf("BIND rejected\n");
        sense = LUA_INVALID_SESSION_PARAMETERS;
      }
    }
    else if (verb->common.lua_message_type == LUA_MESSAGE_TYPE_UNBIND)
    {
      printf("UNBIND\n");
      send_state = BETB;
      lu_session = 0;
    }

    /*******************************************************************/
    /* Respond to any RQD request.                                     */
    /*******************************************************************/
    if ((verb->common.lua_message_type != LUA_MESSAGE_TYPE_RSP)  &&
        (verb->common.lua_rh.ri == 0))            /* definite response */
    {
      action = 1;
    }

    if (verb->common.lua_flag2.bid_enable == 1)
    {
      bid_outstanding = 1;
    }
  }
  else
  {
    /*************************************************************************/
    /* Primary rc not OK, ie the READ failed. Stop now. If we are already in */
    /* the process of terminating, then no further action is required.       */
    /*************************************************************************/
    if (terminating == 0)
    {
      printf("READ ERROR, primary rc = %4.4x, secondary rc = %8.8x\n",
             verb->common.lua_prim_rc, verb->common.lua_sec_rc);
      if (verb->common.lua_prim_rc != LUA_SESSION_FAILURE)
      {
        closedown();
      }
      else
      {
        issue_reinit();
        reinit_issued = 1;
      }
    }
  }

  /*************************************************************************/
  /* Continue processing with either another read or a write positive rsp. */
  /* (The callback from issuing response will then issue another read)     */
  /* If we've already gone into closedown, stop here.                       */
  /*************************************************************************/
  if (terminating == 0)
  {
    if (action)
    {
      issue_rsp(&sense,verb);
    }
    else
    {
      if (reinit_issued != 1)
      {
        issue_read_or_bid();
      }
    }
  }
} /* read_done */


/**PROC+**********************************************************************/
/* Name:      rsp_done                                                       */
/*                                                                           */
/* Purpose:   Write response callback                                        */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN   verb           -   Verb record                            */
/*                                                                           */
/* Operation: Report status to the screen.  If there is no error, then issue */
/* another READ.                                                             */
/*                                                                           */
/*****************************************************************************/
void rsp_done(verb)
LUA_VERB_RECORD *verb;
{
  if (verb->common.lua_prim_rc != LUA_OK)
  {
    /*************************************************************************/
    /* There was a write error. If the session failed, just close down, else */
    /* issue a REINIT                                                        */
    /*************************************************************************/
    printf("WRITE for response failed, (%4.4x, %8.8x)\n",
            verb->common.lua_prim_rc, verb->common.lua_sec_rc);
    if (verb->common.lua_prim_rc != LUA_SESSION_FAILURE)
    {
      closedown();
    }
    else
    {
      issue_reinit();
    }
  }
  else
  {
    /*************************************************************************/
    /* OK, so issue another READ to continue the loop                        */
    /*************************************************************************/
    issue_read_or_bid();
  }
} /* rsp_done */


/**PROC+**********************************************************************/
/* Name:       closedown                                                     */
/*                                                                           */
/* Purpose:    Close down the RUI                                            */
/*                                                                           */
/* Returns:    None                                                          */
/*                                                                           */
/* Params:     None                                                          */
/*                                                                           */
/**PROC-**********************************************************************/
void closedown()
{
  /***************************************************************************/
  /* if we are using PURGE, do so now                                        */
  /***************************************************************************/
  if (use_purge)
  {
    issue_purges();
  }

  /***************************************************************************/
  /* Protect against entering this arm more than once                        */
  /***************************************************************************/
  if (terminating == 0)
  {
    terminating = 1;
    printf("Closedown\n");
    if (issue_verb((unsigned int) LUA_OPCODE_RUI_TERM))
    {
      exit(1);
    }
    exit(0);
  }
} /* closedown */


/**PROC+**********************************************************************/
/* Name:      do_select                                                      */
/*                                                                           */
/* Purpose:   Loop, checking file descriptors for events                     */
/*                                                                           */
/* Returns:   1 - if an event has occurred on stdin (user has hit enter)     */
/*            0 - otherwise                                                  */
/*                                                                           */
/* Params:    IN check_stdin      - 1 if we are interested in stdin          */
/*                                  0 otherwise                              */
/*                                                                           */
/* Operation: A select() loop.                                               */
/*                                                                           */
/**PROC-**********************************************************************/
int do_select(check_stdin)
int check_stdin;
{
  /***************************************************************************/
  /* Local variables                                                         */
  /***************************************************************************/
  int           rc;
  fd_set        read_table;
  fd_set        write_table;
  fd_set        except_table;
  int           nfds;

  /***************************************************************************/
  /* The two values semaphore and check_stdin are never both set at the same */
  /* time.                                                                   */
  /*                                                                         */
  /* Either we are blocking, waiting for a write/init/term to complete.      */
  /* (semaphore = 1, check_stdin = 0)                                        */
  /*                                                                         */
  /* Or we have an outstanding read/bid, in which case check_stdin is set to */
  /* indicate that we are also taking user input.                            */
  /* (semaphore = 0, check_stdin = 1)                                        */
  /*                                                                         */
  /***************************************************************************/
  while(semaphore || check_stdin)
  {
    /*************************************************************************/
    /* Get the SNA file descriptor (must do this every time through).        */
    /*                                                                       */
    /* Note that a value of -1 implies that we have lost our communication   */
    /* path to the core SNA software. In this sample application we treat    */
    /* this as a fatal error. A real application may need to handle this (eg */
    /* if the application can start before SNA itself is started).           */
    /*************************************************************************/
    sna_fd = SNA_GET_FD();
    if (sna_fd == -1)
    {
      printf("Internal error - failed to get file descriptor.\nExiting\n");
      exit(-1);
    }

    /*************************************************************************/
    /* Prepare to issue a select() call                                      */
    /*************************************************************************/
    FD_ZERO(&read_table);
    FD_ZERO(&write_table);
    FD_ZERO(&except_table);

    FD_SET(sna_fd, &except_table);
    FD_SET(sna_fd, &read_table);

    /*************************************************************************/
    /* If we need to check stdin (file descriptor ZERO) then set it in the   */
    /* read and exception tables                                             */
    /*************************************************************************/
    if (check_stdin == 1)
    {
      FD_SET(0, &except_table);
      FD_SET(0, &read_table);
    }

    /*************************************************************************/
    /* Go ahead and issue the select.                                        */
    /*************************************************************************/
    rc = select(FD_SETSIZE,
                &read_table,
                &write_table,
                &except_table,
                NULL);

    /*************************************************************************/
    /* select() returns the number of file descriptors on which an event has */
    /* occurred, or zero if the timer has expired.  Since we have an         */
    /* infinite timeout, we expect to get 1 or 2 here.                       */
    /*************************************************************************/
    if ((rc == 1) || (rc == 2))
    {
      if (FD_ISSET(sna_fd, &except_table) || FD_ISSET(sna_fd, &read_table))
      {
        /*********************************************************************/
        /* We are using application scheduling. An event has occurred on the */
        /* SNA file descriptor, so we must make the required call to the     */
        /* event handler.                                                    */
        /*********************************************************************/
        SNA_EVENT_FD();
      }

      if (check_stdin && 
          (FD_ISSET(0, &except_table) || FD_ISSET(0, &read_table)))
      {
        /*********************************************************************/
        /* The user has just hit <enter> so return 1 to indicate that there  */
        /* is work waiting on stdin.                                         */
        /*********************************************************************/
        return (1);
      }
    }
    else
    {
      /***********************************************************************/
      /* We cannot hope to survive such a failure, so simply exit.           */
      /***********************************************************************/
      printf("Internal error - select call failed, rc = %d.\nExiting\n", rc);
      exit (-1);
    }
  }
  return (0);
} /* do_select */
