/********************************************************************/
/*                                                                  */
/* Licensed Materials - Property of IBM                             */
/*                                                                  */
/* US Government Users Restricted Rights - Use, duplication or      */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.*/
/*                                                                  */
/* OCO Source Materials                                             */
/*                                                                  */
/* IBM IMS TCP/IP OTMA Connection, Version 2, Release 1             */
/* (C) Copyright IBM Corp. 1997, 1998.  All Rights Reserved.        */
/*                                                                  */
/* The source code for this program is not published or otherwise   */
/* divested of its trade secrets, irrespective of what has been     */
/* deposited with the U.S. Copyright Office.                        */
/*                                                                  */
/********************************************************************/

#ifndef HWSCNVTR_HPP
#define HWSCNVTR_HPP

#include "hwserror.hpp"
#include "hwscnvtr.h"

/*--------------------------------------------------------------------*/
/* The HWSConverter class provides methods for converting user and    */
/* control data between host and workstation formats.                 */
/* It also performs endian coversion between chip architectures.      */
/*--------------------------------------------------------------------*/
class DllExport HWSConverter {

   /*-----------------------------------------------------------------*/
   /* Public methods and attributes.                                  */
   /*-----------------------------------------------------------------*/
   public:

      HWSConverter( char *HWSPath );   // Constructor.
      ~HWSConverter();                 // Destructor.

      /*--------------------------------------------------------------*/
      /* The open method loads the conversion tables that are used    */
      /* by this instance of the HWSConverter object and performs     */
      /* initialization for the HWSConverter class.  It sets the      */
      /* language used to convert user data.  U.S. English is always  */
      /* used for control data.                                       */
      /*--------------------------------------------------------------*/
#ifdef HWS_IMSWEB
      int                              // Error code.
      open( HWSCnvsn  language,        // language identifies set of
                                       // conversion tables to load
                                       // into memory.
            int       clientEndian,    // Endian of machine running
                                       // client (either IMSWeb Runtime
                                       // or application using IMSClient).
            HWSError  *errObj );       // Error object.
#else
      int                              // Error code.
      open( HWSCnvsn  language,        // language identifies set of
                                       // conversion tables to load
                                       // into memory.
            int       clientEndian );  // Endian of machine running
                                       // client (either IMSWeb Runtime
                                       // or application using IMSClient).
#endif

      /*--------------------------------------------------------------*/
      /* The close method unloads (frees storage allocated for) any   */
      /* conversion tables that are loaded in the memory and performs */
      /* cleanup for the HWSConverter class.                          */
      /*--------------------------------------------------------------*/
#ifdef HWS_IMSWEB
      int                              // Error code.
      close( HWSError  *errObj );      // Error object.
#else
      int                              // Error code.
      close();                         // Error object.
#endif

      /*--------------------------------------------------------------*/
      /* The convertUserData method converts transaction data in any  */
      /* supported language between source and target encodings.      */
      /*--------------------------------------------------------------*/
#ifdef HWS_IMSWEB
      int                                  // Error code.
      convertUserData (

               HWSDirection   convDirect,  // HWS_CIENT2IMS or HWS_IMS2CLIENT

               HWSBool        isPureDBCS,  // HWSTrue or HWSFalse.

               char           *src,        // String to be converted.

               unsigned long  srcLength,   // # chars to be converted.

               char           *tar,        // Target area for converted
                                           // string.
               unsigned long  tarLength,   // # chars allotted for
                                           // converted string.
               unsigned long  *cnvtdLength,// Converted length.
               HWSError       *errObj );   // Error information
                                           // returned by method.
#else
      int                                  // Error code.
      convertUserData (

               HWSDirection   convDirect,  // HWS_CIENT2IMS or HWS_IMS2CLIENT

               HWSBool        isPureDBCS,  // HWSTrue or HWSFalse.

               char           *src,        // String to be converted.

               unsigned long  srcLength,   // # chars to be converted.

               char           *tar,        // Target area for converted
                                           // string.
               unsigned long  tarLength,   // # chars allotted for
                                           // converted string.
               unsigned long  *cnvtdLength ); // Converted length.
#endif

      /*--------------------------------------------------------------*/
      /* The convertControlData method converts control data between  */
      /* source and target encodings.  Control data is always in      */
      /* U.S. English.  An example of control data is the product     */
      /* identifier in the message prefix.                            */
      /*--------------------------------------------------------------*/
#ifdef HWS_IMSWEB
      int                                  // Error code.
      convertControlData(

               HWSDirection   convDirect,  // HWS_CIENT2IMS or HWS_IMS2CLIENT

               char           *src,        // String to be converted.

               unsigned long  srcLength,   // # chars to be converted.

               char           *tar,        // Target area for converted
                                           // string.
               unsigned long  tarLength,   // # chars allotted for
                                           // converted string.
               unsigned long  *cnvtdLength,// Converted length.
               HWSError       *errObj );   // Error information
                                           // returned by method.
#else
      int                                  // Error code.
      convertControlData(

               HWSDirection   convDirect,  // HWS_CIENT2IMS or HWS_IMS2CLIENT

               char           *src,        // String to be converted.

               unsigned long  srcLength,   // # chars to be converted.

               char           *tar,        // Target area for converted
                                           // string.
               unsigned long  tarLength,   // # chars allotted for
                                           // converted string.
               unsigned long  *cnvtdLength ); // Converted length.
#endif

      /*--------------------------------------------------------------*/
      /* The convertES method converts the source unsigned short from */
      /* source endian format to target endian format.  The converted */
      /* unsigned short is placed in the target buffer.               */
      /*--------------------------------------------------------------*/
#ifdef HWS_IMSWEB
      int                                  // Error code.
      convertES( unsigned short srcShort,  // Unsigned short to be
                                           // converted.
                 unsigned short *tarShort, // Target for converted
                                           // unsigned short.
                 HWSError       *errObj ); // Error information
                                           // returned by method.
#else
      int                                  // Error code.
      convertES( unsigned short srcShort,  // Unsigned short to be
                                           // converted.
                 unsigned short *tarShort ); // Target for converted
                                             // unsigned short.
#endif

      /*--------------------------------------------------------------*/
      /* Note: Values for srcEndian and tarEndian for both converES   */
      /*       and convertDL are:                                     */
      /*                                                              */
      /*       HWS_INTEL_ENDIAN                                       */
      /*       HWS_RISC_ENDIAN                                        */
      /*       HWS_SPARC_ENDIAN                                       */
      /*       HWS_S370_ENDIAN                                        */
      /*--------------------------------------------------------------*/

      /*--------------------------------------------------------------*/
      /* The convertEL method converts the source unsigned long from  */
      /* source endian format to target endian format.  The converted */
      /* unsigned long is placed in the target buffer.                */
      /*--------------------------------------------------------------*/
#ifdef HWS_IMSWEB
      int                                  // Error code.
      convertEL( unsigned long  srcShort,  // Unsigned long to be
                                           // converted.
                 unsigned long  *tarShort, // Target for converted
                                           // unsigned long.
                 HWSError       *errObj );
#else
      int                                  // Error code.
      convertEL( unsigned long  srcShort,  // Unsigned long to be
                                           // converted.
                 unsigned long  *tarShort ); // Target for converted
                                             // unsigned long.
#endif

   /*-----------------------------------------------------------------*/
   /* Private methods and attributes.                                 */
   /*-----------------------------------------------------------------*/
   private:

     // The convert method converts the source data into the target buffer
     // based on the source and target CCSIDs input to the method.
     // The actual data length after conversion is returned.
     unsigned long                        // Returns converted length.
     convert( HWSError       *errStruc,   // Error structure.
              int            srcCCSID,    // Source CCSID.
              int            tarCCSID,    // Target CCSID.
              char           *src,        // String to be converted.
              unsigned long  srcLength,   // # chars to be converted.
              char           *tar,        // Target area for converted
                                          // string.
              unsigned long  tarLength ); // # chars allotted for
                                          // converted string.

     long  PureSBCS( HWSError      *errStruc
                   , char          **ppSource
                   , unsigned long *pSourceLen
                   , char          **ppTarget
                   , unsigned long *pTargetLen
                   , long          index );

     long  PureDBCS( HWSError      *errStruc
                   , char          **ppSource
                   , unsigned long *pSourceLen
                   , char          **ppTarget
                   , unsigned long *pTargetLen
                   , long          index);

     long  HostToPCMixed( HWSError *errStruc
                   , char          **ppSource
                   , unsigned long *pSourceLen
                   , char          **ppTarget
                   , unsigned long *pTargetLen
                   , long          index );

     long  PCToHostMixed( HWSError *errStruc
                   , char          **ppSource
                   , unsigned long *pSourceLen
                   , char          **ppTarget
                   , unsigned long *pTargetLen
                   , char          *pHead
                   , long          index );

     int XltSBCSMacro(
                     char          **ppSource
                   , char          **ppTarget
                   , unsigned long *pXLTSourceLen
                   , unsigned long *pXLTTargetLen
                   , long          index );

     int XltDBCSMacro(
                     char          **ppSource
                   , char          **ppTarget
                   , unsigned long *pXLTSourceLen
                   , unsigned long *pXLTTargetLen
                   , long          index );

     // Data members.

     private:

     char             *HWSPATH;              // Path to conversion tables.
     TconvTableInfo   *pConvTableInfo;
     TconvIDTable     *pConvIDTable;
     long             convNum;               // Number of conversion records.
     HWSCnvsn         convType;              // Language of user data (passed to open).
     int              clientEndian;          // Endian of client machine (passed to open).
     int              IMSEndian;             // Endian of IMS machine - always
                                             // S370_ENDIAN.
     lng              LNG[HWS_LASTLANG + 1]; // Array of conversion information
                                             // shared by multiple methods.
};

#endif
