/*---------------------------------------------------------------------------*/
/*                                                                           */
/*  NetCentric Computing with Object Rexx                                    */
/*  Programming Example                                                      */
/*                                                                           */
/*  (c) Copyright IBM Corporation 1999                                       */
/*                                                                           */
/*  html.frm - HTML Framwork for dynamic HTML generation                     */
/*                                                                           */
/*  Requirements:                                                            */
/*    html.frm  - HTML Support Framework  (the base HTML Support Class)      */
/*                                                                           */
/*  This code is sample code, provided "AS IS", without warranty of any kind */
/*                                                                           */
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
::METHOD empty                         /* floating method --> parmDir class  */
  return ''                            /* return empty string                */

/*---------------------------------------------------------------------------*/
::METHOD substitute                    /* floating method --> escFilter class*/
  return '#'                           /* return substition string           */


/*****************************************************************************/
::CLASS html PUBLIC SUBCLASS queue

::METHOD workdir ATTRIBUTE 

/*---------------------------------------------------------------------------*/
::METHOD init                          /* initialize an html object          */
  expose type                          /* MIME type                          */

  self~setmethod('[]')                 /* hide these inherited methods       */
  self~setmethod('[]=')
  self~setmethod('REMOVE')
  self~setmethod('PEEK')
  self~setmethod('HASINDEX')

  parse source . . pathname            /* get the pathname of this script    */
  pathname = changestr('\', pathname, '/')  /* transform to unix style       */
  self~workdir= pathname~left(pathname~lastpos('/'))

  if arg(1, 'o') then
    type = 'text/html'                 /* default document type              */
  else 
    type = arg(1)                      /* specified document type            */

/*---------------------------------------------------------------------------*/
::METHOD put                           /* override of the put method         */
  use arg obj
                                       
  if \isOfHtmlKind(obj) then           /* is obj kind of an html object?     */ 
    self~queue(obj)                                        
  else do line over obj                /* embed the html object              */
    self~put(line)                
  end
  return self

/*---------------------------------------------------------------------------*/
::METHOD type                          /* change the document type           */
  expose type                                                              
  use arg type                                                           
                                                                           
/*===========================================================================*/
::METHOD title                         /* title parms: text, htags, bodyopts */
  self~~queue('<html><head><title>'arg(1)'</title>')
  if arg(2, 'e') then                  /* html object defining header tags   */
    self~put(arg(2))                   /* or specify as html string, if any  */
  return self~~queue('</head><body 'arg(3)'>') 

/*---------------------------------------------------------------------------*/
::METHOD Meta                          /* meta parms: keyword, content, opts */
  return self~~queue('<meta name="'arg(1)'" content="'arg(2)'"' arg(3)'>') 

/*---------------------------------------------------------------------------*/
::METHOD LinkRel                       /* link       parms: type, href, opts */
  return self~~queue('<link rel="'arg(1)'" href="'arg(2)'"' arg(3)'>') 

/*---------------------------------------------------------------------------*/
::METHOD LinkRev                       /* link       parms: type, href, opts */
  return self~~queue('<link rev="'arg(1)'" href="'arg(2)'"' arg(3)'>') 

/*---------------------------------------------------------------------------*/
::METHOD scrollingtitle                /* scrollingtitle  parms: text, opt   */
  use arg text

  self~queue('<html><head>')
  do i = 1 to text~length
    self~queue('<title>'text~substr(1,i)'</title>')
  end
  do i = text~length-1 to 1 by -1
    self~queue('<title>'text~right(i)'</title>')
  end
  return self~~queue('<title>'text'</title></head><body 'arg(2)'>') 

/*---------------------------------------------------------------------------*/
::METHOD address                       /* address                            */
  return self~~queue('<address>'arg(1)'</eaddress>')

/*===========================================================================*/
::METHOD h1                            /* header 1 tag                       */
  return self~~queue('<h1>'arg(1)'</h1>')                                              
                                                                           
/*---------------------------------------------------------------------------*/
::METHOD h2                            /* header 2 tag                       */
  return self~~queue('<h2>'arg(1)'</h2>')                                              
                                                                           
/*---------------------------------------------------------------------------*/
::METHOD h3                            /* header 3 tag                       */
  return self~~queue('<h3>'arg(1)'</h3>')                                              
                                                                           
/*---------------------------------------------------------------------------*/
::METHOD h4                            /* header 4 tag                       */
  return self~~queue('<h4>'arg(1)'</h4>')                                              
                                                                           
/*---------------------------------------------------------------------------*/
::METHOD h5                            /* header 5 tag                       */
  return self~~queue('<h5>'arg(1)'</h5>')                                              
                                                                           
/*---------------------------------------------------------------------------*/
::METHOD h6                            /* header 6 tag                       */
  return self~~queue('<h6>'arg(1)'</h6>')                                              
                                                                           
/*===========================================================================*/
::METHOD a                             /* anchor           parms: name, html */
  self~queue('<a name="'arg(1)'">')
  self~put(arg(2))
  return self~~queue('</a>')

/*---------------------------------------------------------------------------*/
::METHOD aref                          /* anchor reference                   */
  use arg ref, html, pic

  self~queue('<a href="'ref'">')
  if arg() = 3 then
    self~queue('<img align=middle src="'pic'">')
  if arg(2, e) then 
    self~put(html)
  return self~~queue('</a>')

/*---------------------------------------------------------------------------*/
::METHOD arefmouse                     /* anchor reference with mouse events */
  use arg ref, num, pic, picmouse, text, opts

  self~queue('<a href="'ref'"',
              'onMouseOver="document.images['num'].src='''picmouse''';"',
              'onMouseOut="document.images['num'].src='''pic''';">',
              '<img align=middle src="'pic'" 'opts'>') 
  if arg(5, 'e') then
    self~queue('<b>'text '</b></a>')
  else 
    self~queue('</a>')
  return self

/*===========================================================================*/
::METHOD tag                           /* generic tag       parms: name,text */
  return self~~queue('<'arg(1)'>'arg(2))                                                  
                                                                             
/*---------------------------------------------------------------------------*/
::METHOD etag                          /* generic end tag   parms: name,text */
  return self~~queue('</'arg(1)'>'arg(2))                                                  
                                                                             
/*---------------------------------------------------------------------------*/
::METHOD tage                          /* generic tag with matching end      */
  return self~put('<'arg(1)'>'arg(2)'</'arg(1)'>')      /* parms: tag, body */                                     
                                                                             
/*===========================================================================*/
::METHOD hr                            /* horizontal rule                    */
  return self~~queue('<hr>')                                                       
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD t                             /* add raw text to the last enty      */
  if self~items = 0 then
    self~~queue(arg(1))
  else 
    self~~put:super(self~at(self~items) arg(1), self~items)
  return self

/*---------------------------------------------------------------------------*/
::METHOD br                            /* break                              */
  self~queue('<br>')
  return self~put(arg(1))                                                   

                                                                         
/*===========================================================================*/
::METHOD p                             /* paragraph        parms: html, opts */
  self~queue('<p' arg(2)'>')
  if arg() > 0 then
    self~put(arg(1))
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD pl                            /* paragraph, left adjusted           */
  self~queue('<p align=left>')
  if arg() > 0 then
    self~put(arg(1))
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD pc                            /* paragraph, centered                */
  self~queue('<p align=center>')
  if arg() > 0 then
    self~put(arg(1))
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD pr                            /* paragraph, right adjusted          */
  self~queue('<p align=right>')
  if arg() > 0 then
    self~put(arg(1))
  return self
                                                                         
/*===========================================================================*/
::METHOD ul                            /* unordered list                     */
  return self~~queue('<ul>')                                                       
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD eul                           /* end unordered list                 */
  return self~~queue('</ul>')                                                      
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD ol                            /* ordered list                       */
  return self~~queue('<ol>')                                                       
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD eol                           /* end ordered list                   */
  return self~~queue('</ol>')                                                      
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD dir                           /* directory                          */
  return self~~queue('<dir>')                                                      
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD edir                          /* end directory                      */
  return self~~queue('</dir>')                                                     
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD div                           /* division               parms: opts */
  return self~~queue('<div arg(1)>')                                                      
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD ediv                          /* end division                       */
  return self~~queue('</div>')                                                     
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD span                          /* span      parms: opts, sptxt, text */
  self~queue('<span' arg(1)'>')                                                      
  if arg(2, 'e') then 
    self~queue(arg(2)'</span>'arg(3))                                                      
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD espan                         /* end span                           */
  return self~~queue('</span>')                                                     
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD li                            /* list item                          */
  self~queue('<li>')
  if arg() > 0 then
    self~put(arg(1))
  return self
                                                                         
/*===========================================================================*/
::METHOD strong                        /* strong                             */
  if arg(1, 'e') then 
    self~queue('<strong>'arg(1)'</strong>')                                            
  else 
    self~queue('<strong>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD estrong                       /* end strong                         */
  return self~~queue('</strong>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD big                           /* big                                */
  if arg(1, 'e') then 
    self~queue('<big>'arg(1)'</big>')                                            
  else 
    self~queue('<big>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD ebig                           /* end big                           */
  return self~~queue('</big>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD small                         /* small                              */
  if arg(1, 'e') then 
    self~queue('<small>'arg(1)'</small>')                                            
  else 
    self~queue('<samll>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD esmall                        /* end small                          */
  return self~~queue('</small>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD em                            /* emphasis                           */
  if arg(1, 'e') then 
    self~queue('<em>'arg(1)'</em>')                                            
  else 
    self~queue('<em>')                                            
  return self

/*---------------------------------------------------------------------------*/
::METHOD eem                           /* end emphasis                       */
  return self~~queue('</em>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD b                             /* bold                               */
  if arg(1, 'e') then 
    self~queue('<b>'arg(1)'</b>')                                            
  else 
    self~queue('<b>')                                            
  return self

/*---------------------------------------------------------------------------*/
::METHOD eb                            /* end bold                           */
  return self~~queue('</b>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD i                             /* italic                             */
  if arg(1, 'e') then 
    self~queue('<i>'arg(1)'</i>')                                            
  else 
    self~queue('<i>')                                            
  return self

/*---------------------------------------------------------------------------*/
::METHOD ei                            /* end italic                         */
  return self~~queue('</i>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD u                             /* underscore                         */
  return self~~queue('<u>'arg(1)'</u>')                                              
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD sub                           /* subscript                          */
  return self~~queue('<sub>'arg(1)'</sub>')                                              
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD sup                           /* superscript                        */
  return self~~queue('<sup>'arg(1)'</sup>')                                              
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD tt                            /* teletype                           */
  return self~~queue('<tt>'arg(1)'</tt>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD blink                         /* blink                              */
  return self~~queue('<blink>'arg(1)'</blink>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD code                          /* code                    parm: text */
  if arg(1, 'e') then 
    self~queue('<code>'arg(1)'</code>')                                            
  else 
    self~queue('<code>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD ecode                         /* end code                           */
  return self~~queue('</code>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD pre                           /* preformatted text                  */
  if arg(1, 'e') then 
    self~queue('<pre>'arg(1)'</pre>')                                            
  else 
    self~queue('<pre>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD epre                          /* end preformatted text              */
  return self~~queue('</pre>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD strike                        /* strikeout text                     */
  if arg(1, 'e') then 
    self~queue('<strike>'arg(1)'</strike>')                                            
  else 
    self~queue('<strike>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD s                             /* strikeout text                     */
  forward message 'strike'                                                        

/*---------------------------------------------------------------------------*/
::METHOD estrike                       /* end strikeout text                 */
  return self~~queue('</strike>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD es                            /* end strikeout text                 */
  forward message 'estrike'                                                        
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD q                             /* quote                              */
  if arg(1, 'e') then 
    self~queue('<q>'text'</q>')                                            
  else 
    self~queue('<q>')                                            
  return self

/*---------------------------------------------------------------------------*/
::METHOD eq                            /* end quote                          */
  return self~~queue('</q>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD font                          /* font         parm: opts, [text]    */
  use arg opts

  if arg(2, 'e') then 
    self~queue('<font' opts'>'arg(2)'</font>')                                            
  else 
    self~queue('<font' opts'>')                                            
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD efont                         /* end font                           */
  return self~~queue('</font>')                                            
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD plaintext                     /* plaintext                          */
  return self~~queue('<plaintext>'arg(1))                                            
                                                                         
/*===========================================================================*/
::METHOD table                         /* table                    parm: opt */
  return self~~queue('<table' arg(1)'>')                                          
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD etable                        /* end table                          */
  return self~~queue('</table>')                                                   
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD td                            /* table data         parm: text, opt */
  self~queue('<td' arg(2)'>')                           
  if arg(1, e) then 
    self~put(arg(1))~queue('</td>') 
  return self
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD etd                           /* end table data                     */
  return self~~queue('</td>')                                                      
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD th                            /* table header     parms: text, opts */
  return self~~queue('<th' arg(2)'>'arg(1)'</th>')                                  
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD tr                            /* table row                          */
  return self~~queue('<tr>')
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD etr                           /* end table row                      */
  return self~~queue('</tr>')
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD tfoot                         /* table footing    parms: text, opts */
  return self~~queue('<tfoot' arg(2)'>'arg(1)'</tfoot>')                                  
                                                                         
/*---------------------------------------------------------------------------*/
::METHOD caption                      /* table caption    parms: text, opts */
  return self~~queue('<caption' arg(2)'>'arg(1)'</caption>')                                  

/*---------------------------------------------------------------------------*/
::METHOD array2table                   /* generate table from array          */
  use arg ar_h, ar_d, ar_o, options 

  if Arg(4, 'o') then 
    options = ""
  self~table(options)
  if Arg(1, 'e') then do i over ar_h
    self~th(i)
  end
  do i = 1 to ar_d~dimension(1)
    self~tr
    do j = 1 to ar_d~dimension(2)
      if Arg(3, 'o') then  
        self~td(ar_d[i,j])
      else if ar_o = "CURRENCY" then
        self~td(ar_d[i,j], 'align=right')
      else 
        self~td(ar_d[i,j], ar_o[j])
    end    
  end    
  self~etable
  return self

/*===========================================================================*/
::METHOD img                           /* image         parms: imgfile, opts */
  use arg imgfile, opts

  if Arg(2, 'o') then 
    opts = 'align="bottom"'
  if imgfile~pos('.') = 0 then
    imgfile = imgfile".gif"
  return self~~queue('<img src='imgfile opts'>')
  
/*---------------------------------------------------------------------------*/
::METHOD play                          /* play soundfile  (Windows only)     */
  use arg soundfile  

  if soundfile~pos('.') = 0 then
    soundfile = soundfile".WAV"
  return self~~queue('<bgsound src='soundfile'>')

/*===========================================================================*/
::METHOD form                          /* form                  parm: action */
  return self~~queue('<form method="GET" action="'arg(1)'">')

/*---------------------------------------------------------------------------*/
::METHOD eform                         /* end form                           */
  return self~~queue('</form>')
  
/*===========================================================================*/
::METHOD input                         /* input  parms:type, name, val, opts */
  self~queue('<input type="'arg(1)'" name="'arg(2)'" value="'arg(3)'"'arg(4)'>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD einput                        /* end input                          */
  return self~~queue('</input>')

/*---------------------------------------------------------------------------*/
::METHOD submit                        /* submit       parms:name, val, opts */
  self~queue('<input type=submit name='arg(1)' value="'arg(2)'"' arg(3)'>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD reset                         /* reset        parms:name, val, opts */
  self~queue('<input type=reset name='arg(1)' value="'arg(2)'"' arg(3)'>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD iText                         /* itext  parms:name, val, opts, text */
  self~queue('<input type=text name='arg(1)' value="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD iPassw                        /* passw  parms:name, val, opts, text */
  self~queue('<input type=password name='arg(1)' value="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD iFile                         /* ifile  parms:name, val, opts, text */
  self~queue('<input type=file name='arg(1)' value="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD iHidden                       /* ihidden parms:name, val, opts, text*/
  self~queue('<input type=hidden name='arg(1)' value="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD iImage                        /* iimage parms:name, img, opts, text */
  self~queue('<input type=image name='arg(1)' src="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD radio                         /* radio  parms:name, val, opts, text */
  self~queue('<input type=radio name="'arg(1)'" value="'arg(2)'"'arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD check                         /* check  parms:name, val, opts, text */
  self~queue('<input type=checkbox name='arg(1)' value="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD button                        /* button parms:name, val, opts, text */
  self~queue('<input type=button name='arg(1)' value="'arg(2)'"' arg(3)'>')
  if arg(4, e) then
    self~queue(arg(4)'</input>')
  return self
      
/*---------------------------------------------------------------------------*/
::METHOD textarea                      /* text area   parm: name, size, opts */
  use arg name, size, opts

  self~queue('<textarea name='name 'rows='size~word(1) 'cols='size~word(2) arg(3)'>')
  if arg(4, e) then 
    self~put(arg(4))~queue('</textarea>') 
  return self
      
/*---------------------------------------------------------------------------*/
::METHOD etextarea                      /* end text area                     */
  return self~~queue('</textarea>')

/*===========================================================================*/
::METHOD ilist                         /* inputlist             parm: action */
  use arg action

  self~queue('<form method="GET" action="'action'">')
  return self~~queue('<table border=0 cellpadding=1>')
  
/*---------------------------------------------------------------------------*/
::METHOD ili        /* inputlistitem  parms:textb,type,name,value,opts,texta */       
  self~tr
  self~td(arg(1),'align=left')
  self~td('<input type="'arg(2)'" name="'arg(3)'" value="'arg(4)'"' arg(5)'>')
  return self~~td(arg(6),'align=left')

/*---------------------------------------------------------------------------*/
::METHOD eilist                        /* end input list                     */
  self~queue('</table>')
  return self~~queue('</form>')
  
/*===========================================================================*/
::METHOD select                        /* select  parms: name, listarr, opts */
  use arg name, listarr

  self~queue('<select name='name arg(3)'>')
  do item over listarr
    self~queue('<option>'item)
  end
  self~queue('</select>')
  if arg(4, e) then
    self~queue(arg(4))
  return self
      
/*---------------------------------------------------------------------------*/
::METHOD selectm                       /* select multiple from list          */
  use arg name, listArr, opts, selIndexArr
                                       /* selIndexArr has indexes of listArr */
  self~queue('<select multiple name='name arg(3)'>')
  if Arg(4, 'e') then do               /* select index array is specified?   */
    do ind = 1 to listArr~items
      selected = .false
      do selind over selIndexArr       /* find out whether item is selected  */
        if selind = ind then do 
          selected = .true
        end
      end
      if selected then                 /* selected list item                 */
        self~queue('<option selected>'listArr[ind])
      else                             /* unselected list item               */ 
        self~queue('<option>'listArr[ind])
    end  
  end
  else do item over listarr            /* no preselected list elements       */
    self~queue('<option>'item)
  end
  self~queue('</select>')
  if arg(5, e) then
    self~queue(arg(5))
  return self

/*===========================================================================*/
::METHOD object                  /* object  parms: type,data,size,param,opts */
  use arg type, data, size
  sizeparms = 'width='size~word(1) 'height='size~word(2)
  self~queue('<object type="'type'" data="'data '"' sizeparms arg(5)'>')                                            
  if arg(4, 'e') then 
    self~put(arg(4))                                            
  self~queue('<embed type="'type'" src="'data'"' sizeparms arg(5)'>')                                            
  self~queue('</object>')
  return self~~queue('<bgsound src="'data'">')

/*---------------------------------------------------------------------------*/
::METHOD param                         /* param                  parms: opts */ 
  return self~~queue('<param' arg(1)'>')                                            

/*===========================================================================*/
::METHOD write                         /* write html to file                 */
  use arg output                       /* output file                        */

  stream = .stream~new(output)         /* get a stream object                */
  do line over self                    /* write data to file line by line    */
    stream~lineout(line)               /* write out the next line            */
  end                                                                       
  stream~close                         /* close the file                     */
                                                                            
/*---------------------------------------------------------------------------*/
::METHOD include                       /* include html file                  */
  use arg infile                       /* input file to be included          */

  if pathname~lastpos('/') = 0 then 
    fullname = self~workdir||infile    /* look for it in the work directoy   */
  else 
    fullname = infile                  /* full path specified                */
  stream = .stream~new(fullname)       /* get a stream object                */
  retstr = stream~open('READ')         /* open file for reading              */ 
  if retstr = "READY:" then do
    do line over stream                /* read the data line by line         */
      self~queue(line)                 /* include line                       */
    end                                                                       
    stream~close                       /* close the file                     */
  end 
  else                                 /* include error message instead      */
    self~queue('<font color=red>Error including file' fullname '--' retstr'</font>') 
  return self

/*---------------------------------------------------------------------------*/
::METHOD end                           /* end HTML document                  */
  expose type                                                               

  self~queue('</body></html>')         /* end body and html                  */
  self~push('<!doctype html public "html4.0">')
  self~push('')
  self~push('Content-Type:' type)      /* add the content type to the top    */ 
  return self


/*****************************************************************************/
::CLASS parmDir SUBCLASS directory PUBLIC /* parameter directory             */

/*---------------------------------------------------------------------------*/
::METHOD init                          /* initialize the parameter directory */
  use arg parmstring                   /* parameter string from evironment   */

  parse var parmstring parm '&' rest   /* initialize parmdir from parmstring */  
  do while parm \= ''                  /* process the whole parmlist         */
    parse var parm name '=' value      /* get each parameter from list ...   */ 
    value = normalize(value)           /* filter out escape sequences        */
    if self~hasindex(name) then do     /* is there an entry already?         */  
      item = self[name]                /* get it                             */
      if item~class=.list then         /* is it a List?                      */
        item~insert(value)             /* add the value to the list          */
      else do
        item = .list~of(item, value)   /* make an item list                  */ 
      end
      self[name]= item                 /* update the parmDir entry           */
    end
    else 
      self[name]= value                /* and  put it in directory           */ 
    parse var rest parm '&' rest       /* get next parameter                 */
  end
                                       /* unknown parameters are handled     */ 
  self~setmethod('UNKNOWN', .methods['EMPTY'])

/*---------------------------------------------------------------------------*/
::METHOD list                          /* Parameter list          parm: name */                   

  item = self[arg(1)]                  /* get parameter by name              */ 
  if item~class=.list then do          /* is it a list?                      */
    list = ''
    do i over item                     /* linearize the list object          */
      if list = '' then 
        list = i 
      else 
        list = list',' i               /* separate items by comma            */
    end
    return list                        /* return list of items               */
  end 
  else                                 
    return item                        /* just return the item               */


/*===========================================================================*/
/*                                                                           */
/* Auxiliary Routines                                                        */
/*                                                                           */
/*===========================================================================*/
::ROUTINE isOfHtmlKind PUBLIC          /* object instance of kind html class?*/
  use arg obj

  return isClassSubclassOf(obj~class, .html)

/*---------------------------------------------------------------------------*/
::ROUTINE isClassSubclassOf            /* is myclass descended from class?   */ 
  use arg myclass, class

  if myclass = class then return .true
  subclasses = class~subclasses 
  do subclass over subclasses
    if isClassSubclassOf(myclass, subclass) then return .true
  end
  return .false

/*---------------------------------------------------------------------------*/ 
::ROUTINE normalize                    /* translate escape seq's  parm: text */ 
  use arg text                         /* e.g. '%3F' --> '?'                 */

  pi = pos('%', text)                  
  do while(pi > 0)                     
    seq = substr(text, pi, 3)
    text = changestr(seq, text, x2c(substr(seq, 2)))
    pi = pos('%', text, pi)   
  end 
  return text                          /* return the normalized text string  */


/*===========================================================================*/
/*                                                                           */
/* Inline Routines (used within text line)                                   */
/*                                                                           */
/*===========================================================================*/
::ROUTINE blink PUBLIC                 /* blink                              */
  return  '<blink>'arg(1)'</blink>'

/*---------------------------------------------------------------------------*/
::ROUTINE sub PUBLIC                   /* subscript                          */
  return  '<sub>'arg(1)'</sub>'

/*---------------------------------------------------------------------------*/
::ROUTINE sup PUBLIC                   /* superscript                        */
  return  '<sup>'arg(1)'</sup>'

/*---------------------------------------------------------------------------*/
::ROUTINE u PUBLIC                     /* underscore                         */
  return  '<u>'arg(1)'</u>'

/*---------------------------------------------------------------------------*/
::ROUTINE i PUBLIC                     /* italic                             */
  return  '<i>'arg(1)'</i>'

/*---------------------------------------------------------------------------*/
::ROUTINE b PUBLIC                     /* bold                               */
  return  '<b>'arg(1)'</b>'

/*---------------------------------------------------------------------------*/
::ROUTINE small PUBLIC                 /* small                              */
  return  '<small>'arg(1)'</small>'

/*---------------------------------------------------------------------------*/
::ROUTINE big PUBLIC                   /* big                                */
  return  '<big>'arg(1)'</big>'

/*---------------------------------------------------------------------------*/
::ROUTINE em PUBLIC                    /* emphasis                           */
  return  '<em>'arg(1)'</em>'

/*---------------------------------------------------------------------------*/
::ROUTINE strong PUBLIC                /* strong                             */
  return  '<strong>'arg(1)'</strong>'

/*---------------------------------------------------------------------------*/
::ROUTINE font PUBLIC                  /* font            parm: option, text */
  return  '<font' arg(1)'>'arg(2)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE size PUBLIC                  /* size              parm: size, text */
  return  '<font size="'arg(1)'">'arg(2)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE color PUBLIC                 /* color            parm: color, text */
  return  '<font color='arg(1)'>'arg(2)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE red PUBLIC                   /* color red               parm: text */
  return  '<font color=#FF0000>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE green PUBLIC                 /* color green             parm: text */
  return  '<font color=#008000>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE blue PUBLIC                  /* color blue              parm: text */
  return  '<font color=#0000FF>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE yellow PUBLIC                /* color yellow            parm: text */
  return  '<font color=#FFFF00>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE brown PUBLIC                 /* color brown             parm: text */
  return  '<font color=#A52A2A>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE white PUBLIC                 /* color white             parm: text */
  return  '<font color=#FFFFFF>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE grey PUBLIC                  /* color brey              parm: text */
  return  '<font color=#808080>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE black PUBLIC                 /* color black             parm: text */
  return '<font color=#000000>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE breaks PUBLIC                /* insert breaks           parm: text */  
  return changestr('0A'x, changestr('0D'x, arg(1), ''), '<br>')


/*===========================================================================*/
/* End of Html framework definition                                          */ 
/*---------------------------------------------------------------------------*/ 

