/*********************************************************/
/*                                                       */
/* CLOUD.CMD - IBM Object REXX Sample Program            */
/*                                                       */
/* ----------------------------------------------------- */
/* (C) Copyright IBM Corp. 1996 - All Rights Reserved.   */
/*                                                       */
/* DISCLAIMER OF WARRANTIES.  The following [enclosed]   */
/* code is sample code created by IBM Corporation. This  */
/* sample code is not part of any standard or IBM        */
/* product and is provided to you solely for the         */
/* purpose of assisting you in the development of your   */
/* applications.  The code is provided "AS IS", without  */
/* warranty of any kind.  IBM shall not be liable for    */
/* any damages arising out of your use of the sample     */
/* code,  even if they have been advised of the          */
/* possibility of such damages.                          */
/*                                                       */
/*********************************************************/

/*********************************************************/
/* Declare the cloud class. It will be a public class    */
/* that means it will be available to all REXX programs  */
/* referring to this program (e.g. by calling it).       */
/*********************************************************/
::CLASS Cloud PUBLIC

/*********************************************************/
/* The INIT method of the cloud expects three arguments: */
/*   1. The name of the cloud                            */
/*   2. The initial size of the cloud                    */
/*   3. The initial temperature of the cloud             */
/*********************************************************/
::METHOD INIT
  Expose Name Size ItsRaining HeavyRain
  Use Arg Name, InitialSize, InitialTemp

  /* set the attributes to their initial states */
  Size = 0
  HeavyRain = .False
  ItsRaining = .False

  /* now let the cloud grow to the specified size */
  Self~Grow(InitialSize)

  /* set the temperature of the cloud, this is done with  */
  /* a separate method that will set the rain/snow flag   */
  Self~Temperature = InitialTemp

  /* start another thread that will show us the rain      */
  Self~ShowRain


/**********************************************************/
/* The GROW method will increase the cloud's size by the  */
/* specified value and set the rain flags accordingly.    */
/**********************************************************/
::METHOD Grow
  Expose Name Size ItsRaining HeavyRain
  Use Arg Growth

  /* let the cloud grow */
  Size = Size + Growth

  /* will it start raining, or even heavy rain? */
  If Size >= 70 Then
    ItsRaining = .True
  If Size >= 100 Then
    HeavyRain = .True

  /* print a short status message with the new cloud size */
  Say "Cloud '" || Name || "' has grown to" Size


/**********************************************************/
/* The SHRINK method will reduce the cloud's size by the  */
/* specified value and set the rain flags accordingly.    */
/**********************************************************/
::METHOD Shrink
  Expose Name Size ItsRaining HeavyRain
  Use Arg Reduction

  /* let the cloud shrink */
  Size = Size - Reduction

  /* will it stop raining? */
  If Size <= 60 Then
    HeavyRain = .False
  If Size <= 40 Then
    ItsRaining = .False


/**********************************************************/
/* The COOL method will reduce the cloud's temperature    */
/* by the specified value.                                */
/**********************************************************/
::METHOD Cool
  Expose Name
  Use Arg Reduction

  /* Set the new temperature and rain/snow flags. The     */
  /* current temperature is retrieved by invoking the     */
  /* TEMPERATURE method of the same object. The result    */
  /* of the subtraction is then used as an argument for   */
  /* the TEMPERATURE= method (special notation).          */
  Self~Temperature = Self~Temperature - Reduction

  /* print a short status message with the new cloud temp */
  Say "Cloud '" || Name || "' has cooled down to",
      Self~Temperature "F"


/**********************************************************/
/* The HEAT method will increase the cloud's temperature  */
/* by the specified value.                                */
/**********************************************************/
::METHOD Heat
  Expose Name
  Use Arg Increase

  /* set the new temperature and rain/snow flags */
  Self~Temperature = Self~Temperature + Increase

  /* print a short status message with the new cloud temp */
  Say "Cloud '" || Name || "' has heated up to",
      Self~Temperature "F"


/**********************************************************/
/* The private TEMPERATURE method will return the         */
/* cloud's current temperature                            */
/**********************************************************/
::METHOD Temperature PRIVATE
  Expose Temperature
  Return Temperature


/**********************************************************/
/* The private TEMPERATURE= method will set the new       */
/* temperature and the rain/snow flag of the cloud.       */
/**********************************************************/
::METHOD "Temperature=" PRIVATE
  Expose Temperature RainAction
  Use Arg Temperature

  /* set the rain/snow flag according to the temperature  */
  If Temperature > 32 Then
    RainAction = "raining"
  Else
    RainAction = "snowing"


/**********************************************************/
/* The private SHOWRAIN method will set a guard on        */
/* the ItsRaining flag. As soon as the cloud's state      */
/* changes to rain, this method will print messages that  */
/* the cloud is pouring rain. While raining the cloud     */
/* will shrink again (due to the loss of water).          */
/**********************************************************/
::METHOD ShowRain PRIVATE
  Expose Name Size ItsRaining HeavyRain RainAction
  Reply /* return immediately to caller */

  /* this loop will run as long as the cloud exists */
  Do ForEver
    /* wait for the cloud to reach the raining state      */
    Guard Off When ItsRaining

    /* print raining message depending on the strength    */
    /* of the rain and reduce the size of the cloud       */
    If HeavyRain Then
      Do
        Say "Cloud '" || Name || "' is" RainAction,
            "heavily (size =" Size || ")."
        Self~Shrink(6)
      End
    Else
      Do
        Say "Cloud '" || Name || "' is" RainAction,
            "(size =" Size || ")."
        Self~Shrink(3)
      End

    /* rain always takes some time */
    Call SysSleep 4
  End
