/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Chat_s.cmd  -  A TCP/IP Server with Interacting Clients                 */
/*                                                                          */
/*  Documentation: see chat.doc in directory docs                           */
/*                                                                          */
/*  History:                                                                */
/*    06/30/97  added client alias option for multiple clients per host     */
/*              added ? method providing list of conversationalists         */
/*              added option to specify server port                         */
/*              improved readibility by minor restructoring/renaming        */
/*    07/02/97  added whisper function  to talk to single client            */
/*    07/09/97  added server introducing new client                         */
/*    07/10/97  added controlled server shudown                             */
/*    07/22/97  added name segment option                                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/

Parse Arg Port 
if Port = '' then Port = 1924          /* Default server port is 1924       */
aChatServer = .ChatServer~new(Port)    /* Create a chat server at Port      */ 
aChatServer~startAccepting             /* Listen at socket for clients      */    
aChatServer~shutdown                   /* server shutdown                   */    
                                       
/****************************************************************************/

::REQUIRES "servers.frm"               /* Load the servers framework        */   

/*--------------------------------------------------------------------------*/
/* ChatServer Class definition                                              */
/*--------------------------------------------------------------------------*/
::CLASS ChatServer SUBCLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init
  expose knownPort Table
  use arg knownPort

  self~init:super(knownPort)           /* Run the superclass init           */
  Table = .Conversation~new(self)      /* Create a Table with empty seats   */                                          
                                          
/*--------------------------------------------------------------------------*/
::METHOD NewClient UNGUARDED           /* Over-write superclass' method     */    
  expose Table
  use arg client, prefix
                                       /* A new client takes a seat         */
  Table~takeSeat(client, prefix~word(1), prefix~word(3))

/*--------------------------------------------------------------------------*/
::METHOD shutdown                      /* Over-write superclass' method     */    
  expose Table

  Table~terminate                      /* Terminate the conversation        */
  self~shutdown:super                  /* Shutdown the tcpServer            */ 


/*--------------------------------------------------------------------------*/
/* Conversation Class definition                                            */
/*--------------------------------------------------------------------------*/
::CLASS Conversation                   /* A place to have a chat            */    

/*--------------------------------------------------------------------------*/
::METHOD init                      
  expose Talkers Server               
  use arg server

  Talkers = .bag~new                   /* A collection of conversationalists*/

/*--------------------------------------------------------------------------*/
::METHOD terminate                     /* Terminate the whole conversation  */
  expose Talkers Server               

  do talker over Talkers               /* Let all other clients know ...    */      
    talker~HeSaid('> Server: conversation terminated!  Bye bye' talker~Xname)
    talker~stopListening               /* Stop him listening                */
    server~removeClient(talker~socket, talker~name)         
  end

/*--------------------------------------------------------------------------*/
::METHOD takeSeat                      /* Seat a client at the table        */
  expose Talkers              
  use arg client, alias, numsegs
                                       /* Create a new talker and ...       */
  aTalker = .Talker~new(client, alias, numsegs, self)

  do talker over Talkers               /* Let all other clients know ...    */      
    talker~HeSaid('* Server:' aTalker~Xname(talker~nameSegs) 'arrived!')
  end

  Talkers~put(aTalker)                 /* and register him                  */

/*--------------------------------------------------------------------------*/
::METHOD leaveSeat                     /* Client leaves seat at the table   */
  expose Talkers Server              
  use arg aTalker  

  if aTalker~listening then do         /* if still beeing listening ...     */
    aTalker~HeSaid('> Server: Bye bye' aTalker~Xname)  
    server~removeClient(aTalker~socket, aTalker~name)         
  end
  Talkers~remove(aTalker)              /* Remove the talker from collection */

/*--------------------------------------------------------------------------*/
::METHOD ISay                          /* A client speaks up                */
  expose Talkers              
  use arg talker, hisWords
          
  do aTalker over Talkers              /* Let all other clients hear it     */      
    if aTalker \= talker then 
      aTalker~HeSaid('*' talker~Xname(aTalker~nameSegs) || ':' hisWords)
  end

/*--------------------------------------------------------------------------*/
::METHOD IWhisper                      /* A talker whispers to other client */
  expose Talkers              
  use arg talker, hisWords, name
 
  upperName = name~translate
  do aTalker over Talkers              /* Find talker by name (abbreviated) */      
    if aTalker~name~translate~abbrev(upperName) then do
      if aTalker \= talker then        /* Don't whisper to yourself         */
        aTalker~HeSaid('>' talker~Xname(aTalker~nameSegs) || ':' hisWords)
      return
    end
  end
                                       /* Tell the talker: wrong name       */
  talker~HeSaid('> Server:' name 'is unknown') 

/*--------------------------------------------------------------------------*/
::METHOD ?                             /* what are the conversationalists   */
  expose Talkers              
  use arg talker
 
  namelist = talker~Xname
  nameSegments = talker~nameSegs
  do aTalker over Talkers              /* List all other client's names     */      
    if aTalker \= talker then          /* with requesting client's precision*/
      namelist = namelist || ',' aTalker~Xname(nameSegments)
  end

  talker~HeSaid('> Server:' namelist)


/*--------------------------------------------------------------------------*/
/* Talker Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS Talker                         /* One is created for each client    */    

/*--------------------------------------------------------------------------*/
::METHOD init                      
  expose mySocket numSegs theGroup
  use arg mySocket, myName, numSegs, theGroup

  self~name = myName                   /* So they'll know who is speaking   */
  self~Listen                          /* Start listening to the group      */

  if \numSegs~datatype('W') then       /* Verify number of name segments    */
    numSegs = 999                      /* for all invalid values ...        */
  else if numSegs < 0 then  
    numSegs = 1                        /* use default: short form           */

/*--------------------------------------------------------------------------*/
::METHOD name ATTRIBUTE                     

/*--------------------------------------------------------------------------*/
::METHOD nameSegs                      /* Provide number of name segments   */
  expose numSegs                          

  return numSegs

/*--------------------------------------------------------------------------*/
::METHOD Xname                         /* Extended name service             */
  expose numSegs

  if arg() = 0 then 
    number = numSegs                  
  else 
    number = arg(1)                    /* default is single name segment    */

  if number = 0 then                   /* use just the alias, if any        */ 
    return self~name~translate(' ', '_.')~word(1) 
  else                                 /* use 'number' of name segments     */       
    return self~name~translate(' ', '.')~subword(1, number)~translate('.', ' ')                    

/*--------------------------------------------------------------------------*/
::METHOD socket                 
  expose mySocket 
  
  return mySocket

/*--------------------------------------------------------------------------*/
::METHOD Listen UNGUARDED              /* When client speaks through socket */
  expose mySocket theGroup 
  self~Listening = .TRUE
  reply                                /* Frees up other concurrent work    */ 
                                       /* Process the rest in another thread*/
  do until myWords~translate = "QUIT"  /* Repeat until the input is 'quit'  */    
    myWords = mySocket~ReceiveData
    if \mySocket~stillOpen then leave

    firstWord = myWords~word(1)        /* Check first word for ?controls    */
    if firstWord~substr(1, 1) = '?' then
      if firstWord~length > 1  then    /* Is there a name specification?    */ 
        theGroup~IWhisper(self, myWords~subword(2), firstWord~substr(2))
      else
        theGroup~'?'(self)             /* Query the conversationalists      */ 
    else
      theGroup~ISay(self, myWords)     /* I say to group my name and words  */     
  end

  theGroup~leaveSeat(self)             /* client leaves seat at the table   */ 
  
/*--------------------------------------------------------------------------*/
::METHOD stopListening UNGUARDED       /* When client speaks through socket */
  self~Listening = .FALSE
  
/*--------------------------------------------------------------------------*/
::METHOD Listening ATTRIBUTE           /* Set/get talker's listen status    */

/*--------------------------------------------------------------------------*/
::METHOD HeSaid                        /* Other speaker's words be sent here*/
  expose mySocket 
  use arg theWords
              
  mySocket~SendData(theWords)          /* then send the words               */

