/*
 *  gdsupport.c      support module for the Gameduino device
 *
 *  This file provides support for the Gameduino VGA adapter and game
 *  system.  This library provides support for all of the functions found
 *  in the GD library (see www.excamera.com or search for Gameduino library
 *  on the web).
 *
 *  This support file is target-independent and can be recompiled for any
 *  microcontroller.
 *
 *  To use this code module with an arbitrary device, your code must
 *  register three callback functions via the GD_register() function call.
 *  These routines must select the GD device, exchange a byte with the GD device,
 *  and deselect the GD device.
 *
 *  Using this type of registering separates the Gameduino support code
 *  from the target hardware implementation.
 *
 *  Before invoking any of the routines in this library, your code must
 *  configure the SPI port to be used by the Gameduino support code!  You
 *  must set the timing of the porrt, set the SPI to send bytes MSB first,
 *  and set the SPI to mode 0 (CPOL=0, CPHA=0).
 *
 *  Refer to the gdsupport.h file for the formal declarations of the functions
 *  you must provide in your target-specific code.
 *
 */


#include <stdio.h>

#include "GD.h"
#include "font8x8.h"
#include "gdsupport.h"


#ifndef  FALSE
#define  FALSE   0
#define  TRUE    !FALSE
#endif




static  unsigned char			spr;			// current sprite, modified by GD_xhide() and GD_xsprite()


static  void					(*select)(void) = (void *)0;
static  char					(*xchg)(char  val) = (void *)0;
static  void					(*deselect)(void) = (void *)0;
static  unsigned char			inited = FALSE;		// record no callback functions have been supplied yet

static  unsigned char			stretch[16] =
{
	0x00, 0x03, 0x0c, 0x0f,
	0x30, 0x33, 0x3c, 0x3f,
	0xc0, 0xc3, 0xcc, 0xcf,
	0xf0, 0xf3, 0xfc, 0xff
};

static  unsigned char			curr_col;		// tracks next text col to write on virtual screen
static  unsigned char			curr_row;		// tracks current text row to write on virtual screen
static  unsigned char			max_col;		// holds maximum displayed col on virtual screen
static	unsigned char			max_row;		// holds maximum displayed row on virtual screen
static  unsigned long			xscroll;		// holds pixel number of left-most pixel in scroll window
static  unsigned long			yscroll;		// holds pixel number of top-most pixel in scroll window




/*
 *  Local functions
 */
static void						scrolltextup(void);
static void						erasecursor(void);
static void						showcursor(void);

static void						dc_begin(unsigned char  *s);
static unsigned int				dc_getn(unsigned char  n);
static unsigned char			dc_get1(void);



/*
 *  GD_register      assign callback functions for accessing the SPI
 *
 *  This routine allows the calling program to assign functions for
 *  enabling and disabling the GD hardware, and for exchanginge a
 *  byte of data with the GD hardware.
 *
 *  You must call this routine BEFORE calling any other routines in
 *  the GD support library!
 */
void  GD_register(void(* _select)(void),
 				   char(* _xchg)(char  val),
				   void(* _deselect)(void))
 {
 	select = _select;							// record the enable function
	xchg = _xchg;								// record the byte-exchange function
	deselect = _deselect;						// record the disable function
	inited = FALSE;								// presume one of the pointers is bad
	if ((select)  &&  (xchg)  &&  (deselect))  inited = TRUE;	// presume OK if all pointers are non-zero
 }



/*
 *  GD_begin      initialize the GD hardware
 *
 *  This routine sets up the GD hardware and resets the adapter.  You must
 *  call this routine after invoking GD_register and BEFORE calling any
 *  other GD functions.
 */

void  GD_begin(void)
{
	unsigned int						i;

	if (!inited)  return;						// guard against illegal callback addresses

	select();
	GD_wr(J1_RESET, 1);							// force reset of J1 CPU
	GD_fill(RAM_PIC, 0, 1024*10);				// zero all chars in RAM
	GD_fill(RAM_SPRPAL, 0, 2048);				// clear sprite palletes block
	GD_fill(RAM_SPR, 0, 64*256);				// clear all sprite data
	GD_fill(VOICES, 0, 256);					// shut off all voices
	GD_fill(PALETTE16A, 0, 128);				// black 16-, 4-palletes, and COMM

	xscroll = 0;
	yscroll = 0;
	GD_wr16(SCROLL_X, xscroll);
	GD_wr16(SCROLL_Y, yscroll);

	GD_wr(JK_MODE, 0);
	GD_wr(SPR_DISABLE, 0);
	GD_wr(SPR_PAGE, 0);
	GD_wr(IOMODE, 0);
	GD_wr16(BG_COLOR, 0);
	GD_wr16(SAMPLE_L, 0);
	GD_wr16(SAMPLE_R, 0);
	GD_wr16(SCREENSHOT_Y, 0);
	GD_wr(MODULATOR, 64);
//	GD_wr(MODULATED, 0);
//	GD_wr(UNMODULATED, 0);

	GD__wstart(RAM_SPR);
	for (i=0; i<512; i++)
	{
		GD_xhide();
	}

	curr_row = 0;
	curr_col = 0;
	GD__end();
}





/*
 *  GD_fill      fill a region of GD memory with count bytes
 */
void  GD_fill(unsigned int  addr, unsigned char  v,  unsigned int  count)
{
	if (!inited)  return;

	GD__wstart(addr);
	while (count--)
	{
		xchg(v);
	}
	GD__end();
}




unsigned char  GD_rd(unsigned int  addr)
{
	unsigned char				r;

	if (!inited)  return  (unsigned char)-1;

	GD__start(addr);
	r = xchg(0);
	GD__end();
	return  r;
}



unsigned int  GD_rd16(unsigned int  addr)
{
	unsigned int				r;

	if (!inited)  return  (unsigned char)-1;

	GD__start(addr);
	r = xchg(0);					// read the low byte
	r = r | (xchg(0) << 8);			// read the high byte
	GD__end();
	return  r;
}




void  GD_wr(unsigned int  addr, unsigned char  v)
{
	if (!inited)  return;

	GD__wstart(addr);
	xchg(v);
	GD__end();
}



void  GD_wr16(unsigned int  addr, unsigned int  v)
{
	if (!inited)  return;

	GD__wstart(addr);
	xchg(v&0xff);
	xchg(v>>8);
	GD__end();
}



void  GD_copy(unsigned int  addr, unsigned char  *src, int  count)
{
	if (!inited)  return;

	GD__wstart(addr);
	while (count--)
	{
		xchg(*src);
		src++;
	}
	GD__end();
}



void  GD_microcode(unsigned char  *src, int  count)
{
	if (!inited)  return;

	GD_wr(J1_RESET, 1);
	GD_copy(J1_CODE, src, count);
	GD_wr(J1_RESET, 0);
}




void  GD_xhide(void)
{
	if (!inited)  return;

	xchg(400&0xff);
	xchg(400>>8);
	xchg(400&0xff);
	xchg(400>>8);
	spr++;
}




void  GD_waitvblank(void)
{
	if (!inited)  return;

	while (GD_rd(VBLANK) == 1)  ;
	while (GD_rd(VBLANK) == 0)  ;
}




void  GD_ascii(void)
{
	int						i;
	unsigned char			b;
	unsigned char			l;
	unsigned char   		h;
	
	if (!inited)  return;

	for (i=0; i<768; i++)
	{
		b = font8x8[i];
		h = stretch[b>>4];
		l = stretch[b&0x0f];
		GD_wr(0x1000 + (16 * ' ') + (2 * i), h);
		GD_wr(0x1000 + (16 * ' ') + (2 * i) + 1, l);
	}

	for (i=0x20; i<0x80; i++)
	{
		GD_setpal(4 * i + 0, TRANSPARENT);
		GD_setpal(4 * i + 3, RGB_WHITE);
	}
	GD_fill(RAM_PIC, ' ', MAX_TEXT_COLS*MAX_TEXT_ROWS);

	max_col = 50;							// limit displayed columns to a subset of GD text area
	max_row = 37;							// limit displayed rows to a subset of GD text area
}




void  GD_putstr(unsigned int  x, unsigned int  y, const char  *s)
{
	if (!inited)  return;

	GD__wstart((y<<6) + x);
	while (*s)
	{
		xchg(*s++);
	}
	GD__end();
}



void  GD_cls(void)
{
	int						n;

	if (!inited)  return;

	GD__wstart(0);
	for (n=0; n<(max_col * max_row); n++)
	{
		xchg(' ');
	}
	GD__end();
	curr_col = 0;
	curr_row = 0;
}



/*
 *  GD_outch      a simple version of outch that supports the GD adapter
 *
 *  This routine displays the character in argument c on the GD adapter.
 *  The character will be displayed in the text area at the row number
 *  in curr_row and the column number in curr_col.  These variables
 *  will then be updated to indicate the new current character location.
 *
 *  If the character to be displayed is a form-feed, the GD text screen
 *  will be cleared (written to spaces) and the character location set
 *  to 0, 0.
 *
 *  If the character to be displayed is a carriage-return, the current
 *  location will be set to the next row in the same column.
 *
 *  If the character to be displayed is a line-feed, the current location
 *  will be set to the first column in the same row.
 *
 *  If any of the above steps cause the current row to increment beyond
 *  the visible window, this routine will scroll the text up one line
 *  (using hardware scrolling), erase the next line of chars, and adjust
 *  the current location to the start of the newly erased line.
 */
void  GD_outch(char  c)
{
	unsigned int					ty;

	if (!inited)  return;

	erasecursor();
	if (c == FF)  GD_cls();
	else if (c == LF)
	{
		curr_row++;
	}
	else if (c == CR)
	{
		curr_col = 0;
	}
	else
	{
		ty = (yscroll / CHAR_HEIGHT_PIXELS) + curr_row;			// calc absolute row number of char
		ty = ty % MAX_TEXT_ROWS;								// keep row number in text area
		GD_wr((ty * MAX_TEXT_COLS) + curr_col, c);				// write char to selected row,col
		curr_col++;
		if (curr_col == max_col)
		{
			curr_col = 0;
			curr_row++;
		}
	}
	
	if (curr_row == max_row)
	{
		scrolltextup();											// adjust location of virtual window down one row
		ty = (yscroll / CHAR_HEIGHT_PIXELS) + curr_row;			// calc absolute row number of next open row
		ty = ty % MAX_TEXT_ROWS;								// keep row number in text area
		GD_fill(ty * MAX_TEXT_COLS, ' ', MAX_TEXT_COLS);		// erase next row
		curr_row--;												// keep current row in virtual window
	}
	showcursor();
}




void  GD_setpal(int  pal, unsigned int  rgb)
{
	if (inited)  GD_wr16(RAM_PAL + (pal<<1), rgb);
}




void  GD_sprite(int  spr, int  x, int  y, unsigned char  image,
					unsigned char  palette, unsigned char  rot, unsigned char  jk)
{
	if (!inited)  return;

	GD__wstart(RAM_SPR + (spr << 2));
	xchg(x&0xff);
	xchg((palette << 4) | (rot << 1) | ((x>>8) & 1));
	xchg(y&0xff);
	xchg((jk << 7) | (image << 1) | ((y>>8) & 1));
	GD__end();
}



void  GD_xsprite(int  ox, int  oy, signed char  x, signed char  y, unsigned char  image,
					unsigned char  palette, unsigned char  rot, unsigned char  jk)
{
	int					s;

	if (rot & MASK_ROTATE_X)  x = -16-x;
	if (rot & MASK_ROTATE_Y)  y = -16-y;
	if (rot & MASK_EXCHANGE_XY)
	{
		s = x;
		x = y;
		y = s;
	}

	ox += x;
	oy += y;
	xchg(ox&0xff);
	xchg((palette << 4) | (rot << 1) | ((ox>>8) & 1));
	xchg(oy&0xff);
	xchg((jk << 7) | (image << 1) | ((oy>>8) & 1));
	spr++;
}



void  GD_sprite2x2(int  spr, int  x, int  y, unsigned char  image,
						unsigned char  palette, unsigned char  rot, unsigned char  jk)
{
	if (!inited)  return;

	GD__wstart(0x3000 + (spr << 2));
	GD_xsprite(x, y, -16, -16, image + 0, palette, rot, jk);
	GD_xsprite(x, y,   0, -16, image + 1, palette, rot, jk);
	GD_xsprite(x, y, -16,   0, image + 2, palette, rot, jk);
	GD_xsprite(x, y,   0,   0, image + 3, palette, rot, jk);
	GD__end();
}



void  GD_voice(int  v, unsigned char  wave, unsigned int  freq,
						unsigned char  lamp, unsigned char  ramp)
{
	if (!inited)  return;

	GD__wstart(VOICES + (v << 2));
	xchg(freq&0xff);
	xchg((freq>>8) | (wave << 7));
	xchg(lamp);
	xchg(ramp);
	GD__end();
}



void  GD__start(unsigned int  addr)
{
	if (!inited)  return;

	select();
	xchg(addr>>8);
	xchg(addr&0xff);
}



void  GD__wstart(unsigned int  addr)
{
	GD__start(addr | 0x8000);
}



void GD__wstartspr(unsigned int sprnum)
{
	GD__start((0x8000 | RAM_SPR) + (sprnum << 2));
	spr = 0;
}



void  GD__end(void)
{
	if (inited)  deselect();
}



/*
 *  The following block provides tools for decompressing files common among the
 *  Gameduino community.  The original code is here: http://excamera.com/sphinx/article-compression.html
 */

unsigned char			*src;
unsigned char			mask;


void  GD_uncompress(unsigned int  dst, unsigned char  *src)
{
	unsigned char				b_off;
	unsigned char				b_len;
	unsigned char				minlen;
	unsigned int				items;
	int							offset;
	int							length;

	dc_begin(src);
	b_off = dc_getn(4);
	b_len = dc_getn(4);
	minlen = dc_getn(2);
	items = dc_getn(16);

	while (items--)
	{
		if (dc_get1() == 0)
		{
			GD_wr(dst++, dc_getn(8));
		}
		else
		{
			offset = -dc_getn(b_off) - 1;
			length = dc_getn(b_len) + minlen;
			while (length--)
			{
				GD_wr(dst, GD_rd(dst+offset));
				dst++;
			}
		}
	}
}



static void  dc_begin(unsigned char  *s)
{
	src = s;
	mask = 1;
}



static unsigned char  dc_get1(void)
{
	unsigned char			r;

	r = (*src & mask) != 0;
	mask <<= 1;
	if (!mask)
	{
		mask = 1;
		src++;
	}
	return  r;
}



static unsigned int  dc_getn(unsigned char  n)
{
	unsigned int				r;

	r = 0;
	while (n--)
	{
		r <<= 1;
		r |= dc_get1();
	}
	return  r;
}

	


static void  scrolltextup(void)
{
	yscroll = yscroll + CHAR_HEIGHT_PIXELS;
	yscroll = yscroll % MAX_PIXELS_Y;
	GD_waitvblank();
	GD_wr16(SCROLL_Y, yscroll);
}



static void  erasecursor(void)
{
	unsigned int				ty;

	ty = (yscroll / CHAR_HEIGHT_PIXELS) + curr_row;			// calc absolute row number of char
	ty = ty % MAX_TEXT_ROWS;								// keep row number in text area
	GD_wr((ty * MAX_TEXT_COLS) + curr_col, ' ');			// erase selected row,col
}



static void  showcursor(void)
{
	unsigned int				ty;

	ty = (yscroll / CHAR_HEIGHT_PIXELS) + curr_row;			// calc absolute row number of char
	ty = ty % MAX_TEXT_ROWS;								// keep row number in text area
	GD_wr((ty * MAX_TEXT_COLS) + curr_col, '_');			// put cursor at selected row,col
}




/*
 *  The following section of code acts as a temporary main() so I can compile and
 *  test portions of this code.  To compile this module as a library, comment out
 *  the following code and rebuild with a makefile designed to create a suitable
 *  library.
 *
 *  This code is set up to run on an Atmel Xplained-128A1 board using the internal
 *  RC clock at 32 MHz.
 */

#if  0										// use 1 to add the following code, 0 to comment it out

void  test_select(void);
unsigned char  test_xchg(unsigned char);
void  test_deselect(void);
void  outstr(char  *s);


#define  GD_CS				PORTC
#define  GD_CS_BIT			4
#define  GD_CS_MASK			(1<<GD_CS_BIT)

#define  GD_SPI_PORT		PORTC
#define  GD_SPI_SCK_BIT		7
#define  GD_SPI_MISO_BIT	6
#define  GD_SPI_MOSI_BIT	5
#define  GD_SPI_SS_BIT		4
#define  GD_SPI_SCK_MASK	(1<<GD_SPI_SCK_BIT)
#define  GD_SPI_MOSI_MASK	(1<<GD_SPI_MOSI_BIT)
#define  GD_SPI_SS_MASK		(1<<GD_SPI_SS_BIT)
#define  GD_SPI_MISO_MASK	(1<<GD_SPI_MISO_BIT)


#define  GD_SPI				SPIC

#define  GD_ENABLE			GD_CS.OUTCLR=GD_CS_MASK
#define  GD_DISABLE			GD_CS.OUTSET=GD_CS_MASK



int  main(void)
{
	int						n;
	char					buff[80];

/*
 *  32 MHz internal oscillator setup
 */
	OSC.CTRL |= OSC_RC32MEN_bm;
	while (!(OSC.STATUS & OSC_RC32MRDY_bm));
 	CCP = CCP_IOREG_gc;
	CLK.CTRL = CLK_SCLKSEL_RC32M_gc;
	OSC.CTRL &= (~OSC_RC2MEN_bm);								// disable RC2M
	PORTCFG.CLKEVOUT = (PORTCFG.CLKEVOUT & (~PORTCFG_CLKOUT_gm)) | PORTCFG_CLKOUT_OFF_gc;	// disable peripheral clock

/*
 *  Configure the SPI port for connection to the Gameduino.
 */
	GD_DISABLE;													// pull the chip-select line high
	GD_CS.DIRSET = GD_CS_MASK;									// make the chip-select line an output

	GD_SPI_PORT.OUTSET = (GD_SPI_SCK_MASK | GD_SPI_MOSI_MASK | GD_SPI_SS_MASK);	// force SPI output pins high
	GD_SPI_PORT.DIRSET = (GD_SPI_SCK_MASK | GD_SPI_MOSI_MASK | GD_SPI_SS_MASK);	// define SPI output pins
	GD_SPI_PORT.DIRCLR = GD_SPI_MISO_MASK;						// make sure the MISO line is an input

	GD_SPI.CTRL = (SPI_MODE_0_gc | SPI_PRESCALER_DIV64_gc |		// SPI uses phase=0, polarity=0, /64 prescaler, *2 SPI clock
				   SPI_MASTER_bm  | SPI_CLK2X_bm);				// master mode

	GD_SPI.CTRL = GD_SPI.CTRL | SPI_ENABLE_bm;					// enable the SPI


/*
 *  The SPI is set up.  Now we need to register the necessary callback functions
 *  with the Gameduino support module.
 */
	GD_register(&test_select, &test_xchg, &test_deselect);


/*
 *  All done, strut our stuff!
 */
 	GD_begin();													// set up the GD hardware
	GD_ascii();													// set up for using text
	GD_cls();
	for (n=0; n<100; n++)										// use the GD_outch function...
	{
		sprintf(buff, "%d Forcing screen to scroll...\n\r", n);
		outstr(buff);
	}


/*
 *  Use the direct-write functions in the GD library.
 */
	GD_putstr(0,   20,      "Hello, world!                    ");
	GD_putstr_P(0, 21, PSTR("Hello from flash!                "));
	GD_putstr_P(0, 22, PSTR("01234567890123456789012345678901234567890123456789012345678901234567890123456789"));
	GD_putstr_P(0, 23, PSTR("abcdefghijklmnopqrstuvwxyz       "));


/*
 *  Use some local functions that in turn use the GD_outch library function.
 */
	outstr("Row 0\n\r");
	outstr("Row 1\n\r");
	outstr("Row 2\n\r");
	outstr("Row 3\n\r");
	outstr("Row 4\n\r");
	outstr("Row 5: abcdefghijklmnopqrstuvwxyz\n\r");

	outstr("\n\r");
	outstr("\n\r> ");

	while(1)   ;

	return  0;													// better never get here!
}


/*
 *  test_select      callback function; selects GD adapter
 */
void  test_select(void)
{
	GD_ENABLE;
}


/*
 *  test_xchg      callback function; exchanges a byte with GD adapter
 */
unsigned char  test_xchg(unsigned char  val)
{
	GD_SPI.DATA = val;									// write byte to send
	while ((GD_SPI.STATUS & SPI_IF_bm) == 0)  ;			// loop until data reg is empty
	return  GD_SPI.DATA;								// return the result
}


/*
 *  test_deselect      callback function; deselects GD adapter
 */
void  test_deselect(void)
{
	GD_DISABLE;
}



/*
 *  outstr      quick-and-dirty string print function
 */
void  outstr(char  *s)
{
	while (*s)
	{
		GD_outch(*s);
		s++;
	}
}



#endif



