/*
 * target_linux.h				header file specific to the Linux target compiler
 */

#ifndef  TARGET_LINUX_H
#define  TARGET_LINUX_H

#include  <stdio.h>
#include  <stdint.h>
#include  <limits.h>

/*
 * Define the size (in bits) of various data elements for the Linux target.
 * These definitions apply to the size of stack elements and variables, among
 * other things.
 */
#define		DATA_SIZE			int64_t
#define		DATA_SIZE_MAX		INT64_MAX
#define		UDATA_SIZE			uint64_t
#define		UDATA_SIZE_MAX		UINT64_MAX

/*
 * Define target characteristics and resources of interest to the core program.
 * 
 * T_NUM_TIMERS     n					number of down-counting timers target can support
 * T_STREAM_SERIAL  0					stream number for the console (usually on a serial port)
 * T_STREAM_FILE    1					stream number for file I/O; define as 0 if file I/O not supported
 * T_SUPPORTS_FILES  TRUE|FALSE			supports or does not support file I/O
 */
#define		T_NUM_TIMERS		4
#define		T_STREAM_SERIAL		0
#define		T_STREAM_FILE		1
#define		T_SUPPORTS_FILES	TRUE

/*
 * Define indexes for each of the ports known to the target system.
 * 
 * WARNING: You must ensure that the indexes used in these defines match exactly with the names
 * and order of the ports table defined in the target.c file!  (This is not the proper
 * way to do this; I need to fix this so it isn't so error-prone.)
 */
#define  PORTS_PORTA		0
#define  PORTS_PORTB		1
#define  PORTS_PORTC		2
#define  PORTS_PORTM		3
#define  PORTS_PORTN		4
#define  PORTS_PORTO		5
#define  PORTS_PORTP		6
#define  PORTS_UNKNOWN		7			/* must end the table */

#define  NUM_PORTS  PORTS_UNKNOWN		/* do not count the UNKNOWN entry! */



extern const unsigned char				ports_tab[];



/*
 * Define the amount of RAM available for the user's program, assorted stacks, and all
 * variables.
 * 
 * Note that 23 variables are one DATA_SIZE large each, while three variables (X, Y, and Z)
 * are 400 DATA_SIZE large each.
 */
#define  T_PROGSPACE_SIZE			32000



void			t_ColdBoot(void);				// system prep following reset
void			t_WarmBoot(void);				// system prep following crash
void			t_Shutdown(void);				// clean system shutdown at end of program

void			t_OutChar(int  c);				// output a char to active stream
int				t_GetChar(void);				// waits (blocks) for char from active stream, returns char
int				t_ConsoleCharAvailable(void);	// returns TRUE if user has entered a char at console; does not lock
int				t_GetCharFromConsole(void);		// always waits (blocks) for char from console, ignoring InputStream

void			t_SetTimerRate(unsigned int  usecs);	// sets down-counting timer tic rate in usecs; 0 disables timer
int				t_AddTimer(DATA_SIZE  *t);		// adds a variable at address t to the timer list
int				t_DeleteTimer(DATA_SIZE  *t);	// removes a variable at address to from the timer list

int				t_SetOutputStream(int  s);		// assigns the output device for outchar() data from core
int				t_SetInputStream(int  s);		// assigns the input device for inchar() data to core

FILE			*t_OpenFile(char  *name, char  *mode);	// open a file for text I/O; closes any open files first
int				t_CloseFile(FILE  *fp);			// close file if open

DATA_SIZE		t_ReadPort(int  index);			// reads a port
void			t_WritePort(DATA_SIZE  value, int  index);		// writes value to indexed port

void			t_Test(void);					// triggers a target-specific test from TBL								




#endif				// TARGET_LINUX_H
