/*---------------------------------------------------------------------------
 * geode_init.c
 *
 * This file contains the driver initialization code.  It makes use of the 
 * Durango graphics software support package from National.
 *
 * It is ported from the geode driver developed by Annius Groenink.
 *
 * Copyright (c) 1999-2000 National Semicondictor.
 *---------------------------------------------------------------------------
 */

/* Xfree86 header files */

#include "xf86.h"
#include "xf86Priv.h"
#include "xf86_OSlib.h"
#include "xf86_HWlib.h"
#include "vga.h"
#include "vga256.h"
#define XCONFIG_FLAGS_ONLY
#include "xf86_Config.h"

/* National Semicondictor header files. */

#include "gfx_rtns.h"
#include "gfx_defs.h"
#include "gfx_regs.h"

#define DEBUGMSG(cond, printf_exp)\
	((cond)?(ErrorF printf_exp),1:0)

extern vgaHWCursorRec vgaHWCursor;

extern Bool GeodeCursorInit();
extern void GeodeRestoreCursor();
extern void GeodeWarpCursor();
extern void GeodeQueryBestSize();

/* MEMORY ORGANIZATION */

unsigned long geodeCursorOffset;
unsigned long geodeOffscreenOffset;
unsigned long geodeOffscreenSize;

/* VGA STRUCUTE FOR INIT/SAVE/RESTORE */

typedef struct {
	vgaHWRec std;				/* IBM VGA */
	gfx_vga_struct durangovga;	/* Extended VGA */
	int BPP;					/* bits per pixel */
} vgaGeodeRec, *vgaGeodePtr;

#define newstate ((vgaGeodePtr)vgaNewVideoState)

static unsigned Geode_ExtPorts[] = { 0x22,0x23,0x80, };  /* I/O ports used by gfx_gxm_config_read */
static int Num_Geode_ExtPorts =
       (sizeof(Geode_ExtPorts)/sizeof(Geode_ExtPorts[0]));

/* ROUTINE DECLARATIONS */

static Bool	GeodeProbe();
static char 	*GeodeIdent();
static void	GeodeEnterLeave();
static Bool	GeodeInit();
static int 	GeodeValidMode();
static void 	*GeodeSave();
static void	GeodeRestore();
static void	GeodeAdjust();
static void	GeodeFbInit();
static Bool	GeodePitchAdjust();

/* STILL NAMED geode UNTIL PROPER MAKEFILE TO CHANGE IS FOUND */

vgaVideoChipRec GEODE = {
	GeodeProbe,
	GeodeIdent,
	GeodeEnterLeave,
	GeodeInit,
	GeodeValidMode,
	GeodeSave,
	GeodeRestore,
	GeodeAdjust,
	vgaHWSaveScreen,
	(void (*)())NoopDDA,     /* GeodeGetMode not implemented */
	GeodeFbInit,
	(void (*)())NoopDDA,     /* GeodeSetRead not implemented */
	(void (*)())NoopDDA,     /* GeodeSetWrite not implemented */
	(void (*)())NoopDDA,     /* GeodeSetReadWrite not implemented */
	0x10000,                 /* ChipMapSize */
	0x10000,                 /* ChipSegmentSize */
	16,                      /* ChipSegmentShift */
	0xFFFF,                  /* ChipSegmentMask */
	0x00000, 0x10000,        /* ChipReadBottom/Top */
	0x00000, 0x10000,        /* ChipWriteBottom/Top */
	TRUE,                    /* ChipUse2Banks */
	VGA_NO_DIVIDE_VERT,      /* ChipInterlaceType */
	{0,},                    /* ChipOptionFlags */
	8,                       /* ChipRounding */
	TRUE,                    /* ChipUseLinearAddressing */
	0x40800000,              /* ChipLinearBase */
	0x001FFFFF,              /* ChipLinearSize */
	TRUE,                    /* ChipHas16bpp */
	FALSE,                   /* ChipHas24bpp */
	FALSE,                   /* ChipHas32bpp */
	NULL,                    /* ChipBuiltinModes */
	1,                       /* ChipClockMulFactor */
	1                        /* ChipCLockDivFactor */
};

/*---------------------------------------------------------------------------
 * GeodeIdent
 *
 * The server will call this function repeatedly, until a NULL is returned, 
 * when printing out the list of configured drivers.
 *---------------------------------------------------------------------------
 */

#define NUM_CHIPSETS 4
static char chipsets[NUM_CHIPSETS][8] = {
	"MediaGX", "SC1400A", "SC1200"
};

static char *GeodeIdent(n)
int n;
{
	if (n >= NUM_CHIPSETS) return(NULL);
	else return((char *)&chipsets[n][0]);
}

/*---------------------------------------------------------------------------
 * GeodeProbe
 *
 * This function detects the presence of the hardware and initializes 
 * the Durango register pointers.
 *---------------------------------------------------------------------------
 */
static Bool GeodeProbe()
{
	unsigned long cpu_version;
 
	/* SETUP IO PERMISSIONS */
	/* Works without this for FreeBSD, needed for Linux. */

    xf86ClearIOPortList(vga256InfoRec.scrnIndex);
    xf86AddIOPorts(vga256InfoRec.scrnIndex, Num_VGA_IOPorts, VGA_IOPorts);
    xf86AddIOPorts(vga256InfoRec.scrnIndex, Num_Geode_ExtPorts, Geode_ExtPorts);
	
	/* ENABLE IO PORTS */

    GeodeEnterLeave(ENTER);

	/* USE DURANGO ROUTINE TO DETECT THE CPU */

	cpu_version = gfx_detect_cpu();
	if (!cpu_version)
	{
		/* NO GEODE PROCESSOR DETECTED */

		if (xf86Verbose > 1)
		{
			DEBUGMSG(1, ("%s %s: No Geode Processor Detected.\n",
				XCONFIG_PROBED, vga256InfoRec.name));
		}
		GeodeEnterLeave(LEAVE);
		return(FALSE);
	}

	/* DISPLAY DETECTION MESSAGE */

	DEBUGMSG(1,("%s %s: Geode Processor Detected (Type = %d, Version = %d.%d).\n",
	       XCONFIG_PROBED, vga256InfoRec.name, cpu_version & 0xFF, 
		   (cpu_version >> 8) & 0xFF, (cpu_version >> 16) & 0xFF));

	/* GET AMOUNT OF VIDEO MEMORY */
	/* Determine value if user has not specified it in XF86Config. */

	if (!vga256InfoRec.videoRam)
		vga256InfoRec.videoRam = gfx_get_frame_buffer_size() >> 10;

	/* FILL IN SOME INFORMATION */

	vga256InfoRec.maxClock = 157500; /* 1280x1024 @ 85*/
	vga256InfoRec.chipset = GeodeIdent((cpu_version & 0xFF) - 1);
	vga256InfoRec.bankedMono = TRUE;
	OFLG_SET(CLOCK_OPTION_PROGRAMABLE, &vga256InfoRec.clockOptions);

	/* define option set valid for the driver */
	OFLG_SET(OPTION_SW_CURSOR, &GEODE.ChipOptionFlags);
	OFLG_SET(OPTION_HW_CURSOR, &GEODE.ChipOptionFlags);
	OFLG_SET(OPTION_NOACCEL, &GEODE.ChipOptionFlags);

	/* switch on hardware cursor option */
	if( !OFLG_ISSET(OPTION_SW_CURSOR, &vga256InfoRec.options) )
	    OFLG_SET(OPTION_HW_CURSOR, &vga256InfoRec.options);

	/* HOOK ROUTINE TO ADJUST PITCH (NEED 1K OR 2K FOR ACCELERATION) */

	vgaSetPitchAdjustHook(GeodePitchAdjust);

	/* USE LINEAR ADDRESSING */

	GEODE.ChipLinearBase = gfx_get_frame_buffer_base();
	GEODE.ChipLinearSize = (1024 * vga256InfoRec.videoRam);
	GEODE.ChipUseLinearAddressing = TRUE;

	/* SET DURANGO REGISTER POINTERS */
	/* The method of mapping from a physical address to a linear address */
	/* is operating system independent.  Set variables to linear address. */

	gfx_regptr = (unsigned char *)xf86MapVidMem(vga256InfoRec.scrnIndex,
		EXTENDED_REGION, (void*)gfx_get_cpu_register_base(), 0x20000);
	gfx_vidptr = (unsigned char *)xf86MapVidMem(vga256InfoRec.scrnIndex,
		EXTENDED_REGION, (void*)gfx_get_vid_register_base(), 0x1000);
	gfx_fbptr = (unsigned char *)xf86MapVidMem(vga256InfoRec.scrnIndex,
		EXTENDED_REGION, (void*)gfx_get_frame_buffer_base(), 
		gfx_get_frame_buffer_size());

	/* CHECK IF REGISTERS WERE MAPPED SUCCESSFULLY */

	if ((!gfx_regptr) || (!gfx_vidptr) || (!gfx_fbptr))
	{	
		DEBUGMSG(1, ("%s %s: Could not map hardware registers.\n",
			XCONFIG_PROBED, vga256InfoRec.name));
		GeodeEnterLeave(LEAVE);
		return(FALSE);
	}	
	return(TRUE);
}

/*---------------------------------------------------------------------------
 * GeodeCalculatePitchBytes
 * 
 * Local routine to return pitch of the screen, in bytes.
 *---------------------------------------------------------------------------
 */
static int GeodeCalculatePitchBytes()
{
	int lineDelta = vga256InfoRec.virtualX * (vgaBitsPerPixel >> 3);
	DEBUGMSG(0, ("GeodeCalculatePitchBytes = %d %d %d \n",
		vga256InfoRec.virtualX, vgaBitsPerPixel, lineDelta));

	if( !OFLG_ISSET(OPTION_NOACCEL, &vga256InfoRec.options )) {	
		/* FORCE TO 1K, 2K OR 4K (Pyramid only) IF ACCELERATION IS ENABLED */
		if( (gfx_cpu_version == GFX_CPU_PYRAMID) && ( lineDelta > 2048 )) { 
			lineDelta = 4096;
		} else if (lineDelta > 1024) { 
			lineDelta = 2048;
		} else {
			lineDelta = 1024; 
		}
	} else {
		/* FORCE TO MULTIPLE OF 16 IF NO ACCELERATION */
		lineDelta = (lineDelta + 15) & 0xFFF0; 
	}
	DEBUGMSG(0, ("GeodeCalculatePitchBytes = %d \n", lineDelta));
	return( lineDelta );
}

/*---------------------------------------------------------------------------
 * GeodeFbInit
 * 
 * From README file: "The FbInit() function is required for drivers with
 * accelerated graphics support.  It is used to replace default cfb.banked
 * functions with accelerated chip-specific versions.
 *
 * This routine is also used to allocate graphics memory. 
 *---------------------------------------------------------------------------
 */
static void GeodeFbInit()
{
	unsigned long total_size, fb_size;

	/* ALWAYS PUT THE CURSOR AT THE END OF VIDEO MEMORY */

	total_size = gfx_get_frame_buffer_size();
	geodeCursorOffset = total_size - 512;
	fb_size = vga256InfoRec.virtualY * GeodeCalculatePitchBytes();
	geodeOffscreenOffset = fb_size;
	geodeOffscreenSize = total_size - fb_size - 512;

	/* PRINT RESULTS OF OFFSCREEN MEMORY CONFIGURATION */

	if (geodeOffscreenSize <= 0)
	{
		DEBUGMSG(1, ("%s %s: No offscreen memory available.\n", 
			XCONFIG_PROBED, vga256InfoRec.name));
	}
	else
	{
		DEBUGMSG(1, ("%s %s: Offscreen memory from 0x%8.8X-0x%8.8X.\n",
			XCONFIG_PROBED, vga256InfoRec.name, 
			geodeOffscreenOffset, 
			geodeOffscreenOffset+geodeOffscreenSize-1));
	}

    /* CALL "GeodeAccelInit" ROUTINE TO SET UP XAA ACCELERATED FUNCTIONS */
	/* Local routine to hook desired rendering operations to accelerate. */

	if (!OFLG_ISSET(OPTION_NOACCEL, &vga256InfoRec.options))
		GeodeAccelInit();

   	/* INSTALL HARDWARE CURSOR ROUTINES */

	if (OFLG_ISSET(OPTION_HW_CURSOR, &vga256InfoRec.options))
	{
		vgaHWCursor.Initialized = TRUE;
		vgaHWCursor.Init = GeodeCursorInit;
		vgaHWCursor.Restore = GeodeRestoreCursor;
		vgaHWCursor.Warp = GeodeWarpCursor;
		vgaHWCursor.QueryBestSize = GeodeQueryBestSize;
		DEBUGMSG(1, ("%s %s: Using hardware cursor at offset 0x%8.8X.\n",
			XCONFIG_PROBED, vga256InfoRec.name, geodeCursorOffset));
	}
}

/*---------------------------------------------------------------------------
 * GeodePitchAdjust
 * 
 * This routine is hooked to allow the driver to enforce pitch requirements
 * that the hardware has.  For acceleration to be used, the pitch must be
 * 1K , 2K or 4K (Pyramid only) bytes.
 *---------------------------------------------------------------------------
 */
static int GeodePitchAdjust()
{
	int pitch, lineDelta = GeodeCalculatePitchBytes();

	/* PYRAMID HAS A MAX OF 4K, OTHER HARDWARE HAS A MAX OF 2K */

	if( gfx_cpu_version != GFX_CPU_PYRAMID) { 
		if( lineDelta > 2048 ) {
			FatalError("virtual display width requires a line size of more than 2048 bytes\n");
		}
	} else {
		if( lineDelta > 4096 ) {
			FatalError("virtual display width requires a line size of more than 4096 bytes\n");
		}
	}

	/* CONVERT FROM BYTES TO PIXELS */
	pitch = lineDelta / (vgaBitsPerPixel >> 3);

	/* DISPLAY MESSAGE IF PITCH WAS CHANGED */

	if (pitch != vga256InfoRec.virtualX)
		DEBUGMSG(1, ("%s %s: Display pitch set to %d %d pixels.\n",
		       XCONFIG_PROBED, vga256InfoRec.name, pitch,
			vga256InfoRec.virtualX));
	return pitch;
}

/*---------------------------------------------------------------------------
 * GeodeEnterLeave
 * 
 * This routine is called before and after accessing VGA registers.
 *---------------------------------------------------------------------------
 */
static void GeodeEnterLeave(enter)
Bool enter;
{
	if (enter) xf86EnableIOPorts(vga256InfoRec.scrnIndex);
	else xf86DisableIOPorts(vga256InfoRec.scrnIndex);
}

#define READ_REG_COUNT 0x78

static void print_gxm_reg(unsigned int count)
{
	int i, base = DC_UNLOCK;
	for(i=0; i<count; i+=4){
		DEBUGMSG(1, ("%x = %08x \n", (base + i), READ_REG32(base + i)));
	}
}

static void print_crtc(crtc)
vgaGeodePtr crtc;
{
	int i;
	DEBUGMSG(1, ("gfx_print_crtc: %d %d %d %d\n", 
		crtc->durangovga.xsize,
		crtc->durangovga.ysize,
		crtc->durangovga.hz,
		crtc->BPP));

	DEBUGMSG(1, ("%x\n", crtc->durangovga.miscOutput));

	for(i=0; i<25; i++){
		DEBUGMSG(1, ("%x ", crtc->durangovga.stdCRTCregs[i]));
	}
	DEBUGMSG(1, ("\n"));
	for(i=0; i<16; i++){
		DEBUGMSG(1, ("%x ", crtc->durangovga.extCRTCregs[i]));
	}
	DEBUGMSG(1, ("\n"));
}

static unsigned long GraphicsModeSaved = 0;
/*---------------------------------------------------------------------------
 * GeodeRestore
 * 
 * This routine restores (or sets for the first time) a display mode 
 * using the VGA registers.
 *---------------------------------------------------------------------------
 */
static void GeodeRestore(restore)
vgaGeodePtr restore;
{
	unsigned long value;
	int graphicsMode;

	graphicsMode = gfx_is_display_mode_supported( restore->durangovga.xsize,
		restore->durangovga.ysize,
		restore->BPP,
		restore->durangovga.hz);

#if 0
	print_crtc(restore);
#endif

	/* text mode and comming from graphics mode */
	if((graphicsMode == -1) || (GraphicsModeSaved == 0)){
		gfx_enable_softvga();

#if 0
		print_gxm_reg(READ_REG_COUNT); 
#endif

		if(graphicsMode == -1)
			DEBUGMSG(0, ("GeodeRestore text mode\n"));
		else
			DEBUGMSG(0, ("GeodeRestore text - graphics mode\n"));

		/* RESET START ADDRESS BACK TO ZERO */
		/* SoftVGA never knows it changed if panning the desktop. */
		gfx_set_display_offset(0);
		gfx_vga_mode_switch(1);

		/* CLEAR EXTENDED VGA REGISTERS */
		/* Need to do before restoring standard VGA state so that the text */
		/* mode really gets set properly when doing the font restore. */

		gfx_vga_clear_extended();

		/* RESTORE STANDARD VGA PORTION */

		vgaHWRestore((vgaHWPtr)restore);

		/* RESTORE EXTENDED VGA PORTION */
		/* Also restore standard CRTC values since we're overwriting values. */
		gfx_vga_restore(&restore->durangovga, 
			GFX_VGA_FLAG_MISC_OUTPUT | 
			GFX_VGA_FLAG_STD_CRTC | 
			GFX_VGA_FLAG_EXT_CRTC);

		/* SIGNAL THE END OF A MODE SWITCH */
		gfx_vga_mode_switch(0);

#ifdef DEBUG
		print_gxm_reg(READ_REG_COUNT);
#endif

		/* put no graphics mode */
		if(graphicsMode == -1)
			GraphicsModeSaved = 0;
	}

	/* TELL GRAPHICS UNIT THE BITS PER PIXEL */
	/* Use the BPP from display controller that was just set by SoftVGA. */
	/* Need to do this, instead of using vgaBitsPerPixel, so that it is */
	/* properly set to 8BPP when returning to text mode from 16 BPP. */

	gfx_set_bpp(gfx_get_display_bpp());

	/* graphics mode */
	if(graphicsMode != -1){
		DEBUGMSG(0, ("GeodeRestore graphics mode\n"));
		DEBUGMSG(0, ("gfx_set_display_mode %d\n", GraphicsModeSaved));
		if(GraphicsModeSaved == 0){
			/* Disable SoftVGA, as we talk to H/W register
			to set the mode directly
			*/
			gfx_disable_softvga();
			GraphicsModeSaved=1;
		}
		gfx_set_bpp(restore->BPP);
		gfx_set_display_mode( restore->durangovga.xsize,
				restore->durangovga.ysize,
				restore->BPP,
				restore->durangovga.hz);
		/* adjust the pitch */
		gfx_set_display_pitch(GeodeCalculatePitchBytes());
		gfx_set_cursor_position(geodeCursorOffset, 0, 0, 0, 0);
		gfx_set_display_offset(0);

#ifdef DEBUG
		print_gxm_reg(READ_REG_COUNT);
#endif

	}
}

/*---------------------------------------------------------------------------
 * GeodeSave
 * 
 * This routine saves the current VGA register settings.
 *---------------------------------------------------------------------------
 */
static void *GeodeSave(save)
vgaGeodePtr save;
{
	gfx_vga_struct vga;
	int bpp, is_graphics;
	
	/* SAVE EXTENDED VGA REGISTERS */
	/* Also save standard CRTC and misc output since we overwrite */
	/* those values. */

	gfx_vga_save(&vga, GFX_VGA_FLAG_MISC_OUTPUT | 
		GFX_VGA_FLAG_STD_CRTC | GFX_VGA_FLAG_EXT_CRTC);

	gfx_get_display_mode(&vga.xsize, &vga.ysize, &bpp, &vga.hz);

	DEBUGMSG(0, ("GeodeSave %d %d %d %d\n", vga.xsize, vga.ysize, bpp, vga.hz));
	/* SAVE STANDARD VGA PORTION */

	save = (vgaGeodePtr)vgaHWSave((vgaHWPtr)save, sizeof(vgaGeodeRec));
	save->durangovga = vga;
	save->BPP = bpp;

#ifdef DEBUG
	print_crtc(save);
#endif

	return ((void *) save);
}

static unsigned long fixedformat_frequency(frequency)
int frequency;
{
	int clock = frequency / 1000;
	unsigned long pll;
	pll = (clock << 16) | ((int)
		((double)(frequency - (clock * 1000)) * (double)65.536));
	DEBUGMSG(0, ("fixedformat_frequency: %d - %d %d %08x\n",
		frequency,
		clock,
		((int)((double)(frequency - (clock * 1000)) * (double)65.536)),
		pll));

	return(pll);
}

/*---------------------------------------------------------------------------
 * GeodeInit
 * 
 * This routine initializes the VGA structure for the specified mode. 
 * It only sets the appropriate register values in the structure.  It 
 * does not load the hardware (that is done in GeodeRestore).
 *---------------------------------------------------------------------------
 */
static Bool GeodeInit(mode)
DisplayModePtr mode;
{
	int frequency = vga256InfoRec.clock[mode->Clock];
	int clock = frequency / 1000;
	int i, hz = 60;
	unsigned long pll;

	/* INITIALIZE STANDARD VGA PORTION */

	if (!vgaHWInit(mode,sizeof(vgaGeodeRec)))
		return(FALSE);

	pll = fixedformat_frequency(frequency);

	DEBUGMSG(0, ("GeodeInit: %dx%d %d @%d\n", 
		mode->CrtcHDisplay,
		mode->CrtcVDisplay, 
		vgaBitsPerPixel, frequency));

	gfx_get_refreshrate_from_frequency(mode->CrtcHDisplay,
		mode->CrtcVDisplay, 
		vgaBitsPerPixel, &hz, pll);

	DEBUGMSG(0, ("%d\n", hz));

	/* INITIALIZE DURANGO VGA STRUCTURE FOR THE MODE */
	/* This does not set the hardware.  That is done later in "restore". */

	/* set the VGA reg's with default 60 Hz and later the 
	* correct freq set by durango 
	*/
	gfx_vga_mode(&newstate->durangovga, mode->CrtcHDisplay, 
		mode->CrtcVDisplay, vgaBitsPerPixel, 60);

	/* UPDATE THE PITCH VALUE */
	/* This is in case no acceleration is used and it is continuous. */

	gfx_vga_pitch(&newstate->durangovga, GeodeCalculatePitchBytes());

	newstate->durangovga.xsize = mode->CrtcHDisplay;
	newstate->durangovga.ysize = mode->CrtcVDisplay;
	newstate->durangovga.hz = hz;
	newstate->BPP = vgaBitsPerPixel;

	return(TRUE);
}

/*---------------------------------------------------------------------------
 * GeodeAdjust
 * 
 * This function is called when the server needs to change the start
 * address of the frame buffer.  This is done when panning across a 
 * virtual desktop that is larger than the displayed frame buffer.
 *---------------------------------------------------------------------------
 */
static void GeodeAdjust(x, y)
int x, y;
{	
	unsigned long offset;

	/* CALCULATE BYTE OFFSET FROM (X,Y) COORDINATE */

	offset = (unsigned long) y * (unsigned long) gfx_get_display_pitch();
	offset += x;
	if (vgaBitsPerPixel > 8) offset += x;

	/* CALL DURANGO ROUTINES TO SET THE START ADDRESS */
	/* Waiting until vertical blank prevents some visual anomalies */
	/* such as "jumping" and "tearing" when panning. */

	gfx_wait_vertical_blank();
	gfx_set_display_offset(offset);
}

/*---------------------------------------------------------------------------
 * GeodeValidMode()
 * 
 * From README file: "The ValidMode() function is required.  It is used to 
 * check for any chipset dependent reasons why a graphics mode might not be
 * valid.  It gets called by higher levels of the code after the Probe()
 * stage.  In many cases no special checking will be required and this 
 * function will simply return TRUE always."
 *---------------------------------------------------------------------------
 */
static int GeodeValidMode(mode, verbose, flag)
DisplayModePtr mode;
Bool verbose;
int flag;
{
	int ret, hz=60;
	/* curently the Pixel freq. in Linux do not comply with VESA, 
	* the code below is commented for this purpose. When we have a 
	* better king of approach then this can be either deleted/re-written
	*/
	unsigned long pll = fixedformat_frequency(mode->Clock);
	DEBUGMSG(0, ("GeodeValidMode: %s %d %d %d %d %X", 
		mode->name,
		mode->CrtcHDisplay,
		mode->CrtcVDisplay, 
		vgaBitsPerPixel,
		mode->Clock,
		pll));

#if 1
	/* check if the frequency provided can be supported by durango */
	hz = gfx_mode_frequency_supported(
		mode->CrtcHDisplay, mode->CrtcVDisplay, 
		vgaBitsPerPixel,
		pll);


	if(hz == -1)
	{
		DEBUGMSG(1, ("%s %s: %s: Mode Not Supported (%dx%dx%d @ %x)\n",
			XCONFIG_PROBED, vga256InfoRec.name,
			vga256InfoRec.chipset,
			mode->CrtcHDisplay, mode->CrtcVDisplay, 
			vgaBitsPerPixel,
			pll));
	    return MODE_BAD;
	} else {
		DEBUGMSG(0, ("%s %s: %s: Mode Supported (%dx%dx%d @ %x = %d)\n",
			XCONFIG_PROBED, vga256InfoRec.name,
			vga256InfoRec.chipset,
			mode->CrtcHDisplay, mode->CrtcVDisplay, 
			vgaBitsPerPixel,
			pll, hz));
	}
#endif
	
	/* can our h/w really support this */
	ret = gfx_is_display_mode_supported( 
		mode->CrtcHDisplay, mode->CrtcVDisplay, 
		vgaBitsPerPixel, hz);
	if(ret == -1)
	{
		DEBUGMSG(1, ("%s %s: %s: Mode Not Supported By Processor (%dx%dx%d @ %d)\n",
			XCONFIG_PROBED, vga256InfoRec.name,
			vga256InfoRec.chipset,
			mode->CrtcHDisplay, mode->CrtcVDisplay, 
			vgaBitsPerPixel, hz));
	    return MODE_BAD;
	}

	return MODE_OK;
}

/* END OF FILE */
