/*
    Helper.c

    This program computes tables for Tantrakr, writing assembler code for 
    the tables to stdout.

    The first table is a table of 128 adjustment factors for computing 
    sample steps for finetune and arpeggio.  We use the following formula:

        t(0) = 0FFFFh  (only a place holder - never used)
        t(i) = round( 2 ** (16 - i/96) )  for i = 1 to 127

    The results are 16-bit unsigned integers, to be written out in hex.  
    The name of the table is FINEADJS.

    The second table consists of 33 values of the sine function for 
    implementing vibrato and tremolo.  We use the following formula:

        v(i) = sin( 2*pi*i / 64 )

    The floating-point result is multiplied by 65536 and rounded to a long.  
    If less than 0, we set it to 0, and if greater than 65535, we set it to 
    65535.  The long is then cast to an unsigned.  We write out the results 
    in hex.  The name of the table is SINETBL.

    The third table consists of 33 values of a ramp for implementing 
    vibrato and tremolo.  We use the following formula:

        v(i) = (-1/32)*i + 1

    The floating-point result is multiplied by 65536 and rounded to a long.  
    If less than 0, we set it to 0, and if greater than 65535, we set it to 
    65535.  The long is then cast to an unsigned.  We write out the results 
    in hex.  The name of the table is RAMPTBL.

    The fourth table consists of 33 values of a square wave for 
    implementing vibrato and tremolo.  All values are 0FFFFh.  We write out 
    the results in hex.  The name of the table is SQUARETBL.

    The fifth table is a table of 120 period values for half-step 
    frequencies from C0 to B9.  We use the following formula:

                     /         7159090.5         \
        t(i) = round|  -------------------------  |   (i = 0 to 119)
                     \ 64 * 440 * 2**((i-33)/12) /

    The results are 16-bit unsigned integers, to be written out in hex.  
    The name of the table is HALFSTEPS.

    The sixth table is the multiplication table for the 8086 version.  It 
    actually consists of 65 tables, one for each volume 0-64.  Each of the 
    65 tables contains 256 values, one for each possible sample value.  We 
    use the following formula:

        t(i,v) = (i-128) * v       (i = 0 to 255, v = 0 to 64)

    The name of the table is MULTBL.
*/

#include <stdio.h>
#include <math.h>

    /* We can't compute the half-step table accurately enough for octaves 0 
       and 1, so we'll just fill in the known values. */
unsigned knownhalfs[24] = {
    1712, 1616, 1525, 1440, 1357, 1281,
    1209, 1141, 1077, 1017,  961,  907,
     856,  808,  762,  720,  678,  640,
     604,  570,  538,  508,  480,  453,
};

void main( void )
{
    int i;              /* table index */
    long templong;      /* for intermediate results */
    double pi;          /* pi value for sine table */
    int v;              /* volume for 8086 multiplication table */

    /* FINEADJS table */
    printf( "\t\t;\n" );
    printf( "\t\t; Table of adjustment factors for finetune and arpeggio.\n" );
    printf( "\t\t;\n" );
    printf( "FINEADJS\tDW\t0FFFFh" );
    for ( i=1; i<128; i++ )
    {
        if ( (i & 3) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( "," );
        printf( "%05Xh", (unsigned) (pow( 2.0, 16.0 - i/96.0 ) + 0.5) );
        if ( (i & 3) == 3 )
            printf( "\n" );
    }

    /* SINETBL table */
    printf( "\t\t;\n" );
    printf( "\t\t; Sine wave table for vibrato and tremolo.\n" );
    printf( "\t\t;\n" );
    printf( "SINETBL\t\tDW\t00000h" );
    pi = 4.0 * atan( 1.0 );
    for ( i=1; i<33; i++ )
    {
        if ( (i & 3) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( "," );
        templong = (long) (65536.0 * sin( (2.0*pi*i) / 64.0 ) + 0.5);
        templong = (templong < 0 ? 0 : (templong > 65535 ? 65535 : templong));
        printf( "%05Xh", (unsigned) templong );
        if ( (i & 3) == 3 )
            printf( "\n" );
    }
    printf( "\n" );

    /* RAMPTBL table */
    printf( "\t\t;\n" );
    printf( "\t\t; Ramp table for vibrato and tremolo.\n" );
    printf( "\t\t;\n" );
    printf( "RAMPTBL\t\tDW\t0FFFFh" );
    for ( i=1; i<33; i++ )
    {
        if ( (i & 3) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( "," );
        templong = (long) (65536.0 * (-i/32.0 + 1.0) + 0.5);
        templong = (templong < 0 ? 0 : (templong > 65535 ? 65535 : templong));
        printf( "%05Xh", (unsigned) templong );
        if ( (i & 3) == 3 )
            printf( "\n" );
    }
    printf( "\n" );

    /* SQUARETBL table */
    printf( "\t\t;\n" );
    printf( "\t\t; Square wave table for vibrato and tremolo.\n" );
    printf( "\t\t;\n" );
    printf( "SQUARETBL\tDW\t0FFFFh" );
    for ( i=1; i<33; i++ )
    {
        if ( (i & 3) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( "," );
        printf( "0FFFFh" );
        if ( (i & 3) == 3 )
            printf( "\n" );
    }
    printf( "\n" );

    /* HALFSTEPS table */
    printf( "\t\t;\n" );
    printf( "\t\t; Half-step periods.\n" );
    printf( "\t\t;\n" );
    printf( "HALFSTEPS\tDW\t%5d", 16*knownhalfs[0] );
    for ( i=1; i<24; i++ )
    {
        if ( (i % 6) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( ", " );
        printf( "%5d", 16*knownhalfs[i] );
        if ( (i % 6) == 5 )
            printf( "\n" );
    }
    for ( i=0; i<24; i++ )
    {
        if ( (i % 6) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( ", " );
        printf( "%5d", 4*knownhalfs[i] );
        if ( (i % 6) == 5 )
            printf( "\n" );
    }
    for ( i=0; i<24; i++ )
    {
        if ( (i % 6) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( ", " );
        printf( "%5d", knownhalfs[i] );
        if ( (i % 6) == 5 )
            printf( "\n" );
    }
    for ( i=24; i<120; i++ )
    {
        if ( (i % 6) == 0 )
            printf( "\t\tDW\t" );
        else
            printf( ", " );
        templong = (long) ((7159090.5 /
            (28160.0*pow( 2.0, (i-33) / 12.0 ))) + 0.5);
        templong = (templong < 0 ? 0 : (templong > 65535 ? 65535 : templong));
        printf( "%5d", (unsigned) templong );
        if ( (i % 6) == 5 )
            printf( "\n" );
    }
    printf( "\n" );

    /* MULTBL table */
    printf( "\t\t;\n" );
    printf( "\t\t; Multiplication table for 8086.\n" );
    printf( "\t\t;\n" );
    printf( "MULTBL\t\tLABEL\tWORD\n" );
    for ( v=0; v<65; v++ )
    {
        printf( "\t\t; volume %d\n", v );
        for ( i=0; i<256; i++ )
        {
            if ( (i & 7) == 0 )
                printf( "\t\tDW\t" );
            else
                printf( ", " );
            printf( "%5d", (i < 128 ? i : i-256)*v );
            if ( (i & 7) == 7 )
                printf( "\n" );
        }
    }
}
