#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

#
# This module provide subroutines to handle authentication.
#
# Any subroutines that require i/o to a file or db should be placed in 
# webcalng_auth_io.pm.
#

#
# This file contains perl subroutines used for session based authentication using cookies.
#

package webcalng_auth;
use strict;
use CGI::Cookie;
use Digest::MD5 qw(md5_hex);
use webcalng_auth_io;
use vars qw($user_cookie $session_cookie);

# This is the name of the cookie that stores a session key in their broswer.
$session_cookie = "webcalng_session";

# This is the name of the cookie that stores the username that they last logged in as.
$user_cookie = "webcalng_user";

#
# Find out who they are logged in as.
#
sub get_username {
	my (%cookies,$username,$time,$session_key);
	$username = "";
	$time     = time;
	%cookies  = fetch CGI::Cookie;
	if ($cookies{$session_cookie}) {
		$session_key = $cookies{$session_cookie}->value;
		my ($tmp_username,$tmp_time) = webcalng_auth_io::get_session_info($session_key);
		if ($tmp_username && $tmp_time) {	
			if (($time - $tmp_time) < $::webcalng_conf{'SESSION_LENGTH'}) {
				$username = $tmp_username;
			}
		}
	}
	return $username;
}

#
# Find out who they are logged in as if they are using a wap device.  Since many
# wap devices do not support cookies, we just pass the session key around as a
# form variable or as part of the URL query string.
#
sub wap_get_username {
	my ($username,$time,$session_key);
	$session_key = $::cleanq->{'sessionkey'} || 0;
	$::username  = "";
	$time        = time;
	if ({$session_key}) {
		my ($tmp_username,$tmp_time) = webcalng_auth_io::get_session_info($session_key);
		if ($tmp_username && $tmp_time) {	
			if (($time - $tmp_time) < $::webcalng_conf{'SESSION_LENGTH'}) {
				$username = $tmp_username;
			}
		}
	}
	return $username;
}

#
# Process the results of the login page.  Return 1 for a successful login,
# and 0 for a failed login attempt.
#
sub process_login {
	my ($valid_login,$username,$password,$crypted_password,$session_key,$cookie1,$cookie2,$http);
	$valid_login      = 0;
	$cookie1          = "";
	$cookie2          = "";
	$username         = $::cleanq->{'username'};
	$password         = $::cleanq->{'password'};
	$crypted_password = crypt_password($username,$password);
	if (webcalng_auth_io::validate_login_io($username,$crypted_password)) {
		$::username  = $username;
		$valid_login = 1;
		$session_key = create_session_key($username);
		#	
		# The first cookie holds the session key, and expires when the
		# user exits their browser.  The second cookie holds the username
		# so that we can remember it when they login again.
		#
		$cookie1 = new CGI::Cookie(-name    => $session_cookie,
		                           -value   => $session_key,
		                           -path    => $::webcalng_conf{'WEBCAL'},
		                           -secure  => '0',
		                          );
		$cookie2 = new CGI::Cookie(-name    => $user_cookie,
		                           -value   => $username,
		                           -path    => $::webcalng_conf{'WEBCAL'},
		                           -secure  => '0',
		                           -expires => '+10y',
		                          );
	}
	return ($valid_login,$cookie1,$cookie2);
}

#
# Process the results of the login attempt from a wap broswer.  Return 1 for a
# successful login, and 0 for a failed login attempt.  
#
sub wap_process_login {
	my ($valid_login,$username,$password,$crypted_password,$session_key,$http);
	$valid_login      = 0;
	$session_key      = "";
	$username         = $::cleanq->{'username'};
	$password         = $::cleanq->{'password'};
	$crypted_password = crypt_password($username,$password);
	if (webcalng_auth_io::validate_login_io($username,$crypted_password)) {
		$::username  = $username;
		$valid_login = 1;
		$session_key = create_session_key($username);
	}
	return ($valid_login,$session_key);
}

#
# Let the user logout.
#
sub logout {
	my (%cookies,$session_key,$cookie);
	%cookies  = fetch CGI::Cookie;
	if ($cookies{$session_cookie}) {
		$session_key = $cookies{$session_cookie}->value;
		webcalng_auth_io::remove_expired_sessions($session_key);
	}
	$cookie = "";
	if ($::username) {
		$session_key = "invalid";
		$cookie = new CGI::Cookie(-name    => $session_cookie,
		                          -value   => $session_key,
		                          -path    => $::webcalng_conf{'WEBCAL'},
		                          -secure  => '0',
		                          -expires => '+10y',
		                          );
	}
	return $cookie;
}

#
# Let the user logout from a wap browser.  All we have to do is remove their session.
#
sub wap_logout {
	my ($session_key);
	$session_key = $::cleanq->{'sessionkey'} || 0;	
	if ($session_key) {
		webcalng_auth_io::remove_expired_sessions($session_key);
	}
	return 1;
}

#
# Create a new session key.
#
sub create_session_key {
	my ($username) = (@_);
	my ($session_key,$tmp_key,$time,$random,$secret);
	webcalng_auth_io::remove_expired_sessions();
	$time        = time;
	$random      = rand 1000000 . $$;
	$secret      = md5_hex($random);
	$session_key = $username . $time . $secret;
	$session_key = md5_hex($session_key);
	$session_key = $secret . $session_key;
	$session_key = md5_hex($session_key);
	webcalng_auth_io::save_session_key($session_key,$username,$time);
	return $session_key;
}

#
# Encrypt a given password.
#
sub crypt_password {
	my ($username,$password) = (@_);
	my (@salts,$salt);
	$salt = webcalng_auth_io::get_salt($username);
	if (! $salt) {
		@salts    = ( 'A' .. 'Z', 'a' .. 'z', 0 .. 9, '.', '/' );
		$salt     = $salts[rand @salts] . $salts[rand @salts];
	}
	$password = crypt($password,$salt) or webcalng_subs::hard_error("Could not encrypt password: $!");
	return $password;
}

#
# The following subroutines for the most part just call a webcalng_auth_io
# subroutine to do their work.  We run everything through the webcalng_auth
# module from the main webcalng code, rather than calling the webcalng_auth_io
# directly.  This will make it easier for a developer to integrate webcalng
# into a custom authentication scheme for their site, as they will most likely
# only need to make changes to the webcalng_auth module to call their own code
# for authentication.
#

sub lookup_user {
	return webcalng_auth_io::lookup_user_io(@_);	
}

#
# One extra step in this subroutine - just need to encrypt the password
# before validating it with the io module.
#
sub validate_login {
	my ($user,$pass) = (@_);
	my ($valid,$crypted_pass);

	$crypted_pass = crypt_password($user,$pass);
	$valid        = webcalng_auth_io::validate_login_io($user,$crypted_pass);	

	return $valid;
}

#
# One extra step in this subroutine - just need to encrypt the password
# before validating it with the io module.
#
sub create_user {
	my ($user,$pass) = (@_);
	my ($crypted_pass);

	$crypted_pass = crypt_password($user,$pass);
	webcalng_auth_io::create_user_io($user,$crypted_pass);	

	return 1;
}

sub get_user_list {
	return webcalng_auth_io::get_user_list_io(@_);	
}

sub remove_user {
	return webcalng_auth_io::remove_user_io(@_);	
}

1;
